<?php

/**
 * kocuj-admin.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html
 * @copyright Copyright (c) 2013 Dominik Kocuj
 * @package kocuj_admin
 */

// security
if (basename($_SERVER['SCRIPT_FILENAME']) == basename(__FILE__)) {
	die ('Please do not load this page directly. Thanks!');
}

// check if class not exists
if (!class_exists('KocujAdminEnumLicensePlace1')) {
	/**
	 * Enumeration of places for display license when there is requirement for confirmation - version 1
	 *
	 * @access public
	 */
	final class KocujAdminEnumLicensePlace1 {
		/**
		 * Empty constructor for blocking of creating an instance of this class
		 *
		 * @access private
		 * @var void
		 */
		private function __construct() {}

		/**
		 * License will be not displayed anywhere
		 *
		 */
		const NONE = 0;

		/**
		 * License will be displayed everywhere
		 *
		 */
		const ALL = 1;

		/**
		 * License will be displayed on plugins page
		 *
		 */
		const PLUGINS = 2;

		/**
		 * License will be displayed on themes page
		 *
		 */
		const THEMES = 3;
	}
}

// check if class not exists
if (!class_exists('KocujAdminConfig1')) {
	/**
	 * Configuration class - version 1
	 *
	 * @access public
	 */
	class KocujAdminConfig1 {
		/**
		 * Directory for files for Kocuj Admin
		 *
		 * @access private
		 * @var string
		 */
		private $directory = 'kocuj-admin-1';

		/**
		 * Options
		 *
		 * @access private
		 * @var array
		 */
		private $options = array();

		/**
		 * Option types
		 *
		 * @access private
		 * @var array
		 */
		private $types = array();

		/**
		 * Last error
		 *
		 * @access private
		 * @var string
		 */
		private $lastError = '';

		/**
		 * Option name
		 *
		 * @access protected
		 * @var string
		 */
		protected $optionName = '';

		/**
		 * Constructor
		 *
		 * @access public
		 * @param string $languageDir Directory with language data
		 * @return void
		 */
		public function __construct($languageDir) {
			// add option types
			$this->types['string'] = array(array(&$this, 'typeString'), array(&$this, 'typeUpdateString'), array(&$this, 'typeGetString'));
			$this->types['checkbox'] = array(array(&$this, 'typeCheckbox'), array(&$this, 'typeUpdateCheckbox'), array(&$this, 'typeGetCheckbox'));
			$this->types['numeric'] = array(array(&$this, 'typeNumeric'), array(&$this, 'typeUpdateNumeric'), array(&$this, 'typeGetNumeric'));
			$this->types['post'] = array(array(&$this, 'typePost'), array(&$this, 'typeUpdatePost'), array(&$this, 'typeGetPost'));
			$this->types['page'] = array(array(&$this, 'typePage'), array(&$this, 'typeUpdatePage'), array(&$this, 'typeGetPage'));
			$this->types['date'] = array(array(&$this, 'typeDate'), array(&$this, 'typeUpdateDate'), array(&$this, 'typeGetDate'));
			$this->types['time'] = array(array(&$this, 'typeTime'), array(&$this, 'typeUpdateTime'), array(&$this, 'typeGetTime'));
			$this->types['datetime'] = array(array(&$this, 'typeDatetime'), array(&$this, 'typeUpdateDatetime'), array(&$this, 'typeGetDatetime'));
		}

		/**
		 * Add option
		 *
		 * @access public
		 * @param string $id Option id
		 * @param string $defaultValue Default option value
		 * @param string $type Option type
		 * @param bool $isArray Option is array of values (true) or not (false)
		 * @param string $label Option label
		 * @return void
		 */
		public function addOption($id, $defaultValue, $type, $isArray, $label) {
			// add option
			$this->options[$id] = array('set_'.$id, $defaultValue, $type, $isArray, $label);
		}

		/**
		 * Add option type
		 *
		 * @access public
		 * @param string $type Option type
		 * @param string $function Callback method name - default: null
		 * @param string $updateFunction Callback method name for update option - default: null
		 * @param string $getFunction Callback method name for get option - default: null
		 * @return void
		 */
		public function addType($type, $function = null, $updateFunction = null, $getFunction = null) {
			// prepare methods
			if (!empty($function)) {
				$func1 = array(&$this, $function);
			} else {
				$func1 = null;
			}
			if (!empty($updateFunction)) {
				$func2 = array(&$this, $updateFunction);
			} else {
				$func2 = null;
			}
			if (!empty($getFunction)) {
				$func3 = array(&$this, $getFunction);
			} else {
				$func3 = null;
			}
			// add option type
			$this->types[$type] = array($func1, $func2, $func3);
		}

		/**
		 * Check option value
		 *
		 * @access private
		 * @param string $name Option name
		 * @param array $option Option array
		 * @param int|string|bool $value Value to check
		 * @return bool Status - true or false
		 */
		private function checkOption($name, $option, $value) {
			// initialize
			$this->lastError = '';
			$ok = false;
			// check option value
			if (isset($option[2])) {
				$error = '';
				if (!empty($this->types[$option[2]][0])) {
					$ok = call_user_func($this->types[$option[2]][0], $name, $option, $value, &$error);
					if ((empty($ok)) && (!empty($error))) {
						$this->lastError .= $error.'<br />';
					}
				}
			}
			// exit
			return $ok;
		}

		/**
		 * Check option value - string
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param int|string|bool $value Value to check
		 * @param string &$error Returned error text
		 * @return bool Status - true or false
		 */
		public function typeString($name, $option, $value, &$error) {
			// exit
			return true;
		}

		/**
		 * Update option value - string
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param bool $inArray This option is in array of options (true) or not (false)
		 * @param bool &$customOptionCheck Return setting which will set that method has custom option checking (true) or allow to check by checkOption method (false)
		 * @param int|string|bool &$value Returned value
		 * @param string &$error Returned error text
		 * @return bool Status - true or false
		 */
		public function typeUpdateString($name, $option, $inArray, &$customOptionCheck, &$value, &$error) {
			// exit
			return true;
		}

		/**
		 * Get option value - string
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param bool $inArray This option is in array of options (true) or not (false)
		 * @param int|string|bool $value Value
		 * @return int|string|bool Modified value
		 */
		public function typeGetString($name, $option, $inArray, $value) {
			// exit
			return $value;
		}

		/**
		 * Check option value - checkbox
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param int|string|bool $value Value to check
		 * @param string &$error Returned error text
		 * @return bool Status - true or false
		 */
		public function typeCheckbox($name, $option, $value, &$error) {
			// check value
			$ok = false;
			if (empty($value)) {
				$value = 0;
			}
			if ((is_numeric($value)) && (($value == 0) || ($value == 1))) {
				$ok = true;
			}
			if (empty($ok)) {
				$error = '"'.$option[4].'" '.__('should be on or off', 'kocujadmin1');
			}
			// exit
			return $ok;
		}

		/**
		 * Update option value - checkbox
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param bool $inArray This option is in array of options (true) or not (false)
		 * @param bool &$customOptionCheck Return setting which will set that method has custom option checking (true) or allow to check by checkOption method (false)
		 * @param int|string|bool &$value Returned value
		 * @param string &$error Returned error text
		 * @return bool Status - true or false
		 */
		public function typeUpdateCheckbox($name, $option, $inArray, &$customOptionCheck, &$value, &$error) {
			// update checkbox
			if (empty($value)) {
				$value = 0;
			}
			// exit
			return true;
		}

		/**
		 * Get option value - checkbox
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param bool $inArray This option is in array of options (true) or not (false)
		 * @param int|string|bool $value Value
		 * @return int|string|bool Modified value
		 */
		public function typeGetCheckbox($name, $option, $inArray, $value) {
			// exit
			return $value;
		}

		/**
		 * Check option value - numeric
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param int|string|bool $value Value to check
		 * @param string &$error Returned error text
		 * @return bool Status - true or false
		 */
		public function typeNumeric($name, $option, $value, &$error) {
			// check value
			$ok = false;
			if (empty($value)) {
				$ok = true;
			}
			if (is_numeric($value)) {
				$ok = true;
			}
			if (empty($ok)) {
				$error = '"'.$option[4].'" '.__('should be numeric', 'kocujadmin1');
			}
			// exit
			return $ok;
		}

		/**
		 * Update option value - numeric
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param bool $inArray This option is in array of options (true) or not (false)
		 * @param bool &$customOptionCheck Return setting which will set that method has custom option checking (true) or allow to check by checkOption method (false)
		 * @param int|string|bool &$value Returned value
		 * @param string &$error Returned error text
		 * @return bool Status - true or false
		 */
		public function typeUpdateNumeric($name, $option, $inArray, &$customOptionCheck, &$value, &$error) {
			// exit
			return true;
		}

		/**
		 * Get option value - numeric
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param bool $inArray This option is in array of options (true) or not (false)
		 * @param int|string|bool $value Value
		 * @return int|string|bool Modified value
		 */
		public function typeGetNumeric($name, $option, $inArray, $value) {
			// exit
			return $value;
		}

		/**
		 * Check option value - post
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param int|string|bool $value Value to check
		 * @param string &$error Returned error text
		 * @return bool Status - true or false
		 */
		public function typePost($name, $option, $value, &$error) {
			// check value
			$ok = false;
			if ((is_numeric($value)) && (get_permalink($value))) {
				$ok = true;
			}
			if (empty($value)) {
				$ok = true;
			}
			if (empty($ok)) {
				$error = '"'.$option[4].'" '.__('must be numeric and should be already existing post', 'kocujadmin1');
			}
			// exit
			return $ok;
		}

		/**
		 * Update option value - post
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param bool $inArray This option is in array of options (true) or not (false)
		 * @param bool &$customOptionCheck Return setting which will set that method has custom option checking (true) or allow to check by checkOption method (false)
		 * @param int|string|bool &$value Returned value
		 * @param string &$error Returned error text
		 * @return bool Status - true or false
		 */
		public function typeUpdatePost($name, $option, $inArray, &$customOptionCheck, &$value, &$error) {
			// exit
			return true;
		}

		/**
		 * Get option value - post
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param bool $inArray This option is in array of options (true) or not (false)
		 * @param int|string|bool $value Value
		 * @return int|string|bool Modified value
		 */
		public function typeGetPost($name, $option, $inArray, $value) {
			// exit
			return $value;
		}

		/**
		 * Check option value - page
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param int|string|bool $value Value to check
		 * @param string &$error Returned error text
		 * @return bool Status - true or false
		 */
		public function typePage($name, $option, $value, &$error) {
			// check value
			$ok = false;
			if ((is_numeric($value)) && (get_permalink($value))) {
				$ok = true;
			}
			if (empty($value)) {
				$ok = true;
			}
			if (($option[2] == 'page') && (!empty($ok)) && (!is_page($value))) {
				$ok = false;
			}
			if (empty($ok)) {
				$error = '"'.$option[4].'" '.__('must be numeric and should be already existing page', 'kocujadmin1');
			}
			// exit
			return $ok;
		}

		/**
		 * Update option value - page
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param bool $inArray This option is in array of options (true) or not (false)
		 * @param bool &$customOptionCheck Return setting which will set that method has custom option checking (true) or allow to check by checkOption method (false)
		 * @param int|string|bool &$value Returned value
		 * @param string &$error Returned error text
		 * @return bool Status - true or false
		 */
		public function typeUpdatePage($name, $option, $inArray, &$customOptionCheck, &$value, &$error) {
			// exit
			return true;
		}

		/**
		 * Get option value - page
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param bool $inArray This option is in array of options (true) or not (false)
		 * @param int|string|bool $value Value
		 * @return int|string|bool Modified value
		 */
		public function typeGetPage($name, $option, $inArray, $value) {
			// exit
			return $value;
		}

		/**
		 * Check option value - date
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param int|string|bool $value Value to check
		 * @param string &$error Returned error text
		 * @return bool Status - true or false
		 */
		public function typeDate($name, $option, $value, &$error) {
			// check value
			$ok = preg_match('/^[0-9][0-9][0-9][0-9]\-[0-9][0-9]\-[0-9][0-9]$/s', trim($value));
			if (empty($value)) {
				$ok = true;
			}
			if (empty($ok)) {
				$error = '"'.$option[4].'" '.__('must be in format "yy-mm-dd"', 'kocujadmin1');
			}
			// exit
			return $ok;
		}

		/**
		 * Update option value - date
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param bool $inArray This option is in array of options (true) or not (false)
		 * @param bool &$customOptionCheck Return setting which will set that method has custom option checking (true) or allow to check by checkOption method (false)
		 * @param int|string|bool &$value Returned value
		 * @param string &$error Returned error text
		 * @return bool Status - true or false
		 */
		public function typeUpdateDate($name, $option, $inArray, &$customOptionCheck, &$value, &$error) {
			// trim value
			$value = trim($value);
			// exit
			return true;
		}

		/**
		 * Get option value - date
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param bool $inArray This option is in array of options (true) or not (false)
		 * @param int|string|bool $value Value
		 * @return int|string|bool Modified value
		 */
		public function typeGetDate($name, $option, $inArray, $value) {
			// exit
			return $value;
		}

		/**
		 * Check option value - time
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param int|string|bool $value Value to check
		 * @param string &$error Returned error text
		 * @return bool Status - true or false
		 */
		public function typeTime($name, $option, $value, &$error) {
			// check value
			$ok = preg_match('/^[0-9][0-9]:[0-9][0-9]$/s', trim($value));
			if (empty($value)) {
				$ok = true;
			}
			if (empty($ok)) {
				$error = '"'.$option[4].'" '.__('must be in format "hh:ii"', 'kocujadmin1');
			}
			// exit
			return $ok;
		}

		/**
		 * Update option value - time
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param bool $inArray This option is in array of options (true) or not (false)
		 * @param bool &$customOptionCheck Return setting which will set that method has custom option checking (true) or allow to check by checkOption method (false)
		 * @param int|string|bool &$value Returned value
		 * @param string &$error Returned error text
		 * @return bool Status - true or false
		 */
		public function typeUpdateTime($name, $option, $inArray, &$customOptionCheck, &$value, &$error) {
			// trim value
			$value = trim($value);
			// exit
			return true;
		}

		/**
		 * Get option value - time
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param bool $inArray This option is in array of options (true) or not (false)
		 * @param int|string|bool $value Value
		 * @return int|string|bool Modified value
		 */
		public function typeGetTime($name, $option, $inArray, $value) {
			// exit
			return $value;
		}

		/**
		 * Check option value - datetime
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param int|string|bool $value Value to check
		 * @param string &$error Returned error text
		 * @return bool Status - true or false
		 */
		public function typeDatetime($name, $option, $value, &$error) {
			// check value
			$ok = preg_match('/^[0-9][0-9][0-9][0-9]\-[0-9][0-9]\-[0-9][0-9] [0-9][0-9]:[0-9][0-9]$/s', trim($value));
			if (empty($value)) {
				$ok = true;
			}
			if (empty($ok)) {
				$error = '"'.$option[4].'" '.__('must be in format "yy-mm-dd hh:ii"', 'kocujadmin1');
			}
			// exit
			return $ok;
		}

		/**
		 * Update option value - datetime
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param bool $inArray This option is in array of options (true) or not (false)
		 * @param bool &$customOptionCheck Return setting which will set that method has custom option checking (true) or allow to check by checkOption method (false)
		 * @param int|string|bool &$value Returned value
		 * @param string &$error Returned error text
		 * @return bool Status - true or false
		 */
		public function typeUpdateDatetime($name, $option, $inArray, &$customOptionCheck, &$value, &$error) {
			// trim value
			$value = trim($value);
			// exit
			return true;
		}

		/**
		 * Get option value - datetime
		 *
		 * @access public
		 * @param string $name Option name
		 * @param array $option Option settings
		 * @param bool $inArray This option is in array of options (true) or not (false)
		 * @param int|string|bool $value Value
		 * @return int|string|bool Modified value
		 */
		public function typeGetDatetime($name, $option, $inArray, $value) {
			// exit
			return $value;
		}

		/**
		 * Update options
		 *
		 * @access public
		 * @return string Output information
		 */
		public function updateOptions() {
			// initialize output
			$output = '';
			// update options
			$array = get_option($this->optionName);
			$array = maybe_unserialize($array);
			if (empty($array)) {
				$array = array();
			}
			if (count($this->options) > 0) {
				foreach ($this->options as $key => $val) {
					$value = $_REQUEST[$val[0]];
					$ok = true;
					$customOptionCheck = false;
					if (!empty($val[3])) {
						$lastError = '';
						foreach ($value as $key2 => $row) {
							if (empty($row)) {
								unset($value[$key2]);
							} else {
								$customOptionCheck = false;
								$error = '';
								if (!empty($this->types[$val[2]][1])) {
									$ok2 = call_user_func($this->types[$val[2]][1], $key, $val, true, &$customOptionCheck, &$row, &$error);
								} else {
									$ok2 = true;
								}
								$value[$key2] = $row;
								if ((empty($ok)) && (!empty($error))) {
									$this->lastError = $error;
								}
								if (empty($customOptionCheck)) {
									$ok2 = $this->checkOption($key, $val, $row);
								}
								if (empty($ok2)) {
									if (empty($lastError)) {
										$lastError = $this->lastError;
									}
									$ok = false;
								}
							}
						}
						$array2 = array();
						foreach ($value as $key2 => $row) {
							$array2[] = $row;
						}
						$value = $array2;
						$this->lastError = $lastError;
					} else {
						$customOptionCheck = false;
						$error = '';
						if (!empty($this->types[$val[2]][1])) {
							$ok = call_user_func($this->types[$val[2]][1], $key, $val, false, &$customOptionCheck, &$value, &$error);
						} else {
							$ok = true;
						}
						if ((empty($ok)) && (!empty($error))) {
							$this->lastError = $error;
						}
					}
					if ((empty($val[3])) && (empty($customOptionCheck))) {
						$ok = $this->checkOption($key, $val, $value);
					}
					if (!empty($ok)) {
						$array[$key] = $value;
					} else {
						$output .= $this->lastError;
					}
				}
			}
			update_option($this->optionName, maybe_serialize($array));
			// exit
			return $output;
		}

		/**
		 * Delete options and restore to default values
		 *
		 * @access public
		 * @return void
		 */
		public function deleteOptions() {
			// update options
			$array = get_option($this->optionName);
			$array = maybe_unserialize($array);
			if (empty($array)) {
				$array = array();
			}
			foreach ($this->options as $key => $val) {
				$array[$key] = '';
				if (!empty($val[3])) {
					$array[$key] = array();
				} else {
					$array[$key] = $val[1];
				}
			}
			update_option($this->optionName, maybe_serialize($array));
		}

		/**
		 * Get option value
		 *
		 * @access public
		 * @param string $name Option name
		 * @return int|string|bool|array Option value
		 */
		public function getOption($name) {
			// initialize
			$value = '';
			// get option value
			$array = get_option($this->optionName);
			if ((!is_array($array)) && ($array !== false)) {
				$array = maybe_unserialize($array);
				if (!is_array($array)) {
					$array = maybe_unserialize($array);
				}
			}
			if (empty($array)) {
				$array = array();
			}
			// check if option is empty
			$value = false;
			if (!isset($array[$name])) {
				if (isset($this->options[$name][1])) {
					if (!empty($this->options[$name][3])) {
						$value = false;
					} else {
						$value = $this->options[$name][1];
					}
				}
			} else {
				$value = $array[$name];
			}
			if (!empty($this->options[$name][3])) {
				if (!empty($value)) {
					foreach ($value as $onekey => $onevalue) {
						$ok = $this->checkOption($name, $this->options[$name], $onevalue);
						if (empty($ok)) {
							$value = false;
							break;
						} else {
							if (!empty($this->types[$this->options[$name][2]][2])) {
								$value[$onekey] = call_user_func($this->types[$this->options[$name][2]][2], $name, $this->options[$name], true, $onevalue);
							}
						}
					}
				} else {
					$value = false;
				}
			} else {
				$ok = $this->checkOption($name, $this->options[$name], $value);
				if (empty($ok)) {
					$value = $this->options[$name][1];
					if (!empty($this->types[$this->options[$name][2]][2])) {
						$value = call_user_func($this->types[$this->options[$name][2]][2], $name, $this->options[$name], false, $value);
					}
				}
			}
			// exit
			return $value;
		}

		/**
		 * Get option label
		 *
		 * @access public
		 * @string $name Option name
		 * @return string Option label
		 */
		public function getOptionLabel($name) {
			// get option label
			if (isset($this->options[$name][4])) {
				$value = $this->options[$name][4];
			} else {
				$value = '';
			}
			// exit
			return $value;
		}

		/**
		 * Check if option is array
		 *
		 * @access public
		 * @string $name Option name
		 * @return bool Option is array (true) or not (false)
		 */
		public function checkOptionArray($name) {
			// get option label
			if (isset($this->options[$name][3])) {
				$value = $this->options[$name][3];
			} else {
				$value = false;
			}
			// exit
			return $value;
		}

		/**
		 * Get option type
		 *
		 * @access public
		 * @param string $name Option name
		 * @return string Option type
		 */
		public function getOptionType($name) {
			// get option type
			if (isset($this->options[$name][2])) {
				$value = $this->options[$name][2];
			} else {
				$value = '';
			}
			// exit
			return $value;
		}

		/**
		 * Get option default value
		 *
		 * @access public
		 * @param string $name Option name
		 * @return string|int Option default value
		 */
		public function getOptionDefaultValue($name) {
			// get option type
			if (isset($this->options[$name][1])) {
				$value = $this->options[$name][1];
			} else {
				$value = '';
			}
			// exit
			return $value;
		}

		/**
		 * Get all options value
		 *
		 * @access public
		 * @return array Options value
		 */
		public function getAllOptions() {
			// initialize
			$output = array();
			// get all options
			foreach ($this->options as $key => $option) {
				$output[$key] = $this->getOption($key);
			}
			// exit
			return $output;
		}

		/**
		 * Uninstall options
		 *
		 * @access public
		 * @return void
		 */
		public function uninstallOptions() {
			// uninstall options
			delete_option($this->optionName);
		}
	}
}

// check if class not exists
if (!class_exists('KocujAdminAdmin1')) {
	/**
	 * Admin class - version 1
	 *
	 * @access public
	 */
	class KocujAdminAdmin1 {
		/**
		 * Directory for files for Kocuj Admin
		 *
		 * @access private
		 * @var string
		 */
		private $directory = 'kocuj-admin-1';

		/**
		 * Minimal full supported WordPress version
		 *
		 * @access protected
		 * @var string
		 */
		protected $minimalFullSupportedVersion = '';

		/**
		 * Random value used for unique simulated methods names
		 *
		 * @access private
		 * @var int
		 */
		private $random = 0;

		/**
		 * KocujAdminConfig1 class object
		 *
		 * @access private
		 * @var object
		 */
		private $configClass = null;

		/**
		 * Controllers list
		 *
		 * @access private
		 * @var array
		 */
		private $controller = array();

		/**
		 * Input helpers list
		 *
		 * @access private
		 * @var array
		 */
		private $inputHelper = array();

		/**
		 * Block helpers id list
		 *
		 * @access private
		 * @var array
		 */
		private $blockHelperId = array();

		/**
		 * Settings menus list
		 *
		 * @access private
		 * @var array
		 */
		private $settingsMenu = array();

		/**
		 * Settings menus containers list
		 *
		 * @access private
		 * @var array
		 */
		private $settingsMenuContainer = array();

		/**
		 * Theme menus list
		 *
		 * @access private
		 * @var array
		 */
		private $themeMenu = array();

		/**
		 * Theme menus containers list
		 *
		 * @access private
		 * @var array
		 */
		private $themeMenuContainer = array();

		/**
		 * Meta boxes containers list
		 *
		 * @access private
		 * @var array
		 */
		private $metaBox = array();

		/**
		 * Title for settings page in administration panel
		 *
		 * @access protected
		 * @var string
		 */
		protected $title = '';

		/**
		 * Full title
		 *
		 * @access protected
		 * @var string
		 */
		protected $fullTitle = '';

		/**
		 * Page parameters names list
		 *
		 * @access private
		 * @var array
		 */
		private $pageNames = array();

		/**
		 * Tabs containers list
		 *
		 * @access private
		 * @var array
		 */
		private $tabsContainers = array();

		/**
		 * Tabs list
		 *
		 * @access private
		 * @var array
		 */
		private $tabs = array();

		/**
		 * Tabs container started (true) or not (false)
		 *
		 * @access private
		 * @var bool
		 */
		private $tabsContainerStarted = false;

		/**
		 * Tab started (true) or not (false)
		 *
		 * @access private
		 * @var bool
		 */
		private $tabStarted = false;

		/**
		 * Table started (true) or not (false)
		 *
		 * @access private
		 * @var bool
		 */
		private $tableStarted = false;

		/**
		 * Internal name
		 *
		 * @access protected
		 * @var string
		 */
		protected $internalName = '';

		/**
		 * Form name for settings page in administration panel
		 *
		 * @access protected
		 * @var string
		 */
		protected $formName = '';

		/**
		 * Nonce action
		 *
		 * @access protected
		 * @var string
		 */
		protected $nonceAction = '';

		/**
		 * License option name
		 *
		 * @access protected
		 * @var string
		 */
		protected $licenseOption = '';

		/**
		 * License URL
		 *
		 * @access protected
		 * @var string
		 */
		protected $licenseURL = '';

		/**
		 * License directory
		 *
		 * @access protected
		 * @var string
		 */
		protected $licenseDir = '';

		/**
		 * License filename
		 *
		 * @access private
		 * @var string
		 */
		private $licenseFilename = '';

		/**
		 * License name
		 *
		 * @access protected
		 * @var string
		 */
		protected $licenseName = '';

		/**
		 * Random value used for license
		 *
		 * @access private
		 * @var int
		 */
		private $licenseRandom = 0;

		/**
		 * Force license confirmation on some pages or anywhere; it should have values based on KocujAdminEnumLicensePlace1 class
		 *
		 * @access protected
		 * @var int
		 */
		protected $licenseForce = KocujAdminEnumLicensePlace1::NONE;

		/**
		 * Thanks option name
		 *
		 * @access protected
		 * @var string
		 */
		protected $thanksOption = '';

		/**
		 * Thanks websites
		 *
		 * @access protected
		 * @var array
		 */
		protected $thanksWebsites = array();

		/**
		 * Random value used for thanks script
		 *
		 * @access private
		 * @var int
		 */
		private $thanksRandom = 0;

		/**
		 * CSS URL
		 *
		 * @access protected
		 * @var string
		 */
		protected $cssURL = '';

		/**
		 * JavaScript URL
		 *
		 * @access protected
		 * @var string
		 */
		protected $javascriptURL = '';

		/**
		 * Images URL
		 *
		 * @access protected
		 * @var string
		 */
		protected $imagesURL = '';

		/**
		 * Icon for menu option and settings page in administration panel
		 *
		 * @access protected
		 * @var string
		 */
		protected $adminIcon = '';

		/**
		 * Directory of image for about page in administration panel
		 *
		 * @access protected
		 * @var string
		 */
		protected $aboutImageDir = '';

		/**
		 * URL of image for about page in administration panel
		 *
		 * @access protected
		 * @var string
		 */
		protected $aboutImageURL = '';

		/**
		 * Thanks script API login
		 *
		 * @access protected
		 * @var string
		 */
		protected $thanksAPILogin = '';

		/**
		 * Thanks script API password
		 *
		 * @access protected
		 * @var string
		 */
		protected $thanksAPIPassword = '';

		/**
		 * Constructor
		 *
		 * @access public
		 * @param object $configClass KocujAdminConfig1 class object
		 * @param string $languageDir Directory with language data
		 * @return void
		 */
		public function __construct($configClass, $languageDir) {
			// set random value
			$this->random = rand(11111111, 99999999);
			// set license timestamp
			$this->licenseRandom = time().rand(111, 999);
			// set thanks script timestamp
			$this->thanksRandom = time().rand(111, 999);
			// set config class
			if (!is_a($configClass, 'KocujAdminConfig1')) {
				wp_die('Class error!');
			}
			$this->configClass = $configClass;
			// load language
			$locale = get_locale();
			$filename = $languageDir.'/'.$this->directory.'/kocuj-admin-1_'.$locale.'.mo';
			if (file_exists($filename)) {
				load_textdomain('kocujadmin1', $filename);
			}
			// add admin initialization
			add_action('admin_init', array($this, 'init'));
			// add admin header
			add_action('admin_head', array($this, 'addHeader'));
			// add admin menu
			add_action('admin_menu', array($this, 'pageController'));
			// add meta boxes
			add_action('add_meta_boxes', array($this, 'filterAddMetaBox'), 9999);
			add_action('save_post', array($this, 'filterSaveMetaBox'), 9999);
			// add controllers
			$this->controller['save'] = array(array(&$this, 'controllerSave'), __('Settings saved.', 'kocujadmin1'));
			$this->controller['reset'] = array(array(&$this, 'controllerReset'), __('Settings reset.', 'kocujadmin1'));
			$this->controller['uninstall'] = array(array(&$this, 'controllerUninstall'), __('Settings has been uninstalled.', 'kocujadmin1'));
			// add input helpers
			$this->inputHelper['hidden'] = array(&$this, 'inputHelperHidden');
			$this->inputHelper['text'] = array(&$this, 'inputHelperText');
			$this->inputHelper['textarea'] = array(&$this, 'inputHelperTextarea');
			$this->inputHelper['checkbox'] = array(&$this, 'inputHelperCheckbox');
			$this->inputHelper['select'] = array(&$this, 'inputHelperSelect');
			$this->inputHelper['select_end'] = array(&$this, 'inputHelperSelectEnd');
			$this->inputHelper['option'] = array(&$this, 'inputHelperOption');
			global $wp_version;
			if (version_compare($wp_version, '3.0.0', '>=')) {
				$this->inputHelper['selectmenu'] = array(&$this, 'inputHelperSelectmenu');
			}
			$this->inputHelper['submit'] = array(&$this, 'inputHelperSubmit');
			$this->inputHelper['reset'] = array(&$this, 'inputHelperReset');
			$this->inputHelper['uninstall'] = array(&$this, 'inputHelperUninstall');
		}

		/**
		 * Initialize
		 *
		 * @access public
		 * @return void
		 */
		public function init() {
			// get license filename
			$this->licenseFilename = 'license.txt';
			$language = get_bloginfo('language');
			if (empty($language)) {
				$language = 'en_US';
			} else {
				$language = str_replace('-', '_', $language);
			}
			if (file_exists($this->licenseDir.'/license-'.$language.'.txt')) {
				$this->licenseFilename = 'license-'.$language.'.txt';
			}
			// check if display license
			if (is_admin()) {
				if (isset($_GET['kocujadmin1license'])) {
					if ($_GET['kocujadmin1license'] == $this->internalName) {
						// display license
						$license = file($this->licenseDir.'/'.$this->licenseFilename);
						foreach ($license as $line) {
							$st = trim($line);
							if ((strlen($st) > 1) && (substr($st, 0, 2) == '--')) {
								echo '<hr /><br />';
							} else {
								if (((strlen($st) > 6) && (substr($st, 0, 7) == 'http://')) || ((strlen($st) > 7) && (substr($st, 0, 8) == 'https://'))) {
									echo '<a href="'.$st.'" rel="external">'.$st.'</a><br />';
								} else {
									$upper = mb_convert_case($st, MB_CASE_UPPER, 'UTF-8');
									if ($st == $upper) {
										echo '<strong>'.$st.'</strong><br />';
									} else {
										echo $st.'<br />';
									}
								}
							}
						}
						exit();
					}
				}
			}
			// check if license accept license
			if (is_admin()) {
				if (isset($_GET['kocujadmin1licenseconfirm'])) {
					if ($_GET['kocujadmin1licenseconfirm'] == $this->internalName) {
						// accept license
						add_option($this->licenseOption, 1, '', 'no');
						exit();
					}
				}
			}
			// check if display more information about thanks
			if (is_admin()) {
				if (isset($_GET['kocujadmin1thanks'])) {
					if ($_GET['kocujadmin1thanks'] == $this->internalName) {
						// display more information about thanks
						echo sprintf(__('If you want to thanks the author, please allow to send an information about your website address to him. This information will be used only for statistical purposes and for adding public information about your website on the following websites: %s. Please, keep in mind, that this information can be used in future on another websites which will belong to author - but always for statistical and information purposes.<br /><br />Please, support this project by doing this - it is for free, so it is the best way to thanks the author for his contribution.<br /><br />Only the following information about your website will be send to author:<ul><li>website address (URL);</li><li>website title;</li><li>website description.</li></ul>No other information will be send.<br /><br />These information will not be send anywhere else, only to author, and it will be saved in database belonging to author only.', 'kocujadmin1'), $this->getThanksWebsitesText());
						exit();
					}
				}
			}
			// check if send thanks data
			if (is_admin()) {
				if (isset($_GET['kocujadmin1thanksconfirm'])) {
					if ($_GET['kocujadmin1thanksconfirm'] == $this->internalName) {
						// accept thanks
						update_option($this->thanksOption, 'send', '', 'no');
						exit();
					}
				}
			}
			// check if cancel sending thanks data
			if (is_admin()) {
				if (isset($_GET['kocujadmin1thankscancel'])) {
					if ($_GET['kocujadmin1thankscancel'] == $this->internalName) {
						// cancel thanks
						add_option($this->thanksOption, 'cancel', '', 'no');
						exit();
					}
				}
			}
			// add styles and scripts
			if ($this->checkPagePermission()) {
				wp_enqueue_style('kocujadmin1-jquery-ui', $this->cssURL.'/'.$this->directory.'/jquery-ui/jquery-ui.css', array(), '1');
				wp_enqueue_script('jquery-ui-core');
				wp_enqueue_script('jquery-ui-tabs');
			}
			if (($this->checkLicenseDisplay()) || ($this->checkPagePermission()) || ($this->checkThanks(true))) {
				wp_enqueue_script('kocujadmin1-modal', $this->javascriptURL.'/kocuj-admin-1/modal.js', array('jquery'), '1');
			}
			if ($this->checkThanks(true)) {
				wp_enqueue_script('kocujadmin1-thanks', $this->javascriptURL.'/kocuj-admin-1/thanks.js', array('kocujadmin1-modal'), '1');
				if (function_exists('network_admin_url')) {
					$adminurl = network_admin_url('index.php');
				} else {
					$adminurl = admin_url('index.php');
				}
				if (function_exists('get_home_url')) {
					$homeurl = get_home_url();
				} else {
					$homeurl = get_bloginfo('home');
				}
				wp_localize_script('kocujadmin1-thanks', 'kocujAdmin1ThanksSettings', array(
					'apiUrl' => 'http://api.kocuj.pl/',
					'apiLogin' => $this->thanksAPILogin,
					'apiPassword' => $this->thanksAPIPassword,
					'websiteURL' => $homeurl,
					'websiteTitle' => get_bloginfo('name'),
					'websiteDescription' => get_bloginfo('description'),
					'loadingImageURL' => $this->imagesURL.'/kocuj-admin-1/loading.gif',
					'adminURL' => $adminurl,
					'textSending' => __('Sending information about your website address, please wait...', 'kocujadmin1'),
					'textInformationAlreadySent' => __('You have already send information about your website address. Thank you.', 'kocujadmin1'),
					'textCorrect' => __('Thank you for sending information about your website address to author.', 'kocujadmin1'),
					'textErrorTryAgain' => __('There was an error during sending information about your website address to author. There will be another attempt to do this in a few seconds.', 'kocujadmin1'),
					'textError' => __('There was an error during sending information about your website address to author. Please try again later.', 'kocujadmin1'),
					'textMoreTitle' => __('More information about sending your website address', 'kocujadmin1'),
					'textLoading' => __('Loading, please wait', 'kocujadmin1'),
					'textLoadingError' => __('Loading error! Please, check your internet connection and refresh page to try again.', 'kocujadmin1'),
				));
			}
			if (($this->checkLicenseDisplay()) || ($this->checkPagePermission())) {
				wp_enqueue_script('kocujadmin1-license', $this->javascriptURL.'/kocuj-admin-1/license.js', array('kocujadmin1-modal'), '1');
				if (function_exists('network_admin_url')) {
					$adminurl = network_admin_url('index.php');
				} else {
					$adminurl = admin_url('index.php');
				}
				wp_localize_script('kocujadmin1-license', 'kocujAdmin1LicenseSettings', array(
					'adminURL' => $adminurl,
					'textLoading' => __('Loading, please wait', 'kocujadmin1'),
					'textLoadingError' => __('Loading error! Please, check your internet connection and refresh page to try again.', 'kocujadmin1'),
					'textLicense' => __('License', 'kocujadmin1'),
					'textLicenseFor' => __('License for', 'kocujadmin1'),
					'textAccept' => __('Click here to accept this license', 'kocujadmin1'),
					'textCancel' => __('If you do not agree with it, please remove all related files; then you can refresh page to continue', 'kocujadmin1')
				));
			}
		}

		/**
		 * Call handler
		 *
		 * @access public
		 * @param string $name Method name
		 * @param array $argument Method arguments
		 * @return void
		 */
		public function __call($name, $arguments) {
			// call handler
			$this->callHandler($name, $arguments);
		}

		/**
		 * Call handler helper method
		 *
		 * @access protected
		 * @param string $name Method name
		 * @param array $argument Method arguments
		 * @return void
		 */
		protected function callHandler($name, $arguments) {
			// get method name
			$div = explode('_', $name);
			if (count($div) > 2) {
				$div0 = $div[0];
				$div1 = $div;
				unset($div1[0]);
				$divend = implode('_', $div1);
				$div = array();
				$div[0] = $div0;
				$div[1] = $divend;
			}
			// methods for menus
			if ((count($div) == 2) && (($div[0] == 'settingsMenu'.$this->random) || ($div[0] == 'themeMenu'.$this->random))) {
				// get method for this menu option
				if ($div[0] == 'settingsMenu'.$this->random) {
					$function = $this->settingsMenu[$this->settingsMenuContainer[$div[1]]][3];
					$permission = $this->settingsMenu[$this->settingsMenuContainer[$div[1]]][1];
				} else {
					$function = $this->themeMenu[$this->themeMenuContainer[$div[1]]][3];
					$permission = $this->themeMenu[$this->themeMenuContainer[$div[1]]][1];
				}
				// check permission
				if (!current_user_can($permission)) {
					_e('You have no permission to view this site.', 'kocujadmin1');
				} else {
					// processing output
					$this->processOutput();
					// show header
					?>
						<div class="wrap">
					<?php if (!empty($this->adminIcon)) : ?>
						<div class="icon32 kocujadmin1-icon">
					<?php else : ?>
						<div id="icon-themes" class="icon32">
					<?php endif; ?>
						<br /></div>
						<h2><?php echo $this->title; ?></h2>
					<?php
					// execute method
					call_user_func(array(&$this, $function));
					// show footer
					?>
						</div>
					<?php
				}
			}
			// methods for meta boxes
			if ((count($div) == 2) && ($div[0] == 'metaBox'.$this->random) && (count($arguments) > 0)) {
				$this->showMetaBox($div[1], $arguments[0]);
			}
		}

		/**
		 * Get config class object
		 *
		 * @access public
		 * @return object Config class object
		 */
		public function getConfigClass() {
			// get config class
			return $this->configClass;
		}

		/**
		 * Get form name
		 *
		 * @access public
		 * @return string Form name
		 */
		public function getFormName() {
			// get form name
			return $this->formName;
		}

		/**
		 * Get license URL
		 *
		 * @access public
		 * @return string License URL
		 */
		public function getLicenseURL() {
			// get license URL
			return $this->licenseURL;
		}

		/**
		 * Get license filename
		 *
		 * @access public
		 * @return string License filename
		 */
		public function getLicenseFilename() {
			// get license filename
			return $this->licenseFilename;
		}

		/**
		 * Get license HTML
		 *
		 * @access public
		 * @return string License HTML URL
		 */
		public function getLicenseHTML() {
			// get URL
			if (function_exists('network_admin_url')) {
				$adminurl = network_admin_url('index.php?kocujadmin1license='.$this->internalName);
			} else {
				$adminurl = admin_url('index.php?kocujadmin1license='.$this->internalName);
			}
			// get license HTML URL
			return $adminurl;
		}

		/**
		 * Get license script
		 *
		 * @access public
		 * @param bool $acceptButton Show license accept button (true) or not (false)
		 * @return string License script
		 */
		public function getLicenseScript($acceptButton) {
			// get license script
			if (!empty($acceptButton)) {
				$st = '1';
			} else {
				$st = '0';
			}
			return 'kocujAdmin1License.showLicense('.$st.', \''.$this->licenseRandom.'\')';
		}

		/**
		 * Get license link
		 *
		 * @access public
		 * @param bool $acceptButton Show license accept button (true) or not (false)
		 * @param string $text Link text
		 * @return string License link
		 */
		public function getLicenseLink($acceptButton, $text) {
			// get license link
			$output = '<a href="#" id="kocujadmin1_licenselink_'.$this->licenseRandom.'">';
			$output .= $text;
			$output .= '</a>';
			$output .= '<script type="text/javascript">'."\n";
			$output .= '/* <![CDATA[ */'."\n";
			$output .= '(function($) {'."\n";
			$output .= '$(document).ready(function() {'."\n";
			$output .= '$("#kocujadmin1_licenselink_'.$this->licenseRandom.'").attr("href", "javascript:void(0);");'."\n";
			$output .= '$("#kocujadmin1_licenselink_'.$this->licenseRandom.'").click(function() {'."\n";
			$output .= $this->getLicenseScript(false)."\n";
			$output .= '});'."\n";
			$output .= '});'."\n";
			$output .= '}(jQuery));'."\n";
			$output .= '/* ]]> */'."\n";
			$output .= '</script>'."\n";
			// exit
			return $output;
		}

		/**
		 * Show about page
		 *
		 * @access public
		 * @param string $title Page title
		 * @param string $about About what is that page
		 * @param string $license License name
		 * @param string $author Author name
		 * @param string $email Author email - default: empty
		 * @param string $www Author WWW - default: empty
		 * @param string $additional Additional information - default: empty
		 * @return void
		 */
		public function showAboutPage($title, $about, $license, $author, $email = '', $www = '', $additional = '') {
			// prepare author information
			$authorString = $author;
			$authorWWW = '';
			if (!empty($email)) {
				$authorString .= ' &lt;<a href="mailto:'.$email.'">'.$email.'</a>&gt;';
			}
			if (!empty($www)) {
				$authorWWW = '<br /><a href="'.$www.'" rel="external">'.$www.'</a>';
			}
			// show about page
			?>
				<div class="bordertitle">
					<h3><?php echo $title; ?></h3>
					<?php if ((!empty($this->aboutImageDir)) && (!empty($this->aboutImageURL))) : ?>
						<div class="kocujadmin1-about-image"></div>
					<?php endif; ?>
					<?php $this->showTableStart('about', false); ?>
						<?php
							$info = '<strong>'.$about.'</strong>
								<br /><br />
								'.__('Author', 'kocujadmin1').': '.$authorString.'
								'.$authorWWW.'
								<br /><br />
								'.__('This is licensed under', 'kocujadmin1').' '.$this->getLicenseLink(false, $license);
							if (!empty($additional)) {
								$info .= '<br /><br />'.$additional;
							}
							if ($this->checkThanks(true)) {
								$info .= '<br /><br /><hr />
										<div id="kocujadmin1aboutthanksdiv">
										<br />
										<div id="kocujadmin1aboutthanksinfodiv">'.sprintf(__('If you want to thanks author, please agree to send information about your website address by clicking on the button below. Click %shere%s to see more information about this procedure.', 'kocujadmin1'), '<a id="kocujadmin1aboutthanksmore" href="#">', '</a>').'</div>
										<input type="button" class="button" value="'.__('Send information about this website address to author', 'kocujadmin1').'" id="kocujadmin1aboutthanksconfirm" />
										<div id="kocujadmin1aboutthankstextdiv" style="display: none;">&nbsp;</div>
									</div>
								';
							}
							$this->blockHelper('', 'statictext', $info);
						?>
					<?php $this->showTableEnd(); ?>
				</div>
			<?php
		}

		/**
		 * Check if thanks should be displayed
		 *
		 * @access private
		 * @param bool $inAbout Information about thanks in about screen (true) or not (false) - default: false
		 * @return bool Thanks should be displayed (true) or not (false)
		 */
		private function checkThanks($inAbout = false) {
			// check if it is administration panel
			if (is_admin()) {
				// check if thanks are supported
				if ((empty($this->thanksOption)) || (empty($this->thanksAPILogin))) {
					return false;
				}
				// check page permission
				if ($this->checkPagePermission()) {
					// check thanks option
					$option = get_option($this->thanksOption);
					if (!empty($option)) {
						if (empty($inAbout)) {
							return false;
						} else {
							if ($option == 'send') {
								return false;
							}
						}
					}
					// exit
					return true;
				}
			}
			// exit
			return false;
		}

		/**
		 * Get text with websites where thanks information will be used
		 *
		 * @access private
		 * @return string Text with websites
		 */
		private function getThanksWebsitesText() {
			// initialize
			$output = 'kocuj.pl';
			// get other websites
			if (!empty($this->thanksWebsites)) {
				for ($z=0; $z<count($this->thanksWebsites); $z++) {
					if ($z < count($this->thanksWebsites)-1) {
						$output .= ', ';
					} else {
						$output .= ' '.__('and', 'kocujadmin1').' ';
					}
					$output .= $this->thanksWebsites[$z];
				}
			}
			// exit
			return $output;
		}

		/**
		 * Add controller
		 *
		 * @access protected
		 * @param string $name Controller name
		 * @param string $function Callback method name - must be in this or child class
		 * @param string $text Text to display after executing controller
		 * @return void
		 */
		protected function addController($name, $function, $text) {
			// add controller
			$this->controller[$name] = array(array(&$this, $function), $text);
		}

		/**
		 * Add input helper
		 *
		 * @access protected
		 * @param string $type Input helper type
		 * @param string $function Callback method name - must be in this or child class
		 * @return void
		 */
		protected function addInputHelper($type, $function) {
			// add input helper
			$this->inputHelper[$type] = array(&$this, $function);
		}

		/**
		 * Add settings menu
		 *
		 * @access protected
		 * @param string $title Menu title
		 * @param string $capability Capabilities required for access to this menu
		 * @param string $id Menu id
		 * @param string $function Callback function or method name - can be global function or method from any class
		 * @param string $parentId Parent menu id - default: empty
		 * @param string $icon Menu icon - only for parent menu - default: empty
		 * @return void
		 */
		protected function addSettingsMenu($title, $capability, $id, $function, $parentId = '', $icon = '') {
			// add settings menu
			$this->settingsMenu[] = array($title, $capability, $id, $function, $parentId, $icon);
		}

		/**
		 * Add theme menu
		 *
		 * @access protected
		 * @param string $title Menu title
		 * @param string $capability Capabilities required for access to this menu
		 * @param string $id Menu id
		 * @param string $function Callback function or method name - can be global function or method from any class
		 * @return void
		 */
		protected function addThemeMenu($title, $capability, $id, $function) {
			// add theme menu
			$this->themeMenu[] = array($title, $capability, $id, $function);
		}

		/**
		 * Add meta box
		 *
		 * @access protected
		 * @param string $id Meta box id
		 * @param string $title Meta box title
		 * @param array $places Places with this meta box, for example, "post" or "page"
		 * @param string $context Part of page with meta box, for example, "normal"
		 * @param string $priority Priority of meta box, for example, "default"
		 * @param string $functionShow Callback method name for show meta box - must be in this or child class
		 * @param string $functionUpdate Callback method name for update meta box - must be in this or child class
		 * @param array $metaKeys Meta keys to update - default: empty
		 * @param array $metaKeysCheckbox Meta keys with checkbox value to update - default: empty
		 * @return void
		 */
		protected function addMetaBox($id, $title, $places, $context, $priority, $functionShow, $functionUpdate, $metaKeys = array(),
			$metaKeysCheckbox = array()) {
			// add meta box
			$this->metaBox[] = array($id, $title, $places, $context, $priority, $id.'_'.md5($id), array(&$this, $functionShow),
				array(&$this, $functionUpdate), $metaKeys, $metaKeysCheckbox);
		}

		/**
		 * Add page name
		 *
		 * @access protected
		 * @param string $name Page name
		 * @return void
		 */
		protected function addPageName($name) {
			// add page name
			$this->pageNames[] = $name;
		}

		/**
		 * Register tabs container
		 *
		 * @access protected
		 * @param string $name Container name
		 * @param int &$containerId Return container id
		 * @return bool Status - true or false
		 */
		protected function registerTabsContainer($name, &$containerId) {
			// check if container with the selected name does not exists
			if (array_search($name, $this->tabsContainers) !== false) {
				return false;
			}
			// add container
			$pos = count($this->tabsContainers);
			$this->tabsContainers[$pos] = $name;
			// show initialization script
			?>
				<script type="text/javascript">
				/* <![CDATA[ */
					(function($) {
						$(function() {
							$("#<?php echo $name; ?>_tabs").tabs();
						});
					}(jQuery));
				/* ]]> */
				</script>
			<?php
			// set container id
			$containerId = $pos;
			// exit
			return true;
		}

		/**
		 * Register tab
		 *
		 * @access protected
		 * @param int $containerId Container id
		 * @param string $title Tab title
		 * @param int &$tabId Return tab id
		 * @return bool Status - true or false
		 */
		protected function registerTab($containerId, $title, &$tabId) {
			// check if container exists
			if ((!isset($this->tabsContainers[$containerId])) || (empty($title))) {
				return false;
			}
			// add tab
			$pos = count($this->tabs);
			$this->tabs[$pos] = array();
			$this->tabs[$pos]['CONTAINERID'] = $containerId;
			$this->tabs[$pos]['TITLE'] = $title;
			// set tab id
			$tabId = $pos;
			// exit
			return true;
		}

		/**
		 * Show tabs container start
		 *
		 * @access protected
		 * @param int $containerId Container id
		 * @return bool Status - true or false
		 */
		protected function showTabsContainerStart($containerId) {
			// check if container exists
			if (!isset($this->tabsContainers[$containerId])) {
				return false;
			}
			// show container start
			echo '<div id="'.$this->tabsContainers[$containerId].'_tabs">';
			// show tabs
			$ul = false;
			foreach ($this->tabs as $key => $tab) {
				if ($tab['CONTAINERID'] == $containerId) {
					if (empty($ul)) {
						echo '<ul>';
						$ul = true;
					}
					echo '<li><a href="#'.$this->tabsContainers[$containerId].'-'.$key.'">'.$tab['TITLE'].'</a></li>';
				}
			}
			if (!empty($ul)) {
				echo '</ul>';
			}
			// set tabs container to start
			$this->tabsContainerStarted = true;
			// exit
			return true;
		}

		/**
		 * Show tabs container end
		 *
		 * @access protected
		 * @return bool Status - true or false
		 */
		protected function showTabsContainerEnd() {
			// check if container has started
			if (empty($this->tabsContainerStarted)) {
				return false;
			}
			// show container end
			echo '</div>';
			// set tabs container to end
			$this->tabsContainerStarted = false;
			// exit
			return true;
		}

		/**
		 * Show tab start
		 *
		 * @access protected
		 * @param int $tabId Tab id
		 * @return bool Status - true or false
		 */
		protected function showTabStart($tabId) {
			// check if tab exists
			if (!isset($this->tabs[$tabId])) {
				return false;
			}
			// check if tab has not started
			if (!empty($this->tabStarted)) {
				return false;
			}
			// show tab start
			echo '<div id="'.$this->tabsContainers[$this->tabs[$tabId]['CONTAINERID']].'-'.$tabId.'">';
			// set tab to start
			$this->tabStarted = true;
			// exit
			return true;
		}

		/**
		 * Show tab end
		 *
		 * @access protected
		 * @return bool Status - true or false
		 */
		protected function showTabEnd() {
			// check if tab has started
			if (empty($this->tabStarted)) {
				return false;
			}
			// show tab end
			echo '</div>';
			// set tab to end
			$this->tabStarted = false;
			// exit
			return true;
		}

		/**
		 * Show table start
		 *
		 * @access protected
		 * @param string $id Table id
		 * @param bool $showMinimalFullSupportedVersionInfo Show if WordPress version is lower than minimal full supported version (true) or not (false)
		 * @return bool Status - true or false
		 */
		protected function showTableStart($id, $showMinimalFullSupportedVersionInfo = true) {
			// check if table has not started
			if (!empty($this->tableStarted)) {
				return false;
			}
			// show table start
			if (empty($id)) {
				$id = rand(1111, 9999);
			}
			echo '<table id="admin_form_'.$id.'" class="form-table kocujadmin1-form-table"><tbody>';
			echo '<tr style="display: none;"><td>&nbsp;</td><td>&nbsp;</td></tr>';
			// optionally show information about not fully supported version
			if (!empty($showMinimalFullSupportedVersionInfo)) {
				global $wp_version;
				if (!empty($this->minimalFullSupportedVersion)) {
					if (version_compare($wp_version, $this->minimalFullSupportedVersion, '<')) {
						$this->blockHelper('', 'statictext', sprintf('<em>'.__('You are using WordPress version below %s. Some options are hidden and some functionality is not available. It will be activated when you will update WordPress to - at least - %s version.', 'kocujadmin1').'</em>', $this->minimalFullSupportedVersion, $this->minimalFullSupportedVersion));
					}
				}
			}
			// set tab to end
			$this->tableStarted = true;
			// exit
			return true;
		}

		/**
		 * Show table end
		 *
		 * @access protected
		 * @return bool Status - true or false
		 */
		protected function showTableEnd() {
			// check if table has started
			if (empty($this->tableStarted)) {
				return false;
			}
			// show table start
			echo '</tbody></table>';
			// set tab to end
			$this->tableStarted = false;
			// exit
			return true;
		}

		/**
		 * Get $_SERVER['SCRIPT_URL'] or guess it if there is any
		 *
		 * @access private
		 * @return string Script URL
		 */
		private function getScriptURL() {
			// initialize
			$scriptURL = '';
			$update = false;
			// get script URL
			if ((isset($_SERVER['SCRIPT_URL'])) && (!empty($_SERVER['SCRIPT_URL']))) {
				$scriptURL = $_SERVER['SCRIPT_URL'];
			} else {
				if ((isset($_SERVER['REDIRECT_URL'])) && (!empty($_SERVER['REDIRECT_URL']))) {
					$scriptURL = $_SERVER['REDIRECT_URL'];
					$update = true;
				} else {
					if ((isset($_SERVER['REQUEST_URI'])) && (!empty($_SERVER['REQUEST_URI']))) {
						$path = parse_url($_SERVER['REQUEST_URI']);
						$scriptURL = $path['path'];
						$update = true;
					}
				}
			}
			// save script URL value
			if (!empty($update)) {
				$_SERVER['SCRIPT_URL'] = $scriptURL;
			}
			// exit
			return $scriptURL;
		}

		/**
		 * Check if this page has permission
		 *
		 * @access private
		 * @return bool Permission (true) or not (false)
		 */
		private function checkPagePermission() {
			// check page
			if (is_admin()) {
				if (isset($_GET['page'])) {
					foreach ($this->pageNames as $pageName) {
						if (substr($_GET['page'], strlen($_GET['page'])-strlen($pageName), strlen($pageName)) == $pageName) {
							return true;
						}
					}
				}
			}
			// exit
			return false;
		}

		/**
		 * Check if license should be displayed
		 *
		 * @access private
		 * @return bool License should be displayed (true) or not (false)
		 */
		private function checkLicenseDisplay() {
			// check if license should be displayed
			if (is_admin()) {
				if ($this->licenseForce != KocujAdminEnumLicensePlace1::NONE) {
					$display = true;
					if ($this->licenseForce != KocujAdminEnumLicensePlace1::ALL) {
						$url = $this->getScriptURL();
						if (empty($url)) {
							return false;
						} else {
							$div = explode('/', $url);
							if (count($div) == 0) {
								return false;
							} else {
								$url = $div[count($div)-1];
								$display = false;
								if (($this->licenseForce == KocujAdminEnumLicensePlace1::PLUGINS) && ($url == 'plugins.php')) {
									$display = true;
								}
								if (($this->licenseForce == KocujAdminEnumLicensePlace1::THEMES) && ($url == 'themes.php')) {
									if ((!isset($_GET)) || (empty($_GET))) {
										$display = true;
									}
								}
								if ($this->checkPagePermission()) {
									$display = true;
								}
							}
						}
					}
					if (!empty($display)) {
						$option = get_option($this->licenseOption);
						if (empty($option)) {
							return true;
						}
					}
				}
			}
			// exit
			return false;
		}

		/**
		 * Get input helper
		 *
		 * @access public
		 * @param string $var Field name
		 * @param string $type Field type
		 * @param string $description Field description - default: empty
		 * @param string $value Field value - default: empty
		 * @param string $selected Selected field - default: empty
		 * @param bool $checked Checked field - default: false
		 * @param string $tooltip Tooltip text - default: empty
		 * @param bool $isArray Option is array - default: false
		 * @param string $eventOnChange Event onChange - default: empty
		 * @return string Input helper
		 */
		public function getInputHelper($var, $type, $description = '', $value = '', $selected = '', $checked = false, $tooltip = '',
			$isArray = false, $eventOnChange = '') {
			// check parameters
			if (!isset($this->inputHelper[$type])) {
				wp_die('Helper type error!');
			}
			// initialize
			$output = '';
			// optionally divide tooltip
			if (!empty($tooltip)) {
				$tooltip = str_replace('"', '`', $tooltip);
			}
			// show input field
			$tip = '';
			if (!empty($tooltip)) {
				$tip = 'title="'.$tooltip.'"';
			}
			$output .= "\n";
			// call method
			$output .= call_user_func($this->inputHelper[$type], $var, $description, $value, $selected, $checked, $tip, $isArray, $eventOnChange);
			// exit
			return $output;
		}

		/**
		 * Show input helper
		 *
		 * @access public
		 * @param string $var Field name
		 * @param string $type Field type
		 * @param string $description Field description - default: empty
		 * @param string $value Field value - default: empty
		 * @param string $selected Selected field - default: empty
		 * @param bool $checked Checked field - default: false
		 * @param string $tooltip Tooltip text - default: empty
		 * @param bool $isArray Option is array - default: false
		 * @param string $eventOnChange Event onChange - default: empty
		 * @return void
		 */
		public function inputHelper($var, $type, $description = '', $value = '', $selected = '', $checked = false, $tooltip = '',
			$isArray = false, $eventOnChange = '') {
			// show input helper
			echo $this->getInputHelper($var, $type, $description, $value, $selected, $checked, $tooltip, $isArray, $eventOnChange);
		}

		/**
		 * Get input helper - hidden
		 *
		 * @access public
		 * @param string $var Field name
		 * @param string $description Field description - default: empty
		 * @param string $value Field value - default: empty
		 * @param string $selected Selected field - default: empty
		 * @param bool $checked Checked field - default: false
		 * @param string $tip Event for mouseover displaying tooltip - default: empty
		 * @param bool $isArray Option is array - default: false
		 * @param string $eventOnChange Event onChange - default: empty
		 * @return void
		 */
		public function inputHelperHidden($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
			$isArray = false, $eventOnChange = '') {
			// prepare onChange event
			$change = '';
			if (!empty($eventOnChange)) {
				$change = 'onchange="'.$eventOnChange.'"';
			}
			// prepare id
			$idst = ' id="'.$var.'"';
			if (strpos($var, '[]') !== false) {
				$idst = '';
			}
			// get input helper
			return '<input name="'.$var.'"'.$idst.' type="hidden" value="'.$value.'" '.$change.' />';
		}

		/**
		 * Get input helper - text
		 *
		 * @access public
		 * @param string $var Field name
		 * @param string $description Field description - default: empty
		 * @param string $value Field value - default: empty
		 * @param string $selected Selected field - default: empty
		 * @param bool $checked Checked field - default: false
		 * @param string $tip Event for mouseover displaying tooltip - default: empty
		 * @param bool $isArray Option is array - default: false
		 * @param string $eventOnChange Event onChange - default: empty
		 * @return void
		 */
		public function inputHelperText($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
			$isArray = false, $eventOnChange = '') {
			// prepare onChange event
			$change = '';
			if (!empty($eventOnChange)) {
				$change = 'onchange="'.$eventOnChange.'"';
			}
			// prepare id
			$idst = ' id="'.$var.'"';
			if (strpos($var, '[]') !== false) {
				$idst = '';
			}
			// get input helper
			return '<input name="'.$var.'"'.$idst.' type="text" value="'.htmlspecialchars($value).'" style="width: 300px;" '.$tip.' '.$change.' />';
		}

		/**
		 * Get input helper - textarea
		 *
		 * @access public
		 * @param string $var Field name
		 * @param string $description Field description - default: empty
		 * @param string $value Field value - default: empty
		 * @param string $selected Selected field - default: empty
		 * @param bool $checked Checked field - default: false
		 * @param string $tip Event for mouseover displaying tooltip - default: empty
		 * @param bool $isArray Option is array - default: false
		 * @param string $eventOnChange Event onChange - default: empty
		 * @return void
		 */
		public function inputHelperTextarea($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
			$isArray = false, $eventOnChange = '') {
			// prepare onChange event
			$change = '';
			if (!empty($eventOnChange)) {
				$change = 'onchange="'.$eventOnChange.'"';
			}
			// prepare id
			$idst = ' id="'.$var.'"';
			if (strpos($var, '[]') !== false) {
				$idst = '';
			}
			// get input helper
			return '<textarea name="'.$var.'"'.$idst.' rows="15" cols="" style="width: 300px; resize: none;" '.$tip.' '.$change.'>'.htmlspecialchars($value).'</textarea>';
		}

		/**
		 * Get input helper - checkbox
		 *
		 * @access public
		 * @param string $var Field name
		 * @param string $description Field description - default: empty
		 * @param string $value Field value - default: empty
		 * @param string $selected Selected field - default: empty
		 * @param bool $checked Checked field - default: false
		 * @param string $tip Event for mouseover displaying tooltip - default: empty
		 * @param bool $isArray Option is array - default: false
		 * @param string $eventOnChange Event onChange - default: empty
		 * @return void
		 */
		public function inputHelperCheckbox($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
			$isArray = false, $eventOnChange = '') {
			// prepare onChange event
			$change = '';
			if (!empty($eventOnChange)) {
				$change = 'onchange="'.$eventOnChange.'"';
			}
			// prepare checked
			$extra = '';
			if (!empty($checked)) {
				$extra = 'checked="checked"';
			}
			// prepare id
			$idst = ' id="'.$var.'"';
			if (strpos($var, '[]') !== false) {
				$idst = '';
			}
			// get input helper
			return '<input name="'.$var.'"'.$idst.' type="checkbox" value="'.$value.'" style="border-width: 0px;" '.$extra.' '.$tip.' '.$change.' />';
		}

		/**
		 * Get input helper - select
		 *
		 * @access public
		 * @param string $var Field name
		 * @param string $description Field description - default: empty
		 * @param string $value Field value - default: empty
		 * @param string $selected Selected field - default: empty
		 * @param bool $checked Checked field - default: false
		 * @param string $tip Event for mouseover displaying tooltip - default: empty
		 * @param bool $isArray Option is array - default: false
		 * @param string $eventOnChange Event onChange - default: empty
		 * @return void
		 */
		public function inputHelperSelect($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
			$isArray = false, $eventOnChange = '') {
			// prepare onChange event
			$change = '';
			if (!empty($eventOnChange)) {
				$change = 'onchange="'.$eventOnChange.'"';
			}
			// prepare id
			$idst = ' id="'.$var.'"';
			if (strpos($var, '[]') !== false) {
				$idst = '';
			}
			// get input helper
			return '<select name="'.$var.'"'.$idst.' style="width: 300px;" '.$tip.' '.$change.' >';
		}

		/**
		 * Get input helper - select_end
		 *
		 * @access public
		 * @param string $var Field name
		 * @param string $description Field description - default: empty
		 * @param string $value Field value - default: empty
		 * @param string $selected Selected field - default: empty
		 * @param bool $checked Checked field - default: false
		 * @param string $tip Event for mouseover displaying tooltip - default: empty
		 * @param bool $isArray Option is array - default: false
		 * @param string $eventOnChange Event onChange - default: empty
		 * @return void
		 */
		public function inputHelperSelectEnd($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
			$isArray = false, $eventOnChange = '') {
			// get input helper
			return '</select>';
		}

		/**
		 * Get input helper - option
		 *
		 * @access public
		 * @param string $var Field name
		 * @param string $description Field description - default: empty
		 * @param string $value Field value - default: empty
		 * @param string $selected Selected field - default: empty
		 * @param bool $checked Checked field - default: false
		 * @param string $tip Event for mouseover displaying tooltip - default: empty
		 * @param bool $isArray Option is array - default: false
		 * @param string $eventOnChange Event onChange - default: empty
		 * @return void
		 */
		public function inputHelperOption($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
			$isArray = false, $eventOnChange = '') {
			// get input helper
			$extra = '';
			if ($selected == $value) {
				$extra = 'selected="selected"';
			}
			return '<option value="'.htmlspecialchars($value).'" '.$extra.' >'.$description.'</option>';
		}

		/**
		 * Get input helper - selectmenu
		 *
		 * @access public
		 * @param string $var Field name
		 * @param string $description Field description - default: empty
		 * @param string $value Field value - default: empty
		 * @param string $selected Selected field - default: empty
		 * @param bool $checked Checked field - default: false
		 * @param string $tip Event for mouseover displaying tooltip - default: empty
		 * @param bool $isArray Option is array - default: false
		 * @param string $eventOnChange Event onChange - default: empty
		 * @return void
		 */
		public function inputHelperSelectmenu($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
			$isArray = false, $eventOnChange = '') {
			// get menus
			$menus = get_terms('nav_menu');
			// get input helper
			$output = $this->inputHelperSelect($var, $description, $value, $selected, $checked, $tip, false, $eventOnChange);
			if (!empty($isArray)) {
				$output .= $this->inputHelperOption($var, '----', '', $selected, $checked, $tip, false, $eventOnChange);
			}
			foreach ($menus as $menu) {
				$output .= $this->inputHelperOption($var, $menu->name, $menu->term_id, $selected, $checked, $tip, false, $eventOnChange);
			}
			$output .= $this->inputHelperSelectEnd($var, $description, $value, $selected, $checked, $tip, false, $eventOnChange);
			// exit
			return $output;
		}

		/**
		 * Get input helper - submit
		 *
		 * @access public
		 * @param string $var Field name
		 * @param string $description Field description - default: empty
		 * @param string $value Field value - default: empty
		 * @param string $selected Selected field - default: empty
		 * @param bool $checked Checked field - default: false
		 * @param string $tip Event for mouseover displaying tooltip - default: empty
		 * @param bool $isArray Option is array - default: false
		 * @param string $eventOnChange Event onChange - default: empty
		 * @return void
		 */
		public function inputHelperSubmit($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
			$isArray = false, $eventOnChange = '') {
			// get input helper
			$output = '<input name="'.$var.'" id="'.$var.'" type="button" class="button-primary" value="'.$value.'" '.$tip.' />';
			$output .= '<script type="text/javascript">'."\n";
			$output .= '/* <![CDATA[ */'."\n";
			$output .= '(function($) {'."\n";
			$output .= '$(document).ready(function() {'."\n";
			$output .= '$("#'.$var.'").attr("href", "javascript:void(0);");'."\n";
			$output .= '$("#'.$var.'").click(function() {'."\n";
			$output .= '$("#action").val("save");'."\n";
			$output .= '$("#'.$this->formName.'").submit();'."\n";
			$output .= '});'."\n";
			$output .= '});'."\n";
			$output .= '}(jQuery));'."\n";
			$output .= '/* ]]> */'."\n";
			$output .= '</script>'."\n";
			// exit
			return $output;
		}

		/**
		 * Get input helper - reset
		 *
		 * @access public
		 * @param string $var Field name
		 * @param string $description Field description - default: empty
		 * @param string $value Field value - default: empty
		 * @param string $selected Selected field - default: empty
		 * @param bool $checked Checked field - default: false
		 * @param string $tip Event for mouseover displaying tooltip - default: empty
		 * @param bool $isArray Option is array - default: false
		 * @param string $eventOnChange Event onChange - default: empty
		 * @return void
		 */
		public function inputHelperReset($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
			$isArray = false, $eventOnChange = '') {
			// get input helper
			$output = '<input name="'.$var.'" id="'.$var.'" type="button" class="button" value="'.$value.'" '.$tip.' />';
			$output .= '<script type="text/javascript">'."\n";
			$output .= '/* <![CDATA[ */'."\n";
			$output .= '(function($) {'."\n";
			$output .= '$(document).ready(function() {'."\n";
			$output .= '$("#'.$var.'").attr("href", "javascript:void(0);");'."\n";
			$output .= '$("#'.$var.'").click(function() {'."\n";
			$output .= 'if (confirm(\''.__('Are you sure do you want to restore default settings?', 'kocujadmin1').'\')) {'."\n";
			$output .= '$("#action").val("reset");'."\n";
			$output .= '$("#'.$this->formName.'").submit();'."\n";
			$output .= '}'."\n";
			$output .= '});'."\n";
			$output .= '});'."\n";
			$output .= '}(jQuery));'."\n";
			$output .= '/* ]]> */'."\n";
			$output .= '</script>'."\n";
			// exit
			return $output;
		}

		/**
		 * Get input helper - uninstall
		 *
		 * @access public
		 * @param string $var Field name
		 * @param string $description Field description - default: empty
		 * @param string $value Field value - default: empty
		 * @param string $selected Selected field - default: empty
		 * @param bool $checked Checked field - default: false
		 * @param string $tip Event for mouseover displaying tooltip - default: empty
		 * @param bool $isArray Option is array - default: false
		 * @param string $eventOnChange Event onChange - default: empty
		 * @return void
		 */
		public function inputHelperUninstall($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
			$isArray = false, $eventOnChange = '') {
			// get input helper
			$output = '<input name="'.$var.'" id="'.$var.'" type="button" class="button" value="'.$value.'" '.$tip.' />';
			$output .= '<script type="text/javascript">'."\n";
			$output .= '/* <![CDATA[ */'."\n";
			$output .= '(function($) {'."\n";
			$output .= '$(document).ready(function() {'."\n";
			$output .= '$("#'.$var.'").attr("href", "javascript:void(0);");'."\n";
			$output .= '$("#'.$var.'").click(function() {'."\n";
			$output .= 'if (confirm(\''.__('Are you sure do you want to uninstall settings?', 'kocujadmin1').'\')) {'."\n";
			$output .= '$("#action").val("uninstall");'."\n";
			$output .= '$("#'.$this->formName.'").submit();'."\n";
			$output .= '}'."\n";
			$output .= '});'."\n";
			$output .= '});'."\n";
			$output .= '}(jQuery));'."\n";
			$output .= '/* ]]> */'."\n";
			$output .= '</script>'."\n";
			// exit
			return $output;
		}

		/**
		 * Get block helper
		 *
		 * @access public
		 * @param string $id Configuration id
		 * @param string $type Field type
		 * @param string $tooltip Tooltip text or, if $type is "statictext", text to show - default: empty
		 * @param array $options Options for select - default: empty arraz
		 * @param string $additionalLabel Additional label text - default: empty
		 * @param string $additional Additional text - default: empty
		 * @param bool $showLabel Show label (true) or not (false) - default: true
		 * @param string $eventOnChange Event onChange - default: empty
		 * @return string
		 */
		public function getBlockHelper($id, $type, $tooltip = '', $options = array(), $additionalLabel = '', $additional = '', $showLabel = true,
			$eventOnChange = '') {
			// initialize
			$output = '';
			// check if type exists
			if (($type != 'submittext') && ($type != 'statictext') && (!isset($this->inputHelper[$type]))) {
				return '';
			}
			// get configuration value
			$value = $this->configClass->getOption($id);
			// prepare values
			$array = $this->configClass->checkOptionArray($id);
			$arrayText = '';
			if (empty($array)) {
				$tmp = $value;
				$value = array();
				$value[0] = $tmp;
			} else {
				$arrayText = '[]';
				$value[count($value)] = '';
			}
			// show block
			$labelDisplayed = false;
			$loopCount = count($value);
			for ($z=0; $z<$loopCount; $z++) {
				// check if display this position
				if (($type == 'selectmenu') && (function_exists('wp_get_nav_menu_object')) && (!wp_get_nav_menu_object($value[$z])) && ($z < $loopCount-1)) {
				} else {
					// prepare values
					$checked = false;
					$selected = '';
					if ($type == 'checkbox') {
						$checked = $value[$z];
						$value[$z] = '1';
					}
					if (($type == 'select') || ($type == 'selectmenu')) {
						$checked = '';
						$selected = $value[$z];
					}
					$labeladd = '';
					if (!empty($additionalLabel)) {
						$labeladd = '<br />'.$additionalLabel;
					}
					// prepare block id
					$blockId = 0;
					if (isset($this->blockHelperId[$id])) {
						$this->blockHelperId[$id]++;
						$blockId = $this->blockHelperId[$id];
					}
					if (!empty($blockId)) {
						$blockId = $id.'_'.$blockId;
					} else {
						$blockId = $id;
					}
					if (!isset($this->blockHelperId[$id])) {
						$this->blockHelperId[$id] = 0;
					}
					// get block
					if (($type == 'statictext') || ($type == 'submittext')) {
						$output .= '<tr id="kocujadmin1_field_'.$blockId.'">';
						$st = '';
						if ($type == 'submittext') {
							$st = ' submit';
						}
						$output .= '<td class="kocujadmin1-'.$type.$st.'" colspan="2">';
						$output .= $tooltip;
						$output .= '</td>';
						$output .= '</tr>';
					} else {
						$output .= '<tr id="kocujadmin1_field_'.$blockId.'">';
						$output .= '<td class="kocujadmin1-label">';
						if ((!empty($showLabel)) && (!$labelDisplayed)) {
							if (empty($arrayText)) {
								$output .= '<label for="set_'.$id.'">';
							}
							$output .= $this->configClass->getOptionLabel($id).$labeladd;
							if (empty($arrayText)) {
								$output .= '</label>';
							}
							$labelDisplayed = true;
						} else {
							$output .= '&nbsp;';
						}
						$output .= '</td>';
						$output .= '<td class="kocujadmin1-content">';
						$output .= $this->getInputHelper('set_'.$id.$arrayText, $type, '', $value[$z], $selected, $checked, $tooltip, $array,
							$eventOnChange);
						if ($type == 'select') {
							if (!empty($array)) {
								$output .= $this->getInputHelper('set_'.$id.$arrayText, 'option', '', '', '', false, '', false, $eventOnChange);
							}
							if (!empty($options)) {
								foreach ($options as $key => $option) {
									$output .= $this->getInputHelper('set_'.$id.$arrayText, 'option', $option, $key, $value[$z], false, '', false,
										$eventOnChange);
								}
							}
							$output .= $this->getInputHelper('set_'.$id.$arrayText, 'select_end', '', '', '', false, '', false, $eventOnChange);
						}
						if ($z == count($value)-1) {
							$output .= $additional;
						}
						$output .= '</td>';
						$output .= '</tr>';
					}
				}
			}
			// exit
			return $output;
		}

		/**
		 * Show block helper
		 *
		 * @access public
		 * @param string $id Configuration id
		 * @param string $type Field type
		 * @param string $tooltip Tooltip text - default: empty
		 * @param array $options Options for select - default: empty arraz
		 * @param string $additionalLabel Additional label text - default: empty
		 * @param string $additional Additional text - default: empty
		 * @param bool $showLabel Show label (true) or not (false) - default: true
		 * @param string $eventOnChange Event onChange - default: empty
		 * @return void
		 */
		public function blockHelper($id, $type, $tooltip = '', $options = array(), $additionalLabel = '', $additional = '', $showLabel = true,
			$eventOnChange = '') {
			// show block helper
			echo $this->getBlockHelper($id, $type, $tooltip, $options, $additionalLabel, $additional, $showLabel, $eventOnChange);
		}

		/**
		 * Add settings menu container
		 *
		 * @access private
		 * @param string $value Value
		 * @return int Position
		 */
		private function addSettingsMenuContainer($value) {
			// check if there is this value already
			$loopCount = count($this->settingsMenuContainer);
			for ($z=0; $z<$loopCount; $z++) {
				if ($this->settingsMenu[$this->settingsMenuContainer[$z]][3] == $this->settingsMenu[$value][3]) {
					return $z;
				}
			}
			// add new information
			$pos = count($this->settingsMenuContainer);
			$this->settingsMenuContainer[$pos] = $value;
			// exit
			return $pos;
		}

		/**
		 * Add theme menu container
		 *
		 * @access private
		 * @param string $value Value
		 * @return int Position
		 */
		private function addThemeMenuContainer($value) {
			// check if there is this value already
			$loopCount = count($this->themeMenuContainer);
			for ($z=0; $z<$loopCount; $z++) {
				if ($this->themeMenu[$this->themeMenuContainer[$z]][3] == $this->themeMenu[$value][3]) {
					return $z;
				}
			}
			// add new information
			$pos = count($this->themeMenuContainer);
			$this->themeMenuContainer[$pos] = $value;
			// exit
			return $pos;
		}

		/**
		 * Add header
		 *
		 * @access public
		 * @return void
		 */
		public function addHeader() {
			// optionally add script to open all external links in a new window
			if ($this->checkPagePermission()) {
				?>
					<script type="text/javascript">
					/* <![CDATA[ */
						(function($) {
							$(document).ready(function() {
								$('a[rel=external]').attr('target', '_blank');
							});
						}(jQuery));
					/* ]]> */
					</script>
				<?php
			}
			// optionally show thanks script settings
			if ($this->checkThanks(true)) {
				?>
					<script type="text/javascript">
					/* <![CDATA[ */
						kocujAdmin1Thanks.setInternalName('<?php echo $this->internalName; ?>', '<?php echo $this->thanksRandom; ?>');
						(function($) {
							$(document).ready(function() {
								kocujAdmin1Thanks.setForm('<?php echo $this->thanksRandom; ?>');
							});
						}(jQuery));
					/* ]]> */
					</script>
				<?php
			}
			// show optional license window
			if ((!empty($this->licenseURL)) && (!empty($this->licenseDir)))  {
				// check if license should be displayed
				$showLicense = $this->checkLicenseDisplay();
				// optionally set license scripts
				if ((!empty($showLicense)) || ($this->checkPagePermission())) {
					?>
						<script type="text/javascript">
						/* <![CDATA[ */
							kocujAdmin1License.setURL('<?php echo $this->getLicenseHTML(); ?>', '<?php echo $this->licenseRandom; ?>');
							kocujAdmin1License.setName('<?php echo $this->licenseName; ?>', '<?php echo $this->licenseRandom; ?>');
							kocujAdmin1License.setInternalName('<?php echo $this->internalName; ?>', '<?php echo $this->licenseRandom; ?>');
						/* ]]> */
						</script>
					<?php
				}
				// optionally show license
				if (!empty($showLicense)) {
					?>
						<script type="text/javascript">
						/* <![CDATA[ */
							(function($) {
								$(document).ready(function() {
									<?php echo $this->getLicenseScript(true); ?>
								});
							}(jQuery));
						/* ]]> */
						</script>
					<?php
				}
			}
			// show style
			if ($this->checkPagePermission()) {
				?>
				<style type="text/css" media="all">
				<!--
					.kocujadmin1-form-table {
						width: 100%;
					}

					.kocujadmin1-label,
					.kocujadmin1-label label {
						text-align: left;
						width: 300px;
					}

					.kocujadmin1-content {
						text-align: left;
					}

					<?php if (!empty($this->adminIcon)) : ?>
						.kocujadmin1-icon {
							background: transparent url('<?php echo $this->adminIcon; ?>') no-repeat left top;
						}
					<?php endif; ?>

					<?php
						if ((!empty($this->aboutImageDir)) && (!empty($this->aboutImageURL))) {
							$size = getimagesize($this->aboutImageDir);
							if (empty($size)) {
								$size = array();
								$size[0] = 0;
								$size[1] = 0;
							}
							?>
								.kocujadmin1-about-image {
									background: transparent url('<?php echo $this->aboutImageURL; ?>') no-repeat left top;
									width: <?php echo $size[0]; ?>px;
									height: <?php echo $size[1]; ?>px;
								}
							<?php
						}
					?>
				-->
				</style>
				<?php
			}
		}

		/**
		 * Page controller
		 *
		 * @access public
		 * @return void
		 */
		public function pageController() {
			// change options
			if ($this->checkPagePermission()) {
				if (isset($_REQUEST['action'])) {
					if (isset($this->controller[$_REQUEST['action']][0])) {
						$errors = call_user_func($this->controller[$_REQUEST['action']][0]);
						if (isset($_GET['page'])) {
							$page = $_GET['page'];
						} else {
							$page = '';
						}
						$url = $this->getScriptURL();
						$div = explode('?', $url);
						if (count($div) > 0) {
							$url = $div[0];
						}
						header('Location: '.$url.'?page='.urlencode($page).'&'.urlencode($_REQUEST['action']).'=true&errors='.urlencode($errors));
						exit();
					} else {
						_e('Wrong parameters.', 'kocujadmin1');
					}
				}
			}
			// add settings menu
			foreach ($this->settingsMenu as $key => $menu) {
				if (empty($menu[4])) {
					$pos = $this->addSettingsMenuContainer($key);
					add_menu_page($menu[0], $menu[0], $menu[1], $menu[2], array(&$this, 'settingsMenu'.$this->random.'_'.$pos), $menu[5]);
				}
			}
			foreach ($this->settingsMenu as $key => $menu) {
				if (!empty($menu[4])) {
					$pos = $this->addSettingsMenuContainer($key);
					add_submenu_page($menu[4], $menu[0], $menu[0], $menu[1], $menu[2], array(&$this, 'settingsMenu'.$this->random.'_'.$pos));
				}
			}
			// add theme menu
			foreach ($this->themeMenu as $key => $menu) {
				$pos = $this->addThemeMenuContainer($key);
				add_theme_page($menu[0], $menu[0], $menu[1], $menu[2], array(&$this, 'themeMenu'.$this->random.'_'.$pos));
			}
		}

		/**
		 * Save controller
		 *
		 * @access public
		 * @return string Output statuc
		 */
		public function controllerSave() {
			// save options
			$errors = $this->configClass->updateOptions();
			// exit
			return $errors;
		}

		/**
		 * Reset controller
		 *
		 * @access public
		 * @return string Output statuc
		 */
		public function controllerReset() {
			// reset options
			$this->configClass->deleteOptions();
			// exit
			return '';
		}

		/**
		 * Uninstall controller
		 *
		 * @access public
		 * @return string Output statuc
		 */
		public function controllerUninstall() {
			// uninstall
			$this->configClass->uninstallOptions();
			// exit
			return '';
		}

		/**
		 * Processing output
		 *
		 * @access protected
		 * @return void
		 */
		protected function processOutput() {
			// process output
			if ($this->checkPagePermission()) {
				// optionally show box for sending thanks
				if ($this->checkThanks()) {
					?>
						<div id="kocujadmin1thanks" class="updated fade">
							<div id="kocujadmin1thankstextdiv">
								<p><?php echo sprintf(__('Thank you for using %s.', 'kocujadmin1'), $this->fullTitle); ?></p>
								<p><?php echo sprintf(__('If you want to thanks the author, please allow to send information about your website address. This will be used for statistical purposes and for adding public information about your website on the following websites: %s. Please, support this project by doing this - it is for free, so it is the best way to thanks the author for his contribution.', 'kocujadmin1'), $this->getThanksWebsitesText()); ?></p>
							</div>
							<div id="kocujadmin1thanksmorediv">
								<p><a href="#" id="kocujadmin1thanksmore"><?php _e('Show more information', 'kocujadmin1'); ?></a></p>
							</div>
							<div id="kocujadmin1thanksemptydiv">
								<p>&nbsp;</p>
							</div>
							<div id="kocujadmin1thanksoptionsdiv">
								<p><strong><a href="#" id="kocujadmin1thanksconfirm"><?php _e('Yes, I agree to thanks author by sending the described information to him', 'kocujadmin1'); ?></a></strong></p>
								<p><a href="#" id="kocujadmin1thankscancel"><?php _e('No, I do not agree to do this (this information will never show again)', 'kocujadmin1'); ?></a></p>
							</div>
						</div>
					<?php
				}
				// get errors text
				$errors = '';
				if (isset($_REQUEST['errors'])) {
					if ($_REQUEST['errors']) {
						$errors = $_REQUEST['errors'];
					}
				}
				// show save message
				foreach ($this->controller as $key => $controller) {
					if (isset($_REQUEST[$key])) {
						if ($_REQUEST[$key]) {
							if (isset($this->controller[$key][1])) {
								if (!empty($this->controller[$key][1])) {
									echo '<div id="message" class="updated fade"><p><strong>',$this->controller[$key][1],'</strong></p></div>';
								}
							}
						}
					}
				}
				if (isset($_REQUEST['uninstall'])) {
					if ($_REQUEST['uninstall']) {
						return;
					}
				}
				// optionally show errors
				if (!empty($errors)) {
					echo '<div id="error" class="updated fade" style="color: red;"><p>',__('There were some errors - these options has not been saved.', 'kocujadmin1'),'</p><p>',stripslashes($errors),'</p></div>';
				}
			}
		}

		/**
		 * Add meta boxes
		 *
		 * @access public
		 * @return void
		 */
		public function filterAddMetaBox() {
			// add meta boxes
			foreach ($this->metaBox as $key => $metaBox) {
				foreach ($metaBox[2] as $place) {
					add_meta_box($metaBox[0], $metaBox[1], array(&$this, 'metaBox'.$this->random.'_'.$key), $place, $metaBox[3], $metaBox[4]);
				}
			}
		}

		/**
		 * Show meta box
		 *
		 * @access private
		 * @param int $pos Position in metaBox array
		 * @param object $post Post
		 * @return void
		 */
		private function showMetaBox($pos, $post) {
			// get meta box data
			$data = $this->metaBox[$pos];
			// get post or page id
			if (isset($post->ID)) {
				$id = $post->ID;
			} else {
				return;
			}
			// add hidden security field
			wp_nonce_field($this->nonceAction, $data[5]);
			// show meta box
			call_user_func($data[6], $id, $post);
		}

		/**
		 * Update meta box
		 *
		 * @access private
		 * @param int $id Post id
		 * @param string $metaKey Meta key
		 * @param bool $checkbox It is checkbox (true) or not (false)
		 * @return void
		 */
		private function updateMetaBox($id, $metaKey, $checkbox) {
			// get new value
			if (isset($_POST[$metaKey])) {
				$newValue = $_POST[$metaKey];
			} else {
				$newValue = '';
			}
			if (!empty($checkbox)) {
				if (empty($newValue)) {
					$newValue = '0';
				} else {
					$newValue = '1';
				}
			}
			// add or update
			$value = ThemeInternalCacheInterface::getClass()->get_post_meta($id, '_'.$metaKey, false);
			if (count($value) > 0) {
				update_post_meta($id, '_'.$metaKey, $newValue);
			} else {
				add_post_meta($id, '_'.$metaKey, $newValue);
			}
		}

		/**
		 * Saving of meta box
		 *
		 * @access public
		 * @return void
		 */
		public function filterSaveMetaBox() {
			// declare globals
			global $post;
			// get post id
			if (isset($post->ID)) {
				$id = $post->ID;
			} else {
				return;
			}
			// check if not autosave
			if ((defined('DOING_AUTOSAVE')) && (DOING_AUTOSAVE)) {
				return;
			}
			// loop
			foreach ($this->metaBox as $metaBox) {
				// check security
				if ((isset($_POST[$metaBox[5]])) && (wp_verify_nonce($_POST[$metaBox[5]], $this->nonceAction))) {
					// check permissions
					if (isset($_POST['post_type'])) {
						if (array_search($_POST['post_type'], $metaBox[2]) !== false) {
							$ok = true;
							foreach ($metaBox[2] as $place) {
								$type = 'edit_page';
								if ($place == 'post') {
									$type == 'edit_post';
								}
								if (!current_user_can($type, $id)) {
									$ok = false;
									break;
								}
							}
							if (!empty($ok)) {
								// callback function for update post meta
								call_user_func($metaBox[7], $id);
								// update post meta
								foreach ($metaBox[8] as $metaKey) {
									$this->updateMetaBox($id, $metaKey, false);
								}
								// update post meta with checkbox
								foreach ($metaBox[9] as $metaKey) {
									$this->updateMetaBox($id, $metaKey, true);
								}
							}
						}
					}
				}
			}
		}
	}
}

?>