<?php

/**
 * sitemap.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html
 * @copyright Copyright (c) 2013 Dominik Kocuj
 * @package kocuj_sitemap
 */

// security
if (basename($_SERVER['SCRIPT_FILENAME']) == basename(__FILE__)) {
	die ('Please do not load this page directly. Thanks!');
}

/**
 * Plugin sitemap class
 *
 * @access public
 */
class KocujSitemapPluginSitemap {
	/**
	 * Singleton instance
	 *
	 * @access private
	 * @var object
	 */
	private static $instance = NULL;

	/**
	 * Constructor
	 *
	 * @access private
	 * @return void
	 */
	private function __construct() {
		// add shortcode
		add_shortcode('KocujSitemap', array($this, 'shortcodeSitemap'));
	}

	/**
	 * Clone
	 *
	 * @access private
	 * @return void
	 */
	private function __clone() {
	}

	/**
	 * Get singleton instance
	 *
	 * @access public
	 * @return object Singleton instance
	 */
	public static function getInstance() {
		// optionally create new instance
		if (!self::$instance) {
			self::$instance = new KocujSitemapPluginSitemap();
		}
		// exit
		return self::$instance;
	}

	/**
	 * Add posts from category to sitemap
	 *
	 * @access private
	 * @param int $parentId Category parent id
	 * @param bool &$first It is first position in list (true) or not (false)
	 * @return string Sitemap string
	 */
	private function addSitemapPostsFromCategory($parentId, &$first) {
		// initialize
		$output = '';
		// show start of list
		$output .= '<ul>';
		// get posts
		if (!empty($parentId)) {
			$posts = KocujSitemapPluginInternalCacheInterface::getClass()->get_posts(array(
				'category'    => $parentId,
				'orderby'     => 'post_date',
				'order'       => 'DESC',
				'post_type'   => 'post',
				'post_status' => 'publish',
				'numberposts' => -1,
			));
			if (!empty($posts)) {
				foreach ($posts as $post) {
					$output .= '<li>';
					$output .= '<a href="'.esc_url(get_permalink($post->ID)).'">'.
						apply_filters('kocujsitemap_linktitle', $post->post_title).'</a>';
					$output .= '</li>';
				}
			}
		}
		// get categories
		$categories = get_categories(array(
			'type'       => 'post',
			'parent'     => $parentId,
			'orderby'    => 'name',
			'order'      => 'ASC',
			'hide_empty' => 1,
			'taxonomy'   => 'category',
		));
		if (!empty($categories)) {
			foreach ($categories as $category) {
				$classText = '';
				$first_class = apply_filters('kocujsitemap_firstclass', '');
				if ((empty($parentId)) && (!empty($first))) {
					$add = '';
					if (!empty($first_class)) {
						$add = ' '.$first_class;
					}
					$classText = ' class="kocujsitemapfirst'.$add.'"';
					$first = false;
				}
				$output .= '<li'.$classText.'>';
				$output .= '<a href="'.esc_url(get_category_link($category->cat_ID)).'">'.
					apply_filters('kocujsitemap_linktitle', $category->name).'</a>';
				$output .= $this->addSitemapPostsFromCategory($category->cat_ID, $first);
				$output .= '</li>';
			}
		}
		// show end of list
		$output .= '</ul>';
		// exit
		return $output;
	}

	/**
	 * Add pages from parent to sitemap
	 *
	 * @access private
	 * @param int $parentId Page parent id
	 * @param bool &$first It is first position in list (true) or not (false)
	 * @return string Sitemap string
	 */
	private function addSitemapPagesFromParent($parentId, &$first) {
		// initialize
		$output = '';
		// show start of list
		if (empty($parentId)) {
			$output .= '<ul>';
		}
		// get pages
		$pages = KocujSitemapPluginInternalCacheInterface::getClass()->get_pages(array(
			'parent'       => $parentId,
			'sort_column'  => 'menu_order',
			'sord_order'   => 'ASC',
			'post_type'    => 'page',
			'post_status'  => 'publish',
			'hierarchical' => 0,
			'number'       => '',
		));
		if (!empty($pages)) {
			if (!empty($parentId)) {
				$output .= '<ul>';
			}
			foreach ($pages as $page) {
				$output .= '<li>';
				$output .= '<a href="'.esc_url(get_permalink($page->ID)).'">'.
					apply_filters('kocujsitemap_linktitle', $page->post_title).'</a>';
				$output .= $this->addSitemapPagesFromParent($page->ID, $first);
				$output .= '</li>';
			}
			if (!empty($parentId)) {
				$output .= '</ul>';
			}
		}
		// show end of list
		if (empty($parentId)) {
			$output .= '</ul>';
		}
		// exit
		return $output;
	}

	/**
	 * Add pages to sitemap
	 *
	 * @access private
	 * @param string $text Input text
	 * @param bool &$first It is first position in list (true) or not (false)
	 * @return string Output text
	 */
	private function addPages($text, &$first) {
		// declare global
		global $classKocujSitemapAdmin;
		// add pages to sitemap
		$value = $classKocujSitemapAdmin->getConfigClass()->getOption('DisplayPages');
		if (!empty($value)) {
			$text .= $this->addSitemapPagesFromParent(0, $first);
		}
		// exit
		return $text;
	}

	/**
	 * Add posts to sitemap
	 *
	 * @access private
	 * @param string $text Input text
	 * @param bool &$first It is first position in list (true) or not (false)
	 * @return string Output text
	 */
	private function addPosts($text, &$first) {
		// declare global
		global $classKocujSitemapAdmin;
		// add posts to sitemap
		$value = $classKocujSitemapAdmin->getConfigClass()->getOption('DisplayPosts');
		if (!empty($value)) {
			$text .= $this->addSitemapPostsFromCategory(0, $first);
		}
		// exit
		return $text;
	}

	/**
	 * Add menus to sitemap
	 *
	 * @access private
	 * @param string $text Input text
	 * @param bool &$first It is first position in list (true) or not (false)
	 * @return string Output text
	 */
	private function addMenus($text, &$first) {
		// add menus to sitemap
		global $wp_version;
		if (version_compare($wp_version, '3.0.0', '>=')) {
			// declare global
			global $classKocujSitemapAdmin;
			// add menus to sitemap
			$value = $classKocujSitemapAdmin->getConfigClass()->getOption('DisplayMenus');
			if (!empty($value)) {
				$menus = $classKocujSitemapAdmin->getConfigClass()->getOption('Menus');
				if (empty($menus)) {
					$menus = apply_filters('kocujsitemap_defaultmenus', array());
				}
				if (!empty($menus)) {
					foreach ($menus as $menu) {
						$exists = wp_get_nav_menu_object($menu);
						if (!empty($exists)) {
							$walker = new KocujSitemapPluginMenu($first);
							$text .= wp_nav_menu(array(
								'menu'           => $menu,
								'menu_class'     => 'kocujsitemap-menu',
								'menu_id'        => 'kocujsitemap-menu',
								'theme_location' => '',
								'container'      => false,
								'echo'           => false,
								'fallback_cb'    => false,
								'walker'         => $walker,
							));
							$first = $walker->getFirst();
						}
					}
				}
			}
		}
		// exit
		return $text;
	}

	/**
	 * Create sitemap
	 *
	 * @access public
	 * @return array Sitemap data
	 */
	public function createSitemap() {
		// declare global
		global $classKocujSitemapAdmin;
		// initialize
		$output = '';
		$first = true;
		$titlepos = -1;
		// show homepage link
		$value = $classKocujSitemapAdmin->getConfigClass()->getOption('LinkToMainSite');
		if (!empty($value)) {
			$classText = '';
			$first_class = apply_filters('kocujsitemap_firstclass', '');
			if (!empty($first)) {
				$add = '';
				if (!empty($first_class)) {
					$add = ' '.$first_class;
				}
				$classText = ' class="kocujsitemapfirst'.$add.'"';
				$first = false;
			}
			if (function_exists('get_home_url')) {
				$homeURL = KocujSitemapPluginInternalCacheInterface::getClass()->get_home_url();
			} else {
				$homeURL = KocujSitemapPluginInternalCacheInterface::getClass()->get_bloginfo('home');
			}
			$output .= '<ul><li'.$classText.'><a href="'.esc_url($homeURL).'">';
			$titlepos = strlen($output);
			$output .= '</a></li></ul>';
		}
		// get elements
		$order = $classKocujSitemapAdmin->getConfigClass()->getOption('Order');
		if (strlen($order) > 0) {
			for ($z=0; $z<strlen($order); $z++) {
				switch ($order[$z]) {
					case 'G':
						$output = $this->addPages($output, $first);
					break;
					case 'M':
						$output = $this->addMenus($output, $first);
					break;
					case 'P':
						$output = $this->addPosts($output, $first);
					break;
				}
			}
		}
		// integrate lists
		$output = preg_replace('/<\/([u|U])([l|L])><([u|U])([l|L])(.*?)>/s', '', $output);
		// remove EOL
		$output = str_replace(array("\r\n", "\n", "\r"), '', $output);
		// prepare array
		$array = array(
			'output'   => $output,
			'titlepos' => $titlepos,
		);
		// exit
		return $array;
	}

	/**
	 * Sitemap shortcode
	 *
	 * @access public
	 * @param array $args Arguments
	 * @return string Parsed output
	 */
	public function shortcodeSitemap($args) {
		// get arguments
		$default_title = apply_filters('kocujsitemap_defaulttitle', KocujSitemapPluginInternalCacheInterface::getClass()->get_bloginfo('name'));
		$default_class = apply_filters('kocujsitemap_defaultclass', '');
		extract(shortcode_atts(array(
			'title' => $default_title,
			'class' => $default_class,
		), $args));
		// exit
		return $this->getSitemap($title, $class);
	}

	/**
	 * Get sitemap to display
	 *
	 * @access public
	 * @param string $title Sitemap title - default: empty
	 * @param string $class Sitemap class - default: empty
	 * @return string Sitemap to display
	 */
	public function getSitemap($title = '', $class = '') {
		// declare global
		global $classKocujSitemapAdmin;
		// get arguments
		if (empty($title)) {
			$title = apply_filters('kocujsitemap_defaulttitle', KocujSitemapPluginInternalCacheInterface::getClass()->get_bloginfo('name'));
		}
		if (empty($class)) {
			$class = apply_filters('kocujsitemap_defaultclass', '');
		}
		$title = apply_filters('kocujsitemap_linktitle', $title);
		// initialize
		$output = '';
		// load cache
		if (!KocujSitemapPluginCache::getInstance()->loadCache($title, $output)) {
			KocujSitemapPluginCache::getInstance()->createCache();
			if (!KocujSitemapPluginCache::getInstance()->loadCache($title, $output)) {
				return '';
			}
		}
		// optionally set footer
		$footer = '';
		$value = $classKocujSitemapAdmin->getConfigClass()->getOption('PoweredBy');
		if (!empty($value)) {
			$footer = '<div class="kocujsitemapfooter">'.sprintf(__('Powered by %s plugin', 'kocuj-sitemap'), '<a href="http://kocujsitemap.wpplugin.kocuj.pl/" rel="external">Kocuj Sitemap</a>').'</div>';
		}
		// show begin and end
		if (!empty($output)) {
			$classText = '';
			if (!empty($class)) {
				$classText = ' '.$class;
			}
			$html5 = $classKocujSitemapAdmin->getConfigClass()->getOption('UseHTML5');
			if (!empty($html5)) {
				$tagBegin = KocujSitemapPluginHTML5::getInstance()->getTagNavBegin('', 'kocujsitemap'.$classText);
				$tagEnd = KocujSitemapPluginHTML5::getInstance()->getTagNavEnd();
			} else {
				$tagBegin = '<div class="kocujsitemap'.$classText.'">';
				$tagEnd = '</div>';
			}
			$output = $tagBegin.$output.$footer.$tagEnd;
		}
		// exit
		return $output;
	}
}

?>