<?php

/**
 * project.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib\kocuj_api_add_thanks_client
 */

// set namespace
namespace KocujApiAddThanksClient\V1a\Classes;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Project class
 *
 * @access public
 */
final class Project {
	/**
	 * Project object for Kocuj Internal Lib
	 *
	 * @access private
	 * @var object
	 */
	private $projectKocujInternaLibObj = NULL;

	/**
	 * Kocuj API URL
	 *
	 * @access private
	 * @var string
	 */
	private $apiUrl = '';

	/**
	 * Kocuj API login
	 *
	 * @access private
	 * @var string
	 */
	private $apiLogin = '';

	/**
	 * Kocuj API password
	 *
	 * @access private
	 * @var string
	 */
	private $apiPassword = '';

	/**
	 * List of websites with API data
	 *
	 * @access private
	 * @var array
	 */
	private $apiWebsites = array();

	/**
	 * Classes list for classes with strings which implement \KocujApiAddThanksClient\V1a\Interfaces\Strings interface
	 *
	 * @access private
	 * @var array
	 */
	private $stringsClassesList = array();

	/**
	 * Classes list for Kocuj API Add Thanks Client which has been replaced by child classed
	 *
	 * @access private
	 * @var array
	 */
	private $libClassesList = array();

	/**
	 * Objects with instances of classes for Kocuj API Add Thanks Client
	 *
	 * @access private
	 * @var array
	 */
	private $libObjects = array();

	/**
	 * Constructor
	 *
	 * @access public
	 * @param object $projectKocujInternaLibObj Project object for Kocuj Internal Lib
	 * @param string $apiUrl Kocuj API URL; should be set without protocol, because "http://" is automatically added to beginning
	 * @param string $apiLogin Kocuj API login
	 * @param string $apiPassword Kocuj API password
	 * @param array $apiWebsites List of websites with API data
	 * @param array $stringsClasses Classes list for classes with strings which implement \KocujApiAddThanksClient\V1a\Interfaces\Strings interface; if some keys are empty or does not exist, the default classes which returns only empty strings will be used for these keys - default: empty
	 * @param array $libClasses Classes list for Kocuj API Add Thanks Client which has been replaced by child classed; if some keys are empty or does not exist, the default classes will be used for these keys - default: empty
	 * @return void
	 */
	public function __construct($projectKocujInternaLibObj, $apiUrl, $apiLogin, $apiPassword, array $apiWebsites = array(), array $stringsClasses = array(), array $libClasses = array()) {
		// set Kocuj Internal Lib project object
		if (!($projectKocujInternaLibObj instanceof \KocujInternalLib\V1a\Classes\Project)) {
			throw new \KocujApiAddThanksClient\V1a\Classes\Exception(\KocujApiAddThanksClient\V1a\Classes\ExceptionCode::OBJECT_IS_NOT_KOCUJ_INTERNAL_LIB_PROJECT, __FILE__, __LINE__);
		}
		$this->projectKocujInternaLibObj = $projectKocujInternaLibObj;
		// set API data
		$this->apiUrl = 'http://'.$apiUrl;
		$this->apiLogin = $apiLogin;
		$this->apiPassword = $apiPassword;
		$this->apiWebsites = $apiWebsites;
		// set strings classes
		$this->stringsClassesList = $stringsClasses;
		// set classes
		$this->libClassesList = $libClasses;
		// initialize objects required in administration panel
		if ((is_admin()) || (is_network_admin())) {
			$this->getObj('backend_add-thanks');
			if ((!$this->getObj('backend_add-thanks')->getThanksAddedOptionValue()) && (!\KocujInternalLib\V1a\Classes\Helper::getInstance()->checkUserIPLocal())) {
				$this->getObj('backend_add-thanks-ajax');
				$this->getObj('backend_page-about-add-thanks');
				$this->getObj('backend_review-message-add-thanks');
			}
		}
	}

	/**
	 * Get project object for Kocuj Internal Lib
	 *
	 * @access public
	 * @return object Project object for Kocuj Internal Lib
	 */
	public function getProjectKocujInternalLibObj() {
		// exit
		return $this->projectKocujInternaLibObj;
	}

	/**
	 * Get Kocuj API URL
	 *
	 * @access public
	 * @return string Kocuj API URL
	 */
	public function getApiUrl() {
		// exit
		return $this->apiUrl;
	}

	/**
	 * Get Kocuj API login
	 *
	 * @access public
	 * @return string Kocuj API login
	 */
	public function getApiLogin() {
		// exit
		return $this->apiLogin;
	}

	/**
	 * Get Kocuj API password
	 *
	 * @access public
	 * @return string Kocuj API password
	 */
	public function getApiPassword() {
		// exit
		return $this->apiPassword;
	}

	/**
	 * Get list of websites with API data
	 *
	 * @access public
	 * @return array List of websites with API data
	 */
	public function getApiWebsites() {
		// exit
		return $this->apiWebsites;
	}

	/**
	 * Get object of class type which implements \KocujApiAddThanksClient\V1a\Interfaces\Strings interface
	 *
	 * @access public
	 * @param string $type Class type
	 * @return object Object of class type which implements \KocujApiAddThanksClient\V1a\Interfaces\Strings interface
	 */
	public function getStringsObj($type) {
		// optionally create instance of class
		if (isset($this->stringsClassesList[$type])) {
			$interfaces = class_implements($this->stringsClassesList[$type]);
			if (in_array('KocujInternalLib\\V1a\\Interfaces\\Strings', $interfaces)) {
				return call_user_func(array($this->stringsClassesList[$type], 'getInstance'));
			}
		}
		// exit
		return \KocujInternalLib\V1a\Classes\StringsEmpty::getInstance();
	}

	/**
	 * Get object of class type from Kocuj API Add Thanks Client
	 *
	 * @access public
	 * @param string $type Class type
	 * @return object Object of class type from Kocuj API Add Thanks Client
	 */
	public function getObj($type) {
		// optionally initialize object
		if (!isset($this->libObjects[$type])) {
			if (!isset($this->libClassesList[$type])) {
				$div = explode('_', $type);
				if (!empty($div)) {
					foreach ($div as $key => $val) {
						$div[$key] = preg_replace_callback('/\\-([a-z])/', function($matches) {
							return strtoupper($matches[1]);
						}, $val);
						$div[$key] = strtoupper($div[$key][0]).substr($div[$key], 1);
					}
					$className = '\\KocujApiAddThanksClient\\V1a\\Classes\\Project\\'.implode('\\', $div);
				}
			} else {
				$className = $this->libClassesList[$type];
			}
			$this->libObjects[$type] = (is_subclass_of($className, '\KocujInternalLib\V1a\Classes\ProjectObject')) ?
				new $className($this) :
				NULL;
		}
		// exit
		return $this->libObjects[$type];
	}
}
