<?php

/**
 * helper.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujInternalLib\V1a\Classes;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Helper class
 *
 * @access public
 */
final class Helper {
	/**
	 * Singleton instance
	 *
	 * @access private
	 * @var object
	 */
	private static $instance = NULL;

	/**
	 * Script URL is remembered in $this->scriptUrl property (true) or not and should be updated (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $scriptUrlRemember = false;

	/**
	 * Remembered script URL
	 *
	 * @access private
	 * @var string
	 */
	private $scriptUrl = '';

	/**
	 * Constructor
	 *
	 * @access private
	 * @return void
	 */
	private function __construct() {
	}

	/**
	 * Disable cloning of object
	 *
	 * @access private
	 * @return void
	 */
	private function __clone() {
	}

	/**
	 * Get singleton instance
	 *
	 * @access public
	 * @return object Singleton instance
	 */
	public static function getInstance() {
		// optionally create new instance
		if (!self::$instance) {
			self::$instance = new \KocujInternalLib\V1a\Classes\Helper();
		}
		// exit
		return self::$instance;
	}

	/**
	 * Get prefix for some names in library
	 *
	 * @access public
	 * @return string Prefix
	 */
	public function getPrefix() {
		// exit
		return 'kocujinternallib'.\KocujInternalLib\V1a\Classes\Version::getInstance()->getVersionInternal();
	}

	/**
	 * Calculate the highest priority of WordPress filter or action
	 *
	 * @access public
	 * @param string $name WordPress filter or action name
	 * @return int Calculated highest priority of WordPress filter or action
	 */
	public function calculateMaxPriority($name) {
		// calculate new priority
		global $wp_filter;
		$newPriority = ((isset($wp_filter[$name])) && (!empty($wp_filter[$name]))) ?
			max(array_keys($wp_filter[$name]))+1 :
			1;
		// exit
		return ($newPriority < 999) ?
			999 :
			$newPriority;
	}

	/**
	 * Get real URL; it is similar to "realpath()" but it parse URL-s
	 *
	 * @access public
	 * @param string $url URL to parse
	 * @return string Parsed URL
	 */
	public function getRealUrl($url) {
		// get real URL
		$url = explode('/', $url);
		$keys = array_keys($url, '..');
		if (!empty($keys)) {
			foreach ($keys as $keyPos => $key) {
				array_splice($url, $key-($keyPos*2+1), 2);
			}
		}
		$url = implode('/', $url);
		$url = str_replace('./', '', $url);
		// exit
		return $url;
	}

	/**
	 * Check if IP address is in local network
	 *
	 * @access public
	 * @param string $ip IP to check
	 * @return bool IP is in local network (true) or not (false)
	 */
	public function checkIPLocal($ip) {
		// set list of IP prefixes in local network
		$localIP = array(
			'10.',
			'172.16.',
			'172.17.',
			'172.18.',
			'172.19.',
			'172.20.',
			'172.21.',
			'172.22.',
			'172.23.',
			'172.24.',
			'172.25.',
			'172.26.',
			'172.27.',
			'172.28.',
			'172.29.',
			'172.30.',
			'172.31.',
			'192.168.',
		);
		// check if IP is in local network
		foreach ($localIP as $oneIP) {
			if (substr($ip, 0, strlen($oneIP)) === $oneIP) {
				return true;
			}
		}
		// exit
		return false;
	}

	/**
	 * Check if user IP address is in local network
	 *
	 * @access public
	 * @return bool User IP is in local network (true) or not (false)
	 */
	public function checkUserIPLocal() {
		// exit
		return $this->checkIPLocal((isset($_SERVER['REMOTE_ADDR'])) ?
			$_SERVER['REMOTE_ADDR'] :
			''
		);
	}

	/**
	 * Check if server IP address is in local network
	 *
	 * @access public
	 * @return bool Server IP is in local network (true) or not (false)
	 */
	public function checkServerIPLocal() {
		// exit
		return $this->checkIPLocal((isset($_SERVER['SERVER_ADDR'])) ?
			$_SERVER['SERVER_ADDR'] :
			''
		);
	}

	/**
	 * Get protocol for current URL
	 *
	 * @access public
	 * @return string Protocol for current URL
	 */
	public function getUrlProtocol() {
		// check if forwarded
		if ((isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] == 'https')) {
			$_SERVER['HTTPS'] = 'on';
		}
		// exit
		return (((isset($_SERVER['HTTPS'])) && ($_SERVER['HTTPS'] !== 'off')) || ((isset($_SERVER['SERVER_PORT'])) && ($_SERVER['SERVER_PORT'] === '443'))) ?
			'https' :
			'http';
	}

	/**
	 * Get script URL
	 *
	 * @access public
	 * @return string Script URL
	 */
	public function getScriptUrl() {
		// optionally return remembered script URL
		if ($this->scriptUrlRemember) {
			return $this->scriptUrl;
		}
		// initialize
		$scriptURL = '';
		// get script URL
		if ((isset($_SERVER['SCRIPT_URL'])) && (isset($_SERVER['SCRIPT_URL'][0]) /* strlen($_SERVER['SCRIPT_URL']) > 0 */ )) {
			$scriptURL = $_SERVER['SCRIPT_URL'];
		} else {
			if ((isset($_SERVER['REDIRECT_URL'])) && (isset($_SERVER['REDIRECT_URL'][0]) /* strlen($_SERVER['REDIRECT_URL']) > 0 */ )) {
				$scriptURL = $_SERVER['REDIRECT_URL'];
			} else {
				if ((isset($_SERVER['REQUEST_URI'])) && (isset($_SERVER['REQUEST_URI'][0]) /* strlen($_SERVER['REQUEST_URI']) > 0 */ )) {
					$path = parse_url($_SERVER['REQUEST_URI']);
					$scriptURL = $path['path'];
				}
			}
		}
		// remember script URL
		if (!$this->scriptUrlRemember) {
			$this->scriptUrl = $scriptURL;
			$this->scriptUrlRemember = true;
		}
		// exit
		return $scriptURL;
	}

	/**
	 * Add option for network or site
	 *
	 * @access public
	 * @param string $optionName Option name
	 * @param array|bool|float|int|string $optionValue Option value
	 * @param bool $autoload Automatic loading of option (true) or not (false); works only if option is for site, not network - default: false
	 * @return void
	 */
	public function addOptionForNetworkOrSite($optionName, $optionValue, $autoload = false) {
		// add option
		if (is_multisite()) {
			add_site_option($optionName, $optionValue);
		} else {
			add_option($optionName, $optionValue, '', $autoload);
		}
	}

	/**
	 * Update option for network or site
	 *
	 * @access public
	 * @param string $optionName Option name
	 * @param array|bool|float|int|string $optionValue Option value
	 * @param bool $autoload Automatic loading of option (true) or not (false); works only if option is for site, not network - default: false
	 * @return void
	 */
	public function updateOptionForNetworkOrSite($optionName, $optionValue, $autoload = false) {
		// update option
		if (is_multisite()) {
			update_site_option($optionName, $optionValue);
		} else {
			update_option($optionName, $optionValue, $autoload);
		}
	}

	/**
	 * Delete option for network or site
	 *
	 * @access public
	 * @param string $optionName Option name
	 * @return void
	 */
	public function deleteOptionForNetworkOrSite($optionName) {
		// update option
		if (is_multisite()) {
			delete_site_option($optionName);
		} else {
			delete_option($optionName);
		}
	}

	/**
	 * Get option value for network or site
	 *
	 * @access public
	 * @param string $optionName Option name
	 * @param array|bool|float|int|string $defaultOptionValue Default option value - default: false
	 * @return array|bool|float|int|string Option value
	 */
	public function getOptionForNetworkOrSite($optionName, $defaultOptionValue = false) {
		// exit
		return (is_multisite()) ?
			get_site_option($optionName, $defaultOptionValue) :
			get_option($optionName, $defaultOptionValue);
	}

	/**
	 * Add or update option for network or site
	 *
	 * @access public
	 * @param string $optionName Option name
	 * @param array|bool|float|int|string $optionValue Option value
	 * @param bool $autoload Automatic loading of option (true) or not (false); works only if option is for site, not network and if options is creating, not updating - default: false
	 * @return void
	 */
	public function addOrUpdateOptionForNetworkOrSite($optionName, $optionValue, $autoload = false) {
		// check if option exists
		$exists = $this->getOptionForNetworkOrSite($optionName);
		// add or update option
		if ($exists === false) {
			$this->addOptionForNetworkOrSite($optionName, $optionValue, $autoload);
		} else {
			$this->updateOptionForNetworkOrSite($optionName, $optionValue);
		}
	}

	/**
	 * Get HTML tag attributes
	 *
	 * @access private
	 * @param array $attr Additional attributes; there are available the following attributes: "class" (string type; class of HTML element), "id" (string type; id of HTML element), "style" (string type; style of HTML element), "styleclassfilter" (array type; it contains two fields: "projectobj" with object of class \KocujInternalLib\V1a\Classes\Project for current project and "filter" with filter name which will be added to project prefix)
	 * @return string HTML tag attributes
	 */
	private function getHTMLTagIdStyleClassAttributes(array $attr) {
		// parse style and class
		if (isset($attr['styleclassfilter'])) {
			$styleClassString = $attr['styleclassfilter']['projectobj']->getObj('project-helper')->applyFiltersForHTMLStyleAndClass($attr['styleclassfilter']['filter'], '', array(
				'defaultclass' => (isset($attr['class'])) ?
					$attr['class'] :
					'',
				'defaultstyle' => (isset($attr['style'])) ?
					$attr['style'] :
					'',
			));
		} else {
			$styleClassString = ((isset($attr['style'])) ?
					' style="'.$attr['style'].'"' :
					''
				).((isset($attr['class'])) ?
					' class="'.$attr['class'].'"' :
					''
				);
		}
		// exit
		return ((isset($attr['id'])) ?
				' id="'.$attr['id'].'"' :
				''
			).$styleClassString;
	}

	/**
	 * Get beginning of link anchor
	 *
	 * @access public
	 * @param string $link Link URL
	 * @param array $attr Additional attributes; there are available the following attributes: "class" (string type; class of HTML element), "external" (bool type; if true, link is external and will be opened in new tab), "externalwithouttarget" (bool type; if true, link is external, but there will be no "target" attribute; if there is also an "external" attribute in $attr array, the "external" will be used instead), "id" (string type; id of HTML element), "style" (string type; style of HTML element), "styleclassfilter" (array type; it contains two fields: "projectobj" with object of class \KocujInternalLib\V1a\Classes\Project for current project and "filter" with filter name which will be added to project prefix) - default: empty
	 * @return string Beginning of link anchor
	 */
	public function getLinkAnchorBegin($link, array $attr = array()) {
		// exit
		return '<a href="'.esc_url($link).'"'.((((isset($attr['external'])) && ($attr['external'])) || ((isset($attr['externalwithouttarget'])) && ($attr['externalwithouttarget']))) ?
				' rel="external"'.(((!isset($attr['externalwithouttarget'])) || (!$attr['externalwithouttarget'])) ?
						' target="_blank"' :
						''
					) :
				''
			).$this->getHTMLTagIdStyleClassAttributes($attr).'>';
	}

	/**
	 * Get ending of link anchor
	 *
	 * @access public
	 * @return string Ending of link anchor
	 */
	public function getLinkAnchorEnd() {
		// exit
		return '</a>';
	}

	/**
	 * Get link anchor
	 *
	 * @access public
	 * @param string $link Link URL
	 * @param string $name Link name; if empty, link URL will be used instead - default: empty
	 * @param array $attr Additional attributes; there are available the following attributes: "class" (string type; class of HTML element), "external" (bool type; if true, link is external and will be opened in new tab), "externalwithouttarget" (bool type; if true, link is external, but there will be no "target" attribute; if there is also an "external" attribute in $attr array, the "external" will be used instead), "id" (string type; id of HTML element), "style" (string type; style of HTML element), "styleclassfilter" (array type; it contains two fields: "projectobj" with object of class \KocujInternalLib\V1a\Classes\Project for current project and "filter" with filter name which will be added to project prefix) - default: empty
	 * @return string Link anchor
	 */
	public function getLinkAnchor($link, $name = '', array $attr = array()) {
		// exit
		return $this->getLinkAnchorBegin($link, $attr).((isset($name[0]) /* strlen($name) > 0 */ ) ?
				$name :
				((substr($link, 0, 2) === '//') ?
					$this->getUrlProtocol().':'.$link :
					$link
				)
			).$this->getLinkAnchorEnd();
	}

	/**
	 * Get e-mail link anchor
	 *
	 * @access public
	 * @param string $mail E-mail address
	 * @param array $attr Additional attributes; there are available the following attributes: "class" (string type; class of HTML element), "id" (string type; id of HTML element), "style" (string type; style of HTML element), "styleclassfilter" (array type; it contains two fields: "projectobj" with object of class \KocujInternalLib\V1a\Classes\Project for current project and "filter" with filter name which will be added to project prefix) - default: empty
	 * @return string E-mail link anchor
	 */
	public function getMailAnchor($mail, array $attr = array()) {
		// exit
		return '<a href="mailto:'.$mail.'"'.$this->getHTMLTagIdStyleClassAttributes($attr).'>'.$mail.'</a>';
	}

	/**
	 * Get HTML image
	 *
	 * @access public
	 * @param string $url Image URL
	 * @param array $attr Additional attributes; there are available the following attributes: "class" (string type; class of HTML element), "id" (string type; id of HTML element), "style" (string type; style of HTML element), "styleclassfilter" (array type; it contains two fields: "projectobj" with object of class \KocujInternalLib\V1a\Classes\Project for current project and "filter" with filter name which will be added to project prefix) - default: empty
	 * @return string HTML image
	 */
	public function getHTMLImage($url, array $attr = array()) {
		// add default style
		$style = 'border-width:0;border-style:none;';
		if (isset($attr['style'])) {
			$attr['style'] .= $style;
		} else {
			$attr['style'] = $style;
		}
		// exit
		return '<img src="'.esc_url($url).'" alt=""'.$this->getHTMLTagIdStyleClassAttributes($attr).'/>';
	}
}
