<?php

/**
 * version-check.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujInternalLib\V1a\Classes\Project\Backend;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Set version check class
 *
 * @access public
 */
class VersionCheck extends \KocujInternalLib\V1a\Classes\ProjectObject {
	/**
	 * Current version of project
	 *
	 * @access private
	 * @var string
	 */
	private $currentVersion = '';

	/**
	 * Update callback
	 *
	 * @access private
	 * @var array|string
	 */
	private $updateCallback = '';

	/**
	 * Update informations
	 *
	 * @access private
	 * @var array
	 */
	private $updateInformation = array();

	/**
	 * Information is set to be displayed  (true) or information will not be displayed (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $informationDisplay = false;

	/**
	 * Data for information to display
	 *
	 * @access private
	 * @var array
	 */
	private $informationDisplayData = array();

	/**
	 * Constructor
	 *
	 * @access public
	 * @param object $projectObj \KocujInternalLib\V1a\Classes\Project object for current project
	 * @return void
	 */
	public function __construct($projectObj) {
		// execute parent constructor
		parent::__construct($projectObj);
		// add uninstall action
		add_action($this->getProjectObj()->getObj('project-helper')->getPrefix().'__plugin_uninstall', array($this, 'actionPluginUninstall'), 1);
		// check if it is backend
		if ((!is_admin()) && (!is_network_admin())) {
			return;
		}
		// add actions
		add_action('admin_init', array($this, 'actionAdminInit'), \KocujInternalLib\V1a\Classes\Helper::getInstance()->calculateMaxPriority('admin_init'));
		add_action('admin_print_footer_scripts', array($this, 'actionPrintFooterScripts'), \KocujInternalLib\V1a\Classes\Helper::getInstance()->calculateMaxPriority('admin_print_footer_scripts'));
		// add "helper.js"
		\KocujInternalLib\V1a\Classes\JsHelper::getInstance()->addHelperJS();
		// add "modal.js"
		\KocujInternalLib\V1a\Classes\JsHelper::getInstance()->addModalJS();
	}

	/**
	 * Get version option name
	 *
	 * @access public
	 * @return string Version option name
	 */
	public function getVersionOptionName() {
		// exit
		return $this->getProjectObj()->getSettingInternalName().'__version';
	}

	/**
	 * Get version number from database
	 *
	 * @access public
	 * @return string Version number from database
	 */
	public function getVersionOptionValue() {
		// exit
		return \KocujInternalLib\V1a\Classes\Helper::getInstance()->getOptionForNetworkOrSite($this->getVersionOptionName());
	}

	/**
	 * Get request for update information option name
	 *
	 * @access public
	 * @return string Request for update information option name
	 */
	public function getRequestUpdateInfoOptionName() {
		// exit
		return $this->getProjectObj()->getSettingInternalName().'__req_update_info';
	}

	/**
	 * Get request for update information from database
	 *
	 * @access public
	 * @return array Request for update information from database
	 */
	public function getRequestUpdateInfoValue() {
		// exit
		return \KocujInternalLib\V1a\Classes\Helper::getInstance()->getOptionForNetworkOrSite($this->getRequestUpdateInfoOptionName());
	}

	/**
	 * Set current version of project
	 *
	 * @access public
	 * @param string $version Current version of project
	 * @return void
	 */
	public function setCurrentVersion($currentVersion) {
		// set current version of project
		$this->currentVersion = $currentVersion;
	}

	/**
	 * Get current version of project
	 *
	 * @access public
	 * @return string Current version of project
	 */
	public function getCurrentVersion() {
		// exit
		return $this->currentVersion;
	}

	/**
	 * Set update callback
	 *
	 * @access public
	 * @param array|string $updateCallback Callback function or method for update
	 * @return void
	 */
	public function setUpdateCallback($updateCallback) {
		// set update callback
		$this->updateCallback = $updateCallback;
	}

	/**
	 * Get update callback
	 *
	 * @access public
	 * @return array|string Update callback
	 */
	public function getUpdateCallback() {
		// exit
		return $this->updateCallback;
	}

	/**
	 * Add information for update from the selected version
	 *
	 * @access public
	 * @param string $fromVersion Version from which update will show information; it can be only fragment of version number, but then it must be ended with dot character ("."), for example, "1.", "1.0.", etc.
	 * @param string $information Information to display
	 * @return void
	 */
	public function addUpdateInformation($fromVersion, $information) {
		// set update information
		$this->updateInformation[$fromVersion] = $information;
	}

	/**
	 * Get information for update from the selected version
	 *
	 * @access public
	 * @param string $fromVersion Version from which update will show information; it can be only fragment of version number, but then it must be ended with dot character ("."), for example, "1.", "1.0.", etc.
	 * @return bool|string Information for the selected version or false if not exists
	 */
	public function getInformation($fromVersion) {
		// get information for version
		if (!empty($this->updateInformation)) {
			foreach ($this->updateInformation as $key => $val) {
				if ($key[strlen($key)-1] !== '.') {
					if ($key === $fromVersion) {
						return $val;
					}
				} else {
					if ($key === substr($fromVersion, 0, strlen($key))) {
						return $val;
					}
				}
			}
		}
		// exit
		return false;
	}

	/**
	 * Action for checking version of project
	 *
	 * @access public
	 * @return void
	 */
	public function actionAdminInit() {
		// check if current version of project is set
		if (!isset($this->currentVersion[0]) /* strlen($this->currentVersion) === 0 */ ) {
			return;
		}
		// optionally show information about this update
		$value = $this->getRequestUpdateInfoValue();
		if (($value !== false) && ((current_user_can('manage_network')) || (current_user_can('manage_options')))) {
			if ($this->getInformation($value['from']) !== false) {
				wp_enqueue_script(\KocujInternalLib\V1a\Classes\Helper::getInstance()->getPrefix().'-backend-version-check', \KocujInternalLib\V1a\Classes\LibUrls::getInstance()->getJavaScriptUrl().'/backend/version-check.js', array(
					'jquery',
					\KocujInternalLib\V1a\Classes\Helper::getInstance()->getPrefix().'-helper',
					\KocujInternalLib\V1a\Classes\Helper::getInstance()->getPrefix().'-modal',
				), \KocujInternalLib\V1a\Classes\Version::getInstance()->getVersion(), true);
				wp_localize_script(\KocujInternalLib\V1a\Classes\Helper::getInstance()->getPrefix().'-backend-version-check', 'kocujInternalLibV1aBackendVersionCheckSettings', array(
					'prefix'           => \KocujInternalLib\V1a\Classes\Helper::getInstance()->getPrefix(),
					'security'         => wp_create_nonce(\KocujInternalLib\V1a\Classes\Helper::getInstance()->getPrefix().'__version_info'),
					'textLoading'      => $this->getProjectObj()->getStringsObj('backend_version-check')->getString('ACTION_ADMIN_INIT_SCRIPT_LOADING'),
					'textLoadingError' => $this->getProjectObj()->getStringsObj('backend_version-check')->getString('ACTION_ADMIN_INIT_SCRIPT_LOADING_ERROR'),
					'textTitlePlugin'  => $this->getProjectObj()->getStringsObj('backend_version-check')->getString('ACTION_ADMIN_INIT_SCRIPT_TITLE_PLUGIN'),
					'textTitleTheme'   => $this->getProjectObj()->getStringsObj('backend_version-check')->getString('ACTION_ADMIN_INIT_SCRIPT_TITLE_THEME'),
				));
				// set information to display
				$this->informationDisplay = true;
				// set data for information to display
				$this->informationDisplayData = $value;
			} else {
				// remove request for information about update
				\KocujInternalLib\V1a\Classes\Helper::getInstance()->deleteOptionForNetworkOrSite($this->getRequestUpdateInfoOptionName());
			}
		}
		// check version of project
		$version = $this->getVersionOptionValue();
		if ($version === false) {
			// update version of project in database
			\KocujInternalLib\V1a\Classes\Helper::getInstance()->addOrUpdateOptionForNetworkOrSite($this->getVersionOptionName(), $this->currentVersion);
		} else {
			if (version_compare($version, $this->currentVersion, '<')) {
				// execute update callback
				if ($this->updateCallback !== '') {
					call_user_func_array($this->updateCallback, array(
						$version,
						$this->currentVersion,
					));
				}
				// save request for information about this update
				$value = array(
					'from' => $version,
					'to'   => $this->currentVersion,
				);
				\KocujInternalLib\V1a\Classes\Helper::getInstance()->addOrUpdateOptionForNetworkOrSite($this->getRequestUpdateInfoOptionName(), $value);
				// update version of project in database
				\KocujInternalLib\V1a\Classes\Helper::getInstance()->addOrUpdateOptionForNetworkOrSite($this->getVersionOptionName(), $this->currentVersion);
			}
		}
	}

	/**
	 * Action for displaying update information
	 *
	 * @access public
	 * @return void
	 */
	public function actionPrintFooterScripts() {
		// initialize information script
		if ($this->informationDisplay) {
			?>
				<script type="text/javascript">
				/* <![CDATA[ */
					(function($) {
						$(document).ready(function() {
							kocujInternalLibV1aBackendVersionCheck.addProject('<?php echo esc_js($this->getProjectObj()->getSettingInternalName()); ?>', '<?php echo esc_js($this->getProjectObj()->getSettingTitleOriginal()); ?>', '<?php echo ($this->getProjectObj()->getSettingType() === \KocujInternalLib\V1a\Enums\ProjectType::PLUGIN) ?
									'plugin' :
									'theme'
								?>', '<?php echo esc_js($this->informationDisplayData['from']); ?>', '<?php echo esc_js($this->informationDisplayData['to']); ?>');
							kocujInternalLibV1aBackendVersionCheck.showVersionCheck('<?php echo esc_js($this->getProjectObj()->getSettingInternalName()); ?>');
						});
					}(jQuery));
				/* ]]> */
				</script>
			<?php
		}
	}

	/**
	 * Action for uninstall plugin
	 *
	 * @access public
	 * @return void
	 */
	public function actionPluginUninstall() {
		// remove option
		\KocujInternalLib\V1a\Classes\Helper::getInstance()->deleteOptionForNetworkOrSite($this->getRequestUpdateInfoOptionName());
		\KocujInternalLib\V1a\Classes\Helper::getInstance()->deleteOptionForNetworkOrSite($this->getVersionOptionName());
	}
}
