<?php

/**
 * project.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujInternalLib\V1a\Classes;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Project class
 *
 * @access public
 */
final class Project {
	/**
	 * Project settings
	 *
	 * @access private
	 * @var array
	 */
	private $settings = array();

	/**
	 * Classes list for classes with strings which implement \KocujInternalLib\V1a\Interfaces\Strings interface
	 *
	 * @access private
	 * @var array
	 */
	private $stringsClassesList = array();

	/**
	 * Classes list for Kocuj Internal Lib which has been replaced by child classed
	 *
	 * @access private
	 * @var array
	 */
	private $libClassesList = array();

	/**
	 * Objects with instances of classes for Kocuj Internal Lib
	 *
	 * @access private
	 * @var array
	 */
	private $libObjects = array();

	/**
	 * Constructor
	 *
	 * @access public
	 * @param array $settings Project settings; project settings have the following fields: "licensename" (string type; name of license, for example, "GPL"), "mainfilename" (string type; main filename in project), "title" (string type; title of project), "type" (int type; type of project; it should be one of the constants from \KocujInternalLib\V1a\Enums\ProjectType); there are also the following fields which can exists or not: "author" (array type; information about author of this project; it can contains the following fields: "email" with e-mail to author, "name" with author name, "url" with author homepage URL), "name" (string type; name of the project), "newschannels" (array type; information about news channels for this project; if can contains the following fields: "facebook" with URL to Facebook, "rss" with URL to RSS, "twitter" with URL to Twitter), "support" (array type; information about support; if can contains the following fields: "email" with e-mail address to support team, "url" with support URL such as forum or bug tracking system), "tellothers" (array type; URL-s to telling other about this project; if can contains the following fields: "facebook" with bool value set to true, "twitter" with text which should be added to Twitter), "titleoriginal" (string type; original title of project; should not be translated), "url" (string type; URL to website from the project), "version" (string type; version of the project)
	 * @param array $stringsClasses Classes list for classes with strings which implement \KocujInternalLib\V1a\Interfaces\Strings interface; if some keys are empty or does not exist, the default classes which returns only empty strings will be used for these keys - default: empty
	 * @param array $libClasses Classes list for Kocuj Internal Lib which has been replaced by child classed; if some keys are empty or does not exist, the default classes will be used for these keys - default: empty
	 * @return void
	 */
	public function __construct(array $settings, array $stringsClasses = array(), array $libClasses = array()) {
		// check if there are all required settings for project
		$reqSettings = array(
			'type',
			'mainfilename',
			'licensename',
			'title',
		);
		foreach ($reqSettings as $setting) {
			if (!isset($settings[$setting])) {
				throw new \KocujInternalLib\V1a\Classes\Exception(\KocujInternalLib\V1a\Classes\ExceptionCode::NO_REQUIRED_SETTING_DATA, __FILE__, __LINE__, $setting);
			}
		}
		// remember settings
		$this->settings = array(
			'type'            => (isset($settings['type'])) ?
				$settings['type'] :
				\KocujInternalLib\V1a\Enums\ProjectType::PLUGIN,
			'mainfilename'  => $settings['mainfilename'],
			'licensename'   => (isset($settings['licensename'])) ?
				$settings['licensename'] :
				'',
			'title'         => $settings['title'],
			'name'          => (isset($settings['name'])) ?
				$settings['name'] :
				$settings['title'],
			'version'       => (isset($settings['version'])) ?
				$settings['version'] :
				'',
			'url'           => (isset($settings['url'])) ?
				$settings['url'] :
				'',
			'titleoriginal' => (isset($settings['titleoriginal'])) ?
				$settings['titleoriginal'] :
				$settings['title'],
			'author'        => (isset($settings['author'])) ?
				$settings['author'] :
				array(),
			'support'       => (isset($settings['support'])) ?
				$settings['support'] :
				'',
			'newschannels'  => (isset($settings['newschannels'])) ?
				$settings['newschannels'] :
				array(),
			'tellothers'    => (isset($settings['tellothers'])) ?
				$settings['tellothers'] :
				array(),
		);
		$this->settings['internalname'] = str_replace(array(
			'-',
			'__',
		), '_', sanitize_title($this->settings['name']));
		if (isset($this->settings['internalname'][30]) /* strlen($this->settings['internalname']) > 30 */ ) {
			$this->settings['internalname'] = substr($this->settings['internalname'], 0, 30);
		}
		// set strings classes
		$this->stringsClassesList = $stringsClasses;
		// set classes
		$this->libClassesList = $libClasses;
		// initialize required objects
		$this->getObj('config');
		$this->getObj('admin');
		// initialize objects required in administration panel
		if ((is_admin()) || (is_network_admin())) {
			$this->getObj('backend_installation-date');
			$this->getObj('backend_license-ajax');
			$this->getObj('backend_review-message-ajax');
			$this->getObj('backend_version-check');
			$this->getObj('backend_version-check-ajax');
		}
	}

	/**
	 * Get project setting by id
	 *
	 * @access private
	 * @param string $id Settings id
	 * @return string Project setting
	 */
	private function getSetting($id) {
		// exit
		return (isset($this->settings[$id])) ?
			$this->settings[$id] :
			'';
	}

	/**
	 * Get project setting with type
	 *
	 * @access public
	 * @return string Project setting with type
	 */
	public function getSettingType() {
		// exit
		return $this->getSetting('type');
	}

	/**
	 * Get project setting with main filename
	 *
	 * @access public
	 * @return string Project setting with main filename
	 */
	public function getSettingMainFilename() {
		// exit
		return $this->getSetting('mainfilename');
	}

	/**
	 * Get project setting with title
	 *
	 * @access public
	 * @return string Project setting with title
	 */
	public function getSettingTitle() {
		// exit
		return $this->getSetting('title');
	}

	/**
	 * Get project setting with license name
	 *
	 * @access public
	 * @return string Project setting with license name
	 */
	public function getSettingLicenseName() {
		// exit
		return $this->getSetting('licensename');
	}

	/**
	 * Get project setting with name
	 *
	 * @access public
	 * @return string Project setting with name
	 */
	public function getSettingName() {
		// exit
		return $this->getSetting('name');
	}

	/**
	 * Get project setting with internal name
	 *
	 * @access public
	 * @return string Project setting with internal name
	 */
	public function getSettingInternalName() {
		// exit
		return $this->getSetting('internalname');
	}

	/**
	 * Get project setting with version
	 *
	 * @access public
	 * @return string Project setting with version
	 */
	public function getSettingVersion() {
		// exit
		return $this->getSetting('version');
	}

	/**
	 * Get project setting with URL
	 *
	 * @access public
	 * @return string Project setting with URL
	 */
	public function getSettingUrl() {
		// exit
		return $this->getSetting('url');
	}

	/**
	 * Get project setting with original title
	 *
	 * @access public
	 * @return string Project setting with original title
	 */
	public function getSettingTitleOriginal() {
		// exit
		return $this->getSetting('titleoriginal');
	}

	/**
	 * Get project setting with information about author
	 *
	 * @access public
	 * @param string $type Data type; data type should be one of the following: "email" (e-mail to author), "name" (author name), "url" (author homepage URL)
	 * @return string Project setting with information about author
	 */
	public function getSettingAuthor($type) {
		// get author settings
		$author = $this->getSetting('author');
		if ($author === '') {
			return '';
		}
		// exit
		return (isset($author[$type])) ?
			$author[$type] :
			'';
	}

	/**
	 * Get project setting with information about support
	 *
	 * @access public
	 * @param string $type Data type; data type should be one of the following: "email" (e-mail address to support team), "url" (support URL such as forum or bug tracking system)
	 * @return string Project setting with information about support
	 */
	public function getSettingSupport($type) {
		// get support settings
		$support = $this->getSetting('support');
		if ($support === '') {
			return '';
		}
		// exit
		return (isset($support[$type])) ?
			$support[$type] :
			'';
	}

	/**
	 * Get project setting with news channel link
	 *
	 * @access public
	 * @param string $type Data type; data type should be one of the following: "facebook" (URL to Facebook), "rss" (URL to RSS), "twitter" (URL to Twitter)
	 * @return string Project setting with news channel link
	 */
	public function getSettingNewsChannel($type) {
		// get news channels settings
		$newsChannels = $this->getSetting('newschannels');
		if ($newsChannels === '') {
			return '';
		}
		// exit
		return (isset($newsChannels[$type])) ?
			$newsChannels[$type] :
			'';
	}

	/**
	 * Get project setting with telling others about project
	 *
	 * @access public
	 * @param string $type Data type; data type should be one of the following: "facebook", "twitter"
	 * @return bool Project setting with telling others about project
	 */
	public function getSettingSupportTellOthers($type) {
		// get telling others settings
		$tellOthers = $this->getSetting('tellothers');
		if ($tellOthers === '') {
			return '';
		}
		// get telling others URL
		switch ($type) {
			case 'facebook':
				$output = 'http://www.facebook.com/sharer/sharer.php?u='.urlencode($this->getSettingUrl());
				break;
			case 'twitter':
				$output = 'http://twitter.com/share?url='.urlencode($this->getSettingUrl()).'&text='.urlencode($tellOthers['twitter']);
				break;
			default:
				$output = '';
		}
		// exit
		return $output;
	}

	/**
	 * Get object of class type which implements \KocujInternalLib\V1a\Interfaces\Strings interface
	 *
	 * @access public
	 * @param string $type Class type
	 * @return object Object of class type which implements \KocujInternalLib\V1a\Interfaces\Strings interface
	 */
	public function getStringsObj($type) {
		// optionally create instance of class
		if (isset($this->stringsClassesList[$type])) {
			$interfaces = class_implements($this->stringsClassesList[$type]);
			if (in_array('KocujInternalLib\\V1a\\Interfaces\\Strings', $interfaces)) {
				return call_user_func(array($this->stringsClassesList[$type], 'getInstance'));
			}
		}
		// exit
		return \KocujInternalLib\V1a\Classes\StringsEmpty::getInstance();
	}

	/**
	 * Get object of class type from Kocuj Internal Lib
	 *
	 * @access public
	 * @param string $type Class type
	 * @return object Object of class type from Kocuj Internal Lib
	 */
	public function getObj($type) {
		// optionally initialize object
		if (!isset($this->libObjects[$type])) {
			if (!isset($this->libClassesList[$type])) {
				$div = explode('_', $type);
				if (!empty($div)) {
					foreach ($div as $key => $val) {
						$div[$key] = preg_replace_callback('/\\-([a-z])/', function($matches) {
							return strtoupper($matches[1]);
						}, $val);
						$div[$key] = strtoupper($div[$key][0]).substr($div[$key], 1);
					}
					$className = '\\KocujInternalLib\\V1a\\Classes\\Project\\'.implode('\\', $div);
				}
			} else {
				$className = $this->libClassesList[$type];
			}
			$this->libObjects[$type] = (is_subclass_of($className, '\KocujInternalLib\V1a\Classes\ProjectObject')) ?
				new $className($this) :
				NULL;
		}
		// exit
		return $this->libObjects[$type];
	}
}
