<?php

/**
 * message.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujInternalLib\V1a\Classes\Project\Backend;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Set message class
 *
 * @access public
 */
class Message extends \KocujInternalLib\V1a\Classes\ProjectObject {
	/**
	 * Messages
	 *
	 * @access private
	 * @var array
	 */
	private $messages = array();

	/**
	 * Constructor
	 *
	 * @access public
	 * @param object $projectObj \KocujInternalLib\V1a\Classes\Project object for current project
	 * @return void
	 */
	public function __construct($projectObj) {
		// execute parent constructor
		parent::__construct($projectObj);
		// check if it is backend
		if ((!is_admin()) && (!is_network_admin())) {
			return;
		}
		// add actions
		add_action('admin_notices', array($this, 'actionAdminNotices'));
		if (is_multisite()) {
			add_action('network_admin_notices', array($this, 'actionAdminNotices'));
		}
	}

	/**
	 * Add message
	 *
	 * @access private
	 * @param string $id Message identifier; must be unique
	 * @param string $content Message content
	 * @param int $type Message type; must be one of the following constants: \KocujInternalLib\V1a\Enums\MessageType::STANDARD (for standard message), \KocujInternalLib\V1a\Enums\MessageType::ERROR (for message about error) or \KocujInternalLib\V1a\Enums\MessageType::SPECIAL (for special message)
	 * @param bool $allPages Message will be displayed on all pages in administration panel (true) or not (false)
	 * @param array $pages Settings pages list for current project where this message should be displayed; it should be identifiers from settings page for current project only; if it is empty, this message will be displayed on all settings pages for current project
	 * @param array $attr Additional attributes; there are available the following attributes: "class" (string type; CSS class for message div), "style" (string type; CSS style for message div)
	 * @return void
	 */
	private function addMessageForProjectOrAllPages($id, $content, $type, $allPages, array $pages, array $attr) {
		// check if this id does not already exists
		if (isset($this->messages[$id])) {
			return;
		}
		// add message
		$this->messages[$id] = array(
			'content' => $content,
			'type'    => $type,
		);
		if ($allPages) {
			$this->messages[$id]['allpages'] = true;
		} else {
			$this->messages[$id]['pages'] = $pages;
		}
		if ((isset($attr['class'])) && (isset($attr['class'][0]) /* strlen($attr['class']) > 0 */ )) {
			$this->messages[$id]['class'] = $attr['class'];
		}
		if ((isset($attr['style'])) && (isset($attr['style'][0]) /* strlen($attr['style']) > 0 */ )) {
			$this->messages[$id]['style'] = $attr['style'];
		}
	}

	/**
	 * Add message
	 *
	 * @access public
	 * @param string $id Message identifier; must be unique
	 * @param string $content Message content
	 * @param int $type Message type; must be one of the following constants: \KocujInternalLib\V1a\Enums\MessageType::STANDARD (for standard message), \KocujInternalLib\V1a\Enums\MessageType::ERROR (for message about error) or \KocujInternalLib\V1a\Enums\MessageType::SPECIAL (for special message) - default: \KocujInternalLib\V1a\Enums\MessageType::STANDARD
	 * @param array $pages Settings pages list for current project where this message should be displayed; it should be identifiers from settings page for current project only; if it is empty, this message will be displayed on all settings pages for current project - default: empty
	 * @param array $attr Additional attributes; there are available the following attributes: "class" (string type; CSS class for message div), "style" (string type; CSS style for message div) - default: empty
	 * @return void
	 */
	public function addMessage($id, $content, $type = \KocujInternalLib\V1a\Enums\MessageType::STANDARD, array $pages = array(), array $attr = array()) {
		// add message
		$this->addMessageForProjectOrAllPages($id, $content, $type, false, $pages, $attr);
	}

	/**
	 * Add message for all pages in administration panel
	 *
	 * @access public
	 * @param string $id Message identifier; must be unique
	 * @param string $content Message content
	 * @param int $type Message type; must be one of the following constants: \KocujInternalLib\V1a\Enums\MessageType::STANDARD (for standard message), \KocujInternalLib\V1a\Enums\MessageType::ERROR (for message about error) or \KocujInternalLib\V1a\Enums\MessageType::SPECIAL (for special message) - default: \KocujInternalLib\V1a\Enums\MessageType::STANDARD
	 * @param array $attr Additional attributes; there are available the following attributes: "class" (string type; CSS class for message div), "style" (string type; CSS style for message div) - default: empty
	 * @return void
	 */
	public function addMessageForAllPages($id, $content, $type = \KocujInternalLib\V1a\Enums\MessageType::STANDARD, array $attr = array()) {
		// add message
		$this->addMessageForProjectOrAllPages($id, $content, $type, true, array(), $attr);
	}

	/**
	 * Get messages data
	 *
	 * @access public
	 * @param string $id Message identifier
	 * @return array Message data; each message data have the following fields: "content" (content of message), "type" (type of message); there are also the following fields which should exists or not: "allpages" (is it is set to true, message will be displayed in all pages in administration panel), "style" (string type; CSS style for message div), "class" (string type; CSS class for message div), "pages" (settings pages list for current project on which message is displayed or empty if it is displayed on all settings pages for current project)
	 */
	public function getMessages() {
		// exit
		return $this->messages;
	}

	/**
	 * Get message data by id
	 *
	 * @access public
	 * @param string $id Settings meta box identifier
	 * @return array|bool Message data or false if not exists; message data have the following fields: "content" (content of message), "type" (type of message); there are also the following fields which should exists or not: "allpages" (is it is set to true, message will be displayed in all pages in administration panel), "style" (string type; CSS style for message div), "class" (string type; CSS class for message div), "pages" (settings pages list for current project on which message is displayed or empty if it is displayed on all settings pages for current project)
	 */
	public function getMessage($id) {
		// exit
		return (isset($this->message[$id])) ?
			$this->message[$id] :
			false;
	}

	/**
	 * Action for admin notices
	 *
	 * @access public
	 * @return void
	 */
	public function actionAdminNotices() {
		// show messages
		if (!empty($this->messages)) {
			foreach ($this->messages as $id => $data) {
				// check if this message should be on this page
				$display = true;
				if (!((isset($data['allpages'])) && ($data['allpages']))) {
					$display = (empty($this->informations[$id]['pages'])) ?
						$this->getProjectObj()->getObj('backend_settings-menu')->checkCurrentPageIsSettingsForProject() :
						in_array($this->getProjectObj()->getObj('backend_settings-menu')->getCurrentSettingsMenu(), $this->informations[$id]['pages']);
				}
				// show message
				if ($display) {
					// get message class
					switch ($data['type']) {
						case \KocujInternalLib\V1a\Enums\MessageType::STANDARD:
							$class = 'updated';
							break;
						case \KocujInternalLib\V1a\Enums\MessageType::ERROR:
							$class = 'error';
							break;
						case \KocujInternalLib\V1a\Enums\MessageType::SPECIAL:
							$class = 'update-nag';
							break;
						default:
							$class = '';
					}
					// prepare style
					$style = (isset($data['style'])) ?
						' style="'.$data['style'].'"' :
						'';
					// prepare additional class
					if (isset($data['class'])) {
						$class .= ' '.$data['class'];
					}
					// show message
					echo '<div'.$style.' id="'.$this->getProjectObj()->getObj('project-helper')->getPrefix().'__message_'.$id.'" class="'.$class.'"><p>'.$data['content'].'</p></div>';
				}
			}
		}
	}
}
