<?php

/**
 * license.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujInternalLib\V1a\Classes\Project\Backend;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Set license class
 *
 * @access public
 */
class License extends \KocujInternalLib\V1a\Classes\ProjectObject {
	/**
	 * License will be no displayed anywhere automatically and there will be no need for accept (false) or license will be displayed anywhere automatically and there will be need for accept (true)
	 *
	 * @access private
	 * @var bool
	 */
	private $forceAccept = false;

	/**
	 * License is set to be displayed by forcing it or by link (true) or license will not be displayed (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $licenseDisplay = false;

	/**
	 * Constructor
	 *
	 * @access public
	 * @param object $projectObj \KocujInternalLib\V1a\Classes\Project object for current project
	 * @return void
	 */
	public function __construct($projectObj) {
		// execute parent constructor
		parent::__construct($projectObj);
		// add uninstall action
		add_action($this->getProjectObj()->getObj('project-helper')->getPrefix().'__plugin_uninstall', array($this, 'actionPluginUninstall'), 1);
		// set that license do not require accepting
		$this->forceAccept = false;
		// check if it is backend and check permissions
		if (((!is_admin()) && (!is_network_admin())) || ((!current_user_can('manage_network')) && (!current_user_can('manage_options')))) {
			return;
		}
		// add actions
		add_action('admin_footer', array($this, 'actionAdminFooter'), 1);
		add_action('admin_print_footer_scripts', array($this, 'actionPrintFooterScripts'), \KocujInternalLib\V1a\Classes\Helper::getInstance()->calculateMaxPriority('admin_print_footer_scripts'));
		// add "modal.js" script
		\KocujInternalLib\V1a\Classes\JsHelper::getInstance()->addModalJS();
	}

	/**
	 * Get license accepting option name
	 *
	 * @access public
	 * @return string License accepting option name
	 */
	public function getLicenseAcceptOptionName() {
		// exit
		return $this->getProjectObj()->getSettingInternalName().'__license_accept';
	}

	/**
	 * Get if license has been accepted (true) or not (false)
	 *
	 * @access public
	 * @return bool License has been accepted (true) or not (false)
	 */
	public function getLicenseAcceptOptionValue() {
		// exit
		return (\KocujInternalLib\V1a\Classes\Helper::getInstance()->getOptionForNetworkOrSite($this->getLicenseAcceptOptionName()) === false) ?
			false :
			true;
	}

	/**
	 * Set if license will be no displayed anywhere automatically and there will be no need for accept (false) or license will be displayed anywhere automatically and there will be need for accept (true)
	 *
	 * @access public
	 * @param bool $accept Set if license will be no displayed anywhere automatically and there will be no need for accept (false) or license will be displayed anywhere automatically and there will be need for accept (true)
	 * @return void
	 */
	public function setLicenseForceAccept($forceAccept) {
		// set license accepting
		$this->forceAccept = $forceAccept;
	}

	/**
	 * Get if license will be no displayed anywhere automatically and there will be no need for accept (false) or license will be displayed anywhere automatically and there will be need for accept (true)
	 *
	 * @access public
	 * @return int License will be no displayed anywhere automatically and there will be no need for accept (false) or license will be displayed anywhere automatically and there will be need for accept (true)
	 */
	public function getLicenseForceAccept() {
		// exit
		return $this->forceAccept;
	}

	/**
	 * Get license filename
	 *
	 * @access public
	 * @return string License filename; if license file does not exist, it returns empty string
	 */
	public function getLicenseFilename() {
		// get license filename
		$language = get_locale();
		if ((isset($language[0]) /* strlen($language) > 0 */ ) && (file_exists($this->getProjectObj()->getObj('dirs')->getLicenseDir().'/license-'.$language.'.txt'))) {
			$licenseFilename = 'license-'.$language.'.txt';
		} else {
			$licenseFilename = 'license.txt';
			if (!file_exists($this->getProjectObj()->getObj('dirs')->getLicenseDir().'/'.$licenseFilename)) {
				$licenseFilename = '';
			}
		}
		// exit
		return $licenseFilename;
	}

	/**
	 * Get license script
	 *
	 * @access private
	 * @return string License script
	 */
	private function getLicenseScript() {
		// exit
		return 'kocujInternalLibV1aBackendLicense.showLicense(\''.esc_js($this->getProjectObj()->getSettingInternalName()).'\', '.((($this->forceAccept) && (!$this->getLicenseAcceptOptionValue())) ?
				'true' :
				'false'
			).')';
	}

	/**
	 * Get license link or just name if license file does not exist
	 *
	 * @access public
	 * @return string License link or just name if license file does not exist
	 */
	public function getLicenseLink() {
		// get license link or name
		$licenseFilename = $this->getProjectObj()->getObj('backend_license')->getLicenseFilename();
		if (isset($licenseFilename[0]) /* strlen($licenseFilename) > 0 */ ) {
			// set license to display
			$this->licenseDisplay = true;
			// set HTML identifier
			$id = $this->getProjectObj()->getObj('project-helper')->getPrefix().'__licenselink_'.$this->getProjectObj()->getSettingInternalName().'_'.rand(111111, 999999);
			// exit
			return \KocujInternalLib\V1a\Classes\Helper::getInstance()->getLinkAnchor('#', $this->getProjectObj()->getSettingLicenseName(), array(
					'id'               => $id,
					'styleclassfilter' => array(
						'projectobj' => $this->getProjectObj(),
						'filter'     => 'license_link',
					),
				)).
				'<script type="text/javascript">'."\n".
				'/* <![CDATA[ */'."\n".
				'(function($) {'."\n".
				'$(document).ready(function() {'."\n".
				'$(\'#'.$id.'\').attr(\'href\', \'javascript:void(0);\');'."\n".
				'$(\'#'.$id.'\').click(function(event) {'."\n".
				'event.preventDefault();'."\n".
				$this->getLicenseScript()."\n".
				'});'."\n".
				'});'."\n".
				'}(jQuery));'."\n".
				'/* ]]> */'."\n".
				'</script>'."\n";
		} else {
			// exit
			return $this->getProjectObj()->getSettingLicenseName();
		}
	}

	/**
	 * Action for adding JavaScript scripts
	 *
	 * @access public
	 * @return void
	 */
	public function actionAdminFooter() {
		// check if license should be forced to display
		if ((!$this->licenseDisplay) && ($this->forceAccept) && (!$this->getLicenseAcceptOptionValue())) {
			$this->licenseDisplay = true;
		}
		// add scripts
		if ($this->licenseDisplay) {
			wp_enqueue_script(\KocujInternalLib\V1a\Classes\Helper::getInstance()->getPrefix().'-backend-license', \KocujInternalLib\V1a\Classes\LibUrls::getInstance()->getJavaScriptUrl().'/backend/license.js', array(
				'jquery',
				\KocujInternalLib\V1a\Classes\Helper::getInstance()->getPrefix().'-helper',
				\KocujInternalLib\V1a\Classes\Helper::getInstance()->getPrefix().'-modal',
			), \KocujInternalLib\V1a\Classes\Version::getInstance()->getVersion(), true);
			wp_localize_script(\KocujInternalLib\V1a\Classes\Helper::getInstance()->getPrefix().'-backend-license', 'kocujInternalLibV1aBackendLicenseSettings', array(
				'prefix'           => \KocujInternalLib\V1a\Classes\Helper::getInstance()->getPrefix(),
				'security'         => wp_create_nonce(\KocujInternalLib\V1a\Classes\Helper::getInstance()->getPrefix().'__license'),
				'textLoading'      => $this->getProjectObj()->getStringsObj('backend_license')->getString('ACTION_ADMIN_FOOTER_LICENSE_SCRIPT_LOADING'),
				'textLoadingError' => $this->getProjectObj()->getStringsObj('backend_license')->getString('ACTION_ADMIN_FOOTER_LICENSE_SCRIPT_LOADING_ERROR'),
				'textLicense'      => $this->getProjectObj()->getStringsObj('backend_license')->getString('ACTION_ADMIN_FOOTER_LICENSE_SCRIPT_LICENSE'),
				'textAccept'       => $this->getProjectObj()->getStringsObj('backend_license')->getString('ACTION_ADMIN_FOOTER_LICENSE_SCRIPT_ACCEPT'),
				'textCancel'       => $this->getProjectObj()->getStringsObj('backend_license')->getString('ACTION_ADMIN_FOOTER_LICENSE_SCRIPT_CANCEL'),
			));
		}
	}

	/**
	 * Action for setting up license and forcing displaying of license
	 *
	 * @access public
	 * @return void
	 */
	public function actionPrintFooterScripts() {
		// initialize license script
		if ($this->licenseDisplay) {
			?>
				<script type="text/javascript">
				/* <![CDATA[ */
					(function($) {
						$(document).ready(function() {
							<?php if ($this->getProjectObj()->getSettingType() === \KocujInternalLib\V1a\Enums\ProjectType::PLUGIN) : ?>
								kocujInternalLibV1aBackendLicense.addProject('<?php echo esc_js($this->getProjectObj()->getSettingInternalName()); ?>', '<?php echo esc_js($this->getProjectObj()->getStringsObj('backend_license')->getString('ACTION_PRINT_FOOTER_SCRIPTS_LICENSE_PLUGIN').': '.$this->getProjectObj()->getSettingTitleOriginal()); ?>');
							<?php else : ?>
								kocujInternalLibV1aBackendLicense.addProject('<?php echo esc_js($this->getProjectObj()->getSettingInternalName()); ?>', '<?php echo esc_js($this->getProjectObj()->getStringsObj('backend_license')->getString('ACTION_PRINT_FOOTER_SCRIPTS_LICENSE_THEME').': '.$this->getProjectObj()->getSettingTitleOriginal()); ?>');
							<?php endif; ?>
						});
					}(jQuery));
				/* ]]> */
				</script>
			<?php
		}
		// optionally force displaying of license
		if (($this->forceAccept) && (!$this->getLicenseAcceptOptionValue())) {
			?>
				<script type="text/javascript">
				/* <![CDATA[ */
					(function($) {
						$(document).ready(function() {
							<?php echo $this->getLicenseScript(); ?>
						});
					}(jQuery));
				/* ]]> */
				</script>
			<?php
		}
	}

	/**
	 * Action for uninstall plugin
	 *
	 * @access public
	 * @return void
	 */
	public function actionPluginUninstall() {
		// remove option
		\KocujInternalLib\V1a\Classes\Helper::getInstance()->deleteOptionForNetworkOrSite($this->getLicenseAcceptOptionName());
	}
}
