<?php

/**
 * cache.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html
 * @copyright Copyright (c) 2013-2016 Dominik Kocuj
 * @package kocuj_sitemap
 */

// set namespace
namespace KocujSitemapPlugin\Classes;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Plugin cache class
 *
 * @access public
 */
class Cache {
	/**
	 * Singleton instance
	 *
	 * @access private
	 * @var object
	 */
	private static $instance = NULL;

	/**
	 * Cache content
	 *
	 * @access private
	 * @var array
	 */
	private $cacheContent = array();

	/**
	 * Cache has been created now (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $cacheCreated = false;

	/**
	 * Constructor
	 *
	 * @access private
	 * @param array $actions Cache actions - default: empty
	 * @param array $filters Cache filters - default: empty
	 * @return void
	 */
	private function __construct(array $actions = array(), array $filters = array()) {
		// add actions and filters
		if ((is_admin()) || (is_network_admin())) {
			add_action('permalink_structure_changed', array($this, 'actionRecreateCache'), \KocujInternalLib\V1a\Classes\Helper::getInstance()->calculateMaxPriority('permalink_structure_changed'));
			$parsCount = (is_multisite()) ?
				2 :
				1;
			add_action('activated_plugin', array($this, 'actionClearCacheForPlugin'), \KocujInternalLib\V1a\Classes\Helper::getInstance()->calculateMaxPriority('activated_plugin'), $parsCount);
			add_action('deactivated_plugin', array($this, 'actionClearCacheForPlugin'), \KocujInternalLib\V1a\Classes\Helper::getInstance()->calculateMaxPriority('deactivated_plugin'), $parsCount);
			$this->addRecreateCacheFiltersOrActions($filters, true);
			$this->addRecreateCacheFiltersOrActions($actions, false);
		}
	}

	/**
	 * Disable cloning of object
	 *
	 * @access private
	 * @return void
	 */
	private function __clone() {
	}

	/**
	 * Get singleton instance
	 *
	 * @access public
	 * @param array $actions Cache actions - default: empty
	 * @param array $filters Cache filters - default: empty
	 * @return object Singleton instance
	 */
	public static function getInstance(array $actions = array(), array $filters = array()) {
		// optionally create new instance
		if (!self::$instance) {
			self::$instance = new \KocujSitemapPlugin\Classes\Cache($actions, $filters);
		}
		// exit
		return self::$instance;
	}

	/**
	 * Add filters or actions for recreating cache
	 *
	 * @access private
	 * @param array $data Filters or actions data
	 * @param bool $isFilter It is filter (true) or action (false)
	 * @return void
	 */
	private function addRecreateCacheFiltersOrActions(array $data, $isFilter) {
		// add filters or actions
		if (!empty($data)) {
			$added = array();
			foreach ($data as $element) {
				if (!in_array($element, $added)) {
					if ($isFilter) {
						add_filter($element, array($this, 'filterRecreateCache'), \KocujInternalLib\V1a\Classes\Helper::getInstance()->calculateMaxPriority($element));
					} else {
						add_action($element, array($this, 'actionRecreateCache'), \KocujInternalLib\V1a\Classes\Helper::getInstance()->calculateMaxPriority($element));
					}
					$added[] = $element;
				}
			}
		}
	}

	/**
	 * Check if the selected directory is writable
	 *
	 * @access private
	 * @param string $dir Directory to check
	 * @return bool Directory is writable (true) or not (false)
	 */
	private function checkDirWritable($dir) {
		// optionally create directory
		if (!is_dir($dir)) {
			@mkdir($dir, 0777, true);
		}
		// exit
		return (!((!is_dir($dir)) || (!is_writable($dir))));
	}

	/**
	 * Check if cache directory is writable
	 *
	 * @access public
	 * @return bool Cache directory is writable (true) or not (false)
	 */
	public function checkWritable() {
		// check cache directory is writable
		if ((!$this->checkDirWritable(self::getCacheRootDirectory())) || (!$this->checkDirWritable(self::getCacheDirectory()))) {
			return false;
		}
		// exit
		return true;
	}

	/**
	 * Get cache root directory
	 *
	 * @access public
	 * @return string Cache root directory
	 */
	public function getCacheRootDirectory() {
		// get cache root directory
		return WP_CONTENT_DIR.'/cache/kocuj-sitemap';
	}

	/**
	 * Get cache directory
	 *
	 * @access public
	 * @return string Cache directory
	 */
	public function getCacheDirectory() {
		// get cache directory
		return $this->getCacheRootDirectory().'/'.((is_multisite()) ?
			get_current_blog_id() :
			1
			);
	}

	/**
	 * Remove cache files from the selected directory
	 *
	 * @access private
	 * @param string $dir Directory
	 * @param bool $removeAllFiles Remove all files (true) or "*.cache" files only (false) - default: false
	 * @return void
	 */
	private function removeCacheFiles($dir, $removeAllFiles = false) {
		// remove cache files
		if (is_dir($dir)) {
			$handle = opendir($dir);
			while ($item = readdir($handle)) {
				if ((is_file($dir.'/'.$item)) && (((!$removeAllFiles) && (substr($item, -6, 6) === '.cache')) || ($removeAllFiles))) {
					unlink($dir.'/'.$item);
				}
			}
		}
	}

	/**
	 * Get cache filename
	 *
	 * @access private
	 * @param string $locale Language locale - default: empty
	 * @return string Cache filename - if empty, there is an error in cache directory
	 */
	private function getCacheFilename($locale = '') {
		// check if directory is writable
		if (!$this->checkWritable()) {
			return '';
		}
		// get cache directory
		$dir = $this->getCacheDirectory();
		// get random value
		$securityFilename = $dir.'/security.cache.php';
		if (!file_exists($securityFilename)) {
			$file = @fopen($securityFilename, 'w');
			if (flock($file, LOCK_EX) === false) {
				fclose($file);
				return '';
			}
			if ($file === false) {
				flock($file, LOCK_UN);
				fclose($file);
				return '';
			}
			if (fwrite($file, '<'.'?php'."\n") === false) {
				flock($file, LOCK_UN);
				fclose($file);
				return '';
			}
			if (fwrite($file, '// this is the random value for secure cache files - it has to be secret, so do not tell it to anyone; you can change it, if you think that this value is not secret and is known to other people'."\n") === false) {
				flock($file, LOCK_UN);
				fclose($file);
				return '';
			}
			if (fwrite($file, '$securityValue = \''.rand(111111111, 999999999).'\';'."\n") === false) {
				flock($file, LOCK_UN);
				fclose($file);
				return '';
			}
			flock($file, LOCK_UN);
			if (fclose($file) === false) {
				return '';
			}
			@chmod($securityFilename, 0644);
		}
		$securityValue = '';
		if (file_exists($securityFilename)) {
			$code = file($securityFilename);
			if (($code !== false) && (isset($code[1]) /* count($code) > 1 */ )) {
				unset($code[0]);
				eval(implode('', $code));
			} else {
				return '';
			}
		} else {
			return '';
		}
		// exit
		return $dir.'/cache'.((isset($locale[0]) /* strlen($locale) > 0 */ ) ?
			'_'.$locale :
			''
			).'_'.$securityValue.'.cache';
	}

	/**
	 * Clear cache
	 *
	 * @access public
	 * @param bool $networkWide Clear cache for the entire network (true) or for site only (false) - default: false
	 * @param bool $removeDirectory Remove directory with cache files (true) or not (false) - default: false
	 * @return void
	 */
	public function clearCache($networkWide = false, $removeDirectory = false) {
		// get cache directory
		$cacheDirectory = $this->getCacheDirectory();
		// remove all cache files
		$this->removeCacheFiles(\KocujSitemapPlugin\Classes\Base::getInstance()->getPluginDir().'/cache'); // for compatibility with 1.x.x
		if (!$networkWide) {
			$this->removeCacheFiles($cacheDirectory);
		} else {
			$dir = $this->getCacheRootDirectory();
			if (is_dir($dir)) {
				$handle = opendir($dir);
				while ($item = readdir($handle)) {
					if ((is_dir($dir.'/'.$item)) && ($item !== '.') && ($item !== '..')) {
						$this->removeCacheFiles($dir.'/'.$item, $removeDirectory);
						rmdir($dir.'/'.$item);
					}
				}
			}
		}
	}

	/**
	 * Purge cache
	 *
	 * @access public
	 * @return void
	 */
	public function purgeCache() {
		// remove cache subdirectories for all sites
		$this->clearCache(true, true);
		// remove cache directory
		$dir = $this->getCacheRootDirectory();
		rmdir($dir);
	}

	/**
	 * Create cache
	 *
	 * @access public
	 * @return bool Status - true or false
	 */
	public function createCache() {
		// check if cache has not been created now
		if ($this->cacheCreated) {
			return true;
		}
		// check if cache is activated
		$cache = \KocujSitemapPlugin\Classes\Base::getInstance()->getKocujInternalLibObj()->getObj('config')->getOption('Cache');
		if (!$cache) {
			return true;
		}
		// remove all cache files
		$this->clearCache();
		// check if cache should be generated
		$cacheFrontend = \KocujSitemapPlugin\Classes\Base::getInstance()->getKocujInternalLibObj()->getObj('config')->getOption('CacheFrontend');
		if ((($cacheFrontend) && (!is_admin()) && (!is_network_admin())) || (!$cacheFrontend)) {
			// get all languages
			$langs = \KocujSitemapPlugin\Classes\MultipleLanguages::getInstance()->getLanguages();
			// if cache is generated for frontend, it will be generated only for current locale
			if ($cacheFrontend) {
				$locale = get_locale();
				if (!in_array($locale, $langs)) {
					$locale = 'en_US';
				}
				$langs = array(
					$locale,
				);
			}
			// create cached sitemap for each language
			$this->cacheContent = array();
			if (!empty($langs)) {
				foreach ($langs as $lang) {
					// create sitemap
					$sitemapData = \KocujSitemapPlugin\Classes\Sitemap::getInstance()->createSitemap($lang);
					// set sitemap data
					$this->cacheContent[$lang] = array(
						'dt' => $sitemapData,
						'vr' => 2,
					);
					// check if cache directory is writable
					if (!$this->checkWritable()) {
						return false;
					}
					// get cache directory
					$cacheDirectory = $this->getCacheDirectory();
					// save "index.html" file for secure directory
					$file = @fopen($cacheDirectory.'/index.html', 'w');
					if ($file === false) {
						return false;
					}
					if (fclose($file) === false) {
						return false;
					}
					// set filename
					$filename = $this->getCacheFilename($lang);
					if (!isset($filename[0]) /* strlen($filename) === 0 */ ) {
						return false;
					}
					// save cache
					if (file_exists($filename)) {
						unlink($filename);
					}
					$file = @fopen($filename, 'w');
					if (flock($file, LOCK_EX) === false) {
						fclose($file);
						return false;
					}
					if ($file === false) {
						flock($file, LOCK_UN);
						fclose($file);
						return false;
					}
					if (fwrite($file, maybe_serialize($this->cacheContent[$lang])) === false) {
						flock($file, LOCK_UN);
						fclose($file);
						return false;
					}
					flock($file, LOCK_UN);
					if (fclose($file) === false) {
						return false;
					}
					// set cache as created
					$this->cacheCreated = true;
				}
			}
		}
		// exit
		return true;
	}

	/**
	 * Check if cache file exists
	 *
	 * @access public
	 * @return bool Cache file exists (true) or not (false)
	 */
	public function checkCacheFile() {
		// set filename
		$filename = $this->getCacheFilename(get_locale());
		if (!isset($filename[0]) /* strlen($filename) === 0 */ ) {
			return false;
		}
		// check if file exists */
		return file_exists($filename);
	}

	/**
	 * Load cache
	 *
	 * @access public
	 * @param string &$buffer Returned buffer
	 * @return bool Status - true or false
	 */
	public function loadCache(&$buffer) {
		// get locale
		$locale = get_locale();
		// check if cache is activated
		$cache = \KocujSitemapPlugin\Classes\Base::getInstance()->getKocujInternalLibObj()->getObj('config')->getOption('Cache');
		if ($cache) {
			// optionally get cache from memory
			if (!empty($this->cacheContent)) {
				$array = (isset($this->cacheContent[$locale])) ?
					$this->cacheContent[$locale] :
					array();
			} else {
				// set filename
				$filename = $this->getCacheFilename($locale);
				if (!isset($filename[0]) /* strlen($filename) === 0 */ ) {
					return false;
				}
				// check if file exists */
				if (!file_exists($filename)) {
					// optionally set filename to default locale
					$filename = $this->getCacheFilename('en_US');
					if ((!isset($filename[0]) /* strlen($filename) === 0 */ ) || (!file_exists($filename))) {
						return false;
					}
				}
				// load file
				$input = file_get_contents($filename);
				$array = maybe_unserialize($input);
				if (empty($array)) {
					$array = maybe_unserialize($array);
				}
			}
			// check cache version
			if ((!isset($array['vr'])) || ($array['vr'] !== 2)) {
				return false;
			}
		} else {
			$sitemapData = \KocujSitemapPlugin\Classes\Sitemap::getInstance()->createSitemap($locale);
			$array = array(
				'dt' => $sitemapData,
			);
		}
		// set output buffer
		$buffer = $array['dt'];
		// exit
		return true;
	}

	/**
	 * Action - recreate cache
	 *
	 * @access public
	 * @return void
	 */
	public function actionRecreateCache() {
		// renew cache
		$this->createCache();
	}

	/**
	 * Filter - recreate cache
	 *
	 * @access public
	 * @param array|bool|float|int|object|string $data Filter data
	 * @return array|bool|float|int|object|string Filter data
	 */
	public function filterRecreateCache($data) {
		// renew cache
		$this->actionRecreateCache();
		// exit
		return $data;
	}

	/**
	 * Action - clear cache for plugin which was activated or deactivated
	 *
	 * @access public
	 * @param string $plugin Plugin name
	 * @param bool $networkWide Plugin is enabled or disabled for the entire network (true) or for site only (false) - default: false
	 * @return void
	 */
	public function actionClearCacheForPlugin($plugin, $networkWide = false) {
		// check if plugin is supported
		$pluginsFiles = \KocujSitemapPlugin\Classes\MultipleLanguagesData::getInstance()->getPluginsFiles();
		if (in_array($plugin, $pluginsFiles)) {
			// clear cache
			$this->clearCache($networkWide);
		}
	}
}
