<?php

/**
 * authors.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2013-2016 Dominik Kocuj
 * @package kocuj_sitemap
 */

// set namespace
namespace KocujSitemapPlugin\Classes\ElementType\Backend;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Authors type administration panel class
 *
 * @access public
 */
class Authors implements \KocujSitemapPlugin\Interfaces\ElementTypeBackend {
	/**
	 * Add authors to sitemap
	 *
	 * @access public
	 * @param string $locale Language locale - default: empty
	 * @return array Output array
	 */
	public function getElementArray($locale = '') {
		// initialize
		$array = array();
		// add authors to sitemap
		$value = \KocujSitemapPlugin\Classes\Base::getInstance()->getKocujInternalLibObj()->getObj('config')->getOption('DisplayAuthors');
		if ($value) {
			// get authors
			$value = \KocujSitemapPlugin\Classes\Base::getInstance()->getKocujInternalLibObj()->getObj('config')->getOption('DisplayAuthorsSort');
			$sortColumn = 'display_name';
			if ((isset($value[0]) /* strlen($value) > 0 */ ) && ($value !== 'name')) {
				switch ($value) {
					case 'login';
						$sortColumn = 'login';
					break;
					case 'id';
						$sortColumn = 'ID';
					break;
					case 'date';
						$sortColumn = 'registered';
					break;
					case 'posts';
						$sortColumn = 'post_count';
					break;
				}
			}
			$value = \KocujSitemapPlugin\Classes\Base::getInstance()->getKocujInternalLibObj()->getObj('config')->getOption('DisplayAuthorsOrder');
			$sortOrder = 'ASC';
			if (isset($value[0]) /* strlen($value) > 0 */ ) {
				switch ($value) {
					case 'desc':
						$sortOrder = 'DESC';
					break;
				}
			}
			\KocujSitemapPlugin\Classes\MultipleLanguages::getInstance()->beforeGetAuthors($locale);
			$authors = get_users(array(
				'orderby' => $sortColumn,
				'order'   => $sortOrder,
				'who'     => 'authors',
				'number'  => '',
				'offset'  => '',
			));
			\KocujSitemapPlugin\Classes\MultipleLanguages::getInstance()->afterGetAuthors($locale);
			if (!empty($authors)) {
				$check = \KocujSitemapPlugin\Classes\Base::getInstance()->getKocujInternalLibObj()->getObj('config')->getOption('DisplayOnlyAuthorsWithArticles');
				global $wpdb;
				if ($check) {
					$types = get_post_types(array(
						'public'   => true,
						'_builtin' => false,
					), 'names');
					$typeString = '';
					if (!empty($types)) {
						foreach ($types as $type) {
							$typeString .= $wpdb->prepare('"%s"', $type).',';
						}
						$typeString = substr($typeString, 0, -1);
					}
				}
				foreach ($authors as $author) {
					$ok = true;
					if ($check) {
						$postsCount = $wpdb->get_var($wpdb->prepare('SELECT COUNT(*) FROM '.$wpdb->posts.' WHERE post_author = "%d" AND post_type IN ("post", "page"'.$typeString.') AND post_status = "publish"', $author->ID));
						if (empty($postsCount)) {
							$ok = false;
						}
					}
					if ($ok) {
						\KocujSitemapPlugin\Classes\MultipleLanguages::getInstance()->beforeGetAuthorItem($author->ID, $locale);
						$linkText = apply_filters('kocujsitemap_linktitle', \KocujSitemapPlugin\Classes\MultipleLanguages::getInstance()->getTranslatedAuthorName($author->display_name, $author->ID, $locale), $author->ID, 'author', $locale); // for compatibility with 1.x.x
						\KocujSitemapPlugin\Classes\Base::getInstance()->filterOldInfo('kocujsitemap_linktitle', 'kocujsitemap_link_text'); // for compatibility with 1.x.x
						$linkText = apply_filters('kocujsitemap_link_text', $linkText, $author->ID, 'author', $locale);
					if (!isset($linkText[0]) /* strlen($linkText) === 0 */ ) {
						$linkText = '-';
					}
						$link = apply_filters('kocujsitemap_element', \KocujInternalLib\V2a\Classes\HtmlHelper::getInstance()->getLinkAnchor(\KocujSitemapPlugin\Classes\Helpers\Url::getInstance()->removeProtocolLocal(\KocujSitemapPlugin\Classes\MultipleLanguages::getInstance()->getTranslatedAuthorURL(get_author_posts_url($author->ID), $author->ID, $locale)), $linkText), $author->ID, 'author', $locale);
						$arrayAdd = array(
							'id' => $author->ID,
							'tp' => 'author',
							'lk' => $link,
						);
						if ($sortColumn === 'login') {
							$arrayAdd['sortname'] = $linkText;
						}
						$array[] = $arrayAdd;
						\KocujSitemapPlugin\Classes\MultipleLanguages::getInstance()->afterGetAuthorItem($author->ID, $locale);
					}
				}
				if ($sortColumn === 'login') {
					$array = \KocujSitemapPlugin\Classes\Helpers\Sort::getInstance()->sortElements($array, $sortOrder);
				}
			}
		}
		// exit
		return $array;
	}

	/**
	 * Get administration page settings
	 *
	 * @access public
	 * @return array Administration page settings
	 */
	public function getAdminPageSettings() {
		// get administration page settings
		return array(
			'title' => __('Authors list options', 'kocuj-sitemap'),
			'id'    => 'authors_options',
			'tabs'  => array(
				'authors_displaying' => __('Displaying authors', 'kocuj-sitemap'),
				'authors_options'    => __('Options', 'kocuj-sitemap'),
			),
			'help'  => array(
				'overview' => array(
					'title'   => __('Overview', 'kocuj-sitemap'),
					'content' =>
						__('This is the place where you can enable or disable displaying of authors in the sitemap. Here you can also change how they will be displayed.', 'kocuj-sitemap').'</p>'.
						'<p>'.__('Options are divided into three tabs: `displaying authors`, `options` and `section title`. Each tab can be selected by clicking on it.', 'kocuj-sitemap').'</p>'.
						'<p>'.__('To save changed settings, click on the button `save authors list options`.', 'kocuj-sitemap'),
				),
				'displaying_authors' => array(
					'title'   => __('`Displaying authors` tab', 'kocuj-sitemap'),
					'content' =>
						__('There are the following options in this tab:', 'kocuj-sitemap').'</p>'.
						'<p><ul>'.
						'<li><em>`'.__('Display authors', 'kocuj-sitemap').'`</em>: '.__('If this option is activated, there will be authors list displayed in the sitemap.', 'kocuj-sitemap').'</li>'.
						'</ul>',
				),
				'options' => array(
					'title'   => __('`Options` tab', 'kocuj-sitemap'),
					'content' =>
						__('There are the following options in this tab:', 'kocuj-sitemap').'</p>'.
						'<p><ul>'.
						'<li><em>`'.__('Display only authors with at least one article', 'kocuj-sitemap').'`</em>: '.__('If this option is activated, only authors with at least one article will be displayed.', 'kocuj-sitemap').
						'<li><em>`'.__('Sort authors by', 'kocuj-sitemap').'`</em>: '.__('You can sort authors in the sitemap using the selected authors properties. There are the following authors properties to select:', 'kocuj-sitemap').
						'<ul>'.
						'<li><em>`'.__('name', 'kocuj-sitemap').'`</em>: '.__('Authors will be sorted by name.', 'kocuj-sitemap').'</li>'.
						'<li><em>`'.__('login', 'kocuj-sitemap').'`</em>: '.__('Authors will be sorted by login.', 'kocuj-sitemap').'</li>'.
						'<li><em>`'.__('ID', 'kocuj-sitemap').'`</em>: '.__('Authors will be sorted by author identifier.', 'kocuj-sitemap').'</li>'.
						'<li><em>`'.__('registered date', 'kocuj-sitemap').'`</em>: '.__('Authors will be sorted by registered date.', 'kocuj-sitemap').'</li>'.
						'<li><em>`'.__('posts count', 'kocuj-sitemap').'`</em>: '.__('Authors will be sorted by posts count created by each author.', 'kocuj-sitemap').'</li>'.
						'</ul>'.
						'</li>'.
						'<li><em>`'.__('Sort authors order', 'kocuj-sitemap').'`</em>: '.__('You can sort authors by ascending or descending order.', 'kocuj-sitemap').'</li>'.
						'</ul>',
				),
				'section_title' => array(
					'title'   => __('`Section title` tab', 'kocuj-sitemap'),
					'content' =>
						__('In this tab you can set the title of the section with authors. It is used if option `divide display into sections` in the main settings of the Kocuj Sitemap plugin is activated.', 'kocuj-sitemap').'</p>'.
						'<p>'.__('There are fields to enter the section title for each language that is available in your WordPress installation. If you have not activated any of the supported plugins for multilingualism, there will be visible only two fields to enter the title - for current WordPress language and for default section title in English language.', 'kocuj-sitemap').'</p>'.
						'<p>'.__('It is not necessary to enter section titles here. However, the place is here in order to be able to display the title of the section in your chosen language if there is no translation for it. If you leave this empty and if you will not have a translation of section title for current language and for default language (English language), there will be displayed the standard section title in English language.', 'kocuj-sitemap'),
				),
			),
		);
	}

	/**
	 * Get administration page data
	 *
	 * @access public
	 * @return array Administration page data
	 */
	public function getAdminPageData() {
		// get administration page data
		$sortArray = array(
			'name'  => __('name', 'kocuj-sitemap'),
			'login' => __('login', 'kocuj-sitemap'),
			'id'    => __('ID', 'kocuj-sitemap'),
			'date'  => __('registered date', 'kocuj-sitemap'),
			'posts' => __('posts count', 'kocuj-sitemap'),
		);
		$sortByArray = array(
			'asc'  => __('ascending', 'kocuj-sitemap'),
			'desc' => __('descending', 'kocuj-sitemap'),
		);
		// exit
		return array(
			'pages'   => array(
				\KocujSitemapPlugin\Classes\Base::getInstance()->getKocujInternalLibObj()->getObj('admin')->getBlockHelper('DisplayAuthors', 'checkbox', __('If this option is activated, there will be authors list displayed in the sitemap.', 'kocuj-sitemap')),
				\KocujSitemapPlugin\Classes\Base::getInstance()->getKocujInternalLibObj()->getObj('admin')->getBlockHelper('DisplayOnlyAuthorsWithArticles', 'checkbox', __('If this option is activated, only authors with at least one article will be displayed.', 'kocuj-sitemap')).
					\KocujSitemapPlugin\Classes\Base::getInstance()->getKocujInternalLibObj()->getObj('admin')->getBlockHelper('DisplayAuthorsSort', 'select', __('You can sort authors in the sitemap using the selected authors properties.', 'kocuj-sitemap'), $sortArray).
					\KocujSitemapPlugin\Classes\Base::getInstance()->getKocujInternalLibObj()->getObj('admin')->getBlockHelper('DisplayAuthorsOrder', 'select', __('You can sort authors by ascending or descending order.', 'kocuj-sitemap'), $sortByArray),
			),
			'submit'  => array(
				'title' => __('Save authors list options', 'kocuj-sitemap'),
				'info'  => __('Save current authors list options.', 'kocuj-sitemap'),
			),
			'buttons' => '',
		);
	}

	/**
	 * Get administration panel order name
	 *
	 * @access public
	 * @return string Administration panel order name
	 */
	public function getAdminOrderName() {
		// get administration panel order name
		return __('Authors', 'kocuj-sitemap');
	}

	/**
	 * Get administration cache actions
	 *
	 * @access public
	 * @return array Administration cache actions
	 */
	public function getAdminCacheActions() {
		// get administration cache actions
		return array(
			'user_register',
			'edit_user_profile_update',
			'deleted_user',
		);
	}

	/**
	 * Get administration cache filters
	 *
	 * @access public
	 * @return array Administration cache filters
	 */
	public function getAdminCacheFilters() {
		// get administration cache filters
		return array();
	}
}
