<?php

/**
 * review-message.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujInternalLib\V2a\Classes\Project\Backend;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Set review message class
 *
 * @access public
 */
class ReviewMessage extends \KocujInternalLib\V2a\Classes\ProjectObject {
	/**
	 * Days count after which message about review will be displayed; 0 means never
	 *
	 * @access private
	 * @var int
	 */
	private $days = 0;

	/**
	 * Vote URL
	 *
	 * @access private
	 * @var string
	 */
	private $voteUrl = '';

	/**
	 * Review message is set to be displayed (true) or review message will not be displayed (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $reviewMessageDisplay = false;

	/**
	 * Constructor
	 *
	 * @access public
	 * @param object $projectObj \KocujInternalLib\V2a\Classes\Project object for current project
	 * @return void
	 */
	public function __construct($projectObj) {
		// execute parent constructor
		parent::__construct($projectObj);
		// add uninstall action
		add_action($this->getProjectObj()->getObj('project-helper')->getPrefix().'__plugin_uninstall', array($this, 'actionPluginUninstall'), 1);
		// check if it is backend and check permissions
		if (((!is_admin()) && (!is_network_admin())) || ((!current_user_can('manage_network')) && (!current_user_can('manage_options')))) {
			return;
		}
		// add actions
		add_action('admin_head', array($this, 'actionAdminHead'), 1);
		add_action('admin_print_footer_scripts', array($this, 'actionPrintFooterScripts'), \KocujInternalLib\V2a\Classes\Helper::getInstance()->calculateMaxPriority('admin_print_footer_scripts'));
	}

	/**
	 * Set days count after which message about review will be displayed; 0 means never
	 *
	 * @access public
	 * @param int $days Days count after which message about review will be displayed; 0 means never
	 * @return void
	 */
	public function setDays($days) {
		// set days count
		$this->days = $days;
	}

	/**
	 * Get days count after which message about review will be displayed; 0 means never
	 *
	 * @access public
	 * @return int Days count after which message about review will be displayed; 0 means never
	 */
	public function getDays() {
		// exit
		return $this->days;
	}

	/**
	 * Set vote URL
	 *
	 * @access public
	 * @param string $url Vote url
	 * @return void
	 */
	public function setVoteUrl($url) {
		// set value
		$this->voteUrl = $url;
	}

	/**
	 * Set vote URL
	 *
	 * @access public
	 * @return string Vote url
	 */
	public function getVoteUrl() {
		// exit
		return $this->voteUrl;
	}

	/**
	 * Get closed window option name
	 *
	 * @access public
	 * @return string Closed window option name
	 */
	public function getClosedWindowOptionName() {
		// exit
		return $this->getProjectObj()->getSettingInternalName().'__review_message_closed';
	}

	/**
	 * Get window has been closed (true) or not (false)
	 *
	 * @access public
	 * @return bool Window has been closed (true) or not (false)
	 */
	public function getClosedWindowOptionValue() {
		// exit
		return (\KocujInternalLib\V2a\Classes\Helper::getInstance()->getOptionForNetworkOrSite($this->getClosedWindowOptionName()) === false) ?
			false :
			true;
	}

	/**
	 * Get message element in "div" container
	 *
	 * @access private
	 * @param string $element Message element
	 * @param bool $smaller Message element is smaller than normal (true) or not (false) - default: false
	 * @return string Message element in "div" container
	 */
	private function getMessageElement($element, $smaller = false) {
		// set horizontal margins
		$margin = ($smaller) ?
			2 :
			5;
		// exit
		return '<div'.$this->getProjectObj()->getObj('project-helper')->applyFiltersForHTMLStyleAndClass('review_message_element'.(($smaller) ?
				'_small' :
				''
			), '', array(
				'defaultstyle' => 'float:left;margin-left:'.$margin.'px;margin-right:'.$margin.'px;vertical-align:top;',
				'defaultclass' => $this->getProjectObj()->getObj('project-helper')->getPrefix().'__review_message_element_'.(($smaller) ?
						'small_' :
						''
					).'div',
			)).'>'.$element.'</div>';
	}

	/**
	 * Add additional elements
	 *
	 * @access private
	 * @param array $elements Additional elements to add
	 * @return string Additional elements in string
	 */
	private function addAdditionalElements(array $elements) {
		// check if there are any additional elements
		if (empty($elements)) {
			return '';
		}
		// add additional elements
		$output = '';
		foreach ($elements as $element) {
			$output .= $this->getMessageElement($element);
		}
		// exit
		return $output;
	}

	/**
	 * Action for display review message
	 *
	 * @access public
	 * @return void
	 */
	public function actionAdminHead() {
		// check if display review message
		if ((!isset($this->voteUrl[0]) /* strlen($this->voteUrl) === 0 */ ) || ($this->getClosedWindowOptionValue()) || ($this->days === 0)) {
			return;
		}
		// get installation date
		$date = $this->getProjectObj()->getObj('backend_installation-date')->getInstallationDateOptionValue();
		if ($date === false) {
			return;
		}
		// check days count
		$days = (int)((strtotime(date('Y-m-d').' 23:59:59')-strtotime($date.' 00:00:00'))/86400);
		// optionally show review message
		if ($days > $this->days) {
			// add CSS stylesheet
			?>
				<style type="text/css" media="all">
				<!--
					#<?php echo $this->getProjectObj()->getObj('project-helper')->getPrefix(); ?>__review_message_news_channels_header_div {
						font-size: 11px;
					}

					.<?php echo $this->getProjectObj()->getObj('project-helper')->getPrefix(); ?>__review_message_element_div {
						height: 40px;
					}

					@media screen and (max-width: 782px) {
						#<?php echo $this->getProjectObj()->getObj('project-helper')->getPrefix(); ?>__review_message_news_channels_header_div {
							font-size: 12px;
						}
					}
				-->
				</style>
			<?php
			// show review message beginning
			$message = $this->getProjectObj()->getObj('project-helper')->applyFilters('before_review_message');
			if ($this->getProjectObj()->getSettingType() === \KocujInternalLib\V2a\Enums\ProjectType::PLUGIN) {
				$message .= sprintf($this->getProjectObj()->getStringsObj('backend_review-message')->getString('ACTION_ADMIN_HEAD_PLUGIN_DAYS'), $this->getProjectObj()->getSettingTitleOriginal(), $this->days);
				$message .= ' '.$this->getProjectObj()->getStringsObj('backend_review-message')->getString('ACTION_ADMIN_HEAD_THINGS_SUPPORT_PLUGIN').' ';
			} else {
				$message .= sprintf($this->getProjectObj()->getStringsObj('backend_review-message')->getString('ACTION_ADMIN_HEAD_THEME_DAYS'), $this->getProjectObj()->getSettingTitleOriginal(), $this->days);
				$message .= ' '.$this->getProjectObj()->getStringsObj('backend_review-message')->getString('ACTION_ADMIN_HEAD_THINGS_SUPPORT_THEME').' ';
			}
			$message .= '<br />';
			$elements = $this->getProjectObj()->getObj('project-helper')->applyFilters('review_message_elements_before', '', '', array());
			// show buttons for telling others
			$message .= $this->addAdditionalElements($elements);
			$message .= $this->getMessageElement('<input type="button" class="button button-small" id="'.esc_attr($this->getProjectObj()->getObj('project-helper')->getPrefix().'__review_message_vote').'" value="'.esc_attr(($this->getProjectObj()->getSettingType() === \KocujInternalLib\V2a\Enums\ProjectType::PLUGIN) ?
					$this->getProjectObj()->getStringsObj('backend_review-message')->getString('ACTION_ADMIN_HEAD_VOTE_PLUGIN') :
					$this->getProjectObj()->getStringsObj('backend_review-message')->getString('ACTION_ADMIN_HEAD_VOTE_THEME')
				).'" />');
			if ($this->getProjectObj()->getSettingSupportTellOthers('facebook')) {
				$message .= $this->getMessageElement('<input type="button" class="button button-small" id="'.esc_attr($this->getProjectObj()->getObj('project-helper')->getPrefix().'__review_message_facebook').'" value="'.esc_attr($this->getProjectObj()->getStringsObj('backend_review-message')->getString('ACTION_ADMIN_HEAD_FACEBOOK')).'" />');
			}
			if ($this->getProjectObj()->getSettingSupportTellOthers('twitter')) {
				$message .= $this->getMessageElement('<input type="button" class="button button-small" id="'.esc_attr($this->getProjectObj()->getObj('project-helper')->getPrefix().'__review_message_twitter').'" value="'.esc_attr($this->getProjectObj()->getStringsObj('backend_review-message')->getString('ACTION_ADMIN_HEAD_TWITTER')).'" />');
			}
			$elements = $this->getProjectObj()->getObj('project-helper')->applyFilters('review_message_elements_after', '', '', array());
			$message .= $this->addAdditionalElements($elements);
			$message .= '<div style="clear:both;"></div>';
			// show news channels links
			$linkRSS = $this->getProjectObj()->getSettingNewsChannel('rss');
			$linkFacebook = $this->getProjectObj()->getSettingNewsChannel('facebook');
			$linkTwitter = $this->getProjectObj()->getSettingNewsChannel('twitter');
			if ((isset($linkRSS[0]) /* strlen($linkRSS) > 0 */ ) || (isset($linkFacebook[0]) /* strlen($linkFacebook) > 0 */ ) || (isset($linkTwitter[0]) /* strlen($linkTwitter) > 0 */ )) {
				if ($this->getProjectObj()->getSettingType() === \KocujInternalLib\V2a\Enums\ProjectType::PLUGIN) {
					$text = $this->getProjectObj()->getStringsObj('backend_review-message')->getString('ACTION_ADMIN_HEAD_NEWS_CHANNELS_TEXT_PLUGIN');
				} else {
					$text = $this->getProjectObj()->getStringsObj('backend_review-message')->getString('ACTION_ADMIN_HEAD_NEWS_CHANNELS_TEXT_THEME');
				}
				$message .= $this->getMessageElement('<div id="'.$this->getProjectObj()->getObj('project-helper')->getPrefix().'__review_message_news_channels_header_div"><em>'.$text.':</em></div>', true);
				if (isset($linkRSS[0]) /* strlen($linkRSS) > 0 */ ) {
					$message .= $this->getMessageElement(\KocujInternalLib\V2a\Classes\HtmlHelper::getInstance()->getLinkAnchor($linkRSS, \KocujInternalLib\V2a\Classes\HtmlHelper::getInstance()->getHTMLImage(\KocujInternalLib\V2a\Classes\LibUrls::getInstance()->getImagesUrl().'/icon-small-rss.png'), array(
						'external' => true,
					)), true);
				}
				if (isset($linkFacebook[0]) /* strlen($linkFacebook) > 0 */ ) {
					$message .= $this->getMessageElement(\KocujInternalLib\V2a\Classes\HtmlHelper::getInstance()->getLinkAnchor($linkFacebook, \KocujInternalLib\V2a\Classes\HtmlHelper::getInstance()->getHTMLImage(\KocujInternalLib\V2a\Classes\LibUrls::getInstance()->getImagesUrl().'/icon-small-facebook.png'), array(
						'external' => true,
					)), true);
				}
				if (isset($linkTwitter[0]) /* strlen($linkTwitter) > 0 */ ) {
					$message .= $this->getMessageElement(\KocujInternalLib\V2a\Classes\HtmlHelper::getInstance()->getLinkAnchor($linkTwitter, \KocujInternalLib\V2a\Classes\HtmlHelper::getInstance()->getHTMLImage(\KocujInternalLib\V2a\Classes\LibUrls::getInstance()->getImagesUrl().'/icon-small-twitter.png'), array(
						'external' => true,
					)), true);
				}
				$message .= '<div style="clear:both;"></div>';
			}
			// show review message end
			$message .= $this->getProjectObj()->getObj('project-helper')->applyFilters('after_review_message');
			$this->getProjectObj()->getObj('backend_message')->addMessageForAllPages('review', $message, \KocujInternalLib\V2a\Enums\MessageType::INFORMATION, \KocujInternalLib\V2a\Enums\MessageClosable::CLOSABLE);
			// add scripts
			\KocujInternalLib\V2a\Classes\JsHelper::getInstance()->addHelperJS(true);
			\KocujInternalLib\V2a\Classes\JsHelper::getInstance()->addExceptionJS(true);
			$deps = array(
				'jquery',
				\KocujInternalLib\V2a\Classes\Helper::getInstance()->getPrefix().'-helper',
			);
			if (\KocujInternalLib\V2a\Classes\JsHelper::getInstance()->getThrowErrors()) {
				$deps[] = \KocujInternalLib\V2a\Classes\Helper::getInstance()->getPrefix().'-exception';
			}
			\KocujInternalLib\V2a\Classes\JsHelper::getInstance()->addScript(\KocujInternalLib\V2a\Classes\Helper::getInstance()->getPrefix().'-backend-review-message', \KocujInternalLib\V2a\Classes\LibUrls::getInstance()->getJavaScriptUrl().'/backend', 'review-message', $deps, \KocujInternalLib\V2a\Classes\Version::getInstance()->getVersion(), true, true, 'kocujILV2aBackendReviewMessageVals', array(
				'prefix'   => \KocujInternalLib\V2a\Classes\Helper::getInstance()->getPrefix(),
				'security' => wp_create_nonce(\KocujInternalLib\V2a\Classes\Helper::getInstance()->getPrefix().'__review_message'),
			), true);
			// set review message to be displayed
			$this->reviewMessageDisplay = true;
		}
	}

	/**
	 * Action for forcing displaying of review message
	 *
	 * @access public
	 * @return void
	 */
	public function actionPrintFooterScripts() {
		// initialize review message script
		if ($this->reviewMessageDisplay) {
			$url = $this->getProjectObj()->getSettingUrl();
			$tellFacebook = $this->getProjectObj()->getSettingSupportTellOthers('facebook');
			$tellTwitter = $this->getProjectObj()->getSettingSupportTellOthers('twitter');
			?>
				<script type="text/javascript">
				/* <![CDATA[ */
					(function($) {
						$(document).ready(function() {
							kocujILV2aBackendReviewMessage.addProject('<?php echo esc_js($this->getProjectObj()->getSettingInternalName()); ?>', '<?php echo esc_js($this->voteUrl); ?>', '<?php echo ((isset($url[0]) /* strlen($url) > 0 */ ) && (isset($tellFacebook[0]) /* strlen($tellFacebook) > 0 */ )) ?
								esc_js($tellFacebook) :
								''
							; ?>', '<?php echo ((isset($url[0]) /* strlen($url) > 0 */ ) && (isset($tellTwitter[0]) /* strlen($tellTwitter) > 0 */ )) ?
								esc_js($tellTwitter) :
								''
							; ?>');
							kocujILV2aBackendReviewMessage.setEvents('<?php echo esc_js($this->getProjectObj()->getSettingInternalName()); ?>');
						});
					}(jQuery));
				/* ]]> */
				</script>
			<?php
		}
	}

	/**
	 * Action for uninstall plugin
	 *
	 * @access public
	 * @return void
	 */
	public function actionPluginUninstall() {
		// remove option
		\KocujInternalLib\V2a\Classes\Helper::getInstance()->deleteOptionForNetworkOrSite($this->getClosedWindowOptionName());
	}
}
