<?php

/**
 * urls.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujInternalLib\V2a\Classes\Project;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * URLS-s class
 *
 * @access public
 */
class Urls extends \KocujInternalLib\V2a\Classes\ProjectObject {
	/**
	 * Project URL
	 *
	 * @access private
	 * @var string
	 */
	private $projectUrl = '';

	/**
	 * Current license directory
	 *
	 * @access private
	 * @var string
	 */
	private $currentLicenseDir = '';

	/**
	 * License URL
	 *
	 * @access private
	 * @var string
	 */
	private $licenseUrl = '';

	/**
	 * Current custom images directory
	 *
	 * @access private
	 * @var string
	 */
	private $currentCustomImagesDir = '';

	/**
	 * Custom images URL
	 *
	 * @access private
	 * @var string
	 */
	private $customImagesUrl = '';

	/**
	 * Current TinyMCE buttons JavaScript directory
	 *
	 * @access private
	 * @var string
	 */
	private $currentTinyMceButtonsJSDir = '';

	/**
	 * TinyMCE buttons JavaScript URL
	 *
	 * @access private
	 * @var string
	 */
	private $tinyMceButtonsJSUrl = '';

	/**
	 * Current TinyMCE buttons PHP directory
	 *
	 * @access private
	 * @var string
	 */
	private $currentTinyMceButtonsPHPDir = '';

	/**
	 * TinyMCE buttons PHP URL
	 *
	 * @access private
	 * @var string
	 */
	private $tinyMceButtonsPHPUrl = '';

	/**
	 * Constructor
	 *
	 * @access public
	 * @param object $projectObj \KocujInternalLib\V2a\Classes\Project object for current project
	 * @return void
	 */
	public function __construct($projectObj) {
		// execute parent constructor
		parent::__construct($projectObj);
		// set URL-s
		$this->projectUrl = ($this->getProjectObj()->getSettingType() === \KocujInternalLib\V2a\Enums\ProjectType::PLUGIN) ?
			plugin_dir_url($this->getProjectObj()->getSettingMainFilename()) :
			get_template_directory_uri();
		$this->currentLicenseDir = $this->getProjectObj()->getObj('dirs')->getLicenseDir();
		$this->licenseUrl = $this->projectUrl;
		$this->currentCustomImagesDir = $this->getProjectObj()->getObj('dirs')->getCustomImagesDir();
		$this->customImagesUrl = $this->projectUrl.'/images';
		$this->currentTinyMceButtonsJSDir = $this->getProjectObj()->getObj('dirs')->getTinyMceButtonsJSDir();
		$this->tinyMceButtonsJSUrl = $this->projectUrl.'/js/tiny-mce';
		$this->currentTinyMceButtonsPHPDir = $this->getProjectObj()->getObj('dirs')->getTinyMceButtonsPHPDir();
		$this->tinyMceButtonsPHPUrl = $this->projectUrl.'/php/tiny-mce';
	}

	/**
	 * Get project URL
	 *
	 * @access public
	 * @return string Project URL
	 */
	public function getProjectUrl() {
		// get project URL
		return $this->projectUrl;
	}

	/**
	 * Get license URL
	 *
	 * @access public
	 * @return string License URL
	 */
	public function getLicenseUrl() {
		// change license URL if license directory has been changed
		$licenseDir = $this->getProjectObj()->getObj('dirs')->getLicenseDir();
		if ($this->currentLicenseDir !== $licenseDir) {
			$this->licenseUrl = $this->projectUrl.'/'.substr($licenseDir, strlen($this->getProjectObj()->getObj('dirs')->getProjectDir()));
			$this->currentLicenseDir = $this->licenseUrl;
		}
		// get license URL
		return $this->licenseUrl;
	}

	/**
	 * Get custom images URL
	 *
	 * @access public
	 * @return string Custom images URL
	 */
	public function getCustomImagesUrl() {
		// change custom images URL if custom images directory has been changed
		$customImagesDir = $this->getProjectObj()->getObj('dirs')->getCustomImagesDir();
		if ($this->currentCustomImagesDir !== $customImagesDir) {
			$this->customImagesUrl = $this->projectUrl.'/'.substr($customImagesDir, strlen($this->getProjectObj()->getObj('dirs')->getProjectDir()));
			$this->currentCustomImagesDir = $this->customImagesUrl;
		}
		// get custom images URL
		return $this->customImagesUrl;
	}

	/**
	 * Get TinyMCE buttons JavaScript URL
	 *
	 * @access public
	 * @return string TinyMCE buttons JavaScript URL
	 */
	public function getTinyMceButtonsJSUrl() {
		// change TinyMCE buttons JavaScript URL if TinyMCE buttons JavaScript directory has been changed
		$tinyMceButtonsJSDir = $this->getProjectObj()->getObj('dirs')->getTinyMceButtonsJSDir();
		if ($this->currentTinyMceButtonsJSDir !== $tinyMceButtonsJSDir) {
			$this->tinyMceButtonsJSUrl = $this->projectUrl.'/'.substr($tinyMceButtonsJSDir, strlen($this->getProjectObj()->getObj('dirs')->getProjectDir()));
			$this->currentTinyMceButtonsJSDir = $this->tinyMceButtonsJSUrl;
		}
		// get TinyMCE buttons JavaScript URL
		return $this->tinyMceButtonsJSUrl;
	}

	/**
	 * Get TinyMCE buttons PHP URL
	 *
	 * @access public
	 * @return string TinyMCE buttons PHP URL
	 */
	public function getTinyMceButtonsPHPUrl() {
		// change TinyMCE buttons PHP URL if TinyMCE buttons PHP directory has been changed
		$tinyMceButtonsPHPDir = $this->getProjectObj()->getObj('dirs')->getTinyMceButtonsPHPDir();
		if ($this->currentTinyMceButtonsPHPDir !== $tinyMceButtonsPHPDir) {
			$this->tinyMceButtonsPHPUrl = $this->projectUrl.'/'.substr($tinyMceButtonsPHPDir, strlen($this->getProjectObj()->getObj('dirs')->getProjectDir()));
			$this->currentTinyMceButtonsPHPDir = $this->tinyMceButtonsPHPUrl;
		}
		// get TinyMCE buttons PHP URL
		return $this->tinyMceButtonsPHPUrl;
	}
}
