<?php

/**
 * project.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib\kocuj_pl_lib
 */

// set namespace
namespace KocujPlLib\V2a\Classes;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Project class
 *
 * @access public
 */
final class Project {
	/**
	 * Project object for Kocuj Internal Lib
	 *
	 * @access private
	 * @var object
	 */
	private $projectKocujInternaLibObj = NULL;

	/**
	 * Kocuj API URL
	 *
	 * @access private
	 * @var string
	 */
	private $apiUrl = '';

	/**
	 * Kocuj API login
	 *
	 * @access private
	 * @var string
	 */
	private $apiLogin = '';

	/**
	 * Kocuj API password
	 *
	 * @access private
	 * @var string
	 */
	private $apiPassword = '';

	/**
	 * List of additional websites
	 *
	 * @access private
	 * @var array
	 */
	private $additionalWebsites = array();

	/**
	 * Classes list for classes with strings which implement \KocujPlLib\V2a\Interfaces\Strings interface
	 *
	 * @access private
	 * @var array
	 */
	private $stringsClassesList = array();

	/**
	 * Classes list for Kocuj.pl Lib which has been replaced by child classed
	 *
	 * @access private
	 * @var array
	 */
	private $libClassesList = array();

	/**
	 * Objects with instances of classes for Kocuj.pl Lib
	 *
	 * @access private
	 * @var array
	 */
	private $libObjects = array();

	/**
	 * Constructor
	 *
	 * @access public
	 * @param object $projectKocujInternaLibObj Project object for Kocuj Internal Lib
	 * @param string $apiUrl Kocuj API URL; should be set without protocol, because "http://" is automatically added to beginning
	 * @param string $apiLogin Kocuj API login
	 * @param string $apiPassword Kocuj API password
	 * @param array $additionalWebsites List of additional websites
	 * @param array $stringsClasses Classes list for classes with strings which implement \KocujPlLib\V2a\Interfaces\Strings interface; if some keys are empty or does not exist, the default classes which returns only empty strings will be used for these keys - default: empty
	 * @param array $libClasses Classes list for Kocuj.pl Lib which has been replaced by child classed; if some keys are empty or does not exist, the default classes will be used for these keys - default: empty
	 * @return void
	 */
	public function __construct($projectKocujInternaLibObj, $apiUrl, $apiLogin, $apiPassword, array $additionalWebsites = array(), array $stringsClasses = array(), array $libClasses = array()) {
		// set Kocuj Internal Lib project object
		if (!($projectKocujInternaLibObj instanceof \KocujInternalLib\V2a\Classes\Project)) {
			throw new Exception(ExceptionCode::OBJECT_IS_NOT_KOCUJ_INTERNAL_LIB_PROJECT, __FILE__, __LINE__);
		}
		$this->projectKocujInternaLibObj = $projectKocujInternaLibObj;
		// set API data
		$this->apiUrl = 'http://'.$apiUrl;
		$this->apiLogin = $apiLogin;
		$this->apiPassword = $apiPassword;
		$this->additionalWebsites = $additionalWebsites;
		// set strings classes
		$this->stringsClassesList = $stringsClasses;
		// set classes
		$this->libClassesList = $libClasses;
		// initialize objects required in administration panel
		if ((is_admin()) || (is_network_admin())) {
			$this->getObj('backend_add-thanks');
			if ((!$this->getObj('backend_add-thanks')->getThanksAddedOptionValue()) && (!\KocujInternalLib\V2a\Classes\Helper::getInstance()->checkUserIPLocal())) {
				$this->getObj('backend_add-thanks-ajax');
				$this->getObj('backend_page-about-add-thanks');
				$this->getObj('backend_review-message-add-thanks');
				$this->getObj('backend_settings-adv-meta-boxes');
			}
		}
	}

	/**
	 * Get project object for Kocuj Internal Lib
	 *
	 * @access public
	 * @return object Project object for Kocuj Internal Lib
	 */
	public function getProjectKocujInternalLibObj() {
		// exit
		return $this->projectKocujInternaLibObj;
	}

	/**
	 * Get Kocuj API URL
	 *
	 * @access public
	 * @return string Kocuj API URL
	 */
	public function getApiUrl() {
		// exit
		return $this->apiUrl;
	}

	/**
	 * Get Kocuj API login
	 *
	 * @access public
	 * @return string Kocuj API login
	 */
	public function getApiLogin() {
		// exit
		return $this->apiLogin;
	}

	/**
	 * Get Kocuj API password
	 *
	 * @access public
	 * @return string Kocuj API password
	 */
	public function getApiPassword() {
		// exit
		return $this->apiPassword;
	}

	/**
	 * Get list of additional websites
	 *
	 * @access public
	 * @return array List of additional websites
	 */
	public function getAdditionalWebsites() {
		// exit
		return $this->additionalWebsites;
	}

	/**
	 * Get object of class type which implements \KocujPlLib\V2a\Interfaces\Strings interface
	 *
	 * @access public
	 * @param string $type Class type
	 * @return object Object of class type which implements \KocujPlLib\V2a\Interfaces\Strings interface
	 */
	public function getStringsObj($type) {
		// optionally create instance of class
		if (isset($this->stringsClassesList[$type])) {
			$interfaces = class_implements($this->stringsClassesList[$type]);
			if (in_array('KocujInternalLib\\V2a\\Interfaces\\Strings', $interfaces)) {
				return call_user_func(array($this->stringsClassesList[$type], 'getInstance'));
			}
		}
		// exit
		return \KocujInternalLib\V2a\Classes\StringsEmpty::getInstance();
	}

	/**
	 * Get object of class type from Kocuj.pl Lib
	 *
	 * @access public
	 * @param string $type Class type
	 * @return object Object of class type from Kocuj.pl Lib
	 */
	public function getObj($type) {
		// optionally initialize object
		if (!isset($this->libObjects[$type])) {
			if (!isset($this->libClassesList[$type])) {
				$div = explode('_', $type);
				if (!empty($div)) {
					foreach ($div as $key => $val) {
						$div[$key] = preg_replace_callback('/\\-([a-z])/', function($matches) {
							return strtoupper($matches[1]);
						}, $val);
						$div[$key] = strtoupper($div[$key][0]).substr($div[$key], 1);
					}
					$className = '\\KocujPlLib\\V2a\\Classes\\Project\\'.implode('\\', $div);
				}
			} else {
				$className = $this->libClassesList[$type];
			}
			$this->libObjects[$type] = (is_subclass_of($className, '\KocujInternalLib\V2a\Classes\ProjectObject')) ?
				new $className($this) :
				NULL;
		}
		// exit
		return $this->libObjects[$type];
	}
}
