<?php

/**
 * q-translate.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2013-2016 Dominik Kocuj
 * @package kocuj_sitemap
 */

// set namespace
namespace KocujSitemapPlugin\Classes\MultipleLanguages;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Multiple languages plugin class - qTranslate
 *
 * @access public
 */
class QTranslate implements \KocujSitemapPlugin\Interfaces\MultipleLanguages {
	/**
	 * Singleton instance
	 *
	 * @access private
	 * @var object
	 */
	private static $instance = NULL;

	/**
	 * Translation plugin functions prefix
	 *
	 * @access protected
	 * @var string
	 */
	protected $functionsPrefix = 'qtrans';

	/**
	 * Languages list prepared (true) or not (false)
	 *
	 * @access private
	 * @var array
	 */
	private $languagesPrepared = false;

	/**
	 * Languages list
	 *
	 * @access protected
	 * @var array
	 */
	protected $languages = array();

	/**
	 * Filter for action before get pages exists (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $beforeGetPagesFilterExists = true;

	/**
	 * Filter for action before get menu exists (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $beforeGetMenuFilterExists = true;

	/**
	 * Constructor
	 *
	 * @access private
	 * @return void
	 */
	private function __construct() {
	}

	/**
	 * Disable cloning of object
	 *
	 * @access private
	 * @return void
	 */
	private function __clone() {
	}

	/**
	 * Get singleton instance
	 *
	 * @access public
	 * @return object Singleton instance
	 */
	public static function getInstance() {
		// optionally create new instance
		if (!self::$instance) {
			self::$instance = new self();
		}
		// exit
		return self::$instance;
	}

	/**
	 * Prepare languages list if any
	 *
	 * @access protected
	 * @return void
	 */
	protected function prepareLanguages() {
		// check if languages list not prepared already
		if ($this->languagesPrepared) {
			return;
		}
		// get languages
		$langs = call_user_func($this->functionsPrefix.'_getSortedLanguages');
		if (!empty($langs)) {
			global $q_config;
			foreach ($langs as $key => $val) {
				if (isset($q_config['locale'][$val])) {
					$this->languages[$q_config['locale'][$val]] = $val;
				}
			}
		}
		// set languages list as prepared
		$this->languagesPrepared = true;
	}

	/**
	 * Get translated data
	 *
	 * @access protected
	 * @param string $origData Original data
	 * @param string $type Data type; can be "text", "url" or "term"
	 * @param string $locale Language locale
	 * @param string $termType Term type; used only when $type="term"; default: empty
	 * @param int $termId Term id; used only when $type="term"; default: empty
	 * @return string Translated data
	 */
	protected function getTranslatedData($origData, $type, $locale, $termType = '', $termId = 0) {
		// prepare languages list
		$this->prepareLanguages();
		// initialize
		$data = $origData;
		// translate data
		if (isset($this->languages[$locale])) {
			switch ($type) {
				case 'text':
					$data = call_user_func_array($this->functionsPrefix.'_use', array(
						$this->languages[$locale],
						$origData,
						false,
					));
					break;
				case 'url':
					$data = call_user_func_array($this->functionsPrefix.'_convertURL', array(
						$origData,
						$this->languages[$locale],
						true,
					));
					break;
				case 'term':
					global $q_config;
					$oldLang = $q_config['language'];
					if (isset($this->languages[$locale])) {
						$q_config['language'] = $this->languages[$locale];
					}
					$data = call_user_func_array($this->functionsPrefix.'_useTermLib', array(
						get_term($termId, $termType),
					));
					$data = (isset($data->name)) ?
						$data->name :
						$origData;
					$q_config['language'] = $oldLang;
					break;
			}
		}
		// exit
		return $data;
	}

	/**
	 * Get translated URL
	 *
	 * @access private
	 * @param string $origURL Original URL
	 * @param string $locale Language locale
	 * @return string Translated URL
	 */
	private function getTranslatedURL($origURL, $locale) {
		// get translated URL
		return $this->getTranslatedData($origURL, 'url', $locale);
	}

	/**
	 * Get translated text
	 *
	 * @access private
	 * @param string $origText Original text
	 * @param string $locale Language locale
	 * @return string Translated text
	 */
	private function getTranslatedText($origText, $locale) {
		// get translated text
		return $this->getTranslatedData($origText, 'text', $locale);
	}

	/**
	 * Get translated term
	 *
	 * @access private
	 * @param string $origText Original term name
	 * @param string $termType Term type
	 * @param int $termId Term id
	 * @param string $locale Language locale
	 * @return string Translated term name
	 */
	private function getTranslatedTerm($origText, $termType, $termId, $locale) {
		// get translated URL
		return $this->getTranslatedData($origText, 'term', $locale, $termType, $termId);
	}

	/**
	 * Get translation plugin name; it is used in administration panel during the selection of multi-lingual plugin
	 *
	 * @access public
	 * @return string Plugin name
	 */
	public function getName() {
		// get plugin name
		return 'qTranslate';
	}

	/**
	 * Get plugin priority; it is used to sort translation plugins usage order; the lowest number is the first plugin in order, the highest number is the last plugin in order
	 *
	 * @access public
	 * @return string Plugin priority
	 */
	public function getPriority() {
		// get plugin priority
		return 12;
	}

	/**
	 * Get languages list
	 *
	 * @access public
	 * @return array Languages list
	 */
	public function getLanguages() {
		// prepare languages list
		$this->prepareLanguages();
		// get languages list
		return $this->languages;
	}

	/**
	 * Get full URL to language flag
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return string URL to language flag
	 */
	public function getLanguageFlag($locale) {
		// initialize
		$output = '';
		// prepare languages list
		$this->prepareLanguages();
		// get locale
		if (isset($this->languages[$locale])) {
			$locale = $this->languages[$locale];
		} else {
			return '';
		}
		// declare global
		global $q_config;
		// check for locale
		if ((!isset($q_config['locale'])) || (!isset($q_config['locale'][$locale]))) {
			return '';
		}
		// get flag image path
		if ((isset($locale[0]) /* strlen($locale) > 0 */ ) && (isset($q_config['flag'][$locale])) && (isset($q_config['flag_location']))) {
			$flagLocation = $q_config['flag_location'].'/'.$q_config['flag'][$locale];
			if (is_file(WP_CONTENT_DIR.'/'.$flagLocation)) {
				$output = content_url().'/'.$flagLocation;
			}
		}
		// exit
		return $output;
	}

	/**
	 * Check if translation plugin exists; this method checks if all functions, classes, methods, properties and global variables, which are used from translation plugin, exists
	 *
	 * @access public
	 * @return bool Translation plugin exists (true) or not (false)
	 */
	public function checkPlugin() {
		// check if qTranslate plugin exists and has required functions, methods and variables implemented
		global $q_config;
		if ((!isset($q_config)) || (!is_array($q_config)) || (!function_exists($this->functionsPrefix.'_getSortedLanguages')) ||
			(!function_exists($this->functionsPrefix.'_use')) || (!function_exists($this->functionsPrefix.'_convertURL')) ||
			(!function_exists($this->functionsPrefix.'_useTermLib')) || (!function_exists($this->functionsPrefix.'_useCurrentLanguageIfNotFoundUseDefaultLanguage'))) {
			return false;
		}
		// exit
		return true;
	}

	/**
	 * Get translation plugin filename with its directory
	 *
	 * @access public
	 * @return string Translation plugin filename
	 */
	public function getPluginFile() {
		// get plugin name
		return 'qtranslate/qtranslate.php';
	}

	/**
	 * Something to do before get blog name
	 *
	 * @access public
	 * @return void
	 */
	public function beforeGetBlogName() {
	}

	/**
	 * Something to do after get blog name
	 *
	 * @access public
	 * @return void
	 */
	public function afterGetBlogName() {
	}

	/**
	 * Get translated blog name
	 *
	 * @access public
	 * @param string $origText Original text
	 * @return string Translated text
	 */
	public function getTranslatedBlogName($origText) {
		// get translated blog name
		return $this->getTranslatedText($origText, get_locale());
	}

	/**
	 * Something to do before get home URL
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetHomeURL($locale) {
	}

	/**
	 * Something to do after get home URL
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetHomeURL($locale) {
	}

	/**
	 * Get translated home URL
	 *
	 * @access public
	 * @param string $origURL Original home URL
	 * @param string $locale Language locale
	 * @return string Translated home URL
	 */
	public function getTranslatedHomeURL($origURL, $locale) {
		// get translated home URL
		return $this->getTranslatedURL($origURL, $locale);
	}

	/**
	 * Something to do before get pages
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetPages($locale) {
		// remove plugin filter
		$value = has_filter('get_pages', $this->functionsPrefix.'_useCurrentLanguageIfNotFoundUseDefaultLanguage');
		if ($value !== false) {
			$this->beforeGetPagesFilterExists = true;
			remove_filter('get_pages', $this->functionsPrefix.'_useCurrentLanguageIfNotFoundUseDefaultLanguage', 0);
		} else {
			$this->beforeGetPagesFilterExists = false;
		}
	}

	/**
	 * Something to do after get pages
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetPages($locale) {
		// add plugin filter again
		if ($this->beforeGetPagesFilterExists) {
			add_filter('get_pages', $this->functionsPrefix.'_useCurrentLanguageIfNotFoundUseDefaultLanguage', 0);
		}
	}

	/**
	 * Something to do before get page item
	 *
	 * @access public
	 * @param int $pageId Page id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetPageItem($pageId, $locale) {
	}

	/**
	 * Something to do after get page item
	 *
	 * @access public
	 * @param int $pageId Page id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetPageItem($pageId, $locale) {
	}

	/**
	 * Get translated page URL
	 *
	 * @access public
	 * @param string $origURL Original page URL
	 * @param int $pageId Page id
	 * @param string $locale Language locale
	 * @return string Translated page URL
	 */
	public function getTranslatedPageURL($origURL, $pageId, $locale) {
		// get translated page URL
		return $this->getTranslatedURL($origURL, $locale);
	}

	/**
	 * Get translated page title
	 *
	 * @access public
	 * @param string $origTitle Original page title
	 * @param int $pageId Page id
	 * @param string $locale Language locale
	 * @return string Translated page title
	 */
	public function getTranslatedPageTitle($origTitle, $pageId, $locale) {
		// get translated page title
		return $this->getTranslatedText($origTitle, $locale);
	}

	/**
	 * Something to do before get posts
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetPosts($locale) {
	}

	/**
	 * Something to do after get posts
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetPosts($locale) {
	}

	/**
	 * Something to do before get post item
	 *
	 * @access public
	 * @param int $postId Post id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetPostItem($postId, $locale) {
	}

	/**
	 * Something to do after get post item
	 *
	 * @access public
	 * @param int $postId Post id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetPostItem($postId, $locale) {
	}

	/**
	 * Get translated post URL
	 *
	 * @access public
	 * @param string $origURL Original post URL
	 * @param int $postId Post id
	 * @param string $locale Language locale
	 * @return string Translated post URL
	 */
	public function getTranslatedPostURL($origURL, $postId, $locale) {
		// get translated post URL
		return $this->getTranslatedURL($origURL, $locale);
	}

	/**
	 * Get translated post title
	 *
	 * @access public
	 * @param string $origTitle Original post title
	 * @param int $postId Post id
	 * @param string $locale Language locale
	 * @return string Translated post title
	 */
	public function getTranslatedPostTitle($origTitle, $postId, $locale) {
		// get translated post title
		return $this->getTranslatedText($origTitle, $locale);
	}

	/**
	 * Something to do before get categories
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetCategories($locale) {
	}

	/**
	 * Something to do after get categories
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetCategories($locale) {
	}

	/**
	 * Something to do before get category item
	 *
	 * @access public
	 * @param int $categoryId Category id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetCategoryItem($categoryId, $locale) {
	}

	/**
	 * Something to do after get category item
	 *
	 * @access public
	 * @param int $categoryId Category id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetCategoryItem($categoryId, $locale) {
	}

	/**
	 * Get translated category URL
	 *
	 * @access public
	 * @param string $origURL Original category URL
	 * @param int $categoryId Category id
	 * @param string $locale Language locale
	 * @return string Translated category URL
	 */
	public function getTranslatedCategoryURL($origURL, $categoryId, $locale) {
		// get translated category URL
		return $this->getTranslatedURL($origURL, $locale);
	}

	/**
	 * Get translated category title
	 *
	 * @access public
	 * @param string $origTitle Original category title
	 * @param int $categoryId Category id
	 * @param string $locale Language locale
	 * @return string Translated category title
	 */
	public function getTranslatedCategoryTitle($origTitle, $categoryId, $locale) {
		// get translated category title
		return $this->getTranslatedTerm($origTitle, 'category', $categoryId, $locale);
	}

	/**
	 * Something to do before get taxonomies
	 *
	 * @access public
	 * @param string $taxonomy Taxonomy type
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetTaxonomies($taxonomy, $locale) {
	}

	/**
	 * Something to do after get taxonomies
	 *
	 * @access public
	 * @param string $taxonomy Taxonomy type
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetTaxonomies($taxonomy, $locale) {
	}

	/**
	 * Something to do before get taxonomy item
	 *
	 * @access public
	 * @param int $termId Term id
	 * @param string $taxonomy Taxonomy type
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetTaxonomyItem($termId, $taxonomy, $locale) {
	}

	/**
	 * Something to do after get taxonomy item
	 *
	 * @access public
	 * @param int $termId Taxonomy id
	 * @param string $taxonomy Taxonomy type
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetTaxonomyItem($termId, $taxonomy, $locale) {
	}

	/**
	 * Get translated taxonomy URL
	 *
	 * @access public
	 * @param string $origURL Original taxonomy URL
	 * @param int $termId Taxonomy id
	 * @param string $taxonomy Taxonomy type
	 * @param string $locale Language locale
	 * @return string Translated taxonomy URL
	 */
	public function getTranslatedTaxonomyURL($origURL, $termId, $taxonomy, $locale) {
		// get translated taxonomy URL
		return $this->getTranslatedURL($origURL, $locale);
	}

	/**
	 * Get translated taxonomy title
	 *
	 * @access public
	 * @param string $origTitle Original taxonomy title
	 * @param int $termId Taxonomy id
	 * @param string $taxonomy Taxonomy type
	 * @param string $locale Language locale
	 * @return string Translated taxonomy title
	 */
	public function getTranslatedTaxonomyTitle($origTitle, $termId, $taxonomy, $locale) {
		// get translated taxonomy title
		return $this->getTranslatedTerm($origTitle, $taxonomy, $termId, $locale);
	}

	/**
	 * Something to do before get menu
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetMenu($locale) {
		// remove plugin filter
		$value = has_filter('wp_setup_nav_menu_item', $this->functionsPrefix.'_useCurrentLanguageIfNotFoundUseDefaultLanguage');
		if ($value !== false) {
			$this->beforeGetMenuFilterExists = true;
			remove_filter('wp_setup_nav_menu_item', $this->functionsPrefix.'_useCurrentLanguageIfNotFoundUseDefaultLanguage');
		} else {
			$this->beforeGetMenuFilterExists = false;
		}
	}

	/**
	 * Something to do after get menu
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetMenu($locale) {
		// add plugin filter again
		if ($this->beforeGetMenuFilterExists) {
			add_filter('wp_setup_nav_menu_item', $this->functionsPrefix.'_useCurrentLanguageIfNotFoundUseDefaultLanguage');
		}
	}

	/**
	 * Something to do before get menu item
	 *
	 * @access public
	 * @param int $itemId Menu item id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetMenuItem($itemId, $locale) {
	}

	/**
	 * Something to do after get menu item
	 *
	 * @access public
	 * @param int $itemId Menu item id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetMenuItem($itemId, $locale) {
	}

	/**
	 * Get translated menu URL
	 *
	 * @access public
	 * @param string $origURL Original menu URL
	 * @param int $menuId Menu id
	 * @param string $locale Language locale
	 * @return string Translated menu URL
	 */
	public function getTranslatedMenuURL($origURL, $menuId, $locale) {
		// get translated menu URL
		return $this->getTranslatedURL($origURL, $locale);
	}

	/**
	 * Get translated menu title
	 *
	 * @access public
	 * @param string $origTitle Original menu title
	 * @param int $menuId Menu id
	 * @param string $locale Language locale
	 * @return string Translated menu title
	 */
	public function getTranslatedMenuTitle($origTitle, $menuId, $locale) {
		// get translated menu title
		return $this->getTranslatedText($origTitle, $locale);
	}

	/**
	 * Something to do before get authors
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetAuthors($locale) {
	}

	/**
	 * Something to do after get authors
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetAuthors($locale) {
	}

	/**
	 * Something to do before get author item
	 *
	 * @access public
	 * @param int $authorId Author id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetAuthorItem($authorId, $locale) {
	}

	/**
	 * Something to do after get author item
	 *
	 * @access public
	 * @param int $authorId Author id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetAuthorItem($authorId, $locale) {
	}

	/**
	 * Get translated author URL
	 *
	 * @access public
	 * @param string $origURL Original author URL
	 * @param int $authorId Author id
	 * @param string $locale Language locale
	 * @return string Translated author URL
	 */
	public function getTranslatedAuthorURL($origURL, $authorId, $locale) {
		// get translated author URL
		return $this->getTranslatedURL($origURL, $locale);
	}

	/**
	 * Get translated author name
	 *
	 * @access public
	 * @param string $origName Original author name
	 * @param int $authorId Author id
	 * @param string $locale Language locale
	 * @return string Translated author name
	 */
	public function getTranslatedAuthorName($origName, $authorId, $locale) {
		// get translated author name
		return $this->getTranslatedText($origName, $locale);
	}

	/**
	 * Something to do before get tags
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetTags($locale) {
	}

	/**
	 * Something to do after get tags
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetTags($locale) {
	}

	/**
	 * Something to do before get tag item
	 *
	 * @access public
	 * @param int $tagId Tag id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetTagItem($tagId, $locale) {
	}

	/**
	 * Something to do after get tag item
	 *
	 * @access public
	 * @param int $tagId Tag id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetTagItem($tagId, $locale) {
	}

	/**
	 * Get translated tag URL
	 *
	 * @access public
	 * @param string $origURL Original tag URL
	 * @param int $tagId Tag id
	 * @param string $locale Language locale
	 * @return string Translated tag URL
	 */
	public function getTranslatedTagURL($origURL, $tagId, $locale) {
		// get translated tag URL
		return $this->getTranslatedURL($origURL, $locale);
	}

	/**
	 * Get translated tag name
	 *
	 * @access public
	 * @param string $origName Original tag name
	 * @param int $tagId Tag id
	 * @param string $locale Language locale
	 * @return string Translated tag name
	 */
	public function getTranslatedTagName($origName, $tagId, $locale) {
		// get translated tag name
		return $this->getTranslatedTerm($origName, 'post_tag', $tagId, $locale);
	}

	/**
	 * Something to do before get custom posts
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetCustomPosts($locale) {
	}

	/**
	 * Something to do after get custom posts
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetCustomPosts($locale) {
	}

	/**
	 * Something to do before get custom post item
	 *
	 * @access public
	 * @param int $customPostId Custom post id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetCustomPostItem($customPostId, $locale) {
	}

	/**
	 * Something to do after get custom post item
	 *
	 * @access public
	 * @param int $customPostId Custom post id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetCustomPostItem($customPostId, $locale) {
	}

	/**
	 * Get translated custom post URL
	 *
	 * @access public
	 * @param string $origURL Original custom post URL
	 * @param int $customPostId Custom post id
	 * @param string $locale Language locale
	 * @return string Translated custom post URL
	 */
	public function getTranslatedCustomPostURL($origURL, $customPostId, $locale) {
		// get translated custom post URL
		return $this->getTranslatedURL($origURL, $locale);
	}

	/**
	 * Get translated custom post title
	 *
	 * @access public
	 * @param string $origTitle Original custom post title
	 * @param int $customPostId Custom post id
	 * @param string $locale Language locale
	 * @return string Translated custom post title
	 */
	public function getTranslatedCustomPostTitle($origTitle, $customPostId, $locale) {
		// get translated custom post title
		return $this->getTranslatedText($origTitle, $locale);
	}
}
