<?php

/**
 * component.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujInternalLib\V3a\Classes\Project\Components\Core\Meta;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Meta class
 *
 * @access public
 */
class Component extends \KocujInternalLib\V3a\Classes\ComponentObject {
	/**
	 * Meta data
	 *
	 * @access private
	 * @var array
	 */
	private $metaData = array();

	/**
	 * Clear meta data (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $clearOptions = false;

	/**
	 * Constructor
	 *
	 * @access public
	 * @param object $projectObj \KocujInternalLib\V3a\Classes\Project object for current project
	 * @return void
	 */
	public function __construct($projectObj) {
		// execute parent constructor
		parent::__construct($projectObj);
		// get meta data
		$value = \KocujInternalLib\V3a\Classes\Helper::getInstance()->getOptionForNetworkOrSite($this->getMetaOptionName());
		if ($value !== false) {
			$this->metaData = maybe_unserialize($value);
		}
	}

	/**
	 * Destructor
	 *
	 * @access public
	 * @return void
	 */
	public function __destruct() {
		// update or clear meta data
		if ($this->clearOptions) {
			$this->forceRealClearDataNow();
		} else {
			$this->forceRealUpdateNow();
		}
	}

	/**
	 * Get meta option name
	 *
	 * @access public
	 * @return string Meta option name
	 */
	public function getMetaOptionName() {
		// exit
		return $this->getProjectObj()->getMainSettingInternalName().'__meta';
	}

	/**
	 * Add or update meta option
	 *
	 * @access public
	 * @param string $optionName Option name
	 * @param array|bool|float|int|string $optionValue Option value
	 * @return void
	 */
	public function addOrUpdateOption($optionName, $optionValue) {
		// set options to not clear
		$this->clearOptions = false;
		// add or update option
		$this->metaData[$optionName] = $optionValue;
	}

	/**
	 * Delete meta option
	 *
	 * @access public
	 * @param string $optionName Option name
	 * @return void
	 */
	public function deleteOption($optionName) {
		// delete option
		if (isset($this->metaData[$optionName])) {
			unset($this->metaData[$optionName]);
		}
	}

	/**
	 * Get meta option value
	 *
	 * @access public
	 * @param string $optionName Option name
	 * @param array|bool|float|int|string $defaultOptionValue Default option value - default: false
	 * @return array|bool|float|int|string Option value
	 */
	public function getOption($optionName, $defaultOptionValue = false) {
		// exit
		return (isset($this->metaData[$optionName])) ?
			$this->metaData[$optionName] :
			$defaultOptionValue;
	}

	/**
	 * Clear options
	 *
	 * @access public
	 * @return void
	 */
	public function clearOptions() {
		// set options to clear
		$this->clearOptions = true;
		// clear options in memory
		$this->metaData = array();
	}

	/**
	 * Force real update of the entire meta data now
	 *
	 * @access public
	 * @return void
	 */
	public function forceRealUpdateNow() {
		// update meta data
		\KocujInternalLib\V3a\Classes\Helper::getInstance()->addOrUpdateOptionForNetworkOrSite($this->getMetaOptionName(), serialize($this->metaData));
	}

	/**
	 * Force real clear entire meta data now
	 *
	 * @access public
	 * @return void
	 */
	public function forceRealClearDataNow() {
		// clear meta data
		$this->clearOptions();
		\KocujInternalLib\V3a\Classes\Helper::getInstance()->deleteOptionForNetworkOrSite($this->getMetaOptionName());
	}
}
