<?php

/**
 * component.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujInternalLib\V3a\Classes\Project\Components\Backend\Message;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Message class
 *
 * @access public
 */
class Component extends \KocujInternalLib\V3a\Classes\ComponentObject {
	/**
	 * Messages
	 *
	 * @access private
	 * @var array
	 */
	private $messages = array();

	/**
	 * Add message
	 *
	 * @access private
	 * @param string $id Message identifier; must be unique
	 * @param string $content Message content
	 * @param int $type Message type; must be one of the following constants: \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::INFORMATION (for information message), \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::WARNING (for warning message), \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::ERROR (for error message) or \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::SUCCESS (for success message)
	 * @param int $closable Message closable status; must be one of the following constants: \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageClosable::NOT_CLOSABLE (for message without closing button) or \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageClosable::CLOSABLE (for message with closing button)
	 * @param bool $allPages Message will be displayed on all pages in administration panel (true) or not (false)
	 * @param array $pages Settings pages list for current project where this message should be displayed; it should be identifiers from settings page for current project only; if it is empty, this message will be displayed on all settings pages for current project
	 * @param array $attr Additional attributes; there are available the following attributes: "class" (string type; CSS class for message div), "closescript" (string type; JavaScript code executed when user clicked on the message close button; it works only if "closable" parameter is set to \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageClosable::CLOSABLE), "style" (string type; CSS style for message div)
	 * @return void
	 */
	private function addMessageForProjectOrAllPages($id, $content, $type, $closable, $allPages, array $pages, array $attr) {
		// check if this id does not already exists
		if (isset($this->messages[$id])) {
			return;
		}
		// add message
		$this->messages[$id] = array(
			'content'  => $content,
			'type'     => $type,
			'closable' => $closable,
		);
		if ($allPages) {
			$this->messages[$id]['allpages'] = true;
		} else {
			$this->messages[$id]['pages'] = $pages;
		}
		if ((isset($attr['class'])) && (isset($attr['class'][0]) /* strlen($attr['class']) > 0 */ )) {
			$this->messages[$id]['class'] = $attr['class'];
		}
		if ((isset($attr['style'])) && (isset($attr['style'][0]) /* strlen($attr['style']) > 0 */ )) {
			$this->messages[$id]['style'] = $attr['style'];
		}
		if (($closable === \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageClosable::CLOSABLE) && ((isset($attr['closescript'])) && (isset($attr['closescript'][0]) /* strlen($attr['closescript']) > 0 */ ))) {
			$this->messages[$id]['closescript'] = $attr['closescript'];
		}
	}

	/**
	 * Add message
	 *
	 * @access public
	 * @param string $id Message identifier; must be unique
	 * @param string $content Message content
	 * @param int $type Message type; must be one of the following constants: \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::INFORMATION (for information message), \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::WARNING (for warning message), \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::ERROR (for error message) or \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::SUCCESS (for success message) - default: \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::INFORMATION
	 * @param int $closable Message closable status; must be one of the following constants: \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageClosable::NOT_CLOSABLE (for message without closing button) or \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageClosable::CLOSABLE (for message with closing button) - default: \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageClosable::NOT_CLOSABLE
	 * @param array $pages Settings pages list for current project where this message should be displayed; it should be identifiers from settings page for current project only; if it is empty, this message will be displayed on all settings pages for current project - default: empty
	 * @param array $attr Additional attributes; there are available the following attributes: "class" (string type; CSS class for message div), "closescript" (string type; JavaScript code executed when user clicked on the message close button; it works only if "closable" parameter is set to \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageClosable::CLOSABLE), "style" (string type; CSS style for message div) - default: empty
	 * @return void
	 */
	public function addMessage($id, $content, $type = \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::INFORMATION, $closable = \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageClosable::NOT_CLOSABLE, array $pages = array(), array $attr = array()) {
		// add message
		$this->addMessageForProjectOrAllPages($id, $content, $type, $closable, false, $pages, $attr);
	}

	/**
	 * Add message for all pages in administration panel
	 *
	 * @access public
	 * @param string $id Message identifier; must be unique
	 * @param string $content Message content
	 * @param int $type Message type; must be one of the following constants: \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::INFORMATION (for information message), \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::WARNING (for warning message), \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::ERROR (for error message) or \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::SUCCESS (for success message) - default: \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::INFORMATION
	 * @param int $closable Message closable status; must be one of the following constants: \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageClosable::NOT_CLOSABLE (for message without closing button) or \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageClosable::CLOSABLE (for message with closing button) - default: \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageClosable::NOT_CLOSABLE
	 * @param array $attr Additional attributes; there are available the following attributes: "class" (string type; CSS class for message div), "closescript" (string type; JavaScript code executed when user clicked on the message close button; it works only if "closable" parameter is set to \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageClosable::CLOSABLE), "style" (string type; CSS style for message div) - default: empty
	 * @return void
	 */
	public function addMessageForAllPages($id, $content, $type = \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::INFORMATION, $closable = \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageClosable::NOT_CLOSABLE, array $attr = array()) {
		// add message
		$this->addMessageForProjectOrAllPages($id, $content, $type, $closable, true, array(), $attr);
	}

	/**
	 * Get messages data
	 *
	 * @access public
	 * @param string $id Message identifier
	 * @return array Message data; each message data have the following fields: "content" (content of message), "type" (type of message); there are also the following fields which should exists or not: "allpages" (if it is set to true, message will be displayed in all pages in administration panel), "style" (string type; CSS style for message div), "class" (string type; CSS class for message div), "closescript" (string type; JavaScript code executed when user clicked on the message close button; it works only if "closable" parameter is set to \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageClosable::CLOSABLE), "pages" (settings pages list for current project on which message is displayed or empty if it is displayed on all settings pages for current project)
	 */
	public function getMessages() {
		// exit
		return $this->messages;
	}

	/**
	 * Get message data by id
	 *
	 * @access public
	 * @param string $id Settings meta box identifier
	 * @return array|bool Message data or false if not exists; message data have the following fields: "content" (content of message), "type" (type of message); there are also the following fields which should exists or not: "allpages" (is it is set to true, message will be displayed in all pages in administration panel), "style" (string type; CSS style for message div), "class" (string type; CSS class for message div), "closescript" (string type; JavaScript code executed when user clicked on the message close button; it works only if "closable" parameter is set to \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageClosable::CLOSABLE), "pages" (settings pages list for current project on which message is displayed or empty if it is displayed on all settings pages for current project)
	 */
	public function getMessage($id) {
		// exit
		return (isset($this->message[$id])) ?
			$this->message[$id] :
			false;
	}

	/**
	 * Action for admin notices
	 *
	 * @access public
	 * @return void
	 */
	public function actionAdminNotices() {
		// show messages
		if (!empty($this->messages)) {
			foreach ($this->messages as $id => $data) {
				// check if this message should be on this page
				$display = true;
				if (!((isset($data['allpages'])) && ($data['allpages']))) {
					$display = (empty($this->messages[$id]['pages'])) ?
						$this->getProjectObj()->getComponent('backend', 'settings-menu')->checkCurrentPageIsSettingsForProject() :
						in_array($this->getProjectObj()->getComponent('backend', 'settings-menu')->getCurrentSettingsMenu(), $this->messages[$id]['pages']);
				}
				// show message
				if ($display) {
					// get message class
					$class = 'notice';
					switch ($data['type']) {
						case \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::INFORMATION:
							$class .= ' notice-info';
							break;
						case \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::WARNING:
							$class .= ' notice-warning';
							break;
						case \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::ERROR:
							$class .= ' notice-error';
							break;
						case \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageType::SUCCESS:
							$class .= ' notice-success';
							break;
						default:
					}
					if ($data['closable'] === \KocujInternalLib\V3a\Enums\Project\Components\Backend\Message\MessageClosable::CLOSABLE) {
						$class .= ' is-dismissible';
					}
					// prepare style
					$style = (isset($data['style'])) ?
						' style="'.esc_attr($data['style']).'"' :
						'';
					// prepare additional class
					if (isset($data['class'])) {
						$class .= ' '.$data['class'];
					}
					// set message identifier
					$messageId = $this->getProjectObj()->getComponent('core', 'project-helper')->getPrefix().'__message_'.$id;
					// show message
					echo '<div'.$style.' id="'.esc_attr($messageId).'" class="'.esc_attr($class).'"><p>'.$data['content'].'</p></div>';
					// show script for closing window
					if (isset($data['closescript'])) {
						?>
							<script type="text/javascript">
							/* <![CDATA[ */
								(function($) {
									$(document).ready(function() {
										$('<?php echo esc_js('#'.$messageId.' .notice-dismiss'); ?>').click(function() {
											<?php echo $data['closescript']; ?>
										});
									});
								}(jQuery));
							/* ]]> */
							</script>
						<?php
					}
				}
			}
		}
	}
}
