<?php

/**
 * component.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujInternalLib\V3a\Classes\Project\Components\Backend\SettingsMetaBoxes;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Meta boxes for settings class
 *
 * @access public
 */
class Component extends \KocujInternalLib\V3a\Classes\ComponentObject {
	/**
	 * Settings meta boxes data
	 *
	 * @access private
	 * @var array
	 */
	private $settingsMetaBoxes = array();

	/**
	 * Add settings meta box
	 *
	 * @access public
	 * @param string $id Settings meta box id; must be unique in this project
	 * @param string $title Settings meta box title
	 * @param string $content Settings meta box content
	 * @param array $pages Settings pages list for current project where settings meta box should be displayed; it should be identifiers from settings page for current project only; if it is empty, meta box will be displayed on all settings pages for current project - default: empty
	 * @return void
	 */
	public function addSettingsMetaBox($id, $title, $content, array $pages = array()) {
		// check if this id does not already exists
		if (isset($this->settingsMetaBoxes[$id])) {
			return;
		}
		// add settings meta box
		$this->settingsMetaBoxes[$id] = array(
			'title'   => $title,
			'content' => $content,
			'pages'   => $pages,
		);
	}

	/**
	 * Get settings meta boxes data
	 *
	 * @access public
	 * @return array Settings meta boxes data; each settings meta box data have the following fields: "pages" (settings pages list for current project on which meta box is displayed or empty if it is displayed on all settings pages for current project), "title" (title of meta box), "content" (content of meta box)
	 */
	public function getSettingsMetaBoxes() {
		// exit
		return $this->settingsMetaBoxes;
	}

	/**
	 * Get settings meta box data by id
	 *
	 * @access public
	 * @param string $id Settings meta box identifier
	 * @return array Selected settings meta box data or false if not exists; meta box data have the following fields: "pages" (settings pages list for current project on which meta box is displayed or empty if it is displayed on all settings pages for current project), "title" (title of meta box), "content" (content of meta box)
	 */
	public function getSettingsMetaBox($id) {
		// exit
		return (isset($this->settingsMetaBoxes[$id])) ?
			$this->settingsMetaBoxes[$id] :
			false;
	}

	/**
	 * Check if settings meta box with the selected identifier should be displayed on current page
	 *
	 * @access private
	 * @param string $id Settings meta box identifier to check
	 * @return bool There should be displayed settings meta box on current page (true) or not (false)
	 */
	private function checkIfDisplaySettingsMetaBox($id) {
		// check if settings meta box exists
		if (!isset($this->settingsMetaBoxes[$id])) {
			return false;
		}
		// exit
		return (empty($this->settingsMetaBoxes[$id]['pages'])) ?
			$this->getProjectObj()->getComponent('backend', 'settings-menu')->checkCurrentPageIsSettingsForProject() :
			in_array($this->getProjectObj()->getComponent('backend', 'settings-menu')->getCurrentSettingsMenu(), $this->settingsMetaBoxes[$id]['pages']);
	}

	/**
	 * Check if any settings meta box should be displayed on current page
	 *
	 * @access private
	 * @return bool There should be displayed settings meta box on current page (true) or not (false)
	 */
	private function checkIfDisplayAnySettingsMetaBox() {
		// check if there are any settings meta boxes
		if (empty($this->settingsMetaBoxes)) {
			return false;
		}
		// check each settings meta box
		foreach ($this->settingsMetaBoxes as $key => $val) {
			if ($this->checkIfDisplaySettingsMetaBox($key)) {
				return true;
			}
		}
		// exit
		return false;
	}

	/**
	 * Action for adding JavaScript scripts
	 *
	 * @access public
	 * @return void
	 */
	public function actionEnqueueScripts() {
		// check if scripts should be added
		if (!$this->checkIfDisplayAnySettingsMetaBox()) {
			return;
		}
		// add scripts
		wp_enqueue_script('post');
	}

	/**
	 * Action after form div element
	 *
	 * @access public
	 * @return void
	 */
	public function actionAfterFormDiv() {
		// check if scripts should be added
		if (!$this->checkIfDisplayAnySettingsMetaBox()) {
			return;
		}
		// show settings meta boxes
		?>
			<script type="text/javascript">
			/* <![CDATA[ */
				(function($) {
					'use strict';
					$(document).ready(function() {
						if ($('#kocujinternallib_div_form').length > 0) {
							$('#kocujinternallib_div_form').width($('.wrap').width()-300);
						}
						if ($('<?php echo esc_js('#'.$this->getProjectObj()->getComponent('core', 'project-helper')->getPrefix().'__div_wrapinside'); ?>').length > 0) {
							$('<?php echo esc_js('#'.$this->getProjectObj()->getComponent('core', 'project-helper')->getPrefix().'__div_wrapinside'); ?>').width($('.wrap').width()-300);
						}
					});
				}(jQuery));
			/* ]]> */
			</script>
			<div id="poststuff">
				<div id="post-body">
					<div style="float:right;width:280px;" class="postbox-container">
						<?php
							foreach ($this->settingsMetaBoxes as $id => $data) {
								if ($this->checkIfDisplaySettingsMetaBox($id)) {
									add_meta_box('kocuj_internal_lib_settings_meta_box_'.$id, $data['title'], array($this, 'metaBoxShow'), 'kocuj_internal_lib', 'advanced', 'default', array(
										'id' => $id,
									));
								}
							}
							do_meta_boxes('kocuj_internal_lib', 'advanced', NULL);
							foreach ($this->settingsMetaBoxes as $id => $data) {
								if ($this->checkIfDisplaySettingsMetaBox($id)) {
									remove_meta_box('kocuj_internal_lib_settings_meta_box_'.$id, 'kocuj_internal_lib', 'advanced', 'default');
								}
							}
							?>
								<script type="text/javascript">
								/* <![CDATA[ */
									(function($) {
										'use strict';
										$(document).ready(function($) {
											$('.if-js-closed').removeClass('if-js-closed').addClass('closed');
											postboxes.add_postbox_toggles('kocujinternallib');
											$('.meta-box-sortables').sortable({
												disabled: true
											});
											$('.meta-box-sortables').css('margin-top', '-10px');
											$('.postbox-container .handlediv').click(function() {
												var classes = $(this).parent().attr('class').split(/\s+/);
												var closed = false;
												if (classes.length > 0) {
													for (var z=0; z<classes.length; z++) {
														if (classes[z] === 'closed') {
															closed = true;
															break;
														}
													}
												}
												if (closed) {
													$(this).parent().removeClass('closed');
												} else {
													$(this).parent().addClass('closed');
												}
											});
										});
									}(jQuery));
								/* ]]> */
								</script>
							<?php
							wp_nonce_field('closedpostboxes', 'closedpostboxesnonce', false);
							wp_nonce_field('meta-box-order', 'meta-box-order-nonce', false);
						?>
					</div>
				</div>
			</div>
		<?php
	}

	/**
	 * Filter for adding style to form div element
	 *
	 * @access public
	 * @param string $style Form div element style
	 * @return string Form div element style
	 */
	public function filterFormDivStyle($style) {
		// check if style should be added to form
		if (!$this->checkIfDisplayAnySettingsMetaBox()) {
			return $style;
		}
		// add style to form
		if (isset($style[0]) /* strlen($style) > 0 */ ) {
			$style .= ' ';
		}
		$style .= 'float:left;';
		// exit
		return $style;
	}

	/**
	 * Show settings meta box
	 *
	 * @access public
	 * @param object $post Post data; it is always set to NULL
	 * @param object $data Data
	 * @return void
	 */
	public function metaBoxShow($post, $data) {
		// show settings meta box content
		$id = $data['args']['id'];
		if (!isset($this->settingsMetaBoxes[$id])) {
			return;
		}
		echo $this->settingsMetaBoxes[$id]['content'];
	}
}
