<?php

/**
 * sitemap.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2013-2016 Dominik Kocuj
 * @package kocuj_sitemap
 */

// set namespace
namespace KocujSitemapPlugin\Classes;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Plugin sitemap class
 *
 * @access public
 */
class Sitemap {
	/**
	 * Minimal length of additional serialization meta data for each link after making it shorter
	 *
	 * @access public
	 */
	const LINK_SHORTER_META_LENGTH = 15;

	/**
	 * Standard position of website root URL when link is in format '<a href="'; if link format is other than that, this constants should be also changed
	 *
	 * @access public
	 */
	const LINK_ROOT_URL_STANDARD_POS = 9;

	/**
	 * Singleton instance
	 *
	 * @access private
	 * @var object
	 */
	private static $instance = NULL;

	/**
	 * Home link text position
	 *
	 * @access private
	 * @var int
	 */
	private $homeLinkTextPos = -1;

	/**
	 * Elements types
	 *
	 * @access private
	 * @var array
	 */
	private $elementsTypes = array();

	/**
	 * Constructor
	 *
	 * @access private
	 * @return void
	 */
	private function __construct() {
		// initialize data types in default order
		$types = array(
			'home',
			'menus',
			'posts',
			'pages',
			'authors',
			'tags',
			'custom',
		);
		// initialize data types letters
		$this->elementsTypes = array();
		foreach ($types as $type) {
			$typeFirstCapital = ucfirst($type);
			$classTemp = '\\KocujSitemapPlugin\\Classes\\ElementType\\Frontend\\'.$typeFirstCapital;
			$obj = new $classTemp();
			$element = array(
				'type'              => $type,
				'object'            => $obj,
				'configurableorder' => $obj->checkConfigurableOrder(),
				'letter'            => $obj->getTypeLetter(),
				'classnamebackend'  => '\\KocujSitemapPlugin\\Classes\\ElementType\\Backend\\'.$typeFirstCapital,
			);
			$letter = $element['object']->getTypeLetter();
			if (!isset($this->elementsTypes[$letter])) {
				$this->elementsTypes[$letter] = $element;
			}
		}
		// set cache actions and filters for administration panel
		$actions = array();
		$filters = array();
		if ((is_admin()) || (is_network_admin())) {
			if (!empty($this->elementsTypes)) {
				foreach ($this->elementsTypes as $type => $element) {
					$object = $this->getElementTypeObject($type);
					$adminobject = $this->getElementTypeAdminObject($type);
					if ($object->checkExists()) {
						$actions = array_merge($actions, $adminobject->getAdminCacheActions());
						$filters = array_merge($filters, $adminobject->getAdminCacheFilters());
					}
				}
			}
		}
		// initialize cache class
		Cache::getInstance($actions, $filters);
		// add shortcode
		add_shortcode('KocujSitemap', array($this, 'shortcodeSitemap'));
	}

	/**
	 * Disable cloning of object
	 *
	 * @access private
	 * @return void
	 */
	private function __clone() {
	}

	/**
	 * Get singleton instance
	 *
	 * @access public
	 * @return object Singleton instance
	 */
	public static function getInstance() {
		// optionally create new instance
		if (!self::$instance) {
			self::$instance = new self();
		}
		// exit
		return self::$instance;
	}

	/**
	 * Get home link text position
	 *
	 * @access public
	 * @return int Home link text position
	 */
	public function getHomeLinkTextPos() {
		// get home link text position
		return $this->homeLinkTextPos;
	}

	/**
	 * Set home link text position
	 *
	 * @access public
	 * @param int $homeLinkTextPos Home link text position
	 * @return void
	 */
	public function setHomeLinkTextPos($homeLinkTextPos) {
		// set home link text position
		$this->homeLinkTextPos = $homeLinkTextPos;
	}

	/**
	 * Get elements types
	 *
	 * @access public
	 * @param bool $onlyConfigurableOrder Get only elements types with configurable order (true) or not (false)
	 * @return array Elements types
	 */
	public function getElementsTypes($onlyConfigurableOrder = false) {
		// get elements types
		if (!$onlyConfigurableOrder) {
			return $this->elementsTypes;
		} else {
			$output = array();
			if (!empty($this->elementsTypes)) {
				foreach ($this->elementsTypes as $key => $type) {
					if ($type['configurableorder']) {
						$output[$key] = $type;
					}
				}
			}
			return $output;
		}
	}

	/**
	 * Get element type
	 *
	 * @access public
	 * @param string $type Element type
	 * @return array Element type
	 */
	public function getElementType($type) {
		// get element type
		if ((isset($this->elementsTypes[$type])) && ($this->elementsTypes[$type]['object']->checkExists())) {
			return $this->elementsTypes[$type];
		}
		// exit
		return NULL;
	}

	/**
	 * Get element type class object
	 *
	 * @access public
	 * @param string $type Element type
	 * @return object Element type class
	 */
	public function getElementTypeObject($type) {
		// get element type
		if ((isset($this->elementsTypes[$type])) && ($this->elementsTypes[$type]['object']->checkExists())) {
			return $this->elementsTypes[$type]['object'];
		}
		// exit
		return NULL;
	}

	/**
	 * Get element type administration panel class object
	 *
	 * @access public
	 * @param string $type Element type
	 * @return object Element type administration panel class
	 */
	public function getElementTypeAdminObject($type) {
		// get element type
		if (isset($this->elementsTypes[$type])) {
			// set object
			$this->elementsTypes[$type]['objectadmin'] = new $this->elementsTypes[$type]['classnamebackend'];
			// get element type
			return $this->elementsTypes[$type]['objectadmin'];
		}
		// exit
		return NULL;
	}

	/**
	 * Get all exclude defaults
	 *
	 * @access public
	 * @param array $labels Output with labels for each suffix
	 * @param string $keyPrefix Prefix for keys in output array - default: exclude
	 * @param string $filterPrefix Prefix for filter with default value; if empty, there will be not filter - default: kocujsitemap_default_exclude_
	 * @return array All exclude defaults
	 */
	public function getExcludeDefaults(array &$labels, $keyPrefix = 'exclude', $filterPrefix = 'kocujsitemap_default_exclude_') {
		// initialize
		$output = array();
		$labels = array();
		// get all exclude defaults
		$elements = $this->getElementsTypes();
		if (!empty($elements)) {
			foreach ($elements as $element) {
				if ($element['object']->checkExcludeParameters()) {
					$suffixes = $element['object']->getExcludeParametersSuffixes();
					if (!empty($suffixes)) {
						foreach ($suffixes as $suffix => $text) {
							if (!isset($output[$keyPrefix.$suffix])) {
								$output[$keyPrefix.$suffix] = (isset($filterPrefix[0]) /* strlen($filterPrefix) > 0 */ ) ?
									apply_filters($filterPrefix.$suffix, '') :
									'';
							}
							if (!isset($labels[$keyPrefix.$suffix])) {
								$labels[$keyPrefix.$suffix] = array();
							}
							$labels[$keyPrefix.$suffix][] = $text;
						}
					}
				}
			}
		}
		// exit
		return $output;
	}

	/**
	 * Compress text
	 *
	 * @access private
	 * @param string $text Text to compress
	 * @return string Compressed text
	 */
	private function compressText($text) {
		// compress text
		if (isset($text[0]) /* strlen($text) > 0 */ ) {
			$textCompressed = chr(255).gzdeflate($text, 9);
			if (strlen($textCompressed) < strlen($text)) {
				$text = $textCompressed;
			}
		}
		// exit
		return $text;
	}

	/**
	 * Decompress text
	 *
	 * @access private
	 * @param string $text Text to decompress
	 * @return string Decompressed text
	 */
	private function decompressText($text) {
		// exit
		return ((isset($text[0])) && ($text[0] === chr(255))) ?
			gzinflate(substr($text, 1)) :
			$text;
	}

	/**
	 * Compress sitemap elements
	 *
	 * @access private
	 * @param array &$data Data to change and its output
	 * @param array &$index Data index to change and its output
	 * @param array $attr Attributes
	 * @return void
	 */
	private function compressElements(&$data, &$index, array $attr) {
		// compress sitemap items
		if (!empty($data)) {
			// check if make links shorter
			$makeShorter = ($attr['rootURLSize'] > self::LINK_SHORTER_META_LENGTH);
 			// process data
			foreach ($data as $key => $val) {
				// make link shorter
				if ($makeShorter) {
					$pos = strpos($val['lk'], $attr['rootURL']);
					if ($pos !== false) {
						$data[$key]['lk'] = array(
							'lk' => substr($val['lk'], 0, $pos).substr($val['lk'], $pos+$attr['rootURLSize']),
						);
						if ($pos !== self::LINK_ROOT_URL_STANDARD_POS) {
							$data[$key]['rp'] = $pos;
						}
						$val = $data[$key];
					}
				}
				// compress text
				if ($attr['additionalCompression']) {
					$text = $this->compressText((is_array($val['lk'])) ?
						$val['lk']['lk'] :
						$val['lk']
						);
					if (is_array($val['lk'])) {
						$data[$key]['lk']['lk'] = $text;
					} else {
						$data[$key]['lk'] = $text;
					}
					if (isset($val['ac'])) {
						foreach ($val['ac'] as $key2 => $val2) {
							$data[$key]['ac'][$key2] = $this->compressText($val2);
						}
					}
					if (isset($val['ad'])) {
						foreach ($val['ad'] as $key2 => $val2) {
							$data[$key]['ad'][$key2] = $this->compressText($val2);
						}
					}
					$val = $data[$key];
				}
				// change type to value
				if (!isset($index['tp'])) {
					$index['tp'] = array();
				}
				$type = $val['tp'];
				$pos = array_search($type, $index['tp']);
				if ($pos === false) {
					$pos = count($index['tp']);
					$index['tp'][] = $type;
				}
				$data[$key]['tp'] = $pos;
				// optionally make reccurence
				if ((isset($val['ch'])) && (!empty($val['ch']))) {
					$this->compressElements($data[$key]['ch'], $index, $attr);
				}
			}
		}
	}

	/**
	 * Decompress sitemap element
	 *
	 * @access private
	 * @param array $element Sitemap element to change
	 * @param array $index Sitemap data index
	 * @param array $attr Attributes
	 * @return array Decompressed sitemap element
	 */
	private function decompressElement($element, array $index, array $attr) {
		// change type to name
		if (isset($index['tp'][$element['tp']])) {
			$element['tp'] = $index['tp'][$element['tp']];
		}
		// decompress text
		if ($attr['additionalCompression']) {
			$text = $this->decompressText((is_array($element['lk'])) ?
				$element['lk']['lk'] :
				$element['lk']
				);
			if (is_array($element['lk'])) {
				$element['lk']['lk'] = $text;
			} else {
				$element['lk'] = $text;
			}
			if (isset($element['ac'])) {
				foreach ($element['ac'] as $key => $val) {
					$element['ac'][$key] = $this->decompressText($val);
				}
			}
			if (isset($element['ad'])) {
				foreach ($element['ad'] as $key => $val) {
					$element['ad'][$key] = $this->decompressText($val);
				}
			}
		}
		// change link if is shortened
		if (is_array($element['lk'])) {
			$pos = (isset($element['lk']['rp'])) ?
				$element['lk']['rp'] :
				self::LINK_ROOT_URL_STANDARD_POS;
			$element['lk'] = substr($element['lk']['lk'], 0, $pos).$attr['rootURL'].substr($element['lk']['lk'], $pos);
		}
		// exit
		return $element;
	}

	/**
	 * Create sitemap
	 *
	 * @access public
	 * @param string $locale Language locale - default: empty
	 * @return array Sitemap data
	 */
	public function createSitemap($locale = '') {
		// prepare locale
		if (!isset($locale[0]) /* strlen($locale) === 0 */ ) {
			$locale = get_locale();
		}
		// check if make additional compression
		if (Base::getInstance()->getKocujInternalLibObj()->getComponent('all', 'config')->getOption('Cache')) {
			$additionalCompression = Base::getInstance()->getKocujInternalLibObj()->getComponent('all', 'config')->getOption('CacheAdditionalCompress');
			if (($additionalCompression) && (!function_exists('gzdeflate'))) {
				$additionalCompression = false;
			}
		} else {
			$additionalCompression = false;
		}
		// set elements order
		$value = Base::getInstance()->getKocujInternalLibObj()->getComponent('all', 'config')->getOption('OrderList');
		if (!isset($value[0]) /* strlen($value) === 0 */ ) {
			$value = Base::getInstance()->getKocujInternalLibObj()->getComponent('all', 'config')->getOptionDefaultValue('OrderList');
		}
		$order = 'H'.implode('', $value);
		// delete WordPress cache
		global $wp_object_cache;
		$wpCache = $wp_object_cache;
		wp_cache_flush();
		// begin multi-lingual processing
		MultipleLanguages::getInstance()->beforeProcess($locale);
		// get blog name
		MultipleLanguages::getInstance()->beforeGetBlogName();
		$defaultHomeLinkText = apply_filters('kocujsitemap_defaulttitle', MultipleLanguages::getInstance()->getTranslatedBlogName($locale), $locale); // for compatibility with 1.x.x
		Base::getInstance()->filterOldInfo('kocujsitemap_defaulttitle', 'kocujsitemap_default_home_link_text'); // for compatibility with 1.x.x
		$defaultHomeLinkText = apply_filters('kocujsitemap_default_home_link_text', $defaultHomeLinkText, $locale);
		MultipleLanguages::getInstance()->afterGetBlogName();
		// clear home link text position
		$this->setHomeLinkTextPos(-1);
		// get elements
		$rootURL = Helpers\Url::getInstance()->getRootURLWithoutProtocol();
		$attr = array(
			'rootURL'               => $rootURL,
			'rootURLSize'           => strlen($rootURL),
			'additionalCompression' => $additionalCompression,
		);
		$index = array(
			'ru' => $rootURL,
			'hl' => $defaultHomeLinkText,
		);
		$sectionsNames = array();
		$array = array();
		$loopLength = strlen($order);
		for ($z=0; $z<$loopLength; $z++) {
			// get element type
			$elementType = $this->getElementTypeAdminObject($order[$z]);
			if (!empty($elementType)) {
				// get elements for current type
				$array[$order[$z]] = $elementType->getElementArray($locale);
				// compress elements
				if (!empty($array[$order[$z]])) {
					$this->compressElements($array[$order[$z]], $index, $attr);
				}
				// get sections names
				$options = Base::getInstance()->getKocujInternalLibObj()->getComponent('all', 'config')->getSitemapTypeLanguagesOptionsNames($order[$z]);
				if (!empty($options)) {
					foreach ($options as $option) {
						$optionValue = Base::getInstance()->getKocujInternalLibObj()->getComponent('all', 'config')->getOption($option['option']);
						if (isset($optionValue[0]) /* strlen($optionValue) > 0 */ ) {
							$sectionsNames[$option['option']] = $optionValue;
						}
					}
				}
			}
		}
		// end multi-lingual processing
		MultipleLanguages::getInstance()->afterProcess($locale);
		// prepare array
		if (!empty($sectionsNames)) {
			$index['sn'] = $sectionsNames;
		}
		$index['hp'] = $this->getHomeLinkTextPos();
		$array = array(
			'dt' => $array,
			'ix' => $index,
		);
		if ($additionalCompression) {
			$array['ac'] = true;
		}
		// return WordPress cache
		$wp_object_cache = $wpCache;
		// exit
		return $array;
	}

	/**
	 * Sitemap shortcode
	 *
	 * @access public
	 * @param array|string $args Arguments
	 * @return string Parsed output
	 */
	public function shortcodeSitemap($args) {
		// get arguments
		$defaultMainCSSClass = apply_filters('kocujsitemap_defaultclass', ''); // for compatibility with 1.x.x
		Base::getInstance()->filterOldInfo('kocujsitemap_defaultclass', 'kocujsitemap_default_main_css_class'); // for compatibility with 1.x.x
		$defaultMainCSSClass = apply_filters('kocujsitemap_default_main_css_class', $defaultMainCSSClass);
		$shortcodes = array(
			'homelinktext' => '',
			'title'        => '', // for compatibility with 1.x.x
			'class'        => $defaultMainCSSClass,
		);
		$labels = array();
		$shortcodeAttr = shortcode_atts(array_merge($shortcodes, $this->getExcludeDefaults($labels)), $args, 'kocujsitemap');
		$homeLinkText = (isset($shortcodeAttr['homelinktext'])) ?
			$shortcodeAttr['homelinktext'] :
			'';
		$title = (isset($shortcodeAttr['title'])) ?
			$shortcodeAttr['title'] :
			'';
		if ((isset($title[0]) /* strlen($title) > 0 */ ) && ($title !== $homeLinkText)) { // for compatibility with 1.x.x
			$homeLinkText = $title; // for compatibility with 1.x.x
		} // for compatibility with 1.x.x
		$excludeData = array();
		$elements = $this->getElementsTypes();
		if (!empty($elements)) {
			foreach ($elements as $element) {
				if ($element['object']->checkExcludeParameters()) {
					$suffixes = $element['object']->getExcludeParametersSuffixes();
					if (!empty($suffixes)) {
						foreach ($suffixes as $suffix => $text) {
							if (!isset($excludeData[$suffix])) {
								$excludeData[$suffix] = (isset($shortcodeAttr['exclude'.$suffix][0]) /* strlen($shortcodeAttr['exclude'.$suffix]) > 0 */ ) ?
									explode(',', trim($shortcodeAttr['exclude'.$suffix])) :
									array();
							}
						}
					}
				}
			}
		}
		// exit
		return $this->getDisplay($homeLinkText, (isset($shortcodeAttr['class'])) ?
				$shortcodeAttr['class'] :
				'',
			$excludeData);
	}

	/**
	 * Prepare sitemap data for text by reccurence when there are any elements excluded
	 *
	 * @access private
	 * @param string $type Sitemap data type
	 * @param array &$data Sitemap data and its output
	 * @param array $index Sitemap data index
	 * @param array $attr Attributes
	 * @param array $exclude List of elements to exclude - it is divided into sections: "post", "category", "author", "term"
	 * @return void
	 */
	private function addSitemapTextReccurencePrepareForExclude($type, &$data, array $index, array $attr, array $exclude) {
		// prepare data for exclude
		if (!empty($data)) {
			// get element type object
			$elementType = $this->getElementTypeObject($type);
			// process data
			$requiresChildren = $elementType->getTypesRequiresChildren();
			$loopCount = count($data);
			for ($z=0; $z<$loopCount; $z++) {
				// decompress element
				$data[$z] = $this->decompressElement($data[$z], $index, $attr);
				// get element
				$element = $data[$z];
				// process children
				if ((isset($element['ch'])) && (!empty($element['ch']))) {
					$this->addSitemapTextReccurencePrepareForExclude($type, $data[$z]['ch'], $index, $attr, $exclude);
				}
				// check if children list after excludes is empty and should be removed
				if ((in_array($element['tp'], $requiresChildren)) && ((!isset($data[$z]['ch'])) || (isset($data[$z]['ch']) && (empty($data[$z]['ch']))))) {
					unset($data[$z]);
				} else {
					// check if exclude this element
					$sectionsCheck = $elementType->getSectionsToCheck($element);
					if (!empty($sectionsCheck)) {
						foreach ($sectionsCheck as $oneSection) {
							if ((isset($exclude[$oneSection])) && (!empty($exclude[$oneSection]))) {
								$idCheck = $elementType->getIDToCheck($element, $oneSection);
								if ($idCheck >= 0) {
									foreach ($exclude[$oneSection] as $id) {
										if ((isset($id[0]) /* strlen($id) > 0 */ ) && (is_numeric($id)) && ($id > 0) && ($id === (string)$idCheck)) {
											unset($data[$z]);
											break;
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}

	/**
	 * Add sitemap text by reccurence
	 *
	 * @access private
	 * @param string $type Sitemap data type
	 * @param array $data Sitemap data
	 * @param array $index Sitemap data index
	 * @param array $attr Attributes
	 * @param bool $firstLevel It is first level on list (true) or not (false)
	 * @param bool &$first Output flag indicate that it is first element (true) or not (false)
	 * @param array $exclude List of elements to exclude - it is divided into sections: "post", "category", "author", "term"
	 * @param bool $isWidget Sitemap for widget (true) or not (false) - default: false
	 * @return string Sitemap text
	 */
	private function addSitemapTextReccurence($type, array $data, array $index, array $attr, $firstLevel, &$first, array $exclude, $isWidget = false) {
		// initialize
		$output = '';
		// set output text
		if (!empty($data)) {
			$elementType = $this->getElementTypeObject($type);
			$showUl = $elementType->checkListTag();
			$elementKeys = array_keys($data);
			foreach ($elementKeys as $elementKey) {
				// get element
				$element = $data[$elementKey];
				// get text before element
				$output .= $elementType->getBeforeElement($element, $isWidget, get_locale());
				// decompress element if there are no elements excluded
				if (empty($exclude)) {
					$element = $this->decompressElement($element, $index, $attr);
				}
				// add title to home element
				if (($element['tp'] === 'home') && ($index['hp'] > -1)) {
					$element['lk'] = substr($element['lk'], 0, $index['hp']).$attr['homeLinkText'].substr($element['lk'], $index['hp']);
				}
				// get element data
				$style = '';
				if (!$firstLevel) {
					$marginLeft = Base::getInstance()->getKocujInternalLibObj()->getComponent('all', 'config')->getOption('MarginLeft');
					if (!empty($marginLeft)) {
						$style = 'margin-left:'.$marginLeft.'px';
					}
				}
				if ((isset($element['sc'])) && ($element['sc'])) {
					$elementData = $element['lk'];
					$containerClassPos = $element['cp'];
				} else {
					$elementData = '<li';
					if (isset($style[0]) /* strlen($style) > 0 */ ) {
						$elementData .= ' style="'.esc_attr($style).'"';
					}
					$containerClassPos = strlen($elementData);
					$elementData .= '>'.$element['lk'];
				}
				// add additional classes
				$addClass = '';
				if ((isset($element['ac'])) && (isset($element['ac'][0]) /* strlen($element['ac']) > 0 */ )) {
					$addClass = implode(' ', $element['ac']);
				}
				// check if element is first
				if ($first) {
					$addClass .= ' kocujsitemapfirst';
					$firstClass = apply_filters('kocujsitemap_firstclass', ''); // for compatibility with 1.x.x
					Base::getInstance()->filterOldInfo('kocujsitemap_firstclass', 'kocujsitemap_first_element_css_class'); // for compatibility with 1.x.x
					$firstClass = apply_filters('kocujsitemap_first_element_css_class', $firstClass);
					if (isset($firstClass[0]) /* strlen($firstClass) > 0 */ ) {
						$addClass .= ' '.$firstClass;
					}
					$first = false;
				}
				// add CSS class based on type
				if (isset($addClass[0]) /* strlen($addClass) > 0 */ ) {
					$addClass .= ' ';
				}
				$addClass .= 'kocujsitemap-'.$element['tp'];
				// add CSS classes
				$classText = ' class="'.esc_attr(trim($addClass)).'"';
				// create output
				$output .= substr($elementData, 0, $containerClassPos).$classText.substr($elementData, $containerClassPos);
				// add children
				if ((isset($element['ch'])) && (!empty($element['ch']))) {
					$output .= $this->addSitemapTextReccurence($type, $element['ch'], $index, $attr, false, $first, $exclude, $isWidget);
				}
				// end tag
				if ($showUl) {
					$output .= '</li>';
				}
				// get text after element
				$output .= $elementType->getAfterElement($element, $isWidget, get_locale());
				// remove element from memory
				unset($data[$elementKey]);
			}
			if ((isset($output[0]) /* strlen($output) > 0 */ ) && ($showUl)) {
				$output = '<ul>'.$output.'</ul>';
			}
		}
		// exit
		return $output;
	}

	/**
	 * Add sitemap text
	 *
	 * @access private
	 * @param string $type Sitemap data type
	 * @param array $attr Attributes
	 * @param array $data Sitemap data
	 * @param array $index Sitemap data index
	 * @param bool &$first Output flag indicate that it is first element (true) or not (false)
	 * @param array $exclude List of elements to exclude - it is divided into sections: "post", "category", "author", "term"
	 * @param bool $isWidget Sitemap for widget (true) or not (false) - default: false
	 * @return string Sitemap text
	 */
	private function addSitemapText($type, array $attr, array $data, array $index, &$first, array $exclude, $isWidget = false) {
		// add sitemap text
		if (!empty($exclude)) {
			$this->addSitemapTextReccurencePrepareForExclude($type, $data, $index, $attr, $exclude);
		}
		return $this->addSitemapTextReccurence($type, $data, $index, $attr, true, $first, $exclude, $isWidget);
	}

	/**
	 * Get sitemap to display
	 *
	 * @access public
	 * @param string $homeLinkText Home link text in the sitemap - default: empty
	 * @param string $class Sitemap class - default: empty
	 * @param array $exclude List of elements to exclude - it is divided into sections: "post", "category", "author", "term" - default: empty
	 * @param bool $isWidget Sitemap is displayed in widget (true) or not (false)
	 * @return string Sitemap to display
	 */
	public function getDisplay($homeLinkText = '', $class = '', array $exclude = array(), $isWidget = false) {
		// load cache
		try {
			$data = Cache::getInstance()->loadCache();
		} catch (Exception $e) {
			return '';
		}
		if ((isset($data['ac'])) && ($data['ac']) && (!function_exists('gzdeflate'))) {
			try {
				Cache::getInstance()->clearCache();
				Cache::getInstance()->createCache();
				$data = Cache::getInstance()->loadCache();
			} catch (Exception $e) {
				return '';
			}
		}
		$index = (isset($data['ix'])) ?
			$data['ix'] :
			array();
		// clear excludes array if all exclude children arrays are empty
		$excludeTemp = array_filter($exclude);
		if (empty($excludeTemp)) {
			$exclude = $excludeTemp;
		}
		// get arguments
		if (!isset($homeLinkText[0]) /* strlen($homeLinkText) === 0 */ ) {
			$homeLinkText = $index['hl'];
		}
		if (!isset($class[0]) /* strlen($class) === 0 */ ) {
			$class = apply_filters('kocujsitemap_defaultclass', ''); // for compatibility with 1.x.x
			Base::getInstance()->filterOldInfo('kocujsitemap_defaultclass', 'kocujsitemap_default_main_css_class'); // for compatibility with 1.x.x
			$class = apply_filters('kocujsitemap_default_main_css_class', $class);
		}
		$homeLinkText = apply_filters('kocujsitemap_linktitle', $homeLinkText, 0, 'home', get_locale()); // for compatibility with 1.x.x
		Base::getInstance()->filterOldInfo('kocujsitemap_linktitle', 'kocujsitemap_link_text'); // for compatibility with 1.x.x
		$homeLinkText = apply_filters('kocujsitemap_link_text', $homeLinkText, 0, 'home', get_locale());
		// integrate data from cache
		$first = true;
		$output = '';
		$sections = Base::getInstance()->getKocujInternalLibObj()->getComponent('all', 'config')->getOption('DisplaySections');
		if ((isset($data['dt'])) && (!empty($data['dt']))) {
			$rootURL = $index['ru'];
			$attr = array(
				'rootURL'               => $rootURL,
				'rootURLSize'           => strlen($rootURL),
				'homeLinkText'          => $homeLinkText,
				'additionalCompression' => (isset($data['ac'])) ?
					$data['ac'] :
					false,
			);
			$hLevel = (!$isWidget) ?
				Base::getInstance()->getKocujInternalLibObj()->getComponent('all', 'config')->getOption('HLevelMain') :
				Base::getInstance()->getKocujInternalLibObj()->getComponent('all', 'config')->getOption('HLevelWidget');
			$types = array_keys($data['dt']);
			foreach ($types as $type) {
				$oneData = $data['dt'][$type];
				$outputSection = '';
				$outputText = '';
				if ($sections) {
					$sectionTitle = '';
					$elementType = $this->getElementTypeObject($type);
					if ((!empty($elementType)) && ($elementType->checkSectionName())) {
						$key = 'SectionName_'.$type.'_'.get_locale();
						if ((isset($index['sn'][$key])) && (isset($index['sn'][$key][0]) /* strlen($index['sn'][$key]) > 0 */ )) {
							$sectionTitle = $index['sn'][$key];
						}
						if (!isset($sectionTitle[0]) /* strlen($sectionTitle) === 0 */ ) {
							$key = 'SectionName_'.$type.'_en_US';
							if ((isset($index['sn'][$key])) && (isset($index['sn'][$key][0]) /* strlen($index['sn'][$key]) > 0 */ )) {
								$sectionTitle = $index['sn'][$key];
							}
						}
						if (!isset($sectionTitle[0]) /* strlen($sectionTitle) === 0 */ ) {
							$sectionTitle = $elementType->getSectionName();
						}
					}
					if (isset($sectionTitle[0]) /* strlen($sectionTitle) > 0 */ ) {
						$outputSection = '<h'.$hLevel.'>'.$sectionTitle.'</h'.$hLevel.'>';
					}
				}
				$outputText = $this->addSitemapText($type, $attr, $oneData, $index, $first, $exclude, $isWidget);
				if (isset($outputText[0]) /* strlen($outputText) > 0 */ ) {
					$output .= $outputSection.$outputText;
				}
				unset($data['dt'][$type]);
			}
		}
		// clean empty lists, integrate lists and remove EOL in output
		$output = str_replace(array(
				"\r\n",
				"\n",
				"\r",
			), '', preg_replace('/<\/([u|U])([l|L])><([u|U])([l|L])(.*?)>/s', '', preg_replace('/<([u|U])([l|L])><([u|U])([l|L])>/s', '', $output)));
		// show begin and end
		if (isset($output[0]) /* strlen($output) > 0 */ ) {
			$classText = '';
			if (isset($class[0]) /* strlen($class) > 0 */ ) {
				$classText = ' '.$class;
			}
			$html5 = Base::getInstance()->getKocujInternalLibObj()->getComponent('all', 'config')->getOption('UseHTML5');
			$footer = '';
			if (!$isWidget) {
				$value = Base::getInstance()->getKocujInternalLibObj()->getComponent('all', 'config')->getOption('PoweredBy');
				if ($value) {
					$footer = (($html5) ?
							Helpers\Html5::getInstance()->getTagFooterBegin('', 'kocujsitemapfooter') :
							'<div class="kocujsitemapfooter">').
						/* translators: 1: link (HTML anchor) to plugin website, 2: link (HTML anchor) to plugin's author website */
						sprintf(__('Powered by %1$s plugin created by %2$s', 'kocuj-sitemap'), \KocujInternalLib\V5a\Classes\HtmlHelper::getInstance()->getLinkAnchor('http://kocujsitemap.wpplugin.kocuj.pl/', 'Kocuj Sitemap', array(
							'externalwithouttarget' => true,
						)), \KocujInternalLib\V5a\Classes\HtmlHelper::getInstance()->getLinkAnchor('http://kocuj.pl/', 'kocuj.pl', array(
							'externalwithouttarget' => true,
						))).(($html5) ?
							Helpers\Html5::getInstance()->getTagFooterEnd() :
							'</div>'
						);
				}
			}
			$output = (($html5) ?
					Helpers\Html5::getInstance()->getTagNavBegin('', 'kocujsitemap'.$classText) :
					'<div class="'.esc_attr('kocujsitemap'.$classText).'">'
				).$output.(($html5) ?
					Helpers\Html5::getInstance()->getTagNavEnd() :
					'</div>'
				).$footer;
		}
		// exit
		return $output;
	}
}
