<?php

/**
 * project-parent.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujIL\V6a\Classes;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Project parent class
 *
 * @access public
 */
class ProjectParent {
	/**
	 * Namespace prefix
	 *
	 * @access protected
	 * @var string
	 */
	protected $namespacePrefix = '';

	/**
	 * Project components
	 *
	 * @access private
	 * @var array
	 */
	private $components = array();

	/**
	 * Project settings
	 *
	 * @access private
	 * @var array
	 */
	private $settings = array();

	/**
	 * Classes list for classes with strings which implement \KocujIL\V6a\Interfaces\Strings interface
	 *
	 * @access private
	 * @var array
	 */
	private $stringsClassesList = array();

	/**
	 * Classes list for components which has been replaced by child classes
	 *
	 * @access private
	 * @var array
	 */
	private $libClassesList = array();

	/**
	 * Objects with instances of classes for components
	 *
	 * @access private
	 * @var array
	 */
	private $libComponentsObjects = array();

	/**
	 * Shutdown flag
	 *
	 * @access private
	 * @var array
	 */
	private $shutdown = false;

	/**
	 * Constructor
	 *
	 * @access public
	 * @param array $components Components to use
	 * @param array $settings Project settings
	 * @param array $stringsClasses Classes list for classes with strings which implement \KocujIL\V6a\Interfaces\Strings interface; if some keys are empty or does not exist, the default classes which returns only empty strings will be used for these keys - default: empty
	 * @param array $libClasses Classes list which has been replaced by child classed; if some keys are empty or does not exist, the default classes will be used for these keys - default: empty
	 * @param array $additionalProjectsForRequirements Additional projects for checking requirements; each element should be objec of class derived from this class (ProjectParent) and its key should be the name of class derived from this class (ProjectParent) - default: empty
	 * @return void
	 */
	public function __construct(array $components, array $settings, array $stringsClasses = array(), array $libClasses = array(), array $additionalProjectsForRequirements = array()) {
		// remember settings
		$this->settings = $settings;
		// set strings classes
		$this->stringsClassesList = $stringsClasses;
		// set classes
		$this->libClassesList = $libClasses;
		// set components
		$this->components = $components;
		// initialize components
		if (!empty($this->components)) {
			foreach ($this->components as $category => $data) {
				if ($category !== 'core') {
					foreach ($data as $component) {
						if (!isset($this->libClassesList[$category][$component])) {
							$classInit = $this->get($category, $component, 'init');
							if ($classInit !== NULL) {
								$requiredComponents = $this->get($category, $component, 'init')->getRequiredComponents();
								if (!empty($requiredComponents)) {
									foreach ($requiredComponents as $requiredLibrary => $requiredLibraryData) {
										if (!empty($requiredLibraryData)) {
											if (!isset($requiredLibrary[0]) /* strlen($requiredLibrary) === 0 */ ) {
												$obj = $this;
											} else {
												if (isset($additionalProjectsForRequirements[$requiredLibrary])) {
													$obj = $additionalProjectsForRequirements[$requiredLibrary];
												} else {
													throw new Exception(ExceptionCode::NO_REQUIRED_LIBRARY, __FILE__, __LINE__, $requiredLibrary);
												}
											}
											foreach ($requiredLibraryData as $requiredCategory => $requiredCategoryData) {
												if (!empty($requiredCategoryData)) {
													foreach ($requiredCategoryData as $requiredComponent) {
														$classInit = $obj->get($requiredCategory, $requiredComponent, 'init');
														if ($classInit === NULL) {
															throw new Exception(ExceptionCode::NO_REQUIRED_COMPONENT, __FILE__, __LINE__, 'library: '.$requiredLibrary.',category: '.$requiredCategory.', component: '.$requiredComponent);
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
		// initialize classes list with child classes
		if (!empty($this->libClassesList)) {
			foreach ($this->libClassesList as $category => $data) {
				if ($category !== 'core') {
					foreach ($data as $component => $componentClass) {
						$this->get($category, $component);
					}
				}
			}
		}
		// add shutdown action
		add_action('shutdown', array($this, 'actionShutdown'), 1);
	}

	/**
	 * Get project setting with string type
	 *
	 * @access public
	 * @param string $type Data type
	 * @return string Project setting
	 */
	public function getSettingString($type) {
		// exit
		return (isset($this->settings[$type])) ?
			$this->settings[$type] :
			'';
	}

	/**
	 * Get project setting with array type
	 *
	 * @access public
	 * @param string $type Data type
	 * @param string $index Data index
	 * @return array|string Project setting
	 */
	public function getSettingArray($type, $index = '') {
		// optionally return entire array
		if (!isset($index[0]) /* strlen($index) === 0 */ ) {
			return ((isset($this->settings[$type])) && (is_array($this->settings[$type]))) ?
				$this->settings[$type] :
				array();
			}
		// exit
		return ((isset($this->settings[$type][$index])) && (is_array($this->settings[$type]))) ?
			$this->settings[$type][$index] :
			'';
	}

	/**
	 * Get object of class type which implements \KocujIL\V6a\Interfaces\Strings interface
	 *
	 * @access public
	 * @param string $category Component category
	 * @param string $type Component type
	 * @return object Object of class type which implements \KocujIL\V6a\Interfaces\Strings interface
	 */
	public function getStringsObj($category, $type) {
		// check if there was not a shutdown
		if ($this->shutdown) {
			throw new Exception(ExceptionCode::CANNOT_USE_COMPONENTS_AFTER_SHUTDOWN, __FILE__, __LINE__);
		}
		// optionally create instance of class
		if (isset($this->stringsClassesList[$category][$type])) {
			$interfaces = class_implements($this->stringsClassesList[$category][$type]);
			if (in_array('KocujIL\\V6a\\Interfaces\\Strings', $interfaces)) {
				return call_user_func(array($this->stringsClassesList[$category][$type], 'getInstance'));
			}
		}
		// exit
		return StringsEmpty::getInstance();
	}

	/**
	 * Get object of class type from component
	 *
	 * @access public
	 * @param string $category Component category
	 * @param string $type Component type
	 * @param string $fragment Component fragment - default: empty
	 * @return object Object of class type from component
	 */
	public function get($category, $type, $fragment = '') {
		// check if there was not a shutdown
		if ($this->shutdown) {
			throw new Exception(ExceptionCode::CANNOT_USE_COMPONENTS_AFTER_SHUTDOWN, __FILE__, __LINE__);
		}
		// check if component has been declared as used by project
		if (($category !== 'core') && (isset($this->components[$category])) && (!in_array($type, $this->components[$category]))) {
			return NULL;
		}
		// check if this component can be used
		if ((($category === 'frontend') && ((is_admin()) || (is_network_admin()))) || (($category === 'backend') && (!is_admin()) && (!is_network_admin()))) {
			return NULL;
		}
		// set fragment value
		$fragmentIndex = (isset($fragment[0]) /* strlen($fragment) === 0 */ ) ?
			$fragment :
			'_';
		// optionally initialize object
		if (!isset($this->libComponentsObjects[$category][$type][$fragmentIndex])) {
			if (!isset($this->libClassesList[$category][$type])) {
				// prepare fragments
				$fragments = array(
					$category,
					$type,
					(isset($fragment[0]) /* strlen($fragment) > 0 */ ) ?
						$fragment :
						'component',
				);
				// get class name
				foreach ($fragments as $key => $val) {
					$fragments[$key] = preg_replace_callback('/\\-([a-z])/', function($matches) {
						return strtoupper($matches[1]);
					}, $val);
					if (isset($fragments[$key][1]) /* strlen($fragments[$key]) > 1 */ ) {
						$fragments[$key] = strtoupper($fragments[$key][0]).substr($fragments[$key], 1);
					}
				}
				$className = $this->namespacePrefix.'\\Classes\\Project\\Components\\'.implode('\\', $fragments);
			} else {
				$className = $this->libClassesList[$category][$type];
				$fragmentIndex = '_';
			}
			// initialize object
			if (!isset($this->libComponentsObjects[$category])) {
				$this->libComponentsObjects[$category] = array();
			}
			if (!isset($this->libComponentsObjects[$category][$type])) {
				$this->libComponentsObjects[$category][$type] = array();
			}
			$this->libComponentsObjects[$category][$type][$fragmentIndex] = (is_subclass_of($className, '\\KocujIL\\V6a\\Classes\\ComponentObject')) ?
				new $className($this) :
				NULL;
		}
		// exit
		return $this->libComponentsObjects[$category][$type][$fragmentIndex];
	}

	/**
	 * Shutdown action
	 *
	 * @access public
	 * @return void
	 */
	public function actionShutdown() {
		// shutdown all components
		unset($this->libComponentsObjects);
		// set shutdown flag
		$this->shutdown = true;
	}
}
