<?php

/**
 * component.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujIL\V6a\Classes\Project\Components\Backend\License;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * License class
 *
 * @access public
 */
class Component extends \KocujIL\V6a\Classes\ComponentObject {
	/**
	 * Option name - license accept
	 *
	 */
	const OPTION_NAME_LICENSE_ACCEPT = 'license_accept';

	/**
	 * License will be no displayed anywhere automatically and there will be no need for accept (false) or license will be displayed anywhere automatically and there will be need for accept (true)
	 *
	 * @access private
	 * @var bool
	 */
	private $forceAccept = false;

	/**
	 * License is set to be displayed by forcing it or by link (true) or license will not be displayed (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $licenseDisplay = false;

	/**
	 * Get if license has been accepted (true) or not (false)
	 *
	 * @access public
	 * @return bool License has been accepted (true) or not (false)
	 */
	public function getLicenseAcceptOptionValue() {
		// exit
		return ($this->getComponent('core', 'meta')->get(self::OPTION_NAME_LICENSE_ACCEPT) === false) ?
			false :
			true;
	}

	/**
	 * Set if license will be no displayed anywhere automatically and there will be no need for accept (false) or license will be displayed anywhere automatically and there will be need for accept (true)
	 *
	 * @access public
	 * @param bool $accept Set if license will be no displayed anywhere automatically and there will be no need for accept (false) or license will be displayed anywhere automatically and there will be need for accept (true)
	 * @return void
	 */
	public function setLicenseForceAccept($forceAccept) {
		// set license accepting
		$this->forceAccept = $forceAccept;
	}

	/**
	 * Get if license will be no displayed anywhere automatically and there will be no need for accept (false) or license will be displayed anywhere automatically and there will be need for accept (true)
	 *
	 * @access public
	 * @return int License will be no displayed anywhere automatically and there will be no need for accept (false) or license will be displayed anywhere automatically and there will be need for accept (true)
	 */
	public function getLicenseForceAccept() {
		// exit
		return $this->forceAccept;
	}

	/**
	 * Get license filename
	 *
	 * @access public
	 * @return string License filename; if license file does not exist, it returns empty string
	 */
	public function getLicenseFilename() {
		// get license filename
		$language = get_locale();
		if ((isset($language[0]) /* strlen($language) > 0 */ ) && (is_file($this->getComponent('core', 'dirs')->getProjectDir().DIRECTORY_SEPARATOR.'license-'.$language.'.txt'))) {
			$licenseFilename = 'license-'.$language.'.txt';
		} else {
			$licenseFilename = 'license.txt';
			if (!is_file($this->getComponent('core', 'dirs')->getProjectDir().DIRECTORY_SEPARATOR.$licenseFilename)) {
				return '';
			}
		}
		// exit
		return $licenseFilename;
	}

	/**
	 * Get license script
	 *
	 * @access private
	 * @return string License script
	 */
	private function getLicenseScript() {
		// exit
		return 'kocujILV6aBackendLicense.showLicense(\''.esc_js($this->getProjectObj()->getMainSettingInternalName()).'\', '.((($this->forceAccept) && (!$this->getLicenseAcceptOptionValue())) ?
				'true' :
				'false'
			).')';
	}

	/**
	 * Get license link or just name if license file does not exist
	 *
	 * @access public
	 * @return string License link or just name if license file does not exist
	 */
	public function getLicenseLink() {
		// get license link or name
		$licenseFilename = $this->getLicenseFilename();
		if (isset($licenseFilename[0]) /* strlen($licenseFilename) > 0 */ ) {
			// set license to display
			$this->licenseDisplay = true;
			// set HTML identifier
			$id = $this->getComponent('core', 'project-helper')->getPrefix().'__licenselink_'.$this->getProjectObj()->getMainSettingInternalName().'_'.rand(111111, 999999);
			// exit
			return \KocujIL\V6a\Classes\HtmlHelper::getInstance()->getLink('#', $this->getProjectObj()->getMainSettingLicenseName(), array(
					'id'               => $id,
					'styleclassfilter' => array(
						'projectobj' => $this->getProjectObj(),
						'filter'     => 'license_link',
					),
				)).
				'<script type="text/javascript">'.PHP_EOL.
				'/* <![CDATA[ */'.PHP_EOL.
				'(function($) {'.PHP_EOL.
				'$(document).ready(function() {'.PHP_EOL.
				'$(\''.esc_js('#'.$id).'\').attr(\'href\', \'javascript:void(0);\');'.PHP_EOL.
				'$(\''.esc_js('#'.$id).'\').click(function(event) {'.PHP_EOL.
				'event.preventDefault();'.PHP_EOL.
				$this->getLicenseScript().PHP_EOL.
				'});'.PHP_EOL.
				'});'.PHP_EOL.
				'}(jQuery));'.PHP_EOL.
				'/* ]]> */'.PHP_EOL.
				'</script>'.PHP_EOL;
		} else {
			// exit
			return $this->getProjectObj()->getMainSettingLicenseName();
		}
	}

	/**
	 * Action for adding JavaScript scripts
	 *
	 * @access public
	 * @return void
	 */
	public function actionAdminFooter() {
		// check if license should be forced to display
		if ((!$this->licenseDisplay) && ($this->forceAccept) && (!$this->getLicenseAcceptOptionValue())) {
			$this->licenseDisplay = true;
		}
		// add scripts
		if ($this->licenseDisplay) {
			$this->getComponent('all', 'js-ajax')->addAjaxJs();
			\KocujIL\V6a\Classes\JsHelper::getInstance()->addLibScript('backend-license', 'project/components/backend/license', 'license', array(
				'helper',
				'modal',
			), array(
				\KocujIL\V6a\Classes\Helper::getInstance()->getPrefix().'-all-js-ajax',
			), 'kocujILV6aBackendLicenseVals', array(
				'prefix'           => \KocujIL\V6a\Classes\Helper::getInstance()->getPrefix(),
				'security'         => wp_create_nonce(\KocujIL\V6a\Classes\Helper::getInstance()->getPrefix().'__license'),
				'textLoading'      => $this->getStrings('backend', 'license')->getString('ACTION_ADMIN_FOOTER_LICENSE_SCRIPT_LOADING'),
				'textLoadingError' => $this->getStrings('backend', 'license')->getString('ACTION_ADMIN_FOOTER_LICENSE_SCRIPT_LOADING_ERROR'),
				'textLicense'      => $this->getStrings('backend', 'license')->getString('ACTION_ADMIN_FOOTER_LICENSE_SCRIPT_LICENSE'),
				'textAccept'       => $this->getStrings('backend', 'license')->getString('ACTION_ADMIN_FOOTER_LICENSE_SCRIPT_ACCEPT'),
				'textCancel'       => $this->getStrings('backend', 'license')->getString('ACTION_ADMIN_FOOTER_LICENSE_SCRIPT_CANCEL'),
			));
		}
	}

	/**
	 * Action for setting up license and forcing displaying of license
	 *
	 * @access public
	 * @return void
	 */
	public function actionPrintFooterScripts() {
		// initialize license script
		if ($this->licenseDisplay) {
			?>
				<script type="text/javascript">
				/* <![CDATA[ */
					(function($) {
						$(document).ready(function() {
							<?php if ($this->getProjectObj()->getMainSettingType() === \KocujIL\V6a\Enums\ProjectType::PLUGIN) : ?>
								kocujILV6aBackendLicense.addProject('<?php echo esc_js($this->getProjectObj()->getMainSettingInternalName()); ?>', '<?php echo esc_js($this->getStrings('backend', 'license')->getString('ACTION_PRINT_FOOTER_SCRIPTS_LICENSE_PLUGIN').': '.$this->getProjectObj()->getMainSettingTitleOriginal()); ?>');
							<?php else : ?>
								kocujILV6aBackendLicense.addProject('<?php echo esc_js($this->getProjectObj()->getMainSettingInternalName()); ?>', '<?php echo esc_js($this->getStrings('backend', 'license')->getString('ACTION_PRINT_FOOTER_SCRIPTS_LICENSE_THEME').': '.$this->getProjectObj()->getMainSettingTitleOriginal()); ?>');
							<?php endif; ?>
						});
					}(jQuery));
				/* ]]> */
				</script>
			<?php
		}
		// optionally force displaying of license
		if (($this->forceAccept) && (!$this->getLicenseAcceptOptionValue())) {
			?>
				<script type="text/javascript">
				/* <![CDATA[ */
					(function($) {
						$(document).ready(function() {
							<?php echo $this->getLicenseScript(); ?>
						});
					}(jQuery));
				/* ]]> */
				</script>
			<?php
		}
	}
}
