/**
 * @file Version check information
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj <dominik@kocuj.pl>
 */

(function() {})(); // empty function for correct minify with comments
//'use strict'; // for jshint uncomment this and comment line above

/* jshint strict: true */
/* jshint -W034 */

/* global document */
/* global jQuery */
/* global window */

/* global ajaxurl */

/* global kocujILV7aModalType */
/* global kocujILV7aHelper */
/* global kocujILV7aModal */
/* global kocujILV7aAllJsAjax */

/* global kocujILV7aBackendVersionCheckVals */

/* global kocujILV7aCException */
/* global kocujILV7aExceptionCode */

/**
 * Version check project types
 *
 * @namespace kocujILV7aBackendVersionCheckProjectType
 * @public
 */
var kocujILV7aBackendVersionCheckProjectType = {
	/**
	 * This project is plugin
	 *
	 * @public
	 * @const {number}
	 */
	PLUGIN : 0,

	/**
	 * This project is theme
	 *
	 * @public
	 * @const {number}
	 */
	THEME : 1,

	/**
	 * Information about maximum constant value; it should not be used in executing the version check script methods
	 *
	 * @public
	 * @const {number}
	 */
	LAST : 1
};

/**
 * Version check prototype constructor
 *
 * @constructs
 * @namespace kocujILV7aCBackendReviewMessage
 * @public
 * @return {void}
 */
function kocujILV7aCBackendVersionCheck() {
	'use strict';
	/* jshint validthis: true */
	// get this object
	var self = this;
	// initialize objects
	self._objHelper = kocujILV7aHelper;
	self._objAllJsAjax = kocujILV7aAllJsAjax;
	self._objModal = kocujILV7aModal;
	// get current script filename
	self._thisFilename = document.scripts[document.scripts.length-1].src;
	// get settings
	if (typeof kocujILV7aBackendVersionCheckVals !== 'undefined') {
		var vals = kocujILV7aBackendVersionCheckVals;
		if (vals.throwErrors !== undefined) {
			if (vals.throwErrors === '1') {
				self._valsThrowErrors = true;
			} else {
				self._valsThrowErrors = false;
			}
		}
		if (vals.prefix !== undefined) {
			self._valsPrefix = vals.prefix;
		}
		if (vals.security !== undefined) {
			self._valsSecurity = vals.security;
		}
		if (vals.textLoading !== undefined) {
			self._valsTextLoading = vals.textLoading;
		}
		if (vals.textLoadingError !== undefined) {
			self._valsTextLoadingError = vals.textLoadingError;
		}
		if (vals.textTitlePlugin !== undefined) {
			self._valsTextTitlePlugin = vals.textTitlePlugin;
		}
		if (vals.textTitleTheme !== undefined) {
			self._valsTextTitleTheme = vals.textTitleTheme;
		}
	}
}

/**
 * Version check prototype
 *
 * @namespace kocujILV7aCBackendReviewMessage
 * @public
 */
kocujILV7aCBackendVersionCheck.prototype = {
	/**
	 * Object kocujILV7aHelper
	 *
	 * @private
	 * @type {Object}
	 */
	_objHelper : null,

	/**
	 * Object kocujILV7aAllJsAjax
	 *
	 * @private
	 * @type {Object}
	 */
	_objAllJsAjax : null,

	/**
	 * Object kocujILV7aModal
	 *
	 * @private
	 * @type {Object}
	 */
	_objModal : null,

	/**
	 * Current script filename
	 *
	 * @private
	 * @type {string}
	 */
	_thisFilename : '',

	/**
	 * Projects list
	 *
	 * @private
	 * @type {Array}
	 */
	_prj : [],

	/**
	 * Script settings - throw errors (true) or not (false)
	 *
	 * @private
	 * @type {string}
	 */
	_valsThrowErrors : false,

	/**
	 * Script settings - prefix
	 *
	 * @private
	 * @type {string}
	 */
	_valsPrefix : '',

	/**
	 * Script settings - security string
	 *
	 * @private
	 * @type {string}
	 */
	_valsSecurity : '',

	/**
	 * Script settings - text for loading
	 *
	 * @private
	 * @type {string}
	 */
	_valsTextLoading : '',

	/**
	 * Script settings - text for loading error
	 *
	 * @private
	 * @type {string}
	 */
	_valsTextLoadingError : '',

	/**
	 * Script settings - text for plugin title
	 *
	 * @private
	 * @type {string}
	 */
	_valsTextTitlePlugin : '',

	/**
	 * Script settings - text for theme title
	 *
	 * @private
	 * @type {string}
	 */
	_valsTextTitleTheme : '',

	/**
	 * Add project
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} projectName Project name
	 * @param {number} projectType Project type; should be one of the constants from kocujILV7aBackendVersionCheckProjectType
	 * @param {string} projectVersionFrom Number of version from which project has been updated
	 * @param {string} projectVersionTo Number of version to which project has been updated
	 * @return {void}
	 * @throws {kocujILV7aCException} kocujILV7aExceptionCode.PROJECT_ALREADY_EXISTS if project identifier entered in projectId already exists
	 */
	addProject : function(projectId, projectName, projectType, projectVersionFrom, projectVersionTo) {
		'use strict';
		// parse arguments
		var args = this._checkAddProject(projectId, projectName, projectType, projectVersionFrom, projectVersionTo);
		// add project
		if (this._prj['prj_' + args.projectId] === undefined) {
			this.addProjectIfNotExists(args.projectId, args.projectName, args.projectType, args.projectVersionFrom, args.projectVersionTo);
		} else {
			this._throwError('PROJECT_ALREADY_EXISTS', args.projectId);
			return;
		}
	},

	/**
	 * Add project if not exists
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} projectName Project name
	 * @param {number} projectType Project type; should be one of the constants from kocujILV7aBackendVersionCheckProjectType
	 * @param {string} projectVersionFrom Number of version from which project has been updated
	 * @param {string} projectVersionTo Number of version to which project has been updated
	 * @return {void}
	 */
	addProjectIfNotExists : function(projectId, projectName, projectType, projectVersionFrom, projectVersionTo) {
		'use strict';
		// parse arguments
		var args = this._checkAddProject(projectId, projectName, projectType, projectVersionFrom, projectVersionTo);
		// add project
		if (this._prj['prj_' + args.projectId] === undefined) {
			this._prj['prj_' + args.projectId] = {
				projectName        : args.projectName,
				projectType        : args.projectType,
				projectVersionFrom : args.projectVersionFrom,
				projectVersionTo   : args.projectVersionTo
			};
		} else {
			return;
		}
		this._objAllJsAjax.addProjectIfNotExists(args.projectId);
		this._objModal.addProjectIfNotExists(args.projectId);
	},

	/**
	 * Show window with information about updated version
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @return {void}
	 */
	showVersionCheck : function(projectId) {
		'use strict';
		// get this object
		var self = this;
		(function($) {
			// parse arguments
			projectId = self._parseProjectId(projectId);
			// prepare modal window
			var title = self._valsTextTitlePlugin;
			if (self._prj['prj_' + projectId].projectType === kocujILV7aBackendVersionCheckProjectType.THEME) {
				title = self._valsTextTitleTheme;
			}
			title = title.replace('%1$s', self._prj['prj_' + projectId].projectVersionTo).replace('%2$s', self._prj['prj_' + projectId].projectName);
			self._objModal.prepareModal(projectId, 'version_check',
				'<em>' +
				self._valsTextLoading +
				'</em>', {
					type  : kocujILV7aModalType.INFO,
					title : title
				});
			// show window
			self._objModal.showPreparedModal(projectId, 'version_check');
			// close window
			self.sendCloseEvent(projectId);
		}(jQuery));
	},

	/**
	 * Close window
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @return {void}
	 */
	sendCloseEvent : function(projectId) {
		'use strict';
		// get this object
		var self = this;
		(function($) {
			// parse arguments
			projectId = self._parseProjectId(projectId);
			// call AJAX
			self._objAllJsAjax.sendPost(projectId, 'version_check_display', ajaxurl, 'text', {
				action             : self._valsPrefix + '_' + projectId + '__version_check_display',
				security           : self._valsSecurity,
				projectVersionFrom : self._prj['prj_' + projectId].projectVersionFrom,
				projectVersionTo   : self._prj['prj_' + projectId].projectVersionTo
			}, {
				success : self.ajaxSuccessCallback,
				error   : self.ajaxErrorCallback
			});
		}(jQuery));
	},

	/**
	 * AJAX loading success callback
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} connectionId Connection identifier
	 * @param {anything} data Data
	 * @param {string} status Text status
	 * @param {Object} obj Request object
	 * @return {void}
	 */
	ajaxSuccessCallback : function(projectId, connectionId, data, status, obj) {
		'use strict';
		// get this object
		var self = kocujILV7aBackendVersionCheck;
		(function($) {
			// parse parameters
			data = self._objHelper.initString(data);
			// get selectors
			var selectorModalDialogContentInside = $(self._objModal.getHTMLSelectorModalDialogContentInside(projectId, 'version_check'));
			// set HTML data
			selectorModalDialogContentInside.html(data);
			// set external links
			selectorModalDialogContentInside.find('a[rel=external]').attr('target', '_blank');
		}(jQuery));
	},

	/**
	 * AJAX loading error callback
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} connectionId Connection identifier
	 * @param {Object} obj Request object
	 * @param {string} status Text status
	 * @param {string} err Error
	 * @return {void}
	 */
	ajaxErrorCallback : function(projectId, connectionId, obj, status, err) {
		'use strict';
		// get this object
		var self = kocujILV7aBackendVersionCheck;
		(function($) {
			// set HTML data
			$(self._objModal.getHTMLSelectorModalDialogContentInside(projectId, 'version_check')).html('<strong>' + self._valsTextLoadingError + '</strong>');
		}(jQuery));
	},

	/**
	 * Parse project identifier
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @return {string} Parsed project identifier
	 * @throws {kocujILV7aCException} kocujILV7aExceptionCode.EMPTY_PROJECT_ID if project identifier entered in projectId is empty
	 * @throws {kocujILV7aCException} kocujILV7aExceptionCode.PROJECT_DOES_NOT_EXIST if project identifier entered in projectId does not exist
	 */
	_parseProjectId : function(projectId) {
		'use strict';
		// parse project identifier
		projectId = this._objHelper.initString(projectId);
		if (projectId === '') {
			this._throwError('EMPTY_PROJECT_ID');
			return;
		}
		// check if project exists
		if (this._prj['prj_' + projectId] === undefined) {
			this._throwError('PROJECT_DOES_NOT_EXIST', projectId);
			return;
		}
		// exit
		return projectId;
	},

	/**
	 * Check arguments for adding project
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @param {string} projectName Project name
	 * @param {number} projectType Project type; should be one of the constants from kocujILV7aBackendVersionCheckProjectType
	 * @param {string} projectVersionFrom Number of version from which project has been updated
	 * @param {string} projectVersionTo Number of version to which project has been updated
	 * @return {Object} Parsed arguments for adding project
	 * @throws {kocujILV7aCException} kocujILV7aExceptionCode.EMPTY_PROJECT_ID if project identifier entered in projectId is empty
	 * @throws {kocujILV7aCException} kocujILV7aExceptionCode.VERSION_CHECK_EMPTY_PROJECT_NAME if project name entered in projectName is empty
	 * @throws {kocujILV7aCException} kocujILV7aExceptionCode.VERSION_CHECK_WRONG_PROJECT_TYPE if project type entered in projectType has wrong value
	 * @throws {kocujILV7aCException} kocujILV7aExceptionCode.VERSION_CHECK_EMPTY_VERSION_FROM if old version number entered in projectVersionFrom has wrong value
	 * @throws {kocujILV7aCException} kocujILV7aExceptionCode.VERSION_CHECK_EMPTY_VERSION_TO if new version number entered in projectVersionTo has wrong value
	 */
	_checkAddProject : function(projectId, projectName, projectType, projectVersionFrom, projectVersionTo) {
		'use strict';
		// parse arguments
		projectId = this._objHelper.initString(projectId);
		if (projectId === '') {
			this._throwError('EMPTY_PROJECT_ID');
			return;
		}
		projectName = this._objHelper.initString(projectName);
		if (projectName === '') {
			this._throwError('VERSION_CHECK_EMPTY_PROJECT_NAME');
			return;
		}
		projectType = this._objHelper.initNumeric(projectType);
		if ((projectType < 0) || (projectType > kocujILV7aBackendVersionCheckProjectType.LAST)) {
			this._throwError('VERSION_CHECK_WRONG_PROJECT_TYPE', projectType);
			return;
		}
		projectVersionFrom = this._objHelper.initString(projectVersionFrom);
		if (projectVersionFrom === '') {
			this._throwError('VERSION_CHECK_EMPTY_VERSION_FROM');
			return;
		}
		projectVersionTo = this._objHelper.initString(projectVersionTo);
		if (projectVersionTo === '') {
			this._throwError('VERSION_CHECK_EMPTY_VERSION_TO');
			return;
		}
		// exit
		return {
			projectId          : projectId,
			projectName        : projectName,
			projectType        : projectType,
			projectVersionFrom : projectVersionFrom,
			projectVersionTo   : projectVersionTo
		};
	},

	/**
	 * Throw an error if debugging is enabled
	 *
	 * @private
	 * @param {string} codeString Error code in string format
	 * @param {string} [param] Parameter for error information
	 * @return {void}
	 */
	_throwError : function(codeString, param) {
		'use strict';
		// parse arguments
		codeString = this._objHelper.initString(codeString);
		if (codeString === '') {
			return;
		}
		param = this._objHelper.initString(param);
		// throw an error
		if (this._valsThrowErrors) {
			/* jshint evil: true */
			eval('throw new kocujILV7aCException(kocujILV7aExceptionCode.' + codeString + ', this._thisFilename, param);');
		}
	}
};

// initialize
var kocujILV7aBackendVersionCheck = new kocujILV7aCBackendVersionCheck();
