/**
 * @file License
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj <dominik@kocuj.pl>
 */

(function() {})(); // empty function for correct minify with comments
//'use strict'; // for jshint uncomment this and comment line above

/* jshint strict: true */
/* jshint -W034 */
/* jshint -W107 */

/* global document */
/* global jQuery */
/* global window */

/* global ajaxurl */

/* global kocujILV7aModalType */
/* global kocujILV7aHelper */
/* global kocujILV7aModal */
/* global kocujILV7aAllJsAjax */

/* global kocujILV7aBackendLicenseVals */

/* global kocujILV7aCException */
/* global kocujILV7aExceptionCode */

/**
 * License prototype constructor
 *
 * @constructs
 * @namespace kocujILV7aCBackendLicense
 * @public
 * @return {void}
 */
function kocujILV7aCBackendLicense() {
	'use strict';
	/* jshint validthis: true */
	// get this object
	var self = this;
	// initialize objects
	self._objHelper = kocujILV7aHelper;
	self._objAllJsAjax = kocujILV7aAllJsAjax;
	self._objModal = kocujILV7aModal;
	// get current script filename
	self._thisFilename = document.scripts[document.scripts.length-1].src;
	// get settings
	if (typeof kocujILV7aBackendLicenseVals !== 'undefined') {
		var vals = kocujILV7aBackendLicenseVals;
		if (vals.throwErrors !== undefined) {
			if (vals.throwErrors === '1') {
				self._valsThrowErrors = true;
			} else {
				self._valsThrowErrors = false;
			}
		}
		if (vals.prefix !== undefined) {
			self._valsPrefix = vals.prefix;
		}
		if (vals.security !== undefined) {
			self._valsSecurity = vals.security;
		}
		if (vals.textLoading !== undefined) {
			self._valsTextLoading = vals.textLoading;
		}
		if (vals.textLoadingError !== undefined) {
			self._valsTextLoadingError = vals.textLoadingError;
		}
		if (vals.textLicense !== undefined) {
			self._valsTextLicense = vals.textLicense;
		}
		if (vals.textAccept !== undefined) {
			self._valsTextAccept = vals.textAccept;
		}
		if (vals.textCancel !== undefined) {
			self._valsTextCancel = vals.textCancel;
		}
	}
}

/**
 * License prototype
 *
 * @namespace kocujILV7aCBackendLicense
 * @public
 */
kocujILV7aCBackendLicense.prototype = {
	/**
	 * Object kocujILV7aHelper
	 *
	 * @private
	 * @type {Object}
	 */
	_objHelper : null,

	/**
	 * Object kocujILV7aAllJsAjax
	 *
	 * @private
	 * @type {Object}
	 */
	_objAllJsAjax : null,

	/**
	 * Object kocujILV7aModal
	 *
	 * @private
	 * @type {Object}
	 */
	_objModal : null,

	/**
	 * Current script filename
	 *
	 * @private
	 * @type {string}
	 */
	_thisFilename : '',

	/**
	 * Projects list
	 *
	 * @private
	 * @type {Array}
	 */
	_prj : [],

	/**
	 * Script settings - throw errors (true) or not (false)
	 *
	 * @private
	 * @type {string}
	 */
	_valsThrowErrors : false,

	/**
	 * Script settings - prefix
	 *
	 * @private
	 * @type {string}
	 */
	_valsPrefix : '',

	/**
	 * Script settings - security string
	 *
	 * @private
	 * @type {string}
	 */
	_valsSecurity : '',

	/**
	 * Script settings - text for loading
	 *
	 * @private
	 * @type {string}
	 */
	_valsTextLoading : '',

	/**
	 * Script settings - text for loading error
	 *
	 * @private
	 * @type {string}
	 */
	_valsTextLoadingError : '',

	/**
	 * Script settings - text for license
	 *
	 * @private
	 * @type {string}
	 */
	_valsTextLicense : '',

	/**
	 * Script settings - text for accept
	 *
	 * @private
	 * @type {string}
	 */
	_valsTextAccept : '',

	/**
	 * Script settings - text for cancel
	 *
	 * @private
	 * @type {string}
	 */
	_valsTextCancel : '',

	/**
	 * Add project
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [projectName] Project name
	 * @return {void}
	 * @throws {kocujILV7aCException} kocujILV7aExceptionCode.PROJECT_ALREADY_EXISTS if project identifier entered in projectId already exists
	 */
	addProject : function(projectId, projectName) {
		'use strict';
		// parse arguments
		var args = this._checkAddProject(projectId, projectName);
		// add project
		if (this._prj['prj_' + args.projectId] === undefined) {
			this.addProjectIfNotExists(args.projectId, args.projectName);
		} else {
			this._throwError('PROJECT_ALREADY_EXISTS', args.projectId);
			return;
		}
	},

	/**
	 * Add project if not exists
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [projectName] Project name
	 * @return {void}
	 */
	addProjectIfNotExists : function(projectId, projectName) {
		'use strict';
		// parse arguments
		var args = this._checkAddProject(projectId, projectName);
		// add project
		if (this._prj['prj_' + args.projectId] === undefined) {
			this._prj['prj_' + args.projectId] = {
				projectName : args.projectName
			};
		} else {
			return;
		}
		this._objAllJsAjax.addProjectIfNotExists(args.projectId);
		this._objModal.addProjectIfNotExists(args.projectId);
	},

	/**
	 * Get HTML selector for license accept
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @return {string} HTML selector for license accept
	 */
	getHTMLSelectorLicenseAccept : function(projectId) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		// exit
		return this._objModal.getHTMLSelectorModalDialogStatus(projectId, 'license') + ' #' + this._getHTMLNameLicenseAccept(projectId);
	},

	/**
	 * Get HTML selector for license cancel
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @return {string} HTML selector for license cancel
	 */
	getHTMLSelectorLicenseCancel : function(projectId) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		// exit
		return this._objModal.getHTMLSelectorModalDialogStatus(projectId, 'license') + ' #' + this._getHTMLNameLicenseCancel(projectId);
	},

	/**
	 * Show license window
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {boolean} [acceptButton] Display accepting button (true) or not (false)
	 * @return {void}
	 */
	showLicense : function(projectId, acceptButton) {
		'use strict';
		// get this object
		var self = this;
		(function($) {
			// parse arguments
			projectId = self._parseProjectId(projectId);
			acceptButton = self._objHelper.initBoolean(acceptButton);
			// prepare modal window
			var licenseTextHeader = self._valsTextLicense;
			if (self._prj['prj_' + projectId].projectName !== '') {
				licenseTextHeader = self._prj['prj_' + projectId].projectName;
			}
			var type = kocujILV7aModalType.INFO;
			var status = '';
			if (acceptButton) {
				type = kocujILV7aModalType.INFO_WITH_STATUS;
				status = '<strong>' +
					'<a href="#" id="' + self._getHTMLNameLicenseAccept(projectId) + '">' +
					self._valsTextAccept +
					'</a>' +
					'</strong>' +
					'<br />' +
					'<div id="' + self._getHTMLNameLicenseCancel(projectId) + '">' +
					self._valsTextCancel +
					'</div>';
			}
			self._objModal.prepareModal(projectId, 'license',
				'<em>' +
				self._valsTextLoading +
				'</em>', {
					type   : type,
					title  : licenseTextHeader,
					status : status
				});
			// optionally remove mask event
			if (acceptButton) {
				self._objModal.removeCloseEventFromMask(projectId, 'license');
			}
			// get selectors
			var selectorModalDialogContentInside = $(self._objModal.getHTMLSelectorModalDialogContentInside(projectId, 'license'));
			// set style
			selectorModalDialogContentInside.css('font-family', '"Courier New", Courier, monospace');
			selectorModalDialogContentInside.css('text-align', 'center');
			// optionally show button
			if (acceptButton) {
				// get selectors
				var selectorLicenseAccept = $(self.getHTMLSelectorLicenseAccept(projectId));
				// prepare accept button
				selectorLicenseAccept.hide();
				$(self.getHTMLSelectorLicenseCancel(projectId)).hide();
				self._objModal.addCloseEvent(projectId, 'license', self.getHTMLSelectorLicenseAccept(projectId));
				selectorLicenseAccept.attr('href', 'javascript:void(0);');
				selectorLicenseAccept.bind('click.' + self._getEventNamePrefix(projectId) + 'license_accept', {
					projectId : projectId
				}, function(event) {
					event.preventDefault();
					self._objAllJsAjax.sendPost(event.data.projectId, 'license_accept', ajaxurl, 'text', {
						action   : self._valsPrefix + '_' + event.data.projectId + '__license_accept',
						security : self._valsSecurity
					});
					self._objModal.removeCloseEvent(event.data.projectId, 'license', self.getHTMLSelectorLicenseAccept(event.data.projectId));
					$(this).unbind('click.' + self._getEventNamePrefix(event.data.projectId) + 'license_accept');
				});
			}
			// show window
			self._objModal.showPreparedModal(projectId, 'license');
			// call AJAX
			self._objAllJsAjax.sendPost(projectId, 'license_display', ajaxurl, 'text', {
				action   : self._valsPrefix + '_' + projectId + '__license_display',
				security : self._valsSecurity
			}, {
				success : self.ajaxSuccessCallback,
				error   : self.ajaxErrorCallback
			});
		}(jQuery));
	},

	/**
	 * AJAX loading success callback
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} connectionId Connection identifier
	 * @param {anything} data Data
	 * @param {string} status Text status
	 * @param {Object} obj Request object
	 * @return {void}
	 */
	ajaxSuccessCallback : function(projectId, connectionId, data, status, obj) {
		'use strict';
		// get this object
		var self = kocujILV7aBackendLicense;
		(function($) {
			// parse parameters
			data = self._objHelper.initString(data);
			// get selectors
			var selectorModalDialogContentInside = $(self._objModal.getHTMLSelectorModalDialogContentInside(projectId, 'license'));
			var selectorLicenseAccept = $(self.getHTMLSelectorLicenseAccept(projectId));
			var selectorLicenseCancel = $(self.getHTMLSelectorLicenseCancel(projectId));
			// set HTML data
			selectorModalDialogContentInside.html(data);
			// set external links
			selectorModalDialogContentInside.find('a[rel=external]').attr('target', '_blank');
			// show license confirmation
			if (selectorLicenseAccept.length > 0) {
				selectorLicenseAccept.fadeTo('slow', 1.0);
			}
			if (selectorLicenseCancel.length > 0) {
				selectorLicenseCancel.fadeTo('slow', 1.0);
			}
		}(jQuery));
	},

	/**
	 * AJAX loading error callback
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} connectionId Connection identifier
	 * @param {Object} obj Request object
	 * @param {string} status Text status
	 * @param {string} err Error
	 * @return {void}
	 */
	ajaxErrorCallback : function(projectId, connectionId, obj, status, err) {
		'use strict';
		// get this object
		var self = kocujILV7aBackendLicense;
		(function($) {
			// set HTML data
			$(self._objModal.getHTMLSelectorModalDialogContentInside(projectId, 'license')).html('<strong>' + self._valsTextLoadingError + '</strong>');
		}(jQuery));
	},

	/**
	 * Get event name prefix
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @return {string} Event name prefix
	 */
	_getEventNamePrefix : function(projectId) {
		'use strict';
		// exit
		return this._valsPrefix + '_' + projectId + '__';
	},

	/**
	 * Get HTML prefix
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @return {string} HTML prefix
	 */
	_getHTMLPrefix : function(projectId) {
		'use strict';
		// exit
		return this._valsPrefix + '_' + projectId + '__';
	},

	/**
	 * Get HTML name for license accept
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @return {string} HTML name for license accept
	 */
	_getHTMLNameLicenseAccept : function(projectId) {
		'use strict';
		// exit
		return this._getHTMLPrefix(projectId) + 'license_accept';
	},

	/**
	 * Get HTML name for license cancel
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @return {string} HTML name for license cancel
	 */
	_getHTMLNameLicenseCancel : function(projectId) {
		'use strict';
		// exit
		return this._getHTMLPrefix(projectId) + 'license_cancel';
	},

	/**
	 * Parse project identifier
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @return {string} Parsed project identifier
	 * @throws {kocujILV7aCException} kocujILV7aExceptionCode.EMPTY_PROJECT_ID if project identifier entered in projectId is empty
	 * @throws {kocujILV7aCException} kocujILV7aExceptionCode.PROJECT_DOES_NOT_EXIST if project identifier entered in projectId does not exist
	 */
	_parseProjectId : function(projectId) {
		'use strict';
		// parse project identifier
		projectId = this._objHelper.initString(projectId);
		if (projectId === '') {
			this._throwError('EMPTY_PROJECT_ID');
			return;
		}
		// check if project exists
		if (this._prj['prj_' + projectId] === undefined) {
			this._throwError('PROJECT_DOES_NOT_EXIST', projectId);
			return;
		}
		// exit
		return projectId;
	},

	/**
	 * Check arguments for adding project
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @param {string} [projectName] Project name
	 * @return {Object} Parsed arguments for adding project
	 * @throws {kocujILV7aCException} kocujILV7aExceptionCode.EMPTY_PROJECT_ID if project identifier entered in projectId is empty
	 */
	_checkAddProject : function(projectId, projectName) {
		'use strict';
		// parse arguments
		projectId = this._objHelper.initString(projectId);
		if (projectId === '') {
			this._throwError('EMPTY_PROJECT_ID');
			return;
		}
		projectName = this._objHelper.initString(projectName);
		// exit
		return {
			projectId   : projectId,
			projectName : projectName
		};
	},

	/**
	 * Throw an error if debugging is enabled
	 *
	 * @private
	 * @param {string} codeString Error code in string format
	 * @param {string} [param] Parameter for error information
	 * @return {void}
	 */
	_throwError : function(codeString, param) {
		'use strict';
		// parse arguments
		codeString = this._objHelper.initString(codeString);
		if (codeString === '') {
			return;
		}
		param = this._objHelper.initString(param);
		// throw an error
		if (this._valsThrowErrors) {
			/* jshint evil: true */
			eval('throw new kocujILV7aCException(kocujILV7aExceptionCode.' + codeString + ', this._thisFilename, param);');
		}
	}
};

// initialize
var kocujILV7aBackendLicense = new kocujILV7aCBackendLicense();
