<?php

/**
 * component.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujIL\V7a\Classes\Project\Components\All\Config;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Configuration class
 *
 * @access public
 */
class Component extends \KocujIL\V7a\Classes\ComponentObject {
	/**
	 * Options data
	 *
	 * @access private
	 * @var array
	 */
	private $options = array();

	/**
	 * Option types
	 *
	 * @access private
	 * @var array
	 */
	private $types = array();

	/**
	 * Widgets data
	 *
	 * @access private
	 * @var array
	 */
	private $widgets = array();

	/**
	 * Last error
	 *
	 * @access private
	 * @var string
	 */
	private $lastError = '';

	/**
	 * Options container name
	 *
	 * @access protected
	 * @var string
	 */
	protected $containerName = '';

	/**
	 * Options container name for multisite (the same for all blogs)
	 *
	 * @access protected
	 * @var string
	 */
	protected $containerMultisiteName = '';

	/**
	 * Options cache
	 *
	 * @access private
	 * @var array
	 */
	private $optionsCache = array();

	/**
	 * Constructor
	 *
	 * @access public
	 * @param object $projectObj \KocujIL\V7a\Classes\Project object for current project
	 * @return void
	 */
	public function __construct($projectObj) {
		// execute parent constructor
		parent::__construct($projectObj);
		// add option types
		$this->types = array(
			'string' => array(
				array($this, 'typeString'),
				array($this, 'typeUpdateString'),
				array($this, 'typeGetString'),
			),
			'checkbox' => array(
				array($this, 'typeCheckbox'),
				array($this, 'typeUpdateCheckbox'),
				array($this, 'typeGetCheckbox'),
			),
			'numeric' => array(
				array($this, 'typeNumeric'),
				array($this, 'typeUpdateNumeric'),
				array($this, 'typeGetNumeric'),
			),
			'numericplus' => array(
				array($this, 'typeNumericPlus'),
				array($this, 'typeUpdateNumeric'),
				array($this, 'typeGetNumeric'),
			),
			'numericminus' => array(
				array($this, 'typeNumericMinus'),
				array($this, 'typeUpdateNumeric'),
				array($this, 'typeGetNumeric'),
			),
			'post' => array(
				array($this, 'typePost'),
				array($this, 'typeUpdatePost'),
				array($this, 'typeGetPost'),
			),
			'page' => array(
				array($this, 'typePage'),
				array($this, 'typeUpdatePage'),
				array($this, 'typeGetPage'),
			),
			'date' => array(
				array($this, 'typeDate'),
				array($this, 'typeUpdateDate'),
				array($this, 'typeGetDate'),
			),
			'time' => array(
				array($this, 'typeTime'),
				array($this, 'typeUpdateTime'),
				array($this, 'typeGetTime'),
			),
			'datetime' => array(
				array($this, 'typeDatetime'),
				array($this, 'typeUpdateDatetime'),
				array($this, 'typeGetDatetime'),
			),
		);
	}

	/**
	 * Initialization after adding to Kocuj Internal Lib object
	 *
	 * @access public
	 * @return void
	 */
	public function initAfterAddToAdmin() {
	}

	/**
	 * Get last error
	 *
	 * @access public
	 * @return string Last error
	 */
	public function getLastError() {
		// get last error
		return $this->lastError;
	}

	/**
	 * Get all options from database or cache
	 *
	 * @access private
	 * @param bool $isWidget It is options container for widget (true) or not (false) - default: false
	 * @param int $widgetId Widget id if it is options container for widget - default: empty
	 * @param bool $isGlobal Options container is for global site (true) or for blogs (false); only in multisite mode - default: true
	 * @return array All options from database or cache
	 */
	private function getOptionsFromDB($isWidget = false, $widgetId = 0, $isGlobal = true) {
		// get options container name
		$containerName = $this->getOptionsContainerName($isWidget, $widgetId, $isGlobal);
		// check options container name
		if (!isset($containerName[0]) /* strlen($containerName) === 0 */ ) {
			return array();
		}
		// check if options are not in cache
		if (isset($this->optionsCache[$containerName])) {
			return $this->optionsCache[$containerName];
		}
		// get options
		$array = maybe_unserialize(((is_multisite()) && (is_main_site()) && (!$isWidget) && ($isGlobal)) ?
			get_site_option($containerName) :
			get_option($containerName));
		if (!is_array($array)) {
			$array = maybe_unserialize($array);
		}
		if (empty($array)) {
			$array = array();
		}
		// add options to cache
		if (!isset($this->optionsCache[$containerName])) {
			$this->optionsCache[$containerName] = $array;
		}
		// exit
		return $array;
	}

	/**
	 * Clear options cache
	 *
	 * @access private
	 * @param bool $isWidget It is options container for widget (true) or not (false) - default: false
	 * @param int $widgetId Widget id if it is options container for widget - default: empty
	 * @param bool $isGlobal Options container is for global site (true) or for blogs (false); only in multisite mode - default: true
	 * @return void
	 */
	private function clearOptionsCache($isWidget = false, $widgetId = 0, $isGlobal = true) {
		// get options container name
		$containerName = $this->getOptionsContainerName($isWidget, $widgetId, $isGlobal);
		// clear options cache
		if (isset($this->optionsCache[$containerName])) {
			unset($this->optionsCache[$containerName]);
		}
	}

	/**
	 * Check if options container is empty
	 *
	 * @access public
	 * @param bool $isWidget It is options container for widget (true) or not (false) - default: false
	 * @param int $widgetId Widget id if it is options container for widget - default: empty
	 * @param bool $isGlobal Options container is for global site (true) or for blogs (false); only in multisite mode - default: true
	 * @return bool Options container is empty (true) or not (false)
	 */
	public function checkOptionsContainerEmpty($isWidget = false, $widgetId = 0, $isGlobal = true) {
		// get options from database
		$options = $this->getOptionsFromDB($isWidget, $widgetId, $isGlobal);
		// exit
		return empty($options);
	}

	/**
	 * Check if option exists in options container
	 *
	 * @access public
	 * @param string $name Option name
	 * @param bool $isWidget It is option for widget (true) or not (false) - default: false
	 * @param int $widgetId Widget id if it is option for widget - default: empty
	 * @param bool $isGlobal Option for global site (true) or for blogs (false); only in multisite mode - default: true
	 * @return bool Option exists in options container (true) or not (false)
	 */
	public function checkOptionExists($name, $isWidget = false, $widgetId = 0, $isGlobal = true) {
		// get options from database
		$options = $this->getOptionsFromDB($isWidget, $widgetId, $isGlobal);
		// exit
		return isset($options[$name]);
	}

	/**
	 * Get options container name
	 *
	 * @access public
	 * @param bool $isWidget It is options container for widget (true) or not (false) - default: false
	 * @param int $widgetId Widget id if it is options container for widget - default: empty
	 * @param bool $isGlobal Options container is for global site (true) or for blogs (false); only in multisite mode - default: true
	 * @return string Options container name
	 */
	public function getOptionsContainerName($isWidget = false, $widgetId = 0, $isGlobal = true) {
		// get options container name
		if ($isWidget) {
			if (!isset($this->widgets[$widgetId])) {
				return '';
			}
			return $this->widgets[$widgetId];
		} else {
			if ((is_multisite()) && (is_main_site()) && ($isGlobal)) {
				return $this->containerMultisiteName;
			}
		}
		// exit
		return $this->containerName;
	}

	/**
	 * Check if option should be used
	 *
	 * @access private
	 * @param array $option Option data
	 * @param bool $isWidget It is option for widget (true) or not (false) - default: false
	 * @param int $widgetId Widget id if it is option for widget - default: empty
	 * @return bool Option should be used (true) or not (false)
	 */
	private function checkOptionToUse(array $option, $isWidget = false, $widgetId = 0) {
		// exit
		return (((!$isWidget) && (!$option[6])) || ((($isWidget) && ($option[6]) && ($option[7] === $widgetId))));
	}

	/**
	 * Add option
	 *
	 * @access public
	 * @param string $id Option id
	 * @param string $defaultValue Default option value
	 * @param string $type Option type
	 * @param string $label Option label
	 * @param bool $isArray Option is array of values (true) or not (false) - default: false
	 * @param bool $arrayOptions Options for array if $isArray=true - default: empty
	 * @param bool $isWidget It is option for widget (true) or not (false) - default: false
	 * @param int $widgetId Widget id if it is option for widget - default: empty
	 * @param bool $networkGlobal Option in multisite mode is global (true) or not (false) - default: true
	 * @return bool Option has been added (true) or not (false)
	 */
	public function addOption($id, $defaultValue, $type, $label, $isArray = false, $arrayOptions = array(), $isWidget = false, $widgetId = 0,
		$multisiteGlobal = true) {
		// check if option does not exists
		if (isset($this->options[$id])) {
			return false;
		}
		// add option
		$this->options[$id] = array(
			'set_'.$id,
			$defaultValue,
			$type,
			$label,
			$isArray,
			$arrayOptions,
			$isWidget,
			$widgetId,
			$multisiteGlobal,
		);
		// exit
		return true;
	}

	/**
	 * Remove option
	 *
	 * @access public
	 * @param string $id Option id
	 * @return void
	 */
	public function removeOption($id) {
		// remove option
		if (isset($this->options[$id])) {
			unset($this->options[$id]);
		}
	}

	/**
	 * Add widget options container
	 *
	 * @access public
	 * @param string $containerName Options container name for widget
	 * @return int Widget id
	 */
	public function addWidgetOptionsContainer($containerName) {
		// check if widget does not exists
		if (in_array($containerName, $this->widgets)) {
			return 0;
		}
		// add widget
		$id = count($this->widgets);
		$this->widgets[$id] = $containerName;
		// exit
		return $id;
	}

	/**
	 * Get widget id by options container name
	 *
	 * @access public
	 * @param string $containerName Options container name for widget
	 * @param int &$id Returned widget id
	 * @return bool Widget has been found (true) or not (false)
	 */
	public function getWidgetIdByOptionsContainerName($containerName, &$id) {
		// get widget id by options container name
		$id = array_search($containerName, $this->widgets);
		if ($id === false) {
			$id = 0;
		}
		// exit
		return ($id !== false);
	}

	/**
	 * Add option type
	 *
	 * @access public
	 * @param string $type Option type
	 * @param string $function Callback method name for check new option value - default: NULL
	 * @param string $updateFunction Callback method name for update option - default: NULL
	 * @param string $getFunction Callback method name for get option - default: NULL
	 * @return void
	 */
	public function addType($type, $checkFunction = NULL, $updateFunction = NULL, $getFunction = NULL) {
		// add option type
		$this->types[$type] = array(
			(isset($checkFunction[0]) /* strlen($checkFunction) > 0 */ ) ?
				array($this, $checkFunction) :
				NULL,
			(isset($updateFunction[0]) /* strlen($updateFunction) > 0 */ ) ?
				array($this, $updateFunction) :
				NULL,
			(isset($getFunction[0]) /* strlen($getFunction) > 0 */ ) ?
				array($this, $getFunction) :
				NULL,
		);
	}

	/**
	 * Check option value
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option array
	 * @param array|bool|float|int|string $value Value to check
	 * @return bool Option is correct (true) or not (false)
	 */
	private function checkOption($name, array $option, $value) {
		// initialize
		$this->lastError = '';
		$ok = false;
		// check option value
		if ((isset($option[2])) && (!empty($this->types[$option[2]][0]))) {
			$error = '';
			$ok = call_user_func_array($this->types[$option[2]][0], array(
				$name,
				$option,
				$value,
				&$error,
			));
			if ((!$ok) && (isset($error[0]) /* strlen($error) > 0 */ )) {
				$this->lastError .= $error.'<br />';
			}
		}
		// exit
		return $ok;
	}

	/**
	 * Check option value by type
	 *
	 * @access public
	 * @param string $type Option type
	 * @param array|bool|float|int|string $value Value to check
	 * @return bool Option is correct (true) or not (false)
	 */
	public function checkOptionByType($type, $value) {
		// initialize
		$ok = false;
		// check option value
		if ((isset($this->types[$type])) && (!empty($this->types[$type][0]))) {
			$error = '';
			$option = array();
			$ok = call_user_func_array($this->types[$type][0], array(
				'',
				$option,
				$value,
				&$error,
			));
		}
		// exit
		return $ok;
	}

	/**
	 * Check option value - string
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param array|bool|float|int|string $value Value to check
	 * @param string &$error Returned error text
	 * @return bool Status - true or false
	 */
	private function typeString($name, array $option, $value, &$error) {
		// exit
		return true;
	}

	/**
	 * Update option value - string
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param bool $inArray This option is in array of options (true) or not (false)
	 * @param bool &$customOptionCheck Return setting which will set that method has custom option checking (true) or allow to check by checkOption method (false)
	 * @param array|bool|float|int|string &$value Returned value
	 * @param string &$error Returned error text
	 * @return bool Status - true or false
	 */
	private function typeUpdateString($name, array $option, $inArray, &$customOptionCheck, &$value, &$error) {
		// exit
		return true;
	}

	/**
	 * Get option value - string
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param bool $inArray This option is in array of options (true) or not (false)
	 * @param array|bool|float|int|string $value Value
	 * @return array|bool|float|int|string Modified value
	 */
	private function typeGetString($name, array $option, $inArray, $value) {
		// exit
		return $value;
	}

	/**
	 * Check option value - checkbox
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param array|bool|float|int|string $value Value to check
	 * @param string &$error Returned error text
	 * @return bool Status - true or false
	 */
	private function typeCheckbox($name, array $option, $value, &$error) {
		// check value
		if (empty($value)) {
			$value = 0;
		}
		if ((is_numeric($value)) && (($value == 0) || ($value == 1))) {
			return true;
		}
		$error = '"'.$option[3].'" '.$this->getStrings('all', 'config')->getString('TYPE_CHECKBOX_ON_OFF');
		// exit
		return false;
	}

	/**
	 * Update option value - checkbox
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param bool $inArray This option is in array of options (true) or not (false)
	 * @param bool &$customOptionCheck Return setting which will set that method has custom option checking (true) or allow to check by checkOption method (false)
	 * @param array|bool|float|int|string &$value Returned value
	 * @param string &$error Returned error text
	 * @return bool Status - true or false
	 */
	private function typeUpdateCheckbox($name, array $option, $inArray, &$customOptionCheck, &$value, &$error) {
		// update checkbox
		if (empty($value)) {
			$value = 0;
		}
		// exit
		return true;
	}

	/**
	 * Get option value - checkbox
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param bool $inArray This option is in array of options (true) or not (false)
	 * @param array|bool|float|int|string $value Value
	 * @return array|bool|float|int|string Modified value
	 */
	private function typeGetCheckbox($name, array $option, $inArray, $value) {
		// exit
		return $value;
	}

	/**
	 * Check option value - numeric
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param array|bool|float|int|string $value Value to check
	 * @param string &$error Returned error text
	 * @return bool Status - true or false
	 */
	private function typeNumeric($name, array $option, $value, &$error) {
		// check value
		if ((is_string($value)) && (!isset($value[0]) /* strlen($value) === 0 */ )) {
			$value = 0;
		}
		if (is_numeric($value)) {
			return true;
		}
		$error = '"'.$option[3].'" '.$this->getStrings('all', 'config')->getString('TYPE_NUMERIC_NUMERIC');
		// exit
		return false;
	}

	/**
	 * Update option value - numeric
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param bool $inArray This option is in array of options (true) or not (false)
	 * @param bool &$customOptionCheck Return setting which will set that method has custom option checking (true) or allow to check by checkOption method (false)
	 * @param array|bool|float|int|string &$value Returned value
	 * @param string &$error Returned error text
	 * @return bool Status - true or false
	 */
	private function typeUpdateNumeric($name, array $option, $inArray, &$customOptionCheck, &$value, &$error) {
		// exit
		return true;
	}

	/**
	 * Get option value - numeric
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param bool $inArray This option is in array of options (true) or not (false)
	 * @param array|bool|float|int|string $value Value
	 * @return array|bool|float|int|string Modified value
	 */
	private function typeGetNumeric($name, array $option, $inArray, $value) {
		// exit
		return $value;
	}

	/**
	 * Check option value - numeric plus
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param array|bool|float|int|string $value Value to check
	 * @param string &$error Returned error text
	 * @return bool Status - true or false
	 */
	private function typeNumericPlus($name, array $option, $value, &$error) {
		// check value
		if (!$this->typeNumeric($name, $option, $value, $error)) {
			return false;
		}
		if ($value >= 0) {
			return true;
		}
		$error = '"'.$option[3].'" '.$this->getStrings('all', 'config')->getString('TYPE_NUMERIC_PLUS_SMALLER_THAN_ZERO');
		// exit
		return false;
	}

	/**
	 * Check option value - numeric minus
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param array|bool|float|int|string $value Value to check
	 * @param string &$error Returned error text
	 * @return bool Status - true or false
	 */
	private function typeNumericMinus($name, array $option, $value, &$error) {
		// check value
		if (!$this->typeNumeric($name, $option, $value, $error)) {
			return false;
		}
		if ($value <= 0) {
			return true;
		}
		$error = '"'.$option[3].'" '.$this->getStrings('all', 'config')->getString('TYPE_NUMERIC_MINUS_BIGGER_THAN_ZERO');
		// exit
		return false;
	}

	/**
	 * Check option value - post
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param array|bool|float|int|string $value Value to check
	 * @param string &$error Returned error text
	 * @return bool Status - true or false
	 */
	private function typePost($name, array $option, $value, &$error) {
		// check value
		if (((is_numeric($value)) && (get_permalink($value))) || (((is_string($value)) && (!isset($value[0]) /* strlen($value) === 0 */ )))) {
			return true;
		}
		$error = '"'.$option[3].'" '.$this->getStrings('all', 'config')->getString('TYPE_POST_NUMERIC_EXIST');
		// exit
		return $ok;
	}

	/**
	 * Update option value - post
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param bool $inArray This option is in array of options (true) or not (false)
	 * @param bool &$customOptionCheck Return setting which will set that method has custom option checking (true) or allow to check by checkOption method (false)
	 * @param array|bool|float|int|string &$value Returned value
	 * @param string &$error Returned error text
	 * @return bool Status - true or false
	 */
	private function typeUpdatePost($name, array $option, $inArray, &$customOptionCheck, &$value, &$error) {
		// exit
		return true;
	}

	/**
	 * Get option value - post
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param bool $inArray This option is in array of options (true) or not (false)
	 * @param array|bool|float|int|string $value Value
	 * @return array|bool|float|int|string Modified value
	 */
	private function typeGetPost($name, array $option, $inArray, $value) {
		// exit
		return $value;
	}

	/**
	 * Check option value - page
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param array|bool|float|int|string $value Value to check
	 * @param string &$error Returned error text
	 * @return bool Status - true or false
	 */
	private function typePage($name, array $option, $value, &$error) {
		// check value
		$ok = false;
		if (((is_numeric($value)) && (get_permalink($value))) || (((is_string($value)) && (!isset($value[0]) /* strlen($value) === 0 */ )))) {
			$ok = true;
		}
		if (($option[2] === 'page') && ($ok) && (!is_page($value))) {
			$ok = false;
		}
		if (!$ok) {
			$error = '"'.$option[3].'" '.$this->getStrings('all', 'config')->getString('TYPE_PAGE_NUMERIC_EXIST');
		}
		// exit
		return $ok;
	}

	/**
	 * Update option value - page
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param bool $inArray This option is in array of options (true) or not (false)
	 * @param bool &$customOptionCheck Return setting which will set that method has custom option checking (true) or allow to check by checkOption method (false)
	 * @param array|bool|float|int|string &$value Returned value
	 * @param string &$error Returned error text
	 * @return bool Status - true or false
	 */
	private function typeUpdatePage($name, array $option, $inArray, &$customOptionCheck, &$value, &$error) {
		// exit
		return true;
	}

	/**
	 * Get option value - page
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param bool $inArray This option is in array of options (true) or not (false)
	 * @param array|bool|float|int|string $value Value
	 * @return array|bool|float|int|string Modified value
	 */
	private function typeGetPage($name, array $option, $inArray, $value) {
		// exit
		return $value;
	}

	/**
	 * Check option value - date
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param array|bool|float|int|string $value Value to check
	 * @param string &$error Returned error text
	 * @return bool Status - true or false
	 */
	private function typeDate($name, array $option, $value, &$error) {
		// check value
		$ok = preg_match('/^[0-9][0-9][0-9][0-9]\-[0-9][0-9]\-[0-9][0-9]$/s', trim($value));
		if (!$ok) {
			$error = '"'.$option[3].'" '.$this->getStrings('all', 'config')->getString('TYPE_DATE_FORMAT');
		}
		// exit
		return $ok;
	}

	/**
	 * Update option value - date
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param bool $inArray This option is in array of options (true) or not (false)
	 * @param bool &$customOptionCheck Return setting which will set that method has custom option checking (true) or allow to check by checkOption method (false)
	 * @param array|bool|float|int|string &$value Returned value
	 * @param string &$error Returned error text
	 * @return bool Status - true or false
	 */
	private function typeUpdateDate($name, array $option, $inArray, &$customOptionCheck, &$value, &$error) {
		// trim value
		$value = trim($value);
		// exit
		return true;
	}

	/**
	 * Get option value - date
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param bool $inArray This option is in array of options (true) or not (false)
	 * @param array|bool|float|int|string $value Value
	 * @return array|bool|float|int|string Modified value
	 */
	private function typeGetDate($name, array $option, $inArray, $value) {
		// exit
		return $value;
	}

	/**
	 * Check option value - time
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param array|bool|float|int|string $value Value to check
	 * @param string &$error Returned error text
	 * @return bool Status - true or false
	 */
	private function typeTime($name, array $option, $value, &$error) {
		// check value
		$ok = preg_match('/^[0-9][0-9]:[0-9][0-9]$/s', trim($value));
		if (!$ok) {
			$error = '"'.$option[3].'" '.$this->getStrings('all', 'config')->getString('TYPE_TIME_FORMAT');
		}
		// exit
		return $ok;
	}

	/**
	 * Update option value - time
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param bool $inArray This option is in array of options (true) or not (false)
	 * @param bool &$customOptionCheck Return setting which will set that method has custom option checking (true) or allow to check by checkOption method (false)
	 * @param array|bool|float|int|string &$value Returned value
	 * @param string &$error Returned error text
	 * @return bool Status - true or false
	 */
	private function typeUpdateTime($name, array $option, $inArray, &$customOptionCheck, &$value, &$error) {
		// trim value
		$value = trim($value);
		// exit
		return true;
	}

	/**
	 * Get option value - time
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param bool $inArray This option is in array of options (true) or not (false)
	 * @param array|bool|float|int|string $value Value
	 * @return array|bool|float|int|string Modified value
	 */
	private function typeGetTime($name, array $option, $inArray, $value) {
		// exit
		return $value;
	}

	/**
	 * Check option value - datetime
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param array|bool|float|int|string $value Value to check
	 * @param string &$error Returned error text
	 * @return bool Status - true or false
	 */
	private function typeDatetime($name, array $option, $value, &$error) {
		// check value
		$ok = preg_match('/^[0-9][0-9][0-9][0-9]\-[0-9][0-9]\-[0-9][0-9] [0-9][0-9]:[0-9][0-9]$/s', trim($value));
		if (!$ok) {
			$error = '"'.$option[3].'" '.$this->getStrings('all', 'config')->getString('TYPE_DATETIME_FORMAT');
		}
		// exit
		return $ok;
	}

	/**
	 * Update option value - datetime
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param bool $inArray This option is in array of options (true) or not (false)
	 * @param bool &$customOptionCheck Return setting which will set that method has custom option checking (true) or allow to check by checkOption method (false)
	 * @param array|bool|float|int|string &$value Returned value
	 * @param string &$error Returned error text
	 * @return bool Status - true or false
	 */
	private function typeUpdateDatetime($name, array $option, $inArray, &$customOptionCheck, &$value, &$error) {
		// trim value
		$value = trim($value);
		// exit
		return true;
	}

	/**
	 * Get option value - datetime
	 *
	 * @access private
	 * @param string $name Option name
	 * @param array $option Option settings
	 * @param bool $inArray This option is in array of options (true) or not (false)
	 * @param array|bool|float|int|string $value Value
	 * @return array|bool|float|int|string Modified value
	 */
	private function typeGetDatetime($name, array $option, $inArray, $value) {
		// exit
		return $value;
	}

	/**
	 * Update options for one container type
	 *
	 * @access private
	 * @param bool $isWidget It is option for widget (true) or not (false) - default: false
	 * @param int $widgetId Widget id if it is option for widget - default: empty
	 * @param bool $isGlobal Options for global site (true) or for blogs (false); only in multisite mode - default: true
	 * @return string Output information
	 */
	private function updateOptionsOneType($isWidget = false, $widgetId = 0, $isGlobal = true) {
		// get option name
		$containerName = $this->getOptionsContainerName($isWidget, $widgetId, $isGlobal);
		// check option name
		if (!isset($containerName[0]) /* strlen($containerName) === 0 */ ) {
			return '';
		}
		// check if force global or local options in multisite mode
		$forceOptionsMultisiteOrSite = false;
		$optionsMultisiteOrSite = false;
		if (!$isWidget) {
			if (is_multisite()) {
				$forceOptionsMultisiteOrSite = true;
				if (is_main_site()) {
					$optionsMultisiteOrSite = $isGlobal;
				} else {
					if ($isGlobal) {
						return '';
					}
					$optionsMultisiteOrSite = false;
				}
			} else {
				if ($isGlobal) {
					return '';
				}
			}
		} else {
			$isGlobal = false;
		}
		// initialize output
		$output = '';
		// update options
		$array = $this->getOptionsFromDB($isWidget, $widgetId, $isGlobal);
		if (!empty($this->options)) {
			foreach ($this->options as $key => $val) {
				if ((!$forceOptionsMultisiteOrSite) || ($val[8] === $optionsMultisiteOrSite)) {
					$correct = $this->checkOptionToUse($val, $isWidget, $widgetId);
					if ($correct) {
						if (isset($_REQUEST[$val[0]])) {
							$value = $_REQUEST[$val[0]];
							$this->getUpdateOption($key, $value, $outputTemp, $array, false, $array, $isWidget, $widgetId);
							$output .= $outputTemp;
						} else {
							$array[$key] = $this->getOption($key);
						}
					}
				}
			}
			if ((is_multisite()) && (is_main_site()) && (!$isWidget) && ($isGlobal)) {
				update_site_option($containerName, $array);
			} else {
				update_option($containerName, $array);
			}
			$this->clearOptionsCache($isWidget, $widgetId, $isGlobal);
		}
		// exit
		return $output;
	}

	/**
	 * Update options
	 *
	 * @access public
	 * @param bool $isWidget It is option for widget (true) or not (false) - default: false
	 * @param int $widgetId Widget id if it is option for widget - default: empty
	 * @return string Output information
	 */
	public function updateOptions($isWidget = false, $widgetId = 0) {
		// update options
		$output = ($isWidget) ?
			$this->updateOptionsOneType($isWidget, $widgetId) :
			$this->updateOptionsOneType($isWidget, $widgetId, true).$this->updateOptionsOneType($isWidget, $widgetId, false);
		// exit
		return $output;
	}

	/**
	 * Get update option
	 *
	 * @access private
	 * @param string $name Option name
	 * @param any $value Value to set to option
	 * @param string &$output Output information
	 * @param array &$outputArray Output array
	 * @param bool $forceUpdate Force options update (true) or not (false) - default: true
	 * @param array $inputArray Input array; if not empty, it will be used, instead of array get from options - default: empty
	 * @param bool $isWidget It is option for widget (true) or not (false) - default: false
	 * @param int $widgetId Widget id if it is option for widget - default: empty
	 * @return bool Status - true or false
	 */
	private function getUpdateOption($name, $value, &$output, array &$outputArray, $forceUpdate = true,
		array $inputArray = array(), $isWidget = false, $widgetId = 0) {
		// initialize output
		$output = '';
		// check if option is global
		$isGlobal = ((!$isWidget) && (isset($this->options[$name])) && ($this->options[$name][8]));
		// get option name
		$containerName = $this->getOptionsContainerName($isWidget, $widgetId, $isGlobal);
		// check option name
		if (!isset($containerName[0]) /* strlen($containerName) === 0 */ ) {
			return false;
		}
		// check if force global or local options in multisite mode
		$forceOptionsMultisiteOrSite = false;
		$optionsMultisiteOrSite = false;
		if (!$isWidget) {
			if (is_multisite()) {
				$forceOptionsMultisiteOrSite = true;
				if (is_main_site()) {
					$optionsMultisiteOrSite = $isGlobal;
				} else {
					if ($isGlobal) {
						return true;
					}
					$optionsMultisiteOrSite = false;
				}
			}
		} else {
			$isGlobal = false;
		}
		// update options
		$array = (!empty($inputArray)) ?
			$inputArray :
			$this->getOptionsFromDB($isWidget, $widgetId, $isGlobal);
		if (isset($this->options[$name])) {
			if ((!$forceOptionsMultisiteOrSite) || ($this->options[$name][8] === $optionsMultisiteOrSite)) {
				$correct = $this->checkOptionToUse($this->options[$name], $isWidget, $widgetId);
				if ($correct) {
					$ok = true;
					$customOptionCheck = false;
					if ($this->options[$name][4]) {
						$lastError = '';
						$keys2 = array_keys($value);
						foreach ($keys2 as $key2) {
							$row = $value[$key2];
							if (empty($row)) {
								unset($value[$key2]);
							} else {
								$customOptionCheck = false;
								$error = '';
								$ok2 = (!empty($this->types[$this->options[$name][2]][1])) ?
									call_user_func_array($this->types[$this->options[$name][2]][1], array(
											$name,
											$this->options[$name],
											true,
											&$customOptionCheck,
											&$row,
											&$error,
										)) :
									true;
								$value[$key2] = $row;
								if ((!$ok) && (isset($error[0]) /* strlen($error) > 0 */ )) {
									$this->lastError = $error;
								}
								if (!$customOptionCheck) {
									$ok2 = $this->checkOption($name, $this->options[$name], $row);
								}
								if (!$ok2) {
									if (!isset($lastError[0]) /* strlen($lastError) === 0 */ ) {
										$lastError = $this->lastError;
									}
									$ok = false;
								}
							}
						}
						$value = array_values($value);
						$this->lastError = $lastError;
					} else {
						$customOptionCheck = false;
						$error = '';
						$ok = (!empty($this->types[$this->options[$name][2]][1])) ?
							call_user_func_array($this->types[$this->options[$name][2]][1], array(
									$name,
									$this->options[$name],
									false,
									&$customOptionCheck,
									&$value,
									&$error,
								)) :
							true;
						if ((!$ok) && (isset($error[0]) /* strlen($error) > 0 */ )) {
							$this->lastError = $error;
						}
					}
					if ((!$this->options[$name][4]) && (!$customOptionCheck)) {
						$ok = $this->checkOption($name, $this->options[$name], $value);
					}
					if ($ok) {
						$array[$name] = $value;
					} else {
						$output .= $this->lastError;
					}
				}
			}
		}
		if ($forceUpdate) {
			if ((is_multisite()) && (is_main_site()) && (!$isWidget) && ($isGlobal)) {
				update_site_option($containerName, $array);
			} else {
				update_option($containerName, $array);
			}
			$this->clearOptionsCache($isWidget, $widgetId, $isGlobal);
		}
		$outputArray = $array;
		// exit
		return !(isset($output[0]) /* strlen($output) > 0 */ );
	}

	/**
	 * Update option
	 *
	 * @access public
	 * @param string $name Option name
	 * @param any $value Value to set to option
	 * @param string &$output Output information
	 * @param bool $isWidget It is option for widget (true) or not (false) - default: false
	 * @param int $widgetId Widget id if it is option for widget - default: empty
	 * @return bool Status - true or false
	 */
	public function updateOption($name, $value, &$output, $isWidget = false, $widgetId = 0) {
		// update option
		$outputArray = array();
		return $this->getUpdateOption($name, $value, $output, $outputArray, true, array(), $isWidget, $widgetId);
	}

	/**
	 * Delete option and restore to default values
	 *
	 * @access public
	 * @param string $name Option name
	 * @param bool $isWidget It is option for widget (true) or not (false) - default: false
	 * @param int $widgetId Widget id if it is option for widget - default: empty
	 * @return void
	 */
	public function deleteOption($name, $isWidget = false, $widgetId = 0) {
		// check if option is global
		$isGlobal = ((!$isWidget) && (isset($this->options[$name])) && ($this->options[$name][8]));
		// get option name
		$containerName = $this->getOptionsContainerName($isWidget, $widgetId, $isGlobal);
		// check option name
		if (!isset($containerName[0]) /* strlen($containerName) === 0 */ ) {
			return;
		}
		// check if force global or local options in multisite mode
		$forceOptionsMultisiteOrSite = false;
		$optionsMultisiteOrSite = false;
		if (!$isWidget) {
			if (is_multisite()) {
				$forceOptionsMultisiteOrSite = true;
				if (is_main_site()) {
					$optionsMultisiteOrSite = $isGlobal;
				} else {
					if ($isGlobal) {
						return;
					}
					$optionsMultisiteOrSite = false;
				}
			} else {
				if ($isGlobal) {
					return;
				}
			}
		} else {
			$isGlobal = false;
		}
		// delete option
		$array = $this->getOptionsFromDB($isWidget, $widgetId, $isGlobal);
		if (isset($this->options[$name])) {
			if ((!$forceOptionsMultisiteOrSite) || ($this->options[$name][8] === $optionsMultisiteOrSite)) {
				$correct = $this->checkOptionToUse($this->options[$name], $isWidget, $widgetId);
				if ($correct) {
					$array[$name] = ($this->options[$name][4]) ?
						array() :
						$this->options[$name][1];
				}
			}
		} else {
			unset($array[$name]);
		}
		if ((is_multisite()) && (is_main_site()) && (!$isWidget) && ($isGlobal)) {
			update_site_option($containerName, $array);
		} else {
			update_option($containerName, $array);
		}
		$this->clearOptionsCache($isWidget, $widgetId, $isGlobal);
	}

	/**
	 * Delete options and restore to default values for one container type
	 *
	 * @access private
	 * @param bool $isWidget It is option for widget (true) or not (false) - default: false
	 * @param int $widgetId Widget id if it is option for widget - default: empty
	 * @param bool $isGlobal Options for global site (true) or for blogs (false); only in multisite mode - default: true
	 * @return void
	 */
	private function deleteOptionsOneType($isWidget = false, $widgetId = 0, $isGlobal = true) {
		// get option name
		$containerName = $this->getOptionsContainerName($isWidget, $widgetId, $isGlobal);
		// check option name
		if (!isset($containerName[0]) /* strlen($containerName) === 0 */ ) {
			return;
		}
		// check if force global or local options in multisite mode
		$forceOptionsMultisiteOrSite = false;
		$optionsMultisiteOrSite = false;
		if (!$isWidget) {
			if (is_multisite()) {
				$forceOptionsMultisiteOrSite = true;
				if (is_main_site()) {
					$optionsMultisiteOrSite = $isGlobal;
				} else {
					if ($isGlobal) {
						return '';
					}
					$optionsMultisiteOrSite = false;
				}
			} else {
				if ($isGlobal) {
					return '';
				}
			}
		} else {
			$isGlobal = false;
		}
		// delete options
		$array = $this->getOptionsFromDB($isWidget, $widgetId, $isGlobal);
		if (!empty($this->options)) {
			foreach ($this->options as $key => $val) {
				if ((!$forceOptionsMultisiteOrSite) || ($val[8] === $optionsMultisiteOrSite)) {
					$correct = $this->checkOptionToUse($val, $isWidget, $widgetId);
					if ($correct) {
						$array[$key] = ($val[4]) ?
							array() :
							$val[1];
					}
				}
			}
			if ((is_multisite()) && (is_main_site()) && (!$isWidget) && ($isGlobal)) {
				update_site_option($containerName, $array);
			} else {
				update_option($containerName, $array);
			}
			$this->clearOptionsCache($isWidget, $widgetId, $isGlobal);
		}
	}

	/**
	 * Delete options and restore to default values
	 *
	 * @access public
	 * @param bool $isWidget It is option for widget (true) or not (false) - default: false
	 * @param int $widgetId Widget id if it is option for widget - default: empty
	 * @return void
	 */
	public function deleteOptions($isWidget = false, $widgetId = 0) {
		// update options
		if ($isWidget) {
			$this->deleteOptionsOneType($isWidget, $widgetId);
		} else {
			$this->deleteOptionsOneType($isWidget, $widgetId, true);
			$this->deleteOptionsOneType($isWidget, $widgetId, false);
		}
	}

	/**
	 * Get option value
	 *
	 * @access public
	 * @param string $name Option name
	 * @return array|bool|float|int|string Option value
	 */
	public function getOption($name) {
		// check option name
		if (!isset($name[0]) /* strlen($name) === 0 */ ) {
			return false;
		}
		// check if it is widget settings
		if (isset($this->options[$name])) {
			$isWidget = $this->options[$name][6];
			$widgetId = $this->options[$name][7];
		} else {
			$isWidget = false;
			$widgetId = 0;
		}
		// optionally change blog
		if ((is_multisite()) && (!$isWidget) && ($this->options[$name][8]) && (!is_main_site())) {
			switch_to_blog((defined('BLOG_ID_CURRENT_SITE')) ?
				BLOG_ID_CURRENT_SITE :
				1);
			$blogChanged = true;
		} else {
			$blogChanged = false;
		}
		// get option name
		$containerName = $this->getOptionsContainerName($isWidget, $widgetId, $this->options[$name][8]);
		// check option name
		if (!isset($containerName[0]) /* strlen($containerName) === 0 */ ) {
			if ($blogChanged) {
				restore_current_blog();
			}
			return false;
		}
		// get option value
		$array = $this->getOptionsFromDB($isWidget, $widgetId, $this->options[$name][8]);
		// check if option is empty
		$value = false;
		if (!isset($array[$name])) {
			if (isset($this->options[$name][1])) {
				$value = ($this->options[$name][4]) ?
					false :
					$this->options[$name][1];
			}
		} else {
			$value = $array[$name];
		}
		if ($this->options[$name][4]) {
			if (!empty($value)) {
				foreach ($value as $oneKey => $oneValue) {
					$ok = $this->checkOption($name, $this->options[$name], $oneValue);
					if (!$ok) {
						$value = false;
						break;
					} else {
						if (!empty($this->types[$this->options[$name][2]][2])) {
							$value[$oneKey] = call_user_func_array($this->types[$this->options[$name][2]][2], array(
								$name,
								$this->options[$name],
								true,
								$oneValue,
							));
						}
					}
				}
			} else {
					$value = false;
			}
		} else {
			if (!isset($this->options[$name])) {
				if ($blogChanged) {
					restore_current_blog();
				}
				return false;
			}
			$ok = $this->checkOption($name, $this->options[$name], $value);
			if (!$ok) {
				$value = $this->options[$name][1];
				if (!empty($this->types[$this->options[$name][2]][2])) {
					$value = call_user_func_array($this->types[$this->options[$name][2]][2], array(
						$name,
						$this->options[$name],
						false,
						$value,
					));
				}
			}
		}
		// optionally restore current blog
		if ($blogChanged) {
			restore_current_blog();
		}
		// exit
		return $value;
	}

	/**
	 * Get option default value
	 *
	 * @access public
	 * @param string $name Option name
	 * @return array|bool|float|int|string Option default value
	 */
	public function getOptionDefaultValue($name) {
		// exit
		return (isset($this->options[$name][1])) ?
			$this->options[$name][1] :
			'';
	}

	/**
	 * Get option type
	 *
	 * @access public
	 * @param string $name Option name
	 * @return string Option type
	 */
	public function getOptionType($name) {
		// exit
		return (isset($this->options[$name][2])) ?
			$this->options[$name][2] :
			'';
	}

	/**
	 * Get option label
	 *
	 * @access public
	 * @param string $name Option name
	 * @return string Option label
	 */
	public function getOptionLabel($name) {
		// exit
		return (isset($this->options[$name][3])) ?
			$this->options[$name][3] :
			'';
	}

	/**
	 * Check if option is array
	 *
	 * @access public
	 * @param string $name Option name
	 * @return bool Option is array (true) or not (false)
	 */
	public function checkOptionArray($name) {
		// exit
		return (isset($this->options[$name][4])) ?
			$this->options[$name][4] :
			false;
	}

	/**
	 * Check if option array can be reorganized
	 *
	 * @access public
	 * @param string $name Option name
	 * @return bool Option array can be reorganized (true) or not (false)
	 */
	public function checkOptionArrayReorganized($name) {
		// exit
		return ((isset($this->options[$name][5])) && (is_array($this->options[$name][5])) && (isset($this->options[$name][5]['allowreorganize'])) && ($this->options[$name][5]['allowreorganize']));
	}

	/**
	 * Check if option array can be closed
	 *
	 * @access public
	 * @param string $name Option name
	 * @return bool Option array can be closed (true) or not (false)
	 */
	public function checkOptionArrayClose($name) {
		// exit
		return ((isset($this->options[$name][5])) && (is_array($this->options[$name][5])) && (isset($this->options[$name][5]['allowclose'])) && ($this->options[$name][5]['allowclose']));
	}

	/**
	 * Check if option array should add new position when last element will be set
	 *
	 * @access public
	 * @param string $name Option name
	 * @return bool Option array should add new position when last element will be set (true) or not (false)
	 */
	public function checkOptionArrayAddOnLastNotEmpty($name) {
		// exit
		return ((isset($this->options[$name][5])) && (is_array($this->options[$name][5])) && (isset($this->options[$name][5]['addonlastnotempty'])) && ($this->options[$name][5]['addonlastnotempty']));
	}

	/**
	 * Check if option array should remove element if it become empty
	 *
	 * @access public
	 * @param string $name Option name
	 * @return bool Option array should remove element if it become empty (true) or not (false)
	 */
	public function checkOptionArrayRemoveOnEmpty($name) {
		// exit
		return ((isset($this->options[$name][5])) && (is_array($this->options[$name][5])) && (isset($this->options[$name][5]['removeonempty'])) && ($this->options[$name][5]['removeonempty']));
	}

	/**
	 * Check if option array should have add button
	 *
	 * @access public
	 * @param string $name Option name
	 * @return bool Option array should have add button (true) or not (false)
	 */
	public function checkOptionArrayAddButton($name) {
		// exit
		return ((isset($this->options[$name][5])) && (is_array($this->options[$name][5])) && (isset($this->options[$name][5]['addbutton'])) && ($this->options[$name][5]['addbutton']));
	}

	/**
	 * Get option array margin left
	 *
	 * @access public
	 * @param string $name Option name
	 * @return int Option array margin left
	 */
	public function getOptionArrayMarginLeft($name) {
		// exit
		return ((isset($this->options[$name][5])) && (is_array($this->options[$name][5])) && (isset($this->options[$name][5]['marginleft'])) && (!empty($this->options[$name][5]['marginleft']))) ?
			$this->options[$name][5]['marginleft'] :
			5;
	}

	/**
	 * Get option array vertical center
	 *
	 * @access public
	 * @param string $name Option name
	 * @return int Option array vertical center
	 */
	public function getOptionArrayVerticalCenter($name) {
		// exit
		return ((isset($this->options[$name][5])) && (is_array($this->options[$name][5])) && (isset($this->options[$name][5]['verticalcenter'])) && (!empty($this->options[$name][5]['verticalcenter']))) ?
			$this->options[$name][5]['verticalcenter'] :
			12;
	}

	/**
	 * Check if option is for widget
	 *
	 * @access public
	 * @param string $name Option name
	 * @return bool Option is for widget (true) or not (false)
	 */
	public function checkOptionForWidget($name) {
		// exit
		return (isset($this->options[$name][6])) ?
			$this->options[$name][6] :
			false;
	}

	/**
	 * Check if option is global in multisite mode
	 *
	 * @access public
	 * @param string $name Option name
	 * @return bool Option is global in multisite mode (true) or not (false)
	 */
	public function checkOptionForMultisite($name) {
		// exit
		return (isset($this->options[$name][8])) ?
			$this->options[$name][8] :
			false;
	}

	/**
	 * Get all options value
	 *
	 * @access public
	 * @param bool $isWidget It is option for widget (true) or not (false) - default: false
	 * @param int $widgetId Widget id if it is option for widget - default: empty
	 * @param bool $isGlobal Options for global site (true) or for blogs (false); only in multisite mode - default: true
	 * @return array All options value
	 */
	public function getAllOptions($isWidget = false, $widgetId = 0, $isGlobal = true) {
		// initialize
		$output = array();
		// check if force global or local options in multisite mode
		$forceOptionsMultisiteOrSite = false;
		$optionsMultisiteOrSite = false;
		if (!$isWidget) {
			if (is_multisite()) {
				$forceOptionsMultisiteOrSite = true;
				if (is_main_site()) {
					$optionsMultisiteOrSite = $isGlobal;
				} else {
					if ($isGlobal) {
						return '';
					}
					$optionsMultisiteOrSite = false;
				}
			} else {
				if ($isGlobal) {
					return '';
				}
			}
		} else {
			$isGlobal = false;
		}
		// get all options
		if (!empty($this->options)) {
			foreach ($this->options as $key => $option) {
				if ((!$forceOptionsMultisiteOrSite) || ($option[8] === $optionsMultisiteOrSite)) {
					$correct = $this->checkOptionToUse($option, $isWidget, $widgetId);
					if ($correct) {
						$output[$key] = $this->getOption($key);
					}
				}
			}
		}
		// exit
		return $output;
	}

	/**
	 * Uninstall options
	 *
	 * @access public
	 * @param bool $currentSiteOnly Uninstall options only for current site in multisite mode (true) or for current site and global options (false) - default: false
	 * @return void
	 */
	public function uninstallOptions($currentSiteOnly = false) {
		// uninstall main option
		if (isset($this->containerName[0]) /* strlen($this->containerName) > 0 */ ) {
			delete_option($this->containerName);
		}
		if (!$currentSiteOnly) {
			if ((is_multisite()) && (isset($this->containerMultisiteName[0]) /* strlen($this->containerMultisiteName) > 0 */ )) {
				delete_site_option($this->containerMultisiteName);
			}
		}
		// uninstall widgets options
		if (!empty($this->widgets)) {
			foreach ($this->widgets as $containerName) {
				delete_option($containerName);
			}
		}
	}
}
