<?php

/**
 * component.class.php
 *
 * @author Dominik Kocuj
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016-2017 kocuj.pl
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujIL\V8a\Classes\Project\Components\All\Options;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Options class
 *
 * @access public
 */
class Component extends \KocujIL\V8a\Classes\ComponentObject {
	/**
	 * Options types
	 *
	 * @access private
	 * @var array
	 */
	private $types = array();

	/**
	 * Options containers
	 *
	 * @access private
	 * @var array
	 */
	private $containers = array();

	/**
	 * Options definitions
	 *
	 * @access private
	 * @var array
	 */
	private $definitions = array();

	/**
	 * Options
	 *
	 * @access private
	 * @var array
	 */
	private $options = array();

	/**
	 * Constructor
	 *
	 * @access public
	 * @param object $projectObj \KocujIL\V8a\Classes\Project object for current project
	 * @return void
	 */
	public function __construct($projectObj) {
		// execute parent constructor
		parent::__construct($projectObj);
		// add options types
		$this->addType('text', NULL, NULL, \KocujIL\V8a\Enums\Project\Components\All\Options\TypeLengthSupport::YES);
		$this->addType('numeric', array(__CLASS__, 'typeDefaultValueNumeric'), array(__CLASS__, 'typeValidationNumeric'), \KocujIL\V8a\Enums\Project\Components\All\Options\TypeLengthSupport::YES);
		$this->addType('checkbox', array(__CLASS__, 'typeDefaultValueCheckbox'), array(__CLASS__, 'typeValidationCheckbox'));
	}

	/**
	 * Add options type
	 *
	 * @access public
	 * @param string $optionType Options type
	 * @param array|string $callbackDefaultValue Callback function or method name for default value of option; can be global function or method from any class; if empty, no callback will be used - default: NULL
	 * @param array|string $callbackValidation Callback function or method name for validation of option; can be global function or method from any class; if empty, no callback will be used - default: NULL
	 * @param int $typeLengthSupport Minimum and maximum length is supported or not; must be one of the following constants from \KocujIL\V8a\Enums\Project\Components\All\Options\TypeLengthSupport: NO (when length limits are not supported) or YES (when length limits are supported) - default: \KocujIL\V8a\Enums\Project\Components\All\Options\TypeLengthSupport::NO
	 * @return void
	 */
	public function addType($optionType, $callbackDefaultValue = NULL, $callbackValidation = NULL, $typeLengthSupport = \KocujIL\V8a\Enums\Project\Components\All\Options\TypeLengthSupport::NO) {
		// check if this option type identifier does not already exist
		if (isset($this->types[$optionType])) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::TYPE_ID_EXISTS, __FILE__, __LINE__, $optionType);
		}
		// add option type
		$this->types[$optionType] = array();
		if ($callbackDefaultValue !== NULL) {
			$this->types[$optionType]['defaultvalue'] = $callbackDefaultValue;
		}
		if ($callbackValidation !== NULL) {
			$this->types[$optionType]['validation'] = $callbackValidation;
		}
		if ($typeLengthSupport !== \KocujIL\V8a\Enums\Project\Components\All\Options\TypeLengthSupport::NO) {
			$this->types[$optionType]['lengthsupport'] = $typeLengthSupport;
		}
	}

	/**
	 * Get options types data
	 *
	 * @access public
	 * @return array Options types data; each options types data has the following fields: "defaultvalue" (array or string type; callback for default value of option), "lengthsupport" (int type; information is minimum and maximum length is supported for this options type), "validation" (array or string type; callback for validation of option)
	 */
	public function getTypes() {
		// prepare types
		$types = $this->types;
		if (!empty($types)) {
			foreach ($types as $key => $val) {
				if (!isset($val['defaultvalue'])) {
					$types[$key]['defaultvalue'] = '';
				}
				if (!isset($val['lengthsupport'])) {
					$types[$key]['lengthsupport'] = \KocujIL\V8a\Enums\Project\Components\All\Options\TypeLengthSupport::NO;
				}
				if (!isset($val['validation'])) {
					$types[$key]['validation'] = '';
				}
			}
		}
		// exit
		return $types;
	}

	/**
	 * Check if options type exists
	 *
	 * @access public
	 * @param string $id Options type identifier
	 * @return bool Options type exists (true) or not (false)
	 */
	public function checkType($id) {
		// exit
		return isset($this->types[$id]);
	}

	/**
	 * Get options type data by id
	 *
	 * @access public
	 * @param string $id Options type identifier
	 * @return array|bool Options type data or false if not exists; options type data have the following fields: "defaultvalue" (array or string type; callback for default value of option), "lengthsupport" (int type; information is minimum and maximum length is supported for this options type), "validation" (array or string type; callback for validation of option)
	 */
	public function getType($id) {
		// get types
		$types = $this->getTypes();
		// exit
		return (isset($types[$id])) ?
			$types[$id] :
			false;
	}

	/**
	 * Remove type
	 *
	 * @access public
	 * @param string $id Type identifier
	 * @return void
	 */
	public function removeType($id) {
		// check if this type identifier exists
		if (!isset($this->types[$id])) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::TYPE_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $id);
		}
		// remove type
		unset($this->types[$id]);
	}

	/**
	 * Callback for option default value - numeric
	 *
	 * @access private
	 * @param object $componentObj Object with this component
	 * @param string $defaultOptionValue Default option value
	 * @param array $additional Additional settings for option; there are the following additional settings which can be used: "maxvalue" (int type; maximum value of number), "minvalue" (int type; minimal value of number)
	 * @return string Parsed default option value
	 */
	private static function typeDefaultValueNumeric($componentObj, $defaultOptionValue, array $additional) {
		// prepare output value
		if ((isset($additional['minvalue'])) && ($defaultOptionValue < $additional['minvalue'])) {
			$defaultOptionValue = $additional['minvalue'];
		} else {
			if ((isset($additional['maxvalue'])) && ($defaultOptionValue > $additional['maxvalue'])) {
				$defaultOptionValue = (isset($additional['minvalue'])) ?
					(string)$additional['minvalue'] :
					'0';
			}
		}
		// exit
		return $defaultOptionValue;
	}

	/**
	 * Callback for option validation - numeric
	 *
	 * @access private
	 * @param object $componentObj Object with this component
	 * @param string $containerId Option container identifier
	 * @param string $optionId Option container identifier
	 * @param int $optionArray Option is array or standard; must be one of the following constants from \KocujIL\V8a\Enums\Project\Components\All\Options\OptionArray: NO (when it is standard option) or YES (when it is array option)
	 * @param string $optionValue Option value
	 * @param array $additional Additional settings for option; there are the following additional settings which can be used: "maxvalue" (int type; maximum value of number), "minvalue" (int type; minimal value of number)
	 * @return string Output text if there was an error or empty string if option value has been validated correctly
	 */
	private static function typeValidationNumeric($componentObj, $containerId, $optionId, $optionArray, $optionValue, array $additional) {
		// initialize
		$output = '';
		// check if it is numeric
		if (is_numeric($optionValue)) {
			if ((isset($additional['minvalue'])) && ($optionValue < $additional['minvalue'])) {
				$output = sprintf($componentObj->getStrings('options', \KocujIL\V8a\Enums\ProjectCategory::ALL)->getString($optionArray === \KocujIL\V8a\Enums\Project\Components\All\Options\OptionArray::YES ?
						'TYPE_VALIDATION_NUMERIC_ARRAY_ERROR_BELOW_MINIMUM_VALUE' :
						'TYPE_VALIDATION_NUMERIC_ERROR_BELOW_MINIMUM_VALUE'
					), $additional['minvalue']);
			} else {
				if ((isset($additional['maxvalue'])) && ($optionValue > $additional['maxvalue'])) {
					$output = sprintf($componentObj->getStrings('options', \KocujIL\V8a\Enums\ProjectCategory::ALL)->getString($optionArray === \KocujIL\V8a\Enums\Project\Components\All\Options\OptionArray::YES ?
							'TYPE_VALIDATION_NUMERIC_ARRAY_ERROR_ABOVE_MAXIMUM_VALUE' :
							'TYPE_VALIDATION_NUMERIC_ERROR_ABOVE_MAXIMUM_VALUE'
						), $additional['maxvalue']);
				}
			}
		} else {
			$output = $componentObj->getStrings('options', \KocujIL\V8a\Enums\ProjectCategory::ALL)->getString($optionArray === \KocujIL\V8a\Enums\Project\Components\All\Options\OptionArray::YES ?
					'TYPE_VALIDATION_NUMERIC_ARRAY_ERROR_NO_NUMERIC' :
					'TYPE_VALIDATION_NUMERIC_ERROR_NO_NUMERIC'
				);
		}
		// exit
		return $output;
	}

	/**
	 * Callback for option default value - checkbox
	 *
	 * @access private
	 * @param object $componentObj Object with this component
	 * @param string $defaultOptionValue Default option value
	 * @param array $additional Additional settings for option
	 * @return string Parsed default option value
	 */
	private static function typeDefaultValueCheckbox($componentObj, $defaultOptionValue, array $additional) {
		// exit
		return (($defaultOptionValue === '0') || ($defaultOptionValue === '1')) ?
			$defaultOptionValue :
			'0';
	}

	/**
	 * Callback for option validation - checkbox
	 *
	 * @access private
	 * @param object $componentObj Object with this component
	 * @param string $containerId Option container identifier
	 * @param string $optionId Option container identifier
	 * @param int $optionArray Option is array or standard; must be one of the following constants from \KocujIL\V8a\Enums\Project\Components\All\Options\OptionArray: NO (when it is standard option) or YES (when it is array option)
	 * @param string $optionValue Option value
	 * @param array $additional Additional settings for option
	 * @return string Output text if there was an error or empty string if option value has been validated correctly
	 */
	private static function typeValidationCheckbox($componentObj, $containerId, $optionId, $optionArray, $optionValue, array $additional) {
		// exit
		return (($optionValue === '0') || ($optionValue === '1')) ?
			'' :
			$componentObj->getStrings('options', \KocujIL\V8a\Enums\ProjectCategory::ALL)->getString($optionArray === \KocujIL\V8a\Enums\Project\Components\All\Options\OptionArray::YES ?
					'TYPE_VALIDATION_CHECKBOX_ARRAY_ERROR' :
					'TYPE_VALIDATION_CHECKBOX_ERROR'
				);
	}

	/**
	 * Add options container
	 *
	 * @access public
	 * @param string $containerId Options container identifier
	 * @param int $optionAutoload Automatic loading of option or not; this argument is ignored if $containerType is set to \KocujIL\V8a\Enums\Project\Components\All\Options\ContainerType::WIDGET; must be one of the following constants from \KocujIL\V8a\Enums\OptionAutoload: NO (when option should not be automatically loaded) or YES (when option should be automatically loaded) - default: \KocujIL\V8a\Enums\OptionAutoload::YES
	 * @param int $containerType Options container type; must be one of the following constants from \KocujIL\V8a\Enums\Project\Components\All\Options\ContainerType: NETWORK_OR_SITE (for network - in multisite installation - or site - in standard installation - type), SITE (for site type), NETWORK (for network type) or WIDGET (for widget type) - default: \KocujIL\V8a\Enums\Project\Components\All\Options\ContainerType::SITE
	 * @return void
	 */
	public function addContainer($containerId, $optionAutoload = \KocujIL\V8a\Enums\OptionAutoload::YES, $containerType = \KocujIL\V8a\Enums\Project\Components\All\Options\ContainerType::SITE) {
		// check if this container identifier does not already exist
		if (isset($this->containers[$containerId])) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::CONTAINER_ID_EXISTS, __FILE__, __LINE__, $containerId);
		}
		// add container
		$this->containers[$containerId] = array(
			'dbkey' => $this->getProjectObj()->getMainSettingInternalName().'__'.$containerId,
		);
		if ($optionAutoload === \KocujIL\V8a\Enums\OptionAutoload::YES) {
			$this->containers[$containerId]['autoload'] = $optionAutoload;
		}
		if ($containerType !== \KocujIL\V8a\Enums\Project\Components\All\Options\ContainerType::SITE) {
			$this->containers[$containerId]['type'] = $containerType;
		}
		// add empty options definitions for container
		$this->definitions[$containerId] = array();
		// add empty options for container
		$this->options[$containerId] = array(
			'options'    => array(),
			'alloptions' => array(),
			'loaded'     => false,
		);
	}

	/**
	 * Get options containers data
	 *
	 * @access public
	 * @return array Options containers data; each options container data has the following fields: "type" (int type; options container is automatic, for network or for site), "autoload" (bool type; if true, options container will be loaded automatically), "dbkey" (string type; database key for options container)
	 */
	public function getContainers() {
		// prepare containers
		$containers = $this->containers;
		if (!empty($containers)) {
			foreach ($containers as $key => $val) {
				if (!isset($val['autoload'])) {
					$containers[$key]['autoload'] = \KocujIL\V8a\Enums\OptionAutoload::NO;
				}
				if (!isset($val['type'])) {
					$containers[$key]['type'] = \KocujIL\V8a\Enums\Project\Components\All\Options\ContainerType::SITE;
				}
			}
		}
		// exit
		return $containers;
	}

	/**
	 * Check if options container exists
	 *
	 * @access public
	 * @param string $id Options container identifier
	 * @return bool Options container exists (true) or not (false)
	 */
	public function checkContainer($id) {
		// exit
		return isset($this->containers[$id]);
	}

	/**
	 * Get options container data by id
	 *
	 * @access public
	 * @param string $id Options container identifier
	 * @return array|bool Options container data or false if not exists; options container data have the following fields: "type" (int type; options container is automatic, for network or for site), "autoload" (bool type; if true, options container will be loaded automatically), "dbkey" (string type; database key for options container)
	 */
	public function getContainer($id) {
		// get containers
		$containers = $this->getContainers();
		// exit
		return (isset($containers[$id])) ?
			$containers[$id] :
			false;
	}

	/**
	 * Remove container
	 *
	 * @access public
	 * @param string $id Container identifier
	 * @return void
	 */
	public function removeContainer($id) {
		// check if this container identifier exists
		if (!isset($this->containers[$id])) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::CONTAINER_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $id);
		}
		// remove container
		unset($this->containers[$id], $this->options[$id]);
	}

	/**
	 * Load options from database if needed
	 *
	 * @access private
	 * @param string $containerId Options container identifier
	 * @return void
	 */
	private function loadOptionsFromDbIfNeeded($containerId) {
		// load container from database
		if (!$this->options[$containerId]['loaded']) {
			if ((!isset($this->containers[$containerId]['type'])) || ((isset($this->containers[$containerId]['type'])) && ($this->containers[$containerId]['type'] !== \KocujIL\V8a\Enums\Project\Components\All\Options\ContainerType::WIDGET))) {
				// load options from database
				$optionsFromDb = \KocujIL\V8a\Classes\DbDataHelper::getInstance()->getOption($this->containers[$containerId]['dbkey'], false, (isset($this->containers[$containerId]['type'])) ?
					$this->changeContainerTypeToArea($this->containers[$containerId]['type']) :
					\KocujIL\V8a\Enums\Area::SITE
				);
				if ($optionsFromDb !== false) {
					$this->options[$containerId]['options'] = maybe_unserialize($optionsFromDb);
					$this->options[$containerId]['alloptions'] = $this->options[$containerId]['options'];
				}
				// set loaded flag
				$this->options[$containerId]['loaded'] = true;
				// change options based on options definitions
				$this->changeOptionsByDefinitions($containerId, true);
			}
		}
	}

	/**
	 * Change options based on options definitions
	 *
	 * @access private
	 * @param string $containerId Options container identifier
	 * @param bool $force Force changing options (true) or not (false)
	 * @return void
	 */
	private function changeOptionsByDefinitions($containerId, $force = false) {
		// check options and definitions count
		if ((!$force) && (count($this->definitions[$containerId]) === count($this->options[$containerId]['options']))) {
			return;
		}
		// update options data
		$options = array();
		if (!empty($this->definitions[$containerId])) {
			foreach ($this->definitions[$containerId] as $optionId => $definition) {
				if (isset($this->options[$containerId]['options'][$optionId])) {
					$options[$optionId] = $this->options[$containerId]['options'][$optionId];
				} else {
					if (isset($this->options[$containerId]['alloptions'][$optionId])) {
						$options[$optionId] = $this->options[$containerId]['alloptions'][$optionId];
					} else {
						$options[$optionId] = $definition['defaultvalue'];
					}
				}
			}
		}
		$this->options[$containerId]['options'] = $options;
	}

	/**
	 * Add options container
	 *
	 * @access public
	 * @param string $containerId Options container identifier
	 * @param string $optionId Option identifier
	 * @param string $type Option type
	 * @param string $defaultValue Option default value
	 * @param string $label Option label
	 * @param int $optionArray Option is array or standard; must be one of the following constants from \KocujIL\V8a\Enums\Project\Components\All\Options\OptionArray: NO (when it is standard option) or YES (when it is array option) - default: \KocujIL\V8a\Enums\Project\Components\All\Options\OptionArray::NO
	 * @param array $arraySettings Array settings if $optionArray is set to \KocujIL\V8a\Enums\Project\Components\All\Options\OptionArray::YES - default: empty
	 * @param array $additional Additional settings for option; each options type can use different additional settings; there are the following additional settings which can be always used: "global_maxlength" (int type; maximum size of string; only for options types which have support for option value length), "global_minlength" (int type; minimal size of string; only for options types which have support for option value length) - default: empty
	 * @return void
	 */
	public function addDefinition($containerId, $optionId, $type, $defaultValue, $label, $optionArray = \KocujIL\V8a\Enums\Project\Components\All\Options\OptionArray::NO, array $arraySettings = array(), array $additional = array()) {
		// check if this container identifier exists
		if (!isset($this->containers[$containerId])) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::CONTAINER_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $containerId);
		}
		// check if this option type exists
		if (!isset($this->types[$type])) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::TYPE_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $type);
		}
		// check if this option definition does not already exist
		if (isset($this->definitions[$containerId][$optionId])) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::DEFINITION_ID_EXISTS, __FILE__, __LINE__, $optionId);
		}
		// check if it is not an array for widget
		if ((isset($this->containers[$containerId]['type'])) && ($this->containers[$containerId]['type'] === \KocujIL\V8a\Enums\Project\Components\All\Options\ContainerType::WIDGET) && ($optionArray === \KocujIL\V8a\Enums\Project\Components\All\Options\OptionArray::YES)) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::CANNOT_USE_ARRAY_OPTION_IN_WIDGET, __FILE__, __LINE__, $optionId);
		}
		// add option definition
		$this->definitions[$containerId][$optionId] = array(
			'type'         => $type,
			'defaultvalue' => (isset($this->types[$type]['defaultvalue'])) ?
				call_user_func_array($this->types[$type]['defaultvalue'], array(
					$this,
					$defaultValue,
					$additional,
				)) :
				$defaultValue,
			'label'        => $label,
		);
		if ($optionArray === \KocujIL\V8a\Enums\Project\Components\All\Options\OptionArray::YES) {
			$this->definitions[$containerId][$optionId]['array'] = $optionArray;
			$this->definitions[$containerId][$optionId]['arraysettings'] = $arraySettings;
		}
		if (!empty($additional)) {
			$this->definitions[$containerId][$optionId]['additional'] = $additional;
		}
		// change options based on options definitions
		$this->changeOptionsByDefinitions($containerId);
	}

	/**
	 * Get options definitions data
	 *
	 * @access public
	 * @param string $containerId Options container identifier
	 * @return array Options definitions data; each option definition data has the following fields: "additional" (array type; additional settings for option), "arraymode" (int type; set if option is standard or array), "arraysettings" (array type; settings for option if it is an array), "defaultvalue" (string type; option default value), "label" (string type; option label), "type" (string type; option type)
	 */
	public function getDefinitions($containerId) {
		// check if this container identifier exists
		if (!isset($this->containers[$containerId])) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::CONTAINER_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $containerId);
		}
		// prepare definitions
		$definitions = $this->definitions[$containerId];
		if (!empty($definitions)) {
			foreach ($definitions as $key => $val) {
				if (!isset($val['array'])) {
					$definitions[$key]['array'] = \KocujIL\V8a\Enums\Project\Components\All\Options\OptionArray::NO;
				}
				if (!isset($val['arraysettings'])) {
					$definitions[$key]['arraysettings'] = array();
				}
				if (!isset($val['additional'])) {
					$definitions[$key]['additional'] = array();
				}
			}
		}
		// exit
		return $definitions;
	}

	/**
	 * Check if option definition exists
	 *
	 * @access public
	 * @param string $containerId Options container identifier
	 * @param string $optionId Option definition identifier
	 * @return bool Option definition exists (true) or not (false)
	 */
	public function checkDefinition($containerId, $optionId) {
		// check if this container identifier exists
		if (!isset($this->containers[$containerId])) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::CONTAINER_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $containerId);
		}
		// exit
		return isset($this->definitions[$containerId][$optionId]);
	}

	/**
	 * Get option definition data by id
	 *
	 * @access public
	 * @param string $containerId Options container identifier
	 * @param string $optionId Option definition identifier
	 * @return array|bool Option definition data or false if not exists; option definition data have the following fields: "additional" (array type; additional settings for option), "arraymode" (int type; set if option is standard or array), "arraysettings" (array type; settings for option if it is an array), "defaultvalue" (string type; option default value), "label" (string type; option label), "type" (string type; option type)
	 */
	public function getDefinition($containerId, $optionId) {
		// get definitions
		$definitions = $this->getDefinitions($containerId);
		// exit
		return (isset($definitions[$optionId])) ?
			$definitions[$optionId] :
			false;
	}

	/**
	 * Remove option definition
	 *
	 * @access public
	 * @param string $containerId Options definition identifier
	 * @param string $optionId Option identifier
	 * @return void
	 */
	public function removeDefinition($containerId, $optionId) {
		// check if this container identifier exists
		if (!isset($this->containers[$containerId])) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::CONTAINER_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $containerId);
		}
		// check if this definition identifier exists
		if (!isset($this->definitions[$containerId][$optionId])) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::DEFINITION_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $optionId);
		}
		// remove option definition
		unset($this->definitions[$containerId][$optionId]);
	}

	/**
	 * Set option value and return text with status
	 *
	 * @access public
	 * @param string $containerId Option container identifier
	 * @param string $optionId Option identifier
	 * @param string $optionValue Option value
	 * @param string &$outputText Text with status of changing option value if there was an error
	 * @return bool Options has been set correctly (true) or not (false)
	 */
	public function setOptionWithReturnedText($containerId, $optionId, $optionValue, &$outputText) {
		// initialize
		$outputText = '';
		// check if this container identifier exists
		if (!isset($this->containers[$containerId])) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::CONTAINER_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $containerId);
		}
		// load options if needed
		$this->loadOptionsFromDbIfNeeded($containerId);
		// change options based on options definitions
		$this->changeOptionsByDefinitions($containerId);
		// check if this definition identifier exists
		if (!isset($this->definitions[$containerId][$optionId])) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::DEFINITION_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $optionId);
		}
		// get option definition type
		$type = $this->definitions[$containerId][$optionId]['type'];
		// check if option is an array
		$array = (isset($this->definitions[$containerId][$optionId]['array'])) ?
			$this->definitions[$containerId][$optionId]['array'] :
			\KocujIL\V8a\Enums\Project\Components\All\Options\OptionArray::NO;
		// prepare option values to parse
		$optionValues = $array === \KocujIL\V8a\Enums\Project\Components\All\Options\OptionArray::YES ?
			$optionValue :
			array(
				$optionValue,
			);
		// optionally check option value length
		if ((isset($this->types[$type]['lengthsupport'])) && (isset($this->definitions[$containerId][$optionId]['additional'])) && (!empty($optionValues))) {
			if (isset($this->definitions[$containerId][$optionId]['additional']['global_minlength'])) {
				$length = $this->definitions[$containerId][$optionId]['additional']['global_minlength'];
				foreach ($optionValues as $val) {
					if (strlen($val) < $length) {
						$outputText = sprintf($this->getStrings('options', \KocujIL\V8a\Enums\ProjectCategory::ALL)->getString($array === \KocujIL\V8a\Enums\Project\Components\All\Options\OptionArray::YES ?
								'SET_OPTION_WITH_RETURNED_ARRAY_TEXT_TOO_FEW_CHARACTERS' :
								'SET_OPTION_WITH_RETURNED_TEXT_TOO_FEW_CHARACTERS'
							), $length);
						return false;
					}
				}
			}
			if (isset($this->definitions[$containerId][$optionId]['additional']['global_maxlength'])) {
				$length = $this->definitions[$containerId][$optionId]['additional']['global_maxlength'];
				foreach ($optionValues as $val) {
					if (strlen($val) > $length) {
						$outputText = sprintf($this->getStrings('options', \KocujIL\V8a\Enums\ProjectCategory::ALL)->getString($array === \KocujIL\V8a\Enums\Project\Components\All\Options\OptionArray::YES ?
								'SET_OPTION_WITH_RETURNED_ARRAY_TEXT_TOO_MANY_CHARACTERS' :
								'SET_OPTION_WITH_RETURNED_TEXT_TOO_MANY_CHARACTERS'
							), $length);
						return false;
					}
				}
			}
		}
		// validate option value
		if ((isset($this->types[$type]['validation'])) && (!empty($optionValues))) {
			$additional = (isset($this->definitions[$containerId][$optionId]['additional'])) ?
				$this->definitions[$containerId][$optionId]['additional'] :
				array();
			foreach ($optionValues as $val) {
				$validationText = call_user_func_array($this->types[$type]['validation'], array(
					$this,
					$containerId,
					$optionId,
					$array,
					$val,
					$additional,
				));
				if (isset($validationText[0]) /* strlen($validationText) > 0 */ ) {
					$outputText = $validationText;
					return false;
				}
			}
		}
		// set option
		$this->options[$containerId]['options'][$optionId] = $optionValue;
		// exit
		return true;
	}

	/**
	 * Set option value
	 *
	 * @access public
	 * @param string $containerId Option container identifier
	 * @param string $optionId Option identifier
	 * @param string $optionValue Option value
	 * @return bool Options has been set correctly (true) or not (false)
	 */
	public function setOption($containerId, $optionId, $optionValue) {
		// initialize
		$text = '';
		// exit
		return $this->setOptionWithReturnedText($containerId, $optionId, $optionValue, $text);
	}

	/**
	 * Get option value
	 *
	 * @access public
	 * @param string $containerId Option container identifier
	 * @param string $optionId Option identifier
	 * @return string Option value
	 */
	public function getOption($containerId, $optionId) {
		// check if this container identifier exists
		if (!isset($this->containers[$containerId])) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::CONTAINER_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $containerId);
		}
		// load options if needed
		$this->loadOptionsFromDbIfNeeded($containerId);
		// change options based on options definitions
		$this->changeOptionsByDefinitions($containerId);
		// check if this definition identifier exists
		if (!isset($this->definitions[$containerId][$optionId])) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::DEFINITION_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $optionId);
		}
		// exit
		return $this->options[$containerId]['options'][$optionId];
	}

	/**
	 * Change container type to area
	 *
	 * @access private
	 * @param int $containerType Options container type; must be one of the following constants from \KocujIL\V8a\Enums\Project\Components\All\Options\ContainerType: NETWORK_OR_SITE (for network - in multisite installation - or site - in standard installation - type), SITE (for site type), NETWORK (for network type) or WIDGET (for widget type) - default: \KocujIL\V8a\Enums\Project\Components\All\Options\ContainerType::SITE
	 * @return int Area type
	 */
	private function changeContainerTypeToArea($containerType) {
		// change container type to area
		$typeToArea = array(
			\KocujIL\V8a\Enums\Project\Components\All\Options\ContainerType::SITE    => \KocujIL\V8a\Enums\Area::SITE,
			\KocujIL\V8a\Enums\Project\Components\All\Options\ContainerType::NETWORK => \KocujIL\V8a\Enums\Area::NETWORK,
		);
		// exit
		return isset($typeToArea[$containerType]) ?
			$typeToArea[$containerType] :
			\KocujIL\V8a\Enums\Area::AUTO;
	}

	/**
	 * Update container in database
	 *
	 * @access public
	 * @param string $containerId Option container identifier
	 * @return void
	 */
	public function updateContainerInDb($containerId) {
		// check if this container identifier exists
		if (!isset($this->containers[$containerId])) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::CONTAINER_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $containerId);
		}
		// check if option type can be used by this method
		if ((isset($this->containers[$containerId]['type'])) && ($this->containers[$containerId]['type'] === \KocujIL\V8a\Enums\Project\Components\All\Options\ContainerType::WIDGET)) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::WRONG_CONTAINER_TYPE_FOR_THIS_METHOD, __FILE__, __LINE__, $containerId);
		}
		// load options if needed
		$this->loadOptionsFromDbIfNeeded($containerId);
		// change options based on options definitions
		$this->changeOptionsByDefinitions($containerId);
		// save options to database
		\KocujIL\V8a\Classes\DbDataHelper::getInstance()->addOrUpdateOption($this->containers[$containerId]['dbkey'], $this->options[$containerId]['options'], (isset($this->containers[$containerId]['autoload'])) ?
				$this->containers[$containerId]['autoload'] :
				\KocujIL\V8a\Enums\OptionAutoload::NO,
			(isset($this->containers[$containerId]['type'])) ?
				$this->changeContainerTypeToArea($this->containers[$containerId]['type']) :
				\KocujIL\V8a\Enums\Area::SITE
			);
	}

	/**
	 * Update all containers in database
	 *
	 * @access public
	 * @return void
	 */
	public function updateAllContainersInDb() {
		// update all containers in database
		if (!empty($this->containers)) {
			foreach ($this->containers as $containerId => $containerData) {
				try {
					$this->updateContainerInDb($containerId);
				} catch (\KocujIL\V8a\Classes\Exception $e) {
				}
			}
		}
	}

	/**
	 * Remove container from database
	 *
	 * @access public
	 * @param string $containerId Option container identifier
	 * @return void
	 */
	public function removeContainerFromDb($containerId) {
		// check if this container identifier exists
		if (!isset($this->containers[$containerId])) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::CONTAINER_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $containerId);
		}
		// check if option type can be used by this method
		if ((isset($this->containers[$containerId]['type'])) && ($this->containers[$containerId]['type'] === \KocujIL\V8a\Enums\Project\Components\All\Options\ContainerType::WIDGET)) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::WRONG_CONTAINER_TYPE_FOR_THIS_METHOD, __FILE__, __LINE__, $containerId);
		}
		// delete options from database
		\KocujIL\V8a\Classes\DbDataHelper::getInstance()->deleteOption($this->containers[$containerId]['dbkey'], (isset($this->containers[$containerId]['type'])) ?
				$this->changeContainerTypeToArea($this->containers[$containerId]['type']) :
				\KocujIL\V8a\Enums\Area::SITE
			);

	}

	/**
	 * Remove all containers from database
	 *
	 * @access public
	 * @return void
	 */
	public function removeAllContainersFromDb() {
		// remove all containers from database
		if (!empty($this->containers)) {
			foreach ($this->containers as $containerId => $containerData) {
				try {
					$this->removeContainerFromDb($containerId);
				} catch (\KocujIL\V8a\Classes\Exception $e) {
				}
			}
		}
	}

	/**
	 * Load container for widget
	 *
	 * @access public
	 * @param string $containerId Option container identifier
	 * @param array $options Options
	 * @return void
	 */
	public function loadContainerForWidget($containerId, array $options) {
		// check if this container identifier exists
		if (!isset($this->containers[$containerId])) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::CONTAINER_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $containerId);
		}
		// check if option type can be used by this method
		if ((!isset($this->containers[$containerId]['type'])) || ((isset($this->containers[$containerId]['type'])) && ($this->containers[$containerId]['type'] !== \KocujIL\V8a\Enums\Project\Components\All\Options\ContainerType::WIDGET))) {
			throw new \KocujIL\V8a\Classes\Exception($this, \KocujIL\V8a\Enums\Project\Components\All\Options\ExceptionCode::WRONG_CONTAINER_TYPE_FOR_THIS_METHOD, __FILE__, __LINE__, $containerId);
		}
		// load container for widget
		$this->options[$containerId] = array(
			'options'    => $options,
			'alloptions' => $options,
			'loaded'     => true,
		);
		// change options based on options definitions
		$this->changeOptionsByDefinitions($containerId, true);
	}
}
