<?php

/**
 * component.class.php
 *
 * @author Dominik Kocuj
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016-2017 kocuj.pl
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujIL\V9a\Classes\Project\Components\Backend\SettingsForm;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Settings form class
 *
 * @access public
 */
class Component extends \KocujIL\V9a\Classes\ComponentObject {
	/**
	 * Settings controllers
	 *
	 * @access private
	 * @var array
	 */
	private $controllers = array();

	/**
	 * Settings forms
	 *
	 * @access private
	 * @var array
	 */
	private $forms = array();

	/**
	 * Forms tabs to display
	 *
	 * @access private
	 * @var array
	 */
	private $formsTabs = array();

	/**
	 * Constructor
	 *
	 * @access public
	 * @param object $projectObj \KocujIL\V9a\Classes\Project object for current project
	 * @return void
	 */
	public function __construct($projectObj) {
		// execute parent constructor
		parent::__construct($projectObj);
		// add controllers
		$this->addController('save', array(__CLASS__, 'controllerSave'));
		$this->addController('restore', array(__CLASS__, 'controllerRestore'));
	}

	/**
	 * Get form id
	 *
	 * @access public
	 * @param object $componentObj Component object
	 * @param string $id Form identifier
	 * @return string Form id
	 */
	public static function getFormId($componentObj, $id) {
		// exit
		return $componentObj->getComponent('project-helper')->getPrefix().'__form';
	}

	/**
	 * Get field id
	 *
	 * @access public
	 * @param object $componentObj Component object
	 * @param string $formId Form identifier
	 * @param string $fieldId Field identifier
	 * @return string Field id
	 */
	public static function getFieldId($componentObj, $formId, $fieldId) {
		// exit
		return $componentObj->getComponent('project-helper')->getPrefix().'__field__'.$formId.'__'.$fieldId;
	}

	/**
	 * Get nonce identifier
	 *
	 * @access public
	 * @param object $componentObj Component object
	 * @param string $id Form identifier
	 * @return string Nonce identifier
	 */
	public static function getNonceId($componentObj, $id) {
		// exit
		return $componentObj->getComponent('project-helper')->getPrefix().'__form__'.$id;
	}

	/**
	 * Get field HTML id for action
	 *
	 * @access public
	 * @param object $componentObj Component object
	 * @param string $id Form identifier
	 * @return string Field HTML id for action
	 */
	public static function getFieldIdAction($componentObj, $id) {
		// exit
		return $componentObj->getComponent('project-helper')->getPrefix().'__action__'.$id;
	}

	/**
	 * Get field name for action
	 *
	 * @access public
	 * @param object $componentObj Component object
	 * @return string Field name for action
	 */
	public static function getFieldNameAction($componentObj) {
		// exit
		return $componentObj->getComponent('project-helper')->getPrefix().'__action';
	}

	/**
	 * Get field name for form id
	 *
	 * @access public
	 * @param object $componentObj Component object
	 * @return string Field name for form id
	 */
	public static function getFieldNameFormId($componentObj) {
		// exit
		return $componentObj->getComponent('project-helper')->getPrefix().'__form_id';
	}

	/**
	 * Get field HTML id for tab
	 *
	 * @access public
	 * @param object $componentObj Component object
	 * @param string $formId Form identifier
	 * @param int $number Position number
	 * @return string Field HTML id for tab
	 */
	public static function getFieldIdTab($componentObj, $formId, $number) {
		// exit
		return $componentObj->getComponent('project-helper')->getPrefix().'__form__'.$formId.'__tab__'.$number;
	}

	/**
	 * Get field HTML id for tab div
	 *
	 * @access public
	 * @param object $componentObj Component object
	 * @param string $formId Form identifier
	 * @param int $number Position number
	 * @return string Field HTML id for tab div
	 */
	public static function getFieldIdTabDiv($componentObj, $formId, $number) {
		// exit
		return $componentObj->getComponent('project-helper')->getPrefix().'__form__'.$formId.'__tab_div__'.$number;
	}

	/**
	 * Get field HTML id for default button
	 *
	 * @access public
	 * @param object $componentObj Component object
	 * @param string $formId Form identifier
	 * @param string $buttonId Button identifier
	 * @return string Field HTML id for default button
	 */
	public static function getFieldIdDefaultButton($componentObj, $formId, $buttonId) {
		// exit
		return $componentObj->getComponent('project-helper')->getPrefix().'__form__'.$formId.'__default_button__'.$buttonId;
	}

	/**
	 * Get field HTML id for button
	 *
	 * @access public
	 * @param object $componentObj Component object
	 * @param string $formId Form identifier
	 * @param string $buttonId Button identifier
	 * @return string Field HTML id for button
	 */
	public static function getFieldIdButton($componentObj, $formId, $buttonId) {
		// exit
		return $componentObj->getComponent('project-helper')->getPrefix().'__form__'.$formId.'__button__'.$buttonId;
	}

	/**
	 * Get array container id
	 *
	 * @access public
	 * @param object $componentObj Component object
	 * @param string $formId Form identifier
	 * @param string $fieldId Field identifier
	 * @return string Array container id
	 */
	public static function getArrayContainerId($componentObj, $formId, $fieldId) {
		// exit
		return $componentObj->getComponent('project-helper')->getPrefix().'__form__'.$formId.'__array_container__'.$fieldId;
	}

	/**
	 * Add controller
	 *
	 * @access public
	 * @param string $id Controller identifier
	 * @param array|string $callbackController Callback function or method name for controller; can be global function or method from any class
	 * @return int Position in controllers list
	 */
	public function addController($id, $callbackController) {
		// add controller
		if (!isset($this->controllers[$id])) {
			$this->controllers[$id] = array();
		}
		$pos = count($this->controllers[$id]);
		$this->controllers[$id][$pos] = $callbackController;
		// exit
		return $pos;
	}

	/**
	 * Get controllers data
	 *
	 * @access public
	 * @return array Controllers data; each controller data has multiple callbacks for controller
	 */
	public function getControllers() {
		// exit
		return $this->controllers;
	}

	/**
	 * Check if controller exists
	 *
	 * @access public
	 * @param string $id Controller identifier
	 * @return bool Controller exists (true) or not (false)
	 */
	public function checkController($id) {
		// exit
		return isset($this->controllers[$id]);
	}

	/**
	 * Get controller data by id
	 *
	 * @access public
	 * @param string $id Controller identifier
	 * @return array|bool Controller data or false if not exists; controller data have multiple callbacks for controller
	 */
	public function getController($id) {
		// exit
		return (isset($this->controllers[$id])) ?
			$this->controllers[$id] :
			false;
	}

	/**
	 * Remove controller
	 *
	 * @access public
	 * @param string $id Controller identifier
	 * @return void
	 */
	public function removeController($id) {
		// check if this controller identifier exists
		if (!isset($this->controllers[$id])) {
			throw new \KocujIL\V9a\Classes\Exception($this, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsForm\ExceptionCode::CONTROLLER_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $id);
		}
		// remove controller
		unset($this->controllers[$id]);
	}

	/**
	 * Controller for save or restore options
	 *
	 * @access private
	 * @param object $componentObj Component object
	 * @param string $formId Form identifier
	 * @param bool $isRestore It is controller for restore (true) or save (false)
	 * @param string &$outputText Output text
	 * @return bool Controller has been executed correctly (true) or not (false)
	 */
	private static function saveOrRestoreInController($componentObj, $formId, $isRestore, &$outputText) {
		// get form
		$form = $componentObj->getForm($formId);
		// get options definitions
		$optionsDefinitions = $componentObj->getComponent('options', \KocujIL\V9a\Enums\ProjectCategory::ALL)->getDefinitions($form['optionscontainerid']);
		// update options
		if (!empty($optionsDefinitions)) {
			foreach ($optionsDefinitions as $optionId => $optionDef) {
				$post = $_POST;
				if (((!$isRestore) && (isset($post[$optionId]))) || ($isRestore)) {
					if ((!$isRestore) && ($optionDef['array']) && (isset($optionDef['arraysettings']['autoadddeleteifempty'])) && ($optionDef['arraysettings']['autoadddeleteifempty']) && (count($post[$optionId]) > 0)) {
						unset($post[$optionId][count($post[$optionId])-1]);
					}
					$optionText = '';
					if (!$componentObj->getComponent('options', \KocujIL\V9a\Enums\ProjectCategory::ALL)->setOptionWithReturnedText($form['optionscontainerid'], $optionId, ($isRestore) ?
								$optionDef['defaultvalue'] :
								$post[$optionId],
							$optionText)) {
						$outputText .= '<li>"'.$optionDef['label'].'": '.$optionText.'</li>';
					}
				}
			}
		}
		// update database
		$componentObj->getComponent('options', \KocujIL\V9a\Enums\ProjectCategory::ALL)->updateContainerInDb($form['optionscontainerid']);
		// add status text
		if (isset($outputText[0]) /* strlen($outputText) > 0 */ ) {
			$outputText = $componentObj->getStrings('settings-form', \KocujIL\V9a\Enums\ProjectCategory::BACKEND)->getString('SAVE_OR_RESTORE_IN_CONTROLLER_ERROR').'<br /><ul>'.$outputText.'</ul>';
			return false;
		} else {
			$outputText = $isRestore ?
				$componentObj->getStrings('settings-form', \KocujIL\V9a\Enums\ProjectCategory::BACKEND)->getString('SAVE_OR_RESTORE_IN_CONTROLLER_RESTORE') :
				$componentObj->getStrings('settings-form', \KocujIL\V9a\Enums\ProjectCategory::BACKEND)->getString('SAVE_OR_RESTORE_IN_CONTROLLER_UPDATE');
		}
		// exit
		return true;
	}

	/**
	 * Controller for save options
	 *
	 * @access private
	 * @param object $componentObj Component object
	 * @param string $formId Form identifier
	 * @param string &$outputText Output text
	 * @return bool Controller has been executed correctly (true) or not (false)
	 */
	private static function controllerSave($componentObj, $formId, &$outputText) {
		// exit
		return self::saveOrRestoreInController($componentObj, $formId, false, $outputText);
	}

	/**
	 * Controller for restore options
	 *
	 * @access private
	 * @param object $componentObj Component object
	 * @param string $formId Form identifier
	 * @param string &$outputText Output text
	 * @return bool Controller has been executed correctly (true) or not (false)
	 */
	private static function controllerRestore($componentObj, $formId, &$outputText) {
		// exit
		return self::saveOrRestoreInController($componentObj, $formId, true, $outputText);
	}

	/**
	 * Add form
	 *
	 * @access public
	 * @param string $id Form identifier
	 * @param string $optionsContainerId Options container identifier for form
	 * @param array $defaultButtons Definition of default buttons to add below form; there are available the following attributes: "isrestore" (bool type; if true, restore button will be displayed), "issubmit" (bool type; if true, submit button will be displayed), "restorelabel" (string type; restore button label), "restoretooltip" (string type; restore button tooltip), "submitlabel" (string type; submit button label), "submittooltip" (string type; submit button tooltip) - default: empty
	 * @param array $buttons Definition of additional buttons to add below form; each button data identifier will be used as button identifier; each button data has the following fields: "events" (array type; HTML events), "isprimary" (bool type; if true, button is primary), "label" (string type; button label), "tooltip" (string type; button tooltip) - default: empty
	 * @return void
	 */
	public function addForm($id, $optionsContainerId, array $defaultButtons = array(), $buttons = array()) {
		// check if this form identifier does not already exist
		if (isset($this->forms[$id])) {
			throw new \KocujIL\V9a\Classes\Exception($this, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsForm\ExceptionCode::FORM_ID_EXISTS, __FILE__, __LINE__, $id);
		}
		// add form
		$this->forms[$id] = array(
			'optionscontainerid' => $optionsContainerId,
		);
		if (!empty($defaultButtons)) {
			$this->forms[$id]['defaultbuttons'] = $defaultButtons;
		}
		if (!empty($buttons)) {
			$this->forms[$id]['buttons'] = $buttons;
		}
	}

	/**
	 * Get forms data
	 *
	 * @access public
	 * @return array Forms data; each form data has the following fields: "buttons" (array type; definition of buttons below form), "defaultbuttons" (array type; definition of default buttons below form), "optionscontainerid" (string type; identifier of options container for form), "tabs" (array type; tabs with form fields)
	 */
	public function getForms() {
		// prepare forms
		$forms = $this->forms;
		if (!empty($forms)) {
			foreach ($forms as $key => $val) {
				if (!isset($val['tabs'])) {
					$forms[$key]['tabs'] = array();
				}
				if (!isset($val['defaultbuttons'])) {
					$forms[$key]['defaultbuttons'] = array();
				}
				if (!isset($val['buttons'])) {
					$forms[$key]['buttons'] = array();
				}
			}
		}
		// exit
		return $forms;
	}

	/**
	 * Check if form exists
	 *
	 * @access public
	 * @param string $id Form identifier
	 * @return bool Form exists (true) or not (false)
	 */
	public function checkForm($id) {
		// exit
		return isset($this->forms[$id]);
	}

	/**
	 * Get form data by id
	 *
	 * @access public
	 * @param string $id Form identifier
	 * @return array|bool Form data or false if not exists; form data have the following fields: "buttons" (array type; definition of buttons below form), "defaultbuttons" (array type; definition of default buttons below form), "optionscontainerid" (string type; identifier of options container for form), "tabs" (array type; tabs with form fields)
	 */
	public function getForm($id) {
		// get forms
		$forms = $this->getForms();
		// exit
		return (isset($forms[$id])) ?
			$forms[$id] :
			false;
	}

	/**
	 * Remove form
	 *
	 * @access public
	 * @param string $id Form identifier
	 * @return void
	 */
	public function removeForm($id) {
		// check if this form identifier exists
		if (!isset($this->forms[$id])) {
			throw new \KocujIL\V9a\Classes\Exception($this, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsForm\ExceptionCode::FORM_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $id);
		}
		// remove form
		unset($this->forms[$id]);
	}

	/**
	 * Add tab to form
	 *
	 * @access public
	 * @param string $formId Form identifier
	 * @param string $tabId Tab identifier; must be unique in this project
	 * @param string $title Tab title; if empty, default title will be used - default: empty
	 * @return void
	 */
	public function addTab($formId, $tabId, $title = '') {
		// check if this form identifier exists
		if (!isset($this->forms[$formId])) {
			throw new \KocujIL\V9a\Classes\Exception($this, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsForm\ExceptionCode::FORM_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $formId);
		}
		// optionally initialize array
		if (!isset($this->forms[$formId]['tabs'])) {
			$this->forms[$formId]['tabs'] = array();
		}
		// check if this tab identifier does not already exist
		if (isset($this->forms[$formId]['tabs'][$tabId])) {
			throw new \KocujIL\V9a\Classes\Exception($this, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsForm\ExceptionCode::TAB_ID_EXISTS, __FILE__, __LINE__, $tabId);
		}
		// add settings tab
		$this->forms[$formId]['tabs'][$tabId] = array(
			'title' => $title,
		);
	}

	/**
	 * Get tab data
	 *
	 * @access public
	 * @param string $id Form identifier
	 * @return array Tab data; each tab data has the following fields: "fields" (array type; fields data; each element contains text with field in HTML to display), "title" (string type; tab title)
	 */
	public function getTabs($id) {
		// check if this form identifier exists
		if (!isset($this->forms[$id])) {
			throw new \KocujIL\V9a\Classes\Exception($this, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsForm\ExceptionCode::FORM_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $id);
		}
		// prepare tabs
		$tabs = $this->forms[$id]['tabs'];
		if (!empty($tabs)) {
			foreach ($tabs as $key => $val) {
				if (!isset($val['fields'])) {
					$tabs[$key]['fields'] = array();
				}
			}
		}
		// exit
		return $tabs;
	}

	/**
	 * Check if tab exists
	 *
	 * @access public
	 * @param string $formId Form identifier
	 * @param string $tabId Tab identifier
	 * @return bool Tab exists (true) or not (false)
	 */
	public function checkTab($formId, $tabId) {
		// check if this form identifier exists
		if (!isset($this->forms[$formId])) {
			throw new \KocujIL\V9a\Classes\Exception($this, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsForm\ExceptionCode::FORM_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $formId);
		}
		// exit
		return isset($this->forms[$formId]['tabs'][$tabId]);
	}

	/**
	 * Get tab data by id
	 *
	 * @access public
	 * @param string $formId Form identifier
	 * @param string $tabId Tab identifier
	 * @return array|bool Tab data or false if not exists; tab data have the following fields: "fields" (array type; fields data; each element contains text with field in HTML to display), "title" (string type; tab title)
	 */
	public function getTab($formId, $tabId) {
		// check if this form identifier exists
		if (!isset($this->forms[$formId])) {
			throw new \KocujIL\V9a\Classes\Exception($this, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsForm\ExceptionCode::FORM_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $formId);
		}
		// get tabs
		$tabs = $this->getTabs($formId);
		// exit
		return (isset($tabs[$tabId])) ?
			$tabs[$tabId] :
			false;
	}

	/**
	 * Remove tab
	 *
	 * @access public
	 * @param string $formId Form identifier
	 * @param string $tabId Tab identifier
	 * @return void
	 */
	public function removeTab($formId, $tabId) {
		// check if this form identifier exists
		if (!isset($this->forms[$formId])) {
			throw new \KocujIL\V9a\Classes\Exception($this, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsForm\ExceptionCode::FORM_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $formId);
		}
		// check if this tab identifier exists
		if (!isset($this->forms[$formId]['tabs'][$tabId])) {
			throw new \KocujIL\V9a\Classes\Exception($this, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsForm\ExceptionCode::TAB_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $tabId);
		}
		// remove tab
		unset($this->forms[$formId]['tabs'][$tabId]);
	}

	/**
	 * Add field or HTML code to form tab
	 *
	 * @access private
	 * @param string $formId Form identifier
	 * @param string $tabId Form tab identifier
	 * @param string $fieldType Field type
	 * @param string $fieldId Field identifier
	 * @param bool $forceFieldValue Force field value (true) or not (false)
	 * @param string $fieldValue Field value
	 * @param string $tipText Tooltip text
	 * @param string $html HTML code to add in field area
	 * @param string $htmlLeft HTML code to add in label area
	 * @param array $events HTML events
	 * @param array $additional Additional settings; each field type can use different additional settings; there are the following additional settings which can be always used: "global_addinfo" (string type; text to add below field), "global_addlabel" (string type; text to add to label), "global_hidelabel" (bool type; if true, label will be hidden), "global_widgetobj" (object type; it must be set to widget object, when it is widget)
	 * @return int Identifier
	 */
	private function addFieldOrHtmlToTab($formId, $tabId, $fieldType, $fieldId, $forceFieldValue, $fieldValue, $tipText, $html, $htmlLeft, array $events, array $additional) {
		// check if this form identifier exists
		if (!isset($this->forms[$formId])) {
			throw new \KocujIL\V9a\Classes\Exception($this, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsForm\ExceptionCode::FORM_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $formId);
		}
		// check if this settings tab identifier exists
		if (!isset($this->forms[$formId]['tabs'][$tabId])) {
			throw new \KocujIL\V9a\Classes\Exception($this, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsForm\ExceptionCode::TAB_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $tabId);
		}
		// get container identifier
		$containerId = $this->forms[$formId]['optionscontainerid'];
		// get option value
		if ($forceFieldValue) {
			$optionValue = $fieldValue;
		} else {
			$optionValue = (isset($fieldType[0]) /* strlen($fieldType) > 0 */ ) ?
				$this->getComponent('options', \KocujIL\V9a\Enums\ProjectCategory::ALL)->getOption($containerId, $fieldId) :
				'';
		}
		// check if this field is for widget settings
		$container = $this->getComponent('options', \KocujIL\V9a\Enums\ProjectCategory::ALL)->getContainer($containerId);
		$widget = ($container['type'] === \KocujIL\V9a\Enums\Project\Components\All\Options\ContainerType::WIDGET) ?
			\KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::YES :
			\KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::NO;
		// get option definition
		$optionDefinition = $this->getComponent('options', \KocujIL\V9a\Enums\ProjectCategory::ALL)->getDefinition($containerId, $fieldId);
		// optionally get option definition
		$arrayMode = \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\OptionArray::NO;
		if (isset($fieldType[0]) /* strlen($fieldType) > 0 */ ) {
			// check if option is array
			if (!$forceFieldValue) {
				if ((isset($fieldType[0]) /* strlen($fieldType) > 0 */ ) && ($widget === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::NO)) {
					if ($optionDefinition['array'] === \KocujIL\V9a\Enums\Project\Components\All\Options\OptionArray::YES) {
						$arrayMode = \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\OptionArray::YES;
					}
					if (($arrayMode === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\OptionArray::YES) && ($optionValue === false)) {
						$optionValue = $optionDefinition['defaultvalue'];
					}
				}
			}
		}
		// check if this field is not for array in widget settings
		if (($widget === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::YES) && ($arrayMode === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\OptionArray::YES)) {
			throw new \KocujIL\V9a\Classes\Exception($this, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsForm\ExceptionCode::CANNOT_USE_ARRAY_OPTION_IN_WIDGET, __FILE__, __LINE__);
		}
		// get label and HTML element order in widget
		$type = $this->getComponent('settings-fields', \KocujIL\V9a\Enums\ProjectCategory::BACKEND)->getType($optionDefinition['type']);
		$orderInWidget = $type['orderinwidget'];
		// add field beginning
		$fieldText = $widget === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::YES ?
			'<p>' :
			'<tr>';
		// add field label
		$optionLabel = '';
		if ((((!isset($fieldType[0]) /* strlen($fieldType) === 0 */ ) && (isset($htmlLeft[0]) /* strlen($htmlLeft) > 0 */ ))) || ((isset($fieldType[0]) /* strlen($fieldType) > 0 */ ) && ((!isset($additional['global_hidelabel'])) || ((isset($additional['global_hidelabel'])) && (!$additional['global_hidelabel']))))) {
			$optionLabel = (isset($fieldType[0]) /* strlen($fieldType) > 0 */ ) ?
				'<label for="'.esc_attr($widget === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::YES ?
						$additional['global_widgetobj']->get_field_id($fieldId) :
						self::getFieldId($this, $formId, $fieldId)
					).'">'.$optionDefinition['label'].(((isset($additional['global_addlabel'])) && (isset($additional['global_addlabel'][0]) /* strlen($additional['global_addlabel']) > 0 */ )) ?
						'<br />'.$additional['global_addlabel'] :
						''
					).(($widget === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::YES) && ($orderInWidget === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\OrderInWidget::FIRST_LABEL) ?
						':' :
						''
					).'</label>' :
				$htmlLeft;
			if ($widget === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::YES) {
				if ($orderInWidget === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\OrderInWidget::FIRST_LABEL) {
					$fieldText .= $optionLabel;
				}
			} else {
				$fieldText .= '<th scope="row">'.
					$optionLabel.
					'</th>';
			}
		}
		// prepare "colspan" tag
		if ($widget === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::YES) {
			$colspanTag = '';
		} else {
			$colspanTag = ((!isset($fieldType[0]) /* strlen($fieldType) === 0 */ ) && (!isset($htmlLeft[0]) /* strlen($htmlLeft) === 0 */ )) ?
				' colspan="2"' :
				'';
		}
		// prepare class and style for option field
		$class = ($widget === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::YES) ?
			'widefat' :
			'';
		$style = ($widget === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::YES) ?
			'' :
			($arrayMode === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\OptionArray::YES ?
					'' :
					'width:50%;'
				);
		$classWithTag = isset($class[0]) /* strlen($class) > 0 */ ?
			' class="'.esc_attr($class).'"' :
			'';
		$styleWithTag = isset($style[0]) /* strlen($style) > 0 */ ?
			' style="'.esc_attr($style).'"' :
			'';
		$classAndStyle = array(
			'class'        => $class,
			'style'        => $style,
			'classwithtag' => $classWithTag,
			'stylewithtag' => $styleWithTag,
			'all'          => $classWithTag.$styleWithTag,
		);
		// add field content
		if (isset($fieldType[0]) /* strlen($fieldType) > 0 */ ) {
			$divElementContainer = '<div data-type="####DATA_TYPE_ELEMENT_CONTAINER####" ####DATA_ATTRS#### style="padding-top:5px;padding-bottom:5px;">';
			$divField = '<div data-type="####DATA_TYPE_FIELD####" style="float:left;width:50%;">';
			$divControls = '<div data-type="####DATA_TYPE_CONTROLS####" style="float:left;">';
			if ($arrayMode === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\OptionArray::YES) {
				// initialize
				$fieldContent = '';
				$pos = 0;
				// show array
				$fieldContent .= '<div id="'.esc_attr(self::getArrayContainerId($this, $formId, $fieldId)).'">';
				if (!empty($optionValue)) {
					foreach ($optionValue as $val) {
						$fieldContent .= str_replace('####DATA_ATTRS####', 'data-number="'.esc_attr($pos).'" data-prev="" data-next=""', str_replace('####DATA_TYPE_ELEMENT_CONTAINER####', 'element-container', $divElementContainer)).
							str_replace('####DATA_TYPE_FIELD####', 'field', $divField).
							$this->getComponent('settings-fields', \KocujIL\V9a\Enums\ProjectCategory::BACKEND)->getField($fieldType, $fieldId, ($widget === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::YES) ?
									$fieldId :
									self::getFieldId($this, $formId, $fieldId),
								$val, $tipText, $classAndStyle, $widget, $arrayMode, $events, $additional).
							'</div>'.
							str_replace('####DATA_TYPE_CONTROLS####', 'controls', $divControls).
							'</div>'.
							'<div style="clear:both;">'.
							'</div>'.
							'</div>';
						++$pos;
					}
				}
				$fieldContent .= '</div>';
				// add script
				if (empty($this->formsArrays)) {
					\KocujIL\V9a\Classes\JsHelper::getInstance()->addLibScript('backend-settings-form-array', 'project/components/backend/settings-form', 'array', array(
						'helper',
						'data-helper',
					), array(), 'kocujILV9aBackendSettingsFormArrayVals', array(
						'htmlIdFormatArrayContainer' => $this->getArrayContainerId($this, '####FORM_ID####', '####FIELD_ID####'),
						'textButtonAddNewElement'    => $this->getStrings('settings-form', \KocujIL\V9a\Enums\ProjectCategory::BACKEND)->getString('ADD_FIELD_OR_HTML_TO_TAB_BUTTON_ADD_NEW_ELEMENT'),
						'divElementContainer'        => $divElementContainer,
						'divField'                   => $divField,
						'divControls'                => $divControls,
						'imageArrowUp'               => \KocujIL\V9a\Classes\LibUrls::getInstance()->get('images').'/project/components/backend/settings-form/arrow-up.png',
						'imageArrowDown'             => \KocujIL\V9a\Classes\LibUrls::getInstance()->get('images').'/project/components/backend/settings-form/arrow-down.png',
						'imageDelete'                => \KocujIL\V9a\Classes\LibUrls::getInstance()->get('images').'/project/components/backend/settings-form/delete.png',
						'imageEmpty'                 => \KocujIL\V9a\Classes\LibUrls::getInstance()->get('images').'/project/components/backend/settings-form/empty.png',
					));
				}
				// remember array data
				if (!isset($this->formsArrays[$formId])) {
					$this->formsArrays[$formId] = array();
				}
				$this->formsArrays[$formId][$fieldId] = array(
					'arraysettings' => $optionDefinition['arraysettings'],
					'count'         => count($optionValue),
					'emptyfield'    => $this->getComponent('settings-fields', \KocujIL\V9a\Enums\ProjectCategory::BACKEND)->getField($fieldType, $fieldId, $widget === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::YES ?
							$fieldId :
							self::getFieldId($this, $formId, $fieldId),
						'', $tipText, $classAndStyle, $widget, $arrayMode, $events, $additional),
				);
			} else {
				$fieldContent = $this->getComponent('settings-fields', \KocujIL\V9a\Enums\ProjectCategory::BACKEND)->getField($fieldType, $fieldId, $widget === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::YES ?
						$fieldId :
						self::getFieldId($this, $formId, $fieldId),
					$optionValue, $tipText, $classAndStyle, $widget, $arrayMode, $events, $additional);
			}
		} else {
			$fieldContent = $html;
		}
		if ($widget === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::YES) {
			$fieldText .= $fieldContent;
		} else {
			$fieldText .= '<td'.$colspanTag.'>'.
				$fieldContent.
				'</td>';
		}
		// optionally add label
		if (($widget === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::YES) && ($orderInWidget === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\OrderInWidget::FIRST_ELEMENT)) {
			$fieldText .= $optionLabel;
		}
		// add field ending
		$fieldText .= ($widget === \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::YES) ?
			'</p>' :
			'</tr>';
		// get new field identifier
		if (!isset($this->forms[$formId]['tabs'][$tabId]['fields'])) {
			$this->forms[$formId]['tabs'][$tabId]['fields'] = array();
		}
		$id = count($this->forms[$formId]['tabs'][$tabId]['fields']);
		// remember field text
		$this->forms[$formId]['tabs'][$tabId]['fields'][$id] = $fieldText;
		// exit
		return $id;
	}

	/**
	 * Add option field to form tab
	 *
	 * @access public
	 * @param string $formId Form identifier
	 * @param string $tabId Form tab identifier
	 * @param string $fieldType Field type
	 * @param string $fieldId Field identifier
	 * @param string $tipText Tooltip text - default: empty
	 * @param array $events HTML events - default: empty
	 * @param array $additional Additional settings; each field type can use different additional settings; there are the following additional settings which can be always used: "global_addinfo" (string type; text to add below field), "global_addlabel" (string type; text to add to label), "global_hidelabel" (bool type; if true, label will be hidden), "global_widgetobj" (object type; it must be set to widget object, when it is widget) - default: empty
	 * @return int Identifier
	 */
	public function addOptionFieldToTab($formId, $tabId, $fieldType, $fieldId, $tipText = '', array $events = array(), array $additional = array()) {
		// exit
		return $this->addFieldOrHtmlToTab($formId, $tabId, $fieldType, $fieldId, false, '', $tipText, '', '', $events, $additional);
	}

	/**
	 * Add field to form tab
	 *
	 * @access public
	 * @param string $formId Form identifier
	 * @param string $tabId Form tab identifier
	 * @param string $fieldType Field type
	 * @param string $fieldId Field identifier
	 * @param string $fieldValue Field value
	 * @param string $tipText Tooltip text - default: empty
	 * @param array $events HTML events - default: empty
	 * @param array $additional Additional settings; each field type can use different additional settings; there are the following additional settings which can be always used: "global_addinfo" (string type; text to add below field), "global_addlabel" (string type; text to add to label), "global_hidelabel" (bool type; if true, label will be hidden), "global_widgetobj" (object type; it must be set to widget object, when it is widget) - default: empty
	 * @return int Identifier
	 */
	public function addFieldToTab($formId, $tabId, $fieldType, $fieldId, $fieldValue, $tipText = '', array $events = array(), array $additional = array()) {
		// exit
		return $this->addFieldOrHtmlToTab($formId, $tabId, $fieldType, $fieldId, true, $fieldValue, $tipText, '', '', $events, $additional);
	}

	/**
	 * Add HTML code to form tab
	 *
	 * @access public
	 * @param string $formId Form identifier
	 * @param string $tabId Form tab identifier
	 * @param string $html HTML code to add in field area
	 * @param string $htmlLeft HTML code to add in label area - default: empty
	 * @return int Identifier
	 */
	public function addHtmlToTab($formId, $tabId, $html, $htmlLeft = '') {
		// exit
		return $this->addFieldOrHtmlToTab($formId, $tabId, '', '', false, '', '', $html, $htmlLeft, array(), array());
	}

	/**
	 * Show form
	 *
	 * @access public
	 * @param string $id Form identifier
	 * @return void
	 */
	public function showForm($id) {
		// check if this form identifier exists
		if (!isset($this->forms[$id])) {
			throw new \KocujIL\V9a\Classes\Exception($this, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsForm\ExceptionCode::FORM_ID_DOES_NOT_EXIST, __FILE__, __LINE__, $id);
		}
		// check if this form is for widget
		$container = $this->getComponent('options', \KocujIL\V9a\Enums\ProjectCategory::ALL)->getContainer($this->forms[$id]['optionscontainerid']);
		$isWidget = ($container['type'] === \KocujIL\V9a\Enums\Project\Components\All\Options\ContainerType::WIDGET);
		// get tabs count
		$tabsCount = (isset($this->forms[$id]['tabs'])) ?
			count($this->forms[$id]['tabs']) :
			0;
		// check if there is exactly one tab when form is for widget
		if (($isWidget) && ($tabsCount !== 1)) {
			throw new \KocujIL\V9a\Classes\Exception($this, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsForm\ExceptionCode::WRONG_TABS_COUNT_IN_WIDGET, __FILE__, __LINE__, $tabsCount);
		}
		// show form header
		if (!$isWidget) {
			// show form start
			$this->getComponent('project-helper')->doAction('before_form_div');
			echo '<div'.$this->getComponent('project-helper')->applyFiltersForHTMLStyleAndClass('form_div').' id="'.esc_attr($this->getComponent('project-helper')->getPrefix().'__div_form').'">';
			$this->getComponent('project-helper')->doAction('before_form');
			echo '<form method="post" action="#" name="'.esc_attr(self::getFormId($this, $id)).'" id="'.esc_attr(self::getFormId($this, $id)).'">';
			$this->getComponent('project-helper')->doAction('inside_form_begin');
			// set nonce
			wp_nonce_field(self::getNonceId($this, $id));
			// show form identifier field
			echo '<input type="hidden" name="'.esc_attr(self::getFieldNameFormId($this)).'" value="'.esc_attr($id).'" />';
			// show action data field
			echo '<input type="hidden" name="'.esc_attr(self::getFieldNameAction($this)).'" id="'.esc_attr(self::getFieldIdAction($this, $id)).'" value="" />';
			// optionally add script
			if ((empty($this->formsTabs)) && ($tabsCount > 1)) {
				\KocujIL\V9a\Classes\JsHelper::getInstance()->addLibScript('backend-settings-form-tabs', 'project/components/backend/settings-form', 'tabs', array(
					'helper',
				), array(), 'kocujILV9aBackendSettingsFormTabsVals', array(
					'prefix' => \KocujIL\V9a\Classes\Helper::getInstance()->getPrefix(),
				));
			}
			// remember form tabs
			if ($tabsCount > 1) {
				$this->formsTabs[$id] = $tabsCount;
			}
			// show tabs header
			if ($tabsCount > 1) {
				echo '<h2 class="nav-tab-wrapper">';
				$tabPos = 0;
				foreach ($this->forms[$id]['tabs'] as $tabId => $tabData) {
					$tabTitle = (isset($tabData['title'][0]) /* strlen($tabData['title']) > 0 */ ) ?
						$tabData['title'] :
						sprintf($this->getStrings('settings-form', \KocujIL\V9a\Enums\ProjectCategory::BACKEND)->getString('SHOW_FORM_TAB'), $tabPos+1);
					echo '<a href="#" id="'.esc_attr(self::getFieldIdTab($this, $id, $tabPos)).'" class="nav-tab">'.$tabTitle.'</a>';
					++$tabPos;
				}
				echo '</h2>';
			} else {
				if (isset($this->forms[$id]['tabs'][0]['title'][0]) /* strlen($this->forms[$id]['tabs'][0]['title']) > 0 */ ) {
					echo '<h2>'.$this->forms[$id]['tabs'][0]['title'].'</h2>';
				}
			}
		}
		// show tabs content
		if (isset($this->forms[$id]['tabs'])) {
			$tabPos = 0;
			foreach ($this->forms[$id]['tabs'] as $tabId => $tabData) {
				if (!$isWidget) {
					echo '<div id="'.esc_attr(self::getFieldIdTabDiv($this, $id, $tabPos)).'"'.(($tabsCount > 1) ?
							' style="visibility:hidden;position:absolute;"' :
							''
						).'>';
				}
				if (isset($tabData['fields'])) {
					if (!$isWidget) {
						echo '<table class="form-table"><tbody><tr style="display:none;"><th scope="row">&nbsp;</th><td>&nbsp;</td></tr>';
					}
					foreach ($tabData['fields'] as $field) {
						echo $field;
					}
					if (!$isWidget) {
						echo '</tbody></table>';
					}
				}
				if (!$isWidget) {
					echo '</div>';
				}
				++$tabPos;
			}
		}
		// initialize bottom buttons
		$bottomButtons = '';
		// initialize class and style for bottom buttons
		$classAndStyle = array(
			'class'        => '',
			'style'        => '',
			'classwithtag' => '',
			'stylewithtag' => '',
			'all'          => '',
		);
		// optionally show submit button
		if ((!$isWidget) && (isset($this->forms[$id]['defaultbuttons']['issubmit'])) && ($this->forms[$id]['defaultbuttons']['issubmit'])) {
			$bottomButtons .= $this->getComponent('settings-fields', \KocujIL\V9a\Enums\ProjectCategory::BACKEND)->getField('button', self::getFieldIdDefaultButton($this, $id, 'submit'), '', '', $this->forms[$id]['defaultbuttons']['submittooltip'], $classAndStyle, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::NO, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\OptionArray::NO, array(
				'click' => '$(\'#'.esc_js(self::getFieldIdAction($this, $id)).'\').val(\'save\');'.PHP_EOL.
					'$(\''.esc_js('#'.self::getFormId($this, $id)).'\').submit();'.PHP_EOL,
			), array(
				'buttonlabel'   => $this->forms[$id]['defaultbuttons']['submitlabel'],
				'buttonprimary' => true,
			));
		}
		// optionally show restore button
		if ((!$isWidget) && (isset($this->forms[$id]['defaultbuttons']['isrestore'])) && ($this->forms[$id]['defaultbuttons']['isrestore'])) {
			$bottomButtons .= $this->getComponent('settings-fields', \KocujIL\V9a\Enums\ProjectCategory::BACKEND)->getField('button', self::getFieldIdDefaultButton($this, $id, 'restore'), '', '', $this->forms[$id]['defaultbuttons']['restoretooltip'], $classAndStyle, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::NO, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\OptionArray::NO, array(
				'click' => 'if (confirm(\''.esc_js($this->getStrings('settings-form', \KocujIL\V9a\Enums\ProjectCategory::BACKEND)->getString('SHOW_FORM_CONFIRM_RESTORE')).'\')) {'.PHP_EOL.

					'$(\'#'.esc_js(self::getFieldIdAction($this, $id)).'\').val(\'restore\');'.PHP_EOL.
					'$(\''.esc_js('#'.self::getFormId($this, $id)).'\').submit();'.PHP_EOL.
					'}'.PHP_EOL,
			), array(
				'buttonlabel' => $this->forms[$id]['defaultbuttons']['restorelabel'],
			));
		}
		// optionally show other buttons
		if (isset($this->forms[$id]['buttons'])) {
			foreach ($this->forms[$id]['buttons'] as $buttonId => $buttonData) {
				$bottomButtons .= $this->getComponent('settings-fields', \KocujIL\V9a\Enums\ProjectCategory::BACKEND)->getField('button', ($isWidget) ?
							$buttonId :
							self::getFieldIdButton($this, $id, $buttonId),
						'', '', $buttonData['tooltip'], $classAndStyle, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\FieldForWidget::NO, \KocujIL\V9a\Enums\Project\Components\Backend\SettingsFields\OptionArray::NO, $buttonData['events'], array(
					'buttonlabel' => $buttonData['label'],
				));
			}
		}
		// optionally display bottom buttons
		if (isset($bottomButtons[0]) /* strlen($bottomButtons) > 0 */) {
			echo '<br />'.$bottomButtons;
		}
		// show form end
		if (!$isWidget) {
			$this->getComponent('project-helper')->doAction('inside_form_end');
			echo '</form>';
			$this->getComponent('project-helper')->doAction('after_form');
			echo '</div>';
			$this->getComponent('project-helper')->doAction('after_form_div');
		}
	}

	/**
	 * Action for controllers
	 *
	 * @access public
	 * @return void
	 */
	public function actionController() {
		// optionally execute controller
		if (($this->getComponent('settings-menu', \KocujIL\V9a\Enums\ProjectCategory::BACKEND)->checkCurrentPageIsSettingsForProject()) && (isset($_POST['_wpnonce'])) && (isset($_POST[self::getFieldNameFormId($this)])) && (wp_verify_nonce($_POST['_wpnonce'], self::getNonceId($this, $_POST[self::getFieldNameFormId($this)]))) && (isset($_POST[self::getFieldNameAction($this)])) && (isset($this->controllers[$_POST[self::getFieldNameAction($this)]])) && (isset($this->forms[$_POST[self::getFieldNameFormId($this)]]))) {
			// execute controller
			$controllerTextCorrect = '';
			$controllerTextError = '';
			$controllerStatus = true;
			foreach ($this->controllers[$_POST[self::getFieldNameAction($this)]] as $callback) {
				$outputText = '';
				$output = call_user_func_array($callback, array(
					$this,
					$_POST[self::getFieldNameFormId($this)],
					&$outputText,
				));
				if (isset($outputText[0]) /* strlen($outputText) > 0 */ ) {
					if (!$output) {
						$controllerStatus = false;
					}
					$controllerText = ($output) ?
						$controllerTextCorrect :
						$controllerTextError;
					if (isset($controllerText[0]) /* strlen($controllerText) > 0 */ ) {
						$controllerText .= '<br /><br />';
					}
					$controllerText .= $outputText;
					if ($output) {
						$controllerTextCorrect = $controllerText;
					} else {
						$controllerTextError = $controllerText;
					}
				}
			}
			// optionally show message with controller output text
			if (isset($controllerText[0]) /* strlen($controllerText) > 0 */ ) {
				$this->getComponent('message', \KocujIL\V9a\Enums\ProjectCategory::BACKEND)->addMessage('settings_form_information', $controllerStatus ?
						$controllerTextCorrect :
						$controllerTextError,
					$controllerStatus ?
						\KocujIL\V9a\Enums\Project\Components\Backend\Message\Type::INFORMATION :
						\KocujIL\V9a\Enums\Project\Components\Backend\Message\Type::ERROR
					);
			}
		}
	}

	/**
	 * Action for tabs script
	 *
	 * @access public
	 * @return void
	 */
	public function actionPrintFooterScripts() {
		// initialize tabs script
		if (!empty($this->formsTabs)) {
			?>
				<script type="text/javascript">
				/* <![CDATA[ */
					(function($) {
						$(document).ready(function() {
							kocujILV9aBackendSettingsFormTabs.addProject('<?php echo esc_js($this->getProjectObj()->getMainSettingInternalName()); ?>', '<?php echo esc_js($this->getProjectObj()->getMainSettingTitleOriginal()); ?>');
							<?php
								foreach ($this->formsTabs as $formId => $tabsCount) {
									?>
									kocujILV9aBackendSettingsFormTabs.process('<?php echo esc_js($this->getProjectObj()->getMainSettingInternalName()); ?>', '<?php echo esc_js($formId); ?>', '<?php echo esc_js($tabsCount); ?>');
									<?php
								}
							?>
						});
					}(jQuery));
				/* ]]> */
				</script>
			<?php
		}
		// initialize array script
		if (!empty($this->formsArrays)) {
			?>
				<script type="text/javascript">
				/* <![CDATA[ */
					(function($) {
						$(document).ready(function() {
							kocujILV9aBackendSettingsFormArray.addProject('<?php echo esc_js($this->getProjectObj()->getMainSettingInternalName()); ?>', '<?php echo esc_js($this->getProjectObj()->getMainSettingTitleOriginal()); ?>');
							<?php
								$jsObjFields = array(
									'addnewbutton',
									'allowchangeorder',
									'autoadddeleteifempty',
									'deletebutton',
								);
								foreach ($this->formsArrays as $formId => $fieldsData) {
									if (!empty($fieldsData)) {
										foreach ($fieldsData as $fieldId => $data) {
											$jsObj = '';
											foreach ($jsObjFields as $objField) {
												$jsObj .= $objField.': '.(((isset($data['arraysettings'][$objField])) && ($data['arraysettings'][$objField])) ?
														'true' :
														'false'
													).', ';
											}
											$jsObj = substr($jsObj, 0, -2);
											?>
											kocujILV9aBackendSettingsFormArray.process('<?php echo esc_js($this->getProjectObj()->getMainSettingInternalName()); ?>', '<?php echo esc_js($formId); ?>', '<?php echo esc_js($fieldId); ?>', '<?php echo str_replace('<', '<\' + \'', str_replace('\'', '\\\'', $data['emptyfield'])); ?>', '<?php echo esc_js($data['count']); ?>', {<?php echo $jsObj; ?>});
											<?php
										}
									}
								}
							?>
						});
					}(jQuery));
				/* ]]> */
				</script>
			<?php
		}
	}
}
