<?php

/**
 * admin.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html
 * @copyright Copyright (c) 2013-2016 Dominik Kocuj
 * @package kocuj_sitemap
 */

// set namespace
namespace KocujSitemapPlugin\Classes;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

// check if load class
if ((is_admin()) || (is_network_admin())) {
	/**
	 * Plugin admin class
	 *
	 * @access public
	 */
	class Admin extends \KocujInternalLib\V1a\Classes\Project\Admin {
		/**
		 * Administration pages data
		 *
		 * @access private
		 * @var array
		 */
		private $adminPagesData = array();

		/**
		 * Widgets settings data
		 *
		 * @access private
		 * @var array
		 */
		private $adminWidgetsSettingsData = array();

		/**
		 * Constructor
		 *
		 * @access public
		 * @param object $projectObj \KocujInternalLib\V1a\Classes\Project object for current project
		 * @return void
		 */
		public function __construct($projectObj) {
			// execute parent constructor
			parent::__construct($projectObj);
			// set plugin activation hook
			$this->pluginActivationHook = array(\KocujSitemapPlugin\Classes\Base::getInstance(), 'activate');
			// set plugin deactivation hook
			$this->pluginDeactivationHook = array(\KocujSitemapPlugin\Classes\Base::getInstance(), 'deactivate');
			// set nonce action
			$this->nonceAction = 'kocujsitemap';
			// set meta boxes with additional information
			$locale = get_locale();
			if ($locale === 'pl_PL') {
				/* translators: %s: Link (HTML anchor) to website */
				$this->getProjectObj()->getObj('backend_settings-meta-boxes')->addSettingsMetaBox('author', __('Order your own unique website', 'kocuj-sitemap'), sprintf(__('Visit %s website for more information about ordering the website creation.', 'kocuj-sitemap'), \KocujInternalLib\V1a\Classes\Helper::getInstance()->getLinkAnchor('http://kocuj.pl/', 'kocuj.pl', array(
						'external' => true,
					))).
					'<br /><br />'.
					'<strong>'.__('About Dominik Kocuj - the author of this plugin and owner of kocuj.pl website', 'kocuj-sitemap').'</strong>'.
					'<ul style="list-style:disc;margin-left:15px;">'.
					'<li>'.__('almost 8 years of working as a websites developer', 'kocuj-sitemap').'</li>'.
					'<li>'.__('experience of paid employment and self-employed', 'kocuj-sitemap').'</li>'.
					'<li>'.__('developing with using my custom solutions which accelerate the creation of website and is giving greater certainty to the reliability of project', 'kocuj-sitemap').'</li>'.
					'<li>'.__('experience in long-term support of websites, including the online shop and blog based on the WordPress, for the client in UK', 'kocuj-sitemap').'</li>'.
					'<li>'.__('experience in creating web services for large companies, such as: Polkomtel (Plus GSM), Grand Parade, Betfair, Ladbrokes', 'kocuj-sitemap').'</li>'.
					'</ul>');
			}
		}

		/**
		 * Initialize
		 *
		 * @access public
		 * @return void
		 */
		public function init() {
			// execute parent method
			parent::init();
			// add menu
			$this->getProjectObj()->getObj('backend_settings-menu')->addSettingsMenu(__('Sitemap', 'kocuj-sitemap'), 'manage_options', 'basic_settings', array($this, 'pluginSettings'), '', array(
				'icon'             => 'dashicons-search',
				'firstoptiontitle' => __('Basic settings', 'kocuj-sitemap'),
				'onpluginslist'    => true,
				'pluginslisttitle' => __('Settings', 'kocuj-sitemap'),
			));
			$this->getProjectObj()->getObj('backend_settings-help')->addHelpTopic('basic_settings', 'overview', __('Overview', 'kocuj-sitemap'), '<p>'.__('This is the place where you can set the basic options for the sitemap.', 'kocuj-sitemap').'</p>'.
				'<p>'.__('Options are divided into four tabs: `sitemap format`, `plugins for multilingualism`, `administration panel options` and `advanced options`. Each tab can be selected by clicking on it.', 'kocuj-sitemap').'</p>'.
				'<p>'.__('To save changed settings, click on the button `save settings`.', 'kocuj-sitemap').'</p>'
			);
			$this->getProjectObj()->getObj('backend_settings-help')->addHelpTopic('basic_settings', 'sitemap_format', __('`Sitemap format` tab', 'kocuj-sitemap'), '<p>'.__('There are the following options in this tab:', 'kocuj-sitemap').'</p>'.
				'<p><ul>'.
				'<li><em>`'.__('Display `powered by` information below sitemap', 'kocuj-sitemap').'`</em>: '.__('If this option is activated, there will be `powered by` information below the sitemap. It will also contain a link to the author website. If you find this plugin useful, please check this option - it is a very good advertisement for its author.', 'kocuj-sitemap').'</li>'.
				'<li><em>`'.__('Use HTML 5 tags', 'kocuj-sitemap').'`</em>: '.__('If this option is activated, the sitemap will display itself in HTML 5 navigation tag (`nav`) and the `powered by` information will display itself in HTML 5 footer tag (`footer`). For Internet Explorer browser which not supports HTML 5 (all versions below 9), these tags will be changed to `div`. All HTML 5 tags (or `div` tags when HTML 5 option is disabled or the Internet Explorer browser is in version below 9) will always have CSS class attached to itself - `tagnav` for `nav` tag and `tagfooter` for `footer` tag.', 'kocuj-sitemap').'</li>'.
				'<li><em>`'.__('Display link to main site', 'kocuj-sitemap').'`</em>: '.__('If this option is activated, there will be link to the main site as first entry in the sitemap.', 'kocuj-sitemap').'</li>'.
				'<li><em>`'.__('Divide display into sections', 'kocuj-sitemap').'`</em>: '.__('If this option is activated, the sitemap will be divided into sections.', 'kocuj-sitemap').'</li>'.
				'<li><em>`'.__('Header text level value for each section', 'kocuj-sitemap').'`</em>: '.__('You can specify the header text level (HTML tag `H*`) for each section of the sitemap displayed by the shortcode or PHP function. This option will not have any effect on sitemap displayed by widget. If you are not sure what is the meaning of this tag in HTML, leave this value unchanged, because its default value should be correct for many WordPress themes.', 'kocuj-sitemap').'</li>'.
				'<li><em>`'.__('Header text level value for each section in widget', 'kocuj-sitemap').'`</em>: '.__('You can specify the header text level (HTML tag `H*`) for each section of the sitemap displayed by the widget. This option will not have any effect on sitemap displayed by the shortcode or PHP function. If you are not sure what is the meaning of this tag in HTML, leave this value unchanged, because its default value should be correct for many WordPress themes.', 'kocuj-sitemap').'</li>'.
				'</ul></p>'
			);
			$this->getProjectObj()->getObj('backend_settings-help')->addHelpTopic('basic_settings', 'plugins_for_multilingualism', __('`Plugins for multilingualism` tab', 'kocuj-sitemap'), '<p>'.__('There are the following options in this tab:', 'kocuj-sitemap').'</p>'.
				'<p><ul>'.
				'<li><em>`'.__('Use plugin for multiple languages', 'kocuj-sitemap').'`</em>: '.__('You can select the plugin which you want to use for multi-lingual website. You must also installed and activated the selected plugin to working it correctly. If you do not want to have a multi-lingual website or if you want to have a plugin selected automatically, you can select the option `automatically select the plugin`. Remember, that automatically selecting the plugin may not always work correctly if you have installed and activated more than one plugin for multilingualism.', 'kocuj-sitemap').'</li>'.
				'</ul></p>'
			);
			$this->getProjectObj()->getObj('backend_settings-help')->addHelpTopic('basic_settings', 'administration_panel_options', __('`Administration panel options` tab', 'kocuj-sitemap'), '<p>'.__('There are the following options in this tab:', 'kocuj-sitemap').'</p>'.
				'<p><ul>'.
				'<li><em>`'.__('Enable sitemap button in visual and HTML editor', 'kocuj-sitemap').'`</em>: '.__('If this option is activated, there will be a button for adding a sitemap shortcode in visual and HTML editor in administration panel.', 'kocuj-sitemap').'</li>'.
				'</ul></p>'
			);
			$this->getProjectObj()->getObj('backend_settings-help')->addHelpTopic('basic_settings', 'advanced_options', __('`Advanced options` tab', 'kocuj-sitemap'), '<p>'.__('There are the following options in this tab:', 'kocuj-sitemap').'</p>'.
				'<p><ul>'.
				/* translators: %s: Root directory name for cache for this plugin (usually "/wp-content/cache/kocuj-sitemap/") */
				'<li><em>`'.__('Enable cache', 'kocuj-sitemap').'`</em>: '.sprintf(__('If this option is activated, the sitemap caching is enabled. Do not disable this option unless you know what you are doing! Cache is an important element of this plugin and disabling it may lower performance of your website. Sometimes disabling of the cache may be needed for some reason, but you should be very careful with this. Remember that cache can works correcly only if the cache directory ("%s") and all its subdirectories have permissions to write.', 'kocuj-sitemap'), \KocujSitemapPlugin\Classes\Cache::getInstance()->getCacheRootDirectory()).'</li>'.
				'<li><em>`'.__('Generate cache on the website instead of in the administration panel', 'kocuj-sitemap').'`</em>: '.__('If this option is activated, cache is generated on the website (frontend) instead of in the administration panel (backend). Remember that, after enabling this option, when you will display the sitemap for the first time, it will take more time due to generation of the cache.', 'kocuj-sitemap').'</li>'.
				'<li><em>`'.__('Enable additional compression of cache', 'kocuj-sitemap').'`</em>: '.__('If this option is activated, there will be additional compression of the cache file. It will make the cache file smaller, so this option is appropriate when usage of the disk space is important on your website. Remember that this option can lower the performance of your system, so use it only if you are sure that your website and server can provide enough resources to do so. This option works only if option `enable cache` is activated and Zlib extension for PHP installed.', 'kocuj-sitemap').'</li>'.
				'</ul></p>'
			);
			$this->getProjectObj()->getObj('backend_settings-menu')->addSettingsMenu(__('Order of elements', 'kocuj-sitemap'), 'manage_options', 'order', array($this, 'pluginOrder'), 'basic_settings');
			$this->getProjectObj()->getObj('backend_settings-help')->addHelpTopic('order', 'overview', __('Overview', 'kocuj-sitemap'), '<p>'.__('This is the place where you can set the order of sections inside the sitemap.', 'kocuj-sitemap').'</p>'.
				'<p>'.__('To save changed order, click on the button `save order of elements`.', 'kocuj-sitemap').'</p>'
			);
			$this->getProjectObj()->getObj('backend_settings-help')->addHelpTopic('order', 'usage', __('Usage', 'kocuj-sitemap'), '<p>'.__('You can change the order of any sitemap section by clicking on one of the arrow next to the entry to move it up or down.', 'kocuj-sitemap').'</p>');
			$types = \KocujSitemapPlugin\Classes\Sitemap::getInstance()->getElementsTypes();
			if (!empty($types)) {
				foreach ($types as $key => $val) {
					if ($val['object']->checkExists()) {
						$adminPageData = $val['objectadmin']->getAdminPageSettings();
						if (!empty($adminPageData)) {
							$pos = count($this->adminPagesData);
							$this->adminPagesData[$pos] = array(
								'type'  => $val,
								'title' => $adminPageData['title'],
								'id'    => $adminPageData['id'],
								'tabs'  => $adminPageData['tabs'],
							);
							if ($val['object']->checkSectionName()) {
								$this->adminPagesData[$pos]['tabs']['section_name'] = __('Section title', 'kocuj-sitemap');
							}
							$this->getProjectObj()->getObj('backend_settings-menu')->addSettingsMenu($adminPageData['title'], 'manage_options', $adminPageData['id'], array($this, 'adminPage'.$this->getRandom().'_'.$pos), 'basic_settings');
							if (!empty($adminPageData['help'])) {
								foreach ($adminPageData['help'] as $key2 => $val2) {
									$this->getProjectObj()->getObj('backend_settings-help')->addHelpTopic($adminPageData['id'], $key2, $val2['title'], '<p>'.$val2['content'].'</p>');
								}
							}
						}
					}
				}
			}
			$this->getProjectObj()->getObj('backend_settings-menu')->addSettingsMenu(__('Restore settings', 'kocuj-sitemap'), 'manage_options', 'restore', array($this, 'pluginRestore'), 'basic_settings');
			/* translators: %s: Name of this plugin ("Kocuj Sitemap") */
			$this->getProjectObj()->getObj('backend_settings-help')->addHelpTopic('restore', 'overview', __('Overview', 'kocuj-sitemap'), '<p>'.sprintf(__('Sometimes there is a need to return to the settings that were set up immediately after installing the %s plugin. These settings are known as `factory settings` or `default settings`. %s plugin gives you possibility to restore these settings.', 'kocuj-sitemap'), 'Kocuj Sitemap', 'Kocuj Sitemap').'</p>');
			$this->getProjectObj()->getObj('backend_settings-help')->addHelpTopic('restore', 'usage', __('Usage', 'kocuj-sitemap'), '<p>'.__('Click on the button `restore default settings`. When a window will appear asking you to confirm this action, click on the `OK` button. Note that this operation can not be undone.', 'kocuj-sitemap').'</p>');
			$this->getProjectObj()->getObj('backend_page-about')->setPageData(array(
				'imagefilename' => 'about2.jpg',
			));
			$this->getProjectObj()->getObj('backend_settings-menu')->addSettingsMenu(__('About', 'kocuj-sitemap'), 'manage_options', 'about', array($this->getProjectObj()->getObj('backend_page-about'), 'showPage'), 'basic_settings');
			// optionally add buttons to editor
			$value = $this->getProjectObj()->getObj('config')->getOption('ButtonEditor');
			if ($value) {
				// add quick tags
				$this->addQuickTag('kocujsitemap', __('sitemap', 'kocuj-sitemap'), '[KocujSitemap]', '', '', __('Add sitemap shortcode', 'kocuj-sitemap'));
				// add TinyMCE buttons
				$this->addMCEPluginJS('kocujsitemap-plugin.js', 'kocujsitemap-plugin-langs.php');
				$this->addMCEButton('kocujsitemap');
			}
			// check if cache directory is writable
			$value = $this->getProjectObj()->getObj('config')->getOption('Cache');
			if (($value) && (!\KocujSitemapPlugin\Classes\Cache::getInstance()->checkWritable())) {
				add_action('admin_head', array($this, 'actionCacheDirNotWritableError'));
			}
			// add input helper
			$this->addInputHelper('kocujsitemaporderofelements', 'inputHelperKocujSitemapOrderOfElements', NULL, false, true);
			// add information about version changes
			$information = '';
			/* translators: 1: name of this plugin ("Kocuj Sitemap"), 2: old version of this plugin, 3: plugin version after update */
			$information .= sprintf(__('Thank you for upgrading the %1$s plugin from version %2$s to version %3$s.', 'kocuj-sitemap'), 'Kocuj Sitemap', '%1$s', '%2$s');
			$information .= '<br /><br />';
			$information .= __('There are many differences between versions with major number 1 and 2. You can read below a few information about the most important changes.', 'kocuj-sitemap');
			$information .= '<br /><br /><br />';
			$information .= '<hr />';
			$information .= '<br />';
			$information .= \KocujInternalLib\V1a\Classes\Helper::getInstance()->getHTMLImage($this->getProjectObj()->getObj('urls')->getCustomImagesUrl().'/changes2-exclude.png', array(
				'style' => 'float:left;margin:0;padding:0;margin-right:5px;',
			)).'<h2 style="margin-top:8px;">'.__('Excluding selected posts, categories, pages, authors, tags or any other custom post types entries', 'kocuj-sitemap').'</h2>';
			$information .= '<br />';
			/* translators: %s: Shortcode name */
			$information .= __('You can exclude selected items from the sitemap. This feature is useful when you have articles that do not fit to the sitemap.', 'kocuj-sitemap').'<br /><br />'.sprintf(__('To exclude selected items simply use the shortcode %s with a parameter indicating the list of identifiers of elements to exclude. There are the following parameters:', 'kocuj-sitemap'), '<em>[KocujSitemap]</em>').'<ul style="list-style:disc;margin-left:15px;"><li><em>excludepost</em> - '.__('for excluding posts of any type (posts, pages, custom post types entries)', 'kocuj-sitemap').';</li><li><em>excludecategory</em> - '.__('for excluding post categories', 'kocuj-sitemap').';</li><li><em>excludeauthor</em> - '.__('for excluding authors', 'kocuj-sitemap').';</li><li><em>excludeterm</em> - '.__('for excluding post tags and custom taxonomies', 'kocuj-sitemap').'.</li></ul>'.__('For example:', 'kocuj-sitemap').'<br /><code>[KocujSitemap excludepost="5,8" excludeterm="12"]</code><br />'.__('excludes posts with identifiers 5 and 8, and post tags and custom taxonomies with identifier 12.', 'kocuj-sitemap').'<br /><br />'.__('This feature is also available for widget in its settings.', 'kocuj-sitemap');
			$information .= '<br /><br />';
			$information .= '<hr />';
			$information .= '<br />';
			$information .= \KocujInternalLib\V1a\Classes\Helper::getInstance()->getHTMLImage($this->getProjectObj()->getObj('urls')->getCustomImagesUrl().'/changes2-divide.png', array(
				'style' => 'float:left;margin:0;padding:0;margin-right:5px;',
			)).'<h2 style="margin-top:8px;">'.__('Dividing sitemap into sections', 'kocuj-sitemap').'</h2>';
			$information .= '<br />';
			$information .= __('You can divide the sitemap into sections. Each section is then described with its title, for example, section with posts will have the title of `posts` and section with tags - `tags`.', 'kocuj-sitemap').'<br /><br />'.__('If the section title does not have a translation, you can manually enter its title in your language. Use the tab `section title`, which contains fields for entering the section titles for the individual languages available in your WordPress installation.', 'kocuj-sitemap');
			$information .= '<br /><br />';
			$information .= '<hr />';
			$information .= '<br />';
			$information .= \KocujInternalLib\V1a\Classes\Helper::getInstance()->getHTMLImage($this->getProjectObj()->getObj('urls')->getCustomImagesUrl().'/changes2-entry.png', array(
				'style' => 'float:left;margin:0;padding:0;margin-right:5px;',
			)).'<h2 style="margin-top:8px;">'.__('Using all types of entries available in WordPress', 'kocuj-sitemap').'</h2>';
			$information .= '<br />';
			/* translators: %s: Name of this plugin ("Kocuj Sitemap") */
			$information .= sprintf(__('Currently %s plugin can use the following types of entries: menus, pages, posts, authors, tags and custom post types entries (with custom taxonomies).', 'kocuj-sitemap'), 'Kocuj Sitemap');
			$information .= '<br /><br />';
			$information .= '<hr />';
			$information .= '<br />';
			$information .= \KocujInternalLib\V1a\Classes\Helper::getInstance()->getHTMLImage($this->getProjectObj()->getObj('urls')->getCustomImagesUrl().'/changes2-control.png', array(
				'style' => 'float:left;margin:0;padding:0;margin-right:5px;',
			)).'<h2 style="margin-top:8px;">'.__('More control of what will be displayed', 'kocuj-sitemap').'</h2>';
			$information .= '<br />';
			/* translators: %s: Name of this plugin ("Kocuj Sitemap") */
			$information .= sprintf(__('%s plugin now provides many options to change the way of what will be displayed on the sitemap. There is a large set of options for almost every type of entry that may be displayed on the sitemap.', 'kocuj-sitemap'), 'Kocuj Sitemap');
			$information .= '<br /><br />';
			$information .= '<hr />';
			$information .= '<br />';
			$information .= \KocujInternalLib\V1a\Classes\Helper::getInstance()->getHTMLImage($this->getProjectObj()->getObj('urls')->getCustomImagesUrl().'/changes2-help.png', array(
				'style' => 'float:left;margin:0;padding:0;margin-right:5px;',
			)).'<h2 style="margin-top:8px;">'.__('Contextual help', 'kocuj-sitemap').'</h2>';
			$information .= '<br />';
			$information .= __('This plugin has a large amount of help texts. In administration panel just click on the `help` button at the top of the screen to get help text about current screen with plugin settings.', 'kocuj-sitemap');
			$information .= '<br /><br />';
			$information .= '<hr />';
			$information .= '<br />';
			$information .= \KocujInternalLib\V1a\Classes\Helper::getInstance()->getHTMLImage($this->getProjectObj()->getObj('urls')->getCustomImagesUrl().'/changes2-multisite.png', array(
				'style' => 'float:left;margin:0;padding:0;margin-right:5px;',
			)).'<h2 style="margin-top:8px;">'.__('Multisite support', 'kocuj-sitemap').'</h2>';
			$information .= '<br />';
			$information .= __('This plugin can work also in multisite environment.', 'kocuj-sitemap');
			$this->getProjectObj()->getObj('backend_version-check')->addUpdateInformation('1.', $information);
		}

		/**
		 * Call handler
		 *
		 * @access public
		 * @param string $name Method name
		 * @param array $argument Method arguments
		 * @return void
		 */
		public function __call($name, array $arguments) {
			// execute parent
			parent::__call($name, $arguments);
			// get method name
			$div = explode('_', $name);
			if (isset($div[2]) /* count($div) > 2 */ ) {
				$div0 = $div[0];
				$div1 = $div;
				unset($div1[0]);
				$divEnd = implode('_', $div1);
				$div = array(
					$div0,
					$divEnd,
				);
			}
			// methods for administration pages
			$divCount = count($div);
			if (($divCount === 2) && ($div[0] === 'adminPage'.$this->getRandom()) && (is_numeric($div[1]))) {
				if (isset($this->adminPagesData[$div[1]])) {
					$data = $this->adminPagesData[$div[1]];
					if (!empty($data['type']['objectadmin'])) {
						// get administration page data
						$page = $data['type']['objectadmin']->getAdminPageData();
						if ((!empty($page)) && (isset($page['pages'])) && (!empty($page['pages']))) {
							// add tab for section name
							if (!empty($data['tabs'])) {
								$pos = count($page['pages']);
								$page['pages'][$pos] = '';
								$options = $this->getProjectObj()->getObj('config')->getSitemapTypeLanguagesOptionsNames($data['type']['letter']);
								if (!empty($options)) {
									foreach ($options as $option) {
										$img = ($option['language'] === 'en_US') ?
											__('default section title', 'kocuj-sitemap').'<br />' :
											'';
										$flag = \KocujSitemapPlugin\Classes\MultipleLanguages::getInstance()->getLanguageFlag($option['language']);
										if (isset($flag[0]) /* strlen($flag) > 0 */ ) {
											$img .= \KocujInternalLib\V1a\Classes\Helper::getInstance()->getHTMLImage($flag, array(
												'style' => 'width:18px;height:12px;',
											)).'&nbsp;';
										}
										$img .= '('.$option['language'].')';
										$page['pages'][$pos] .= $this->getBlockHelper($option['option'], 'text', __('You can set the title of the section for the selected language. If you leave this field empty and your current language will be the same as the language for this field, the default section title will be displayed or standard section title in current language (or in English if no translation is available).', 'kocuj-sitemap'), array(), $img);
									}
								}
							}
							// register tabs
							$tabs = array();
							$containerId = 0;
							if (!empty($data['tabs'])) {
								$ret = $this->registerTabsContainer($data['id'], $containerId);
								foreach ($data['tabs'] as $key => $tab) {
									$pos = count($tabs);
									$tabs[$pos] = array(
										'id'       => 0,
										'idstring' => $key,
									);
									$ret = $this->registerTab($containerId, $tab, $tabs[$pos]['id']);
								}
							}
							// show form begin
							$this->showFormStart();
							// show form
							if (!empty($tabs)) {
								$this->showTabsContainerStart($containerId);
								$loopCount = count($tabs);
								for ($z=0; $z<$loopCount; $z++) {
									$this->showTabStart($tabs[$z]['id']);
									$this->showTableStart($tabs[$z]['idstring']);
									echo $page['pages'][$z];
									$this->showTableEnd();
									$this->showTabEnd();
								}
								$this->showTabsContainerEnd();
							} else {
								$this->showTableStart($tabs[0]['idstring']);
								echo $page['pages'][0];
								$this->showTableEnd();
							}
							// show buttons
							if (!empty($tabs)) {
								$this->showTableStart('submit');
								echo '<tr><td colspan="2" class="submit">';
								$this->inputHelper('save', 'submit', '', $page['submit']['title'], '', false, $page['submit']['info']);
								echo $page['buttons'];
								echo '</td></tr>';
								$this->showTableEnd();
							} else {
								$this->inputHelper('save', 'submit', '', $page['submit']['title'], '', false, $page['submit']['info']);
								echo $page['buttons'];
							}
							// show form end
							$this->showFormEnd();
						}
					}
				}
			}
			// methods for widgets settings
			if (($divCount === 2) && (($div[0] === 'adminWidgetsSettingsShow'.$this->getRandom()) || ($div[0] === 'adminWidgetsSettingsUpdate'.$this->getRandom())) && (is_numeric($div[1]))) {
				if ($div[0] === 'adminWidgetsSettingsShow'.$this->getRandom()) {
					call_user_func_array($this->adminWidgetsSettingsData[$div[1]][0], $arguments);
				} else {
					if ($div[0] === 'adminWidgetsSettingsUpdate'.$this->getRandom()) {
						call_user_func_array($this->adminWidgetsSettingsData[$div[1]][1], $arguments);
					}
				}
			}
		}

		/**
		 * Action - information about error with permissions to cache directory
		 *
		 * @access public
		 * @return void
		 */
		public function actionCacheDirNotWritableError() {
			// show error information
			if (current_user_can('manage_options')) {
				/* translators: 1: name of this plugin ("Kocuj Sitemap"), 2: root directory name with cache for this plugin (usually "/wp-content/cache/kocuj-sitemap/"), 3: subdirectory name with cache for this plugin (for example, "/wp-content/cache/kocuj-sitemap/1/") */
				$this->getProjectObj()->getObj('backend_message')->addMessageForAllPages('cache_error', '<strong>'.sprintf(__('Warning! Your cache directory for sitemap in %1$s plugin (placed in "%2$s") or its current subdirectory (placed in "%3$s") is not writable. Please, create this directory and change its permissions to allow the plugin to use cache for better performance.', 'kocuj-sitemap'), 'Kocuj Sitemap', \KocujSitemapPlugin\Classes\Cache::getInstance()->getCacheRootDirectory(), \KocujSitemapPlugin\Classes\Cache::getInstance()->getCacheDirectory()).'</strong>', \KocujInternalLib\V1a\Enums\MessageType::ERROR);
			} else {
				/* translators: 1: name of this plugin ("Kocuj Sitemap"), 2: root directory name with cache for this plugin (usually "/wp-content/cache/kocuj-sitemap/"), 3: subdirectory name with cache for this plugin (for example, "/wp-content/cache/kocuj-sitemap/1/") */
				$this->getProjectObj()->getObj('backend_message')->addMessageForAllPages('cache_error', '<strong>'.sprintf(__('Warning! Your cache directory for sitemap in %1$s plugin (placed in "%2$s") or its current subdirectory (placed in "%3$s") is not writable. Please, contact with user who is managing permissions to allow the plugin to use cache for better performance.', 'kocuj-sitemap'), 'Kocuj Sitemap', \KocujSitemapPlugin\Classes\Cache::getInstance()->getCacheRootDirectory(), \KocujSitemapPlugin\Classes\Cache::getInstance()->getCacheDirectory()).'</strong>', \KocujInternalLib\V1a\Enums\MessageType::ERROR);
			}
		}

		/**
		 * Add widgets settings
		 *
		 * @access public
		 * @param string $id Widget id
		 * @param string $title Widget title
		 * @param string $functionShow Callback method name for show meta box
		 * @param string $functionUpdate Callback method name for update meta box
		 * @return void
		 */
		public function addWidgetSettings($id, $title, $functionShow, $functionUpdate) {
			// add new widget settings
			$pos = count($this->adminWidgetsSettingsData);
			$this->adminWidgetsSettingsData[$pos] = array(
				$functionShow,
				$functionUpdate,
			);
			$this->addWidget($id, $title, 'adminWidgetsSettingsShow'.$this->getRandom().'_'.$pos, 'adminWidgetsSettingsUpdate'.$this->getRandom().'_'.$pos);
		}

		/**
		 * Save controller
		 *
		 * @access public
		 * @return string Output status
		 */
		public function controllerSave() {
			// execute parent method
			$errors = parent::controllerSave();
			// renew cache
			\KocujSitemapPlugin\Classes\Cache::getInstance()->createCache();
			// exit
			return $errors;
		}

		/**
		 * Reset controller
		 *
		 * @access public
		 * @return string Output status
		 */
		public function controllerReset() {
			// execute parent method
			$errors = parent::controllerReset();
			// renew cache
			\KocujSitemapPlugin\Classes\Cache::getInstance()->createCache();
			// exit
			return '';
		}

		/**
		 * Get input helper - order of elements
		 *
		 * @access public
		 * @param string $var Field name
		 * @param string $description Field description - default: empty
		 * @param string $value Field value - default: empty
		 * @param string $selected Selected field - default: empty
		 * @param bool $checked Checked field - default: false
		 * @param string $tip Event for mouseover displaying tooltip - default: empty
		 * @param bool $isArray Option is array - default: false
		 * @param string $eventOnChange Event onChange - default: empty
		 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
		 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
		 * @param string $forcedName Forced name if $isForcedName is true
		 * @return string Input helper
		 */
		public function inputHelperKocujSitemapOrderOfElements($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
			$isArray = false, $eventOnChange = '', $isWidget = false, $isForcedName = false, $forcedName = '') {
			// prepare id
			$idString = (strpos($var, '[]') !== false) ?
				'' :
				' id="'.$var.'"';
			// prepare additional styles
			$additional = ($isWidget) ?
				' style="display:inline;position:absolute;" class="widefat"' :
				' style="width:300px;display:inline;position:absolute;"';
			// set value text
			$types = \KocujSitemapPlugin\Classes\Sitemap::getInstance()->getElementsTypes(true);
			$valueOutput = (isset($types[$value])) ?
				$types[$value]['objectadmin']->getAdminOrderName() :
				$value;
			// prepare name
			$name = ($isForcedName) ?
				$forcedName :
				$var;
			// get input helper
			return '<input name="'.$name.'"'.$idString.' type="hidden" value="'.htmlspecialchars($value).'" style="width:1px;height:1px;margin:0; padding:0;position:absolute;" /><div'.$additional.'><strong>'.$valueOutput.'</strong></div>';
		}

		/**
		 * Plugin settings page
		 *
		 * @access public
		 * @return void
		 */
		public function pluginSettings() {
			// get multi-language plugins settings
			$multiLangOptions = array();
			$multiLangData = \KocujSitemapPlugin\Classes\MultipleLanguagesData::getInstance()->getData();
			if (!empty($multiLangData)) {
				foreach ($multiLangData as $key => $val) {
					$multiLangOptions[str_replace('\\', '____B_A_C_K_S_P_A_C_E____', $val['admin_id'])] = $val['admin_name'];
				}
				asort($multiLangOptions);
			}
			// register tabs
			$ret = $this->registerTabsContainer('kocujsitemap', $containerId);
			$ret = $this->registerTab($containerId, __('Sitemap format', 'kocuj-sitemap'), $tabSitemapFormatId);
			$ret = $this->registerTab($containerId, __('Plugins for multilingualism', 'kocuj-sitemap'), $tabMultiLangId);
			$ret = $this->registerTab($containerId, __('Administration panel options', 'kocuj-sitemap'), $tabAdminPanelId);
			$ret = $this->registerTab($containerId, __('Advanced options', 'kocuj-sitemap'), $tabAdvancedId);
			// show form
			?>
				<?php $this->showFormStart(); ?>
					<?php $this->showTabsContainerStart($containerId); ?>
						<?php $this->showTabStart($tabSitemapFormatId); ?>
							<?php $this->showTableStart('sitemap_format'); ?>
								<?php
									$this->blockHelper('PoweredBy', 'checkbox', __('If this option is activated, there will be `powered by` information below the sitemap. It will also contain a link to the author website. If you find this plugin useful, please check this option - it is a very good advertisement for its author.', 'kocuj-sitemap'));
									$this->blockHelper('UseHTML5', 'checkbox', __('If this option is activated, the sitemap will display itself in HTML 5 navigation tag (`nav`) and the `powered by` information will display itself in HTML 5 footer tag (`footer`). All HTML 5 tags (or `div` tags when HTML 5 option is disabled or the Internet Explorer browser is in version below 9) will always have CSS class attached to itself - `tagnav` for `nav` tag and `tagfooter` for `footer` tag.', 'kocuj-sitemap'));
									$this->blockHelper('LinkToMainSite', 'checkbox', __('If this option is activated, there will be link to the main site as first entry in the sitemap.', 'kocuj-sitemap'));
									$this->blockHelper('DisplaySections', 'checkbox', __('If this option is activated, the sitemap will be divided into sections.', 'kocuj-sitemap'));
									$this->blockHelper('HLevelMain', 'text', __('You can specify the header text level (HTML tag `H*`) for each section of the sitemap displayed by the shortcode or PHP function.', 'kocuj-sitemap'));
									$this->blockHelper('HLevelWidget', 'text', __('You can specify the header text level (HTML tag `H*`) for each section of the sitemap displayed by the widget.', 'kocuj-sitemap'));
								?>
							<?php $this->showTableEnd(); ?>
						<?php $this->showTabEnd(); ?>
						<?php $this->showTabStart($tabMultiLangId); ?>
							<?php $this->showTableStart('multilang'); ?>
								<?php
									$multiLangOptions = array_merge(array(
										'0' => __('Automatically select the plugin', 'kocuj-sitemap'),
									), $multiLangOptions);
									$pluginName = \KocujSitemapPlugin\Classes\MultipleLanguages::getInstance()->getSelectedPluginName();
									if (!isset($pluginName[0]) /* strlen($pluginName) === 0 */ ) {
										$pluginName = __('none', 'kocuj-sitemap');
									}
									$this->blockHelper('Multilang', 'select', __('You can select the plugin which you want to use for multi-lingual website. You must also installed and activated the selected plugin to working it correctly.', 'kocuj-sitemap'), $multiLangOptions, __('(only installed and activated plugins are on this list)', 'kocuj-sitemap'));
									$this->blockHelper('', 'statictext', __('Currently used plugin:', 'kocuj-sitemap').' '.$pluginName);
								?>
							<?php $this->showTableEnd(); ?>
						<?php $this->showTabEnd(); ?>
						<?php $this->showTabStart($tabAdminPanelId); ?>
							<?php $this->showTableStart('admin_options'); ?>
								<?php
									$this->blockHelper('ButtonEditor', 'checkbox', __('If this option is activated, there will be a button for adding a sitemap shortcode in visual and HTML editor in administration panel.', 'kocuj-sitemap'));
								?>
							<?php $this->showTableEnd(); ?>
						<?php $this->showTabEnd(); ?>
						<?php $this->showTabStart($tabAdvancedId); ?>
							<?php $this->showTableStart('advanced'); ?>
								<?php
									$this->blockHelper('Cache', 'checkbox', __('If this option is activated, the sitemap caching is enabled. Do not disable this option unless you know what you are doing! Cache is an important element of this plugin and disabling it may lower performance of your website. Sometimes disabling of the cache may be needed for some reason, but you should be very careful with this.', 'kocuj-sitemap'), array(), __('(disabling it is not recommended)', 'kocuj-sitemap'));
									$this->blockHelper('CacheFrontend', 'checkbox', __('If this option is activated, cache is generated on the website (frontend) instead of in the administration panel (backend). If this option is activated, when you will display the sitemap for the first time, it will take more time due to generation of the cache.', 'kocuj-sitemap'));
									$this->blockHelper('CacheAdditionalCompress', 'checkbox', __('If this option is activated, there will be additional compression of the cache file. This option works only if option `enable cache` is activated and Zlib extension for PHP installed.', 'kocuj-sitemap'), array(), __('(works only if option `enable cache` is activated and Zlib extension for PHP installed)', 'kocuj-sitemap'));
								?>
							<?php $this->showTableEnd(); ?>
						<?php $this->showTabEnd(); ?>
					<?php $this->showTabsContainerEnd(); ?>
					<?php $this->showTableStart('submit'); ?>
						<tr>
							<td colspan="2" class="submit">
								<?php $this->inputHelper('save', 'submit', '', __('Save settings', 'kocuj-sitemap'), '', false, __('Save current settings', 'kocuj-sitemap')); ?>
							</td>
						</tr>
					<?php $this->showTableEnd(); ?>
				<?php $this->showFormEnd(); ?>
			<?php
		}

		/**
		 * Plugin order of elements page
		 *
		 * @access public
		 * @return void
		 */
		public function pluginOrder() {
			// show form
			?>
				<?php $this->showFormStart(); ?>
				<div class="bordertitle">
					<?php $this->showTableStart('order'); ?>
						<?php
							$this->blockHelper('', 'statictext', __('You can change the order of any sitemap section by clicking on one of the arrow next to the entry to move it up or down.', 'kocuj-sitemap'));
							$this->blockHelper('OrderList', 'kocujsitemaporderofelements', __('Menus which should be included in sitemap; it works only if option `display menus` is checked', 'kocuj-sitemap'));
						?>
					<?php $this->showTableEnd(); ?>
					<?php $this->inputHelper('save', 'submit', '', __('Save order of elements', 'kocuj-sitemap'), '', false, __('Save current order of elements', 'kocuj-sitemap')); ?>
				</div>
				<?php $this->showFormEnd(); ?>
			<?php
		}

		/**
		 * Plugin restore settings page
		 *
		 * @access public
		 * @return void
		 */
		public function pluginRestore() {
			// show form
			?>
				<?php $this->showFormStart(); ?>
				<div class="bordertitle">
					<?php $this->showTableStart('restore'); ?>
						<?php
							$this->blockHelper('', 'statictext', '
								'.__('You can restore default settings for this plugin by clicking the button below.', 'kocuj-sitemap').'
							');
							$this->blockHelper('', 'submittext', '
								'.$this->getInputHelper('reset', 'reset', '', __('Restore default settings', 'kocuj-sitemap'), '', false, __('Reset settings to defaults', 'kocuj-sitemap')).'
							');
						?>
					<?php $this->showTableEnd(); ?>
				</div>
				<?php $this->showFormEnd(); ?>
			<?php
		}
	}
} else {
	/**
	 * Plugin admin class
	 *
	 * @access public
	 */
	class Admin extends \KocujInternalLib\V1a\Classes\Project\Admin {}
}
