<?php

/**
 * urls.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujInternalLib\V1a\Classes\Project;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * URLS-s class
 *
 * @access public
 */
class Urls extends \KocujInternalLib\V1a\Classes\ProjectObject {
	/**
	 * Project URL
	 *
	 * @access private
	 * @var string
	 */
	private $projectUrl = '';

	/**
	 * Current license directory
	 *
	 * @access private
	 * @var string
	 */
	private $currentLicenseDir = '';

	/**
	 * License URL
	 *
	 * @access private
	 * @var string
	 */
	private $licenseUrl = '';

	/**
	 * Current custom images directory
	 *
	 * @access private
	 * @var string
	 */
	private $currentCustomImagesDir = '';

	/**
	 * Custom images URL
	 *
	 * @access private
	 * @var string
	 */
	private $customImagesUrl = '';

	/**
	 * Current TinyMCE plugins directory
	 *
	 * @access private
	 * @var string
	 */
	private $currentTinyMcePluginsDir = '';

	/**
	 * TinyMCE plugins URL
	 *
	 * @access private
	 * @var string
	 */
	private $tinyMcePluginsUrl = '';

	/**
	 * Current TinyMCE plugins languages directory
	 *
	 * @access private
	 * @var string
	 */
	private $currentTinyMcePluginsLanguagesDir = '';

	/**
	 * TinyMCE plugins languages URL
	 *
	 * @access private
	 * @var string
	 */
	private $tinyMcePluginsLanguagesUrl = '';

	/**
	 * Constructor
	 *
	 * @access public
	 * @param object $projectObj \KocujInternalLib\V1a\Classes\Project object for current project
	 * @return void
	 */
	public function __construct($projectObj) {
		// execute parent constructor
		parent::__construct($projectObj);
		// set URL-s
		$this->projectUrl = ($this->getProjectObj()->getSettingType() === \KocujInternalLib\V1a\Enums\ProjectType::PLUGIN) ?
			plugin_dir_url($this->getProjectObj()->getSettingMainFilename()) :
			get_template_directory_uri();
		$this->currentLicenseDir = $this->getProjectObj()->getObj('dirs')->getLicenseDir();
		$this->licenseUrl = $this->projectUrl;
		$this->currentCustomImagesDir = $this->getProjectObj()->getObj('dirs')->getCustomImagesDir();
		$this->customImagesUrl = $this->projectUrl.'/images';
		$this->currentTinyMcePluginsDir = $this->getProjectObj()->getObj('dirs')->getTinyMcePluginsDir();
		$this->tinyMcePluginsUrl = $this->projectUrl.'/js/tiny-mce';
		$this->currentTinyMcePluginsLanguagesDir = $this->getProjectObj()->getObj('dirs')->getTinyMcePluginsLanguagesDir();
		$this->tinyMcePluginsLanguagesUrl = $this->projectUrl.'/php/tiny-mce';
	}

	/**
	 * Get project URL
	 *
	 * @access public
	 * @return string Project URL
	 */
	public function getProjectUrl() {
		// get project URL
		return $this->projectUrl;
	}

	/**
	 * Get license URL
	 *
	 * @access public
	 * @return string License URL
	 */
	public function getLicenseUrl() {
		// change license URL if license directory has been changed
		$licenseDir = $this->getProjectObj()->getObj('dirs')->getLicenseDir();
		if ($this->currentLicenseDir !== $licenseDir) {
			$this->licenseUrl = $this->projectUrl.'/'.substr($licenseDir, strlen($this->getProjectObj()->getObj('dirs')->getProjectDir()));
			$this->currentLicenseDir = $this->licenseUrl;
		}
		// get license URL
		return $this->licenseUrl;
	}

	/**
	 * Get custom images URL
	 *
	 * @access public
	 * @return string Custom images URL
	 */
	public function getCustomImagesUrl() {
		// change custom images URL if custom images directory has been changed
		$customImagesDir = $this->getProjectObj()->getObj('dirs')->getCustomImagesDir();
		if ($this->currentCustomImagesDir !== $customImagesDir) {
			$this->customImagesUrl = $this->projectUrl.'/'.substr($customImagesDir, strlen($this->getProjectObj()->getObj('dirs')->getProjectDir()));
			$this->currentCustomImagesDir = $this->customImagesUrl;
		}
		// get custom images URL
		return $this->customImagesUrl;
	}

	/**
	 * Get TinyMCE plugins URL
	 *
	 * @access public
	 * @return string TinyMCE plugins URL
	 */
	public function getTinyMcePluginsUrl() {
		// change TinyMCE plugins URL if TinyMCE plugins directory has been changed
		$tinyMcePluginsDir = $this->getProjectObj()->getObj('dirs')->getTinyMcePluginsDir();
		if ($this->currentTinyMcePluginsDir !== $tinyMcePluginsDir) {
			$this->tinyMcePluginsUrl = $this->projectUrl.'/'.substr($tinyMcePluginsDir, strlen($this->getProjectObj()->getObj('dirs')->getProjectDir()));
			$this->currentTinyMcePluginsDir = $this->tinyMcePluginsUrl;
		}
		// get TinyMCE plugins URL
		return $this->tinyMcePluginsUrl;
	}

	/**
	 * Get TinyMCE plugins languages URL
	 *
	 * @access public
	 * @return string TinyMCE plugins languages URL
	 */
	public function getTinyMcePluginsLanguagesUrl() {
		// change TinyMCE plugins languages URL if TinyMCE plugins languages directory has been changed
		$tinyMcePluginsLanguagesDir = $this->getProjectObj()->getObj('dirs')->getTinyMcePluginsLanguagesDir();
		if ($this->currentTinyMcePluginsLanguagesDir !== $tinyMcePluginsLanguagesDir) {
			$this->tinyMcePluginsLanguagesUrl = $this->projectUrl.'/'.substr($tinyMcePluginsLanguagesDir, strlen($this->getProjectObj()->getObj('dirs')->getProjectDir()));
			$this->currentTinyMcePluginsLanguagesDir = $this->tinyMcePluginsLanguagesUrl;
		}
		// get TinyMCE plugins languages URL
		return $this->tinyMcePluginsLanguagesUrl;
	}
}
