<?php

/**
 * settings-menu.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujInternalLib\V1a\Classes\Project\Backend;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Set settings menu class
 *
 * @access public
 */
class SettingsMenu extends \KocujInternalLib\V1a\Classes\ProjectObject {
	/**
	 * Settings menus list
	 *
	 * @access private
	 * @var array
	 */
	private $settingsMenu = array();

	/**
	 * Settings menus hooks suffixes
	 *
	 * @access private
	 * @var array
	 */
	private $settingsMenuHookSuffix = array();

	/**
	 * Settings pages screen names
	 *
	 * @access private
	 * @var array
	 */
	private $pageScreenNames = array();

	/**
	 * Constructor
	 *
	 * @access public
	 * @param object $projectObj \KocujInternalLib\V1a\Classes\Project object for current project
	 * @return void
	 */
	public function __construct($projectObj) {
		// execute parent constructor
		parent::__construct($projectObj);
		// check if it is backend
		if ((!is_admin()) && (!is_network_admin())) {
			return;
		}
		// add actions
		add_action('admin_menu', array($this, 'actionAdminMenu'));
		// add filters
		if ($this->getProjectObj()->getSettingType() === \KocujInternalLib\V1a\Enums\ProjectType::PLUGIN) {
			add_filter('plugin_action_links', array($this, 'filterPluginActionLinks'), 10, 2);
		}
	}

	/**
	 * Call handler for non-existing methods
	 *
	 * @access public
	 * @param string $name Method name
	 * @param array $argument Method arguments
	 * @return array|bool|float|int|string|void Value returned by called method
	 */
	public function __call($name, array $arguments) {
		// get type and identifier based on method name
		$div = explode('_', $name);
		if (count($div) > 1) {
			$type = $div[0];
			unset($div[0]);
			$id = implode('_', $div);
			// check type and id
			if (($type === 'settingsMenu') && (isset($this->settingsMenu[$id]))) {
				// check user permission
				if (!current_user_can($this->settingsMenu[$id]['capability'])) {
					wp_die($this->getProjectObj()->getStringsObj('backend_settings-menu')->getString('CALL_NO_PERMISSION'));
				}
				// show header
				$this->getProjectObj()->getObj('project-helper')->doAction('before_wrap_div');
				?>
					<div<?php echo $this->getProjectObj()->getObj('project-helper')->applyFiltersForHTMLStyleAndClass('wrap_div', '', array(
						'defaultclass' => 'wrap',
					)); ?>>
					<?php
						echo '<h1>'.$this->getProjectObj()->getSettingTitle().'</h1>';
						$title = (isset($this->settingsMenu[$id]['fulltitle'])) ?
							$this->settingsMenu[$id]['fulltitle'] :
							$this->settingsMenu[$id]['title'];
						echo '<h2>'.$title.'</h2>';
					?>
				<?php
				// execute method
				call_user_func($this->settingsMenu[$id]['function']);
				// show footer
				?>
					</div>
				<?php
				$this->getProjectObj()->getObj('project-helper')->doAction('after_wrap_div');
			}
		}
	}

	/**
	 * Get settings menus data
	 *
	 * @access public
	 * @return array Settings menus data; each settings menu data have the following fields: "capability" (string type; capability required for access to this menu), "function" (array or string type; callback function or method name), "pagescreenname" (string type; screen name for this page), "title" (string type; menu title); there are also the following fields which should exists or not: "firstoptiontitle" (title for first option if current option menu is without parent; only for menu - not submenu), "icon" (string type; icon name for settings option; only for menu - not submenu), "onpluginslist" (bool type; there will be link to this settings on this plugin information in plugins list with title from this option or from "pluginslisttitle" attribute if exists), "pluginslisttitle" (string type)
	 */
	public function getSettingsMenus() {
		// exit
		return $this->settingsMenu;
	}

	/**
	 * Get the selected settings menu data
	 *
	 * @access public
	 * @param string $id Settings menu identifier
	 * @return array|bool Select settings menu data or false if not exists; settings menu data have the following fields: "capability" (string type; capability required for access to this menu), "function" (array or string type; callback function or method name), "pagescreenname" (string type; screen name for this page), "title" (string type; menu title); there are also the following fields which should exists or not: "firstoptiontitle" (title for first option if current option menu is without parent; only for menu - not submenu), "icon" (string type; icon name for settings option; only for menu - not submenu), "onpluginslist" (bool type; there will be link to this settings on this plugin information in plugins list with title from this option or from "pluginslisttitle" attribute if exists), "pluginslisttitle" (string type)
	 */
	public function getSettingsMenu($id) {
		// exit
		return (isset($this->settingsMenu[$id])) ?
			$this->settingsMenu[$id] :
			false;
	}

	/**
	 * Get hook suffix for the selected settings menu
	 *
	 * @access public
	 * @param string $id Settings menu identifier
	 * @return bool|string Hook suffix for the selected settings menu or false if not exists
	 */
	public function getSettingsMenuHookSuffix($id) {
		// exit
		return (isset($this->settingsMenuHookSuffix[$id])) ?
			$this->settingsMenuHookSuffix[$id] :
			false;
	}

	/**
	 * Check if current page is for settings for current project
	 *
	 * @access public
	 * @return bool Current page is for settings for current project (true) or not (false)
	 */
	public function checkCurrentPageIsSettingsForProject() {
		// check page
		$screen = (function_exists('get_current_screen')) ?
			get_current_screen() :
			NULL;
		// exit
		return ((!empty($screen)) && (!empty($this->pageScreenNames)) && (in_array($screen->id, $this->pageScreenNames)));
	}

	/**
	 * Get current settings menu page
	 *
	 * @access public
	 * @return bool|string Current settings menu page or false if it is not page for current project
	 */
	public function getCurrentSettingsMenu() {
		// get current settings menu
		if (!empty($this->settingsMenu)) {
			// get current screen id
			$screen = get_current_screen();
			if (!empty($screen)) {
				// get current settings menu
				foreach ($this->settingsMenu as $key => $val) {
					if ($val['pagescreenname'] === $screen->id) {
						return $key;
					}
				}
			}
		}
		// exit
		return false;
	}

	/**
	 * Add settings page screen name
	 *
	 * @access private
	 * @param string $pageScreenName Page screen name to add
	 * @return void
	 */
	private function addSettingsPageScreenName($pageScreenName) {
		// add settings page screen name
		if (!in_array($pageScreenName, $this->pageScreenNames)) {
			$this->pageScreenNames[] = $pageScreenName;
		}
	}

	/**
	 * Add settings menu, submenu or submenu to built-in menu
	 *
	 * @access private
	 * @param string $title Menu title
	 * @param string $capability Capability required for access to this menu
	 * @param string $id Menu id
	 * @param array|string $function Callback function or method name; can be global function or method from any class
	 * @param bool $parentIsBuiltInMenu Parent is built-in menu (true) or is settings menu for this project if exists (false)
	 * @param string $parentId Parent menu id (if $parentIsBuiltInMenu=false) or parent menu type (if $parentIsBuiltInMenu=true); for parent menu type it must be one of the following constants: \KocujInternalLib\V1a\Enums\SettingsMenuParentType::DASHBOARD, \KocujInternalLib\V1a\Enums\SettingsMenuParentType::POSTS, \KocujInternalLib\V1a\Enums\SettingsMenuParentType::\KocujInternalLib\V1a\Enums\SettingsMenuParentType::MEDIA, \KocujInternalLib\V1a\Enums\SettingsMenuParentType::LINKS, \KocujInternalLib\V1a\Enums\SettingsMenuParentType::PAGES, \KocujInternalLib\V1a\Enums\SettingsMenuParentType::COMMENTS, \KocujInternalLib\V1a\Enums\SettingsMenuParentType::THEMES, \KocujInternalLib\V1a\Enums\SettingsMenuParentType::PLUGINS, \KocujInternalLib\V1a\Enums\SettingsMenuParentType::USERS, \KocujInternalLib\V1a\Enums\SettingsMenuParentType::TOOLS or \KocujInternalLib\V1a\Enums\SettingsMenuParentType::OPTIONS - default: \KocujInternalLib\V1a\Enums\SettingsMenuParentType::DASHBOARD
	 * @param array $attr Additional attributes; there are available the following attributes: "firstoptiontitle" (title for first option if current option menu is without parent; only for menu - not submenu), "fulltitle" (string type; full title of page), "icon" (string type; icon name for settings option; only for menu - not submenu), "onpluginslist" (bool type; there will be link to this settings on this plugin information in plugins list with title from this option or from "pluginslisttitle" attribute if exists), "pluginslisttitle" (string type)
	 * @return void
	 */
	private function addSettingsMenuOrSubmenu($title, $capability, $id, $function, $parentIsBuiltInMenu, $parentId, array $attr) {
		// check if settings can be add
		if (($this->getProjectObj()->getSettingType() !== \KocujInternalLib\V1a\Enums\ProjectType::PLUGIN) && ((($parentIsBuiltInMenu) && ($parentId !== \KocujInternalLib\V1a\Enums\SettingsMenuParentType::THEMES)) || (!$parentIsBuiltInMenu))) {
			return;
		}
		// set page screen name
		if (!$parentIsBuiltInMenu) {
			$parentTitle = ((isset($parentId[0]) /* strlen($parentId) > 0 */ ) && (isset($this->settingsMenu[$parentId]))) ?
				$this->settingsMenu[$parentId]['title'] :
				'';
			$pageScreenName = (((!isset($parentId[0]) /* strlen($parentId) === 0 */ ) || (!isset($parentTitle[0]) /* strlen($parentTitle) === 0 */ )) ?
				'toplevel' :
				sanitize_title($parentTitle)
			);
		} else {
			switch ($parentId) {
				case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::DASHBOARD:
					$pageScreenName = 'dashboard';
					break;
				case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::POSTS:
					$pageScreenName = 'posts';
					break;
				case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::MEDIA:
					$pageScreenName = 'media';
					break;
				case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::LINKS:
					$pageScreenName = 'admin';
					break;
				case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::PAGES:
					$pageScreenName = 'pages';
					break;
				case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::COMMENTS:
					$pageScreenName = 'comments';
					break;
				case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::THEMES:
					$pageScreenName = 'appearance';
					break;
				case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::PLUGINS:
					$pageScreenName = 'plugins';
					break;
				case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::USERS:
					$pageScreenName = 'users';
					break;
				case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::TOOLS:
					$pageScreenName = 'tools';
					break;
				case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::OPTIONS:
					$pageScreenName = 'settings';
					break;
				default:
					$pageScreenName = '';
			}
		}
		$pageScreenName .= '_page_'.$this->getProjectObj()->getSettingInternalName().'_'.$id;
		// add settings menu
		$this->settingsMenu[$id] = array(
			'title'          => $title,
			'capability'     => $capability,
			'function'       => $function,
			'pagescreenname' => $pageScreenName,
		);
		if (isset($attr['fulltitle'])) {
			$this->settingsMenu[$id]['fulltitle'] = $attr['fulltitle'];
		}
		if (!$parentIsBuiltInMenu) {
			if (isset($parentId[0]) /* strlen($parentId) > 0 */ ) {
				$this->settingsMenu[$id]['parentid'] = $parentId;
			}
		} else {
			$this->settingsMenu[$id]['parentid'] = $parentId;
			$this->settingsMenu[$id]['parentisbuiltinmenu'] = true;
		}
		if ((!$parentIsBuiltInMenu) && (isset($attr['icon']))) {
			$this->settingsMenu[$id]['icon'] = $attr['icon'];
		}
		if (isset($attr['onpluginslist'])) {
			$this->settingsMenu[$id]['onpluginslist'] = $attr['onpluginslist'];
			if (isset($attr['pluginslisttitle'])) {
				$this->settingsMenu[$id]['pluginslisttitle'] = $attr['pluginslisttitle'];
			}
		}
		if (!$parentIsBuiltInMenu) {
			if ((!isset($parentId[0]) /* strlen($parentId) === 0 */ ) && (isset($attr['firstoptiontitle']))) {
				$this->settingsMenu[$id]['firstoptiontitle'] = $attr['firstoptiontitle'];
			}
		}
		// add page internal name
		$this->addSettingsPageScreenName($pageScreenName);
	}

	/**
	 * Add settings menu
	 *
	 * @access public
	 * @param string $title Menu title
	 * @param string $capability Capability required for access to this menu
	 * @param string $id Menu id
	 * @param array|string $function Callback function or method name; can be global function or method from any class
	 * @param string $parentId Parent menu id - default: empty
	 * @param array $attr Additional attributes; there are available the following attributes: "firstoptiontitle" (title for first option if current option menu is without parent; only for menu - not submenu), "fulltitle" (string type; full title of page), "icon" (string type; icon name for settings option; only for menu - not submenu), "onpluginslist" (bool type; there will be link to this settings on this plugin information in plugins list with title from this option or from "pluginslisttitle" attribute if exists), "pluginslisttitle" (string type) - default: empty
	 * @return void
	 */
	public function addSettingsMenu($title, $capability, $id, $function, $parentId = '', array $attr = array()) {
		// add settings menu
		$this->addSettingsMenuOrSubmenu($title, $capability, $id, $function, false, $parentId, $attr);
	}

	/**
	 * Add settings menu to built in menu
	 *
	 * @access public
	 * @param string $title Menu title
	 * @param string $capability Capability required for access to this menu
	 * @param string $id Menu id
	 * @param array|string $function Callback function or method name; can be global function or method from any class
	 * @param int $parentType Parent menu type; must be one of the following constants: \KocujInternalLib\V1a\Enums\SettingsMenuParentType::DASHBOARD, \KocujInternalLib\V1a\Enums\SettingsMenuParentType::POSTS, \KocujInternalLib\V1a\Enums\SettingsMenuParentType::\KocujInternalLib\V1a\Enums\SettingsMenuParentType::MEDIA, \KocujInternalLib\V1a\Enums\SettingsMenuParentType::LINKS, \KocujInternalLib\V1a\Enums\SettingsMenuParentType::PAGES, \KocujInternalLib\V1a\Enums\SettingsMenuParentType::COMMENTS, \KocujInternalLib\V1a\Enums\SettingsMenuParentType::THEMES, \KocujInternalLib\V1a\Enums\SettingsMenuParentType::PLUGINS, \KocujInternalLib\V1a\Enums\SettingsMenuParentType::USERS, \KocujInternalLib\V1a\Enums\SettingsMenuParentType::TOOLS or \KocujInternalLib\V1a\Enums\SettingsMenuParentType::OPTIONS - default: \KocujInternalLib\V1a\Enums\SettingsMenuParentType::DASHBOARD
	 * @param array $attr Additional attributes; there are available the following attributes: "firstoptiontitle" (title for first option if current option menu is without parent; only for menu - not submenu), "fulltitle" (string type; full title of page), "icon" (string type; icon name for settings option; only for menu - not submenu), "onpluginslist" (bool type; there will be link to this settings on this plugin information in plugins list with title from this option or from "pluginslisttitle" attribute if exists), "pluginslisttitle" (string type) - default: empty
	 * @return void
	 */
	public function addSettingsMenuBuiltIn($title, $capability, $id, $function, $parentType = \KocujInternalLib\V1a\Enums\SettingsMenuParentType::DASHBOARD, array $attr = array()) {
		// add settings menu
		$this->addSettingsMenuOrSubmenu($title, $capability, $id, $function, true, $parentType, $attr);
	}

	/**
	 * Action for adding menu
	 *
	 * @access public
	 * @return void
	 */
	public function actionAdminMenu() {
		// add settings menu
		if (!empty($this->settingsMenu)) {
			foreach ($this->settingsMenu as $key => $menu) {
				if (!isset($menu['parentid'])) {
					$hookSuffix = add_menu_page($menu['title'], $menu['title'], $menu['capability'], $this->getProjectObj()->getSettingInternalName().'_'.$key, array($this, 'settingsMenu_'.$key), (isset($menu['icon'])) ?
						$menu['icon'] :
						''
					);
					if ($hookSuffix !== false) {
						$this->settingsMenuHookSuffix[$key] = $hookSuffix;
					}
					if (isset($menu['firstoptiontitle'])) {
						$hookSuffix = add_submenu_page($this->getProjectObj()->getSettingInternalName().'_'.$key, $menu['firstoptiontitle'], $menu['firstoptiontitle'], $menu['capability'], $this->getProjectObj()->getSettingInternalName().'_'.$key, array($this, 'settingsMenu_'.$key));
						if ($hookSuffix !== false) {
							$this->settingsMenuHookSuffix[$key] = $hookSuffix;
						}
					}
				}
			}
			foreach ($this->settingsMenu as $key => $menu) {
				if (isset($menu['parentid'])) {
					$id = $this->getProjectObj()->getSettingInternalName().'_'.$key;
					$callback = array($this, 'settingsMenu_'.$key);
					if ((!isset($menu['parentisbuiltinmenu'])) || ((isset($menu['parentisbuiltinmenu'])) && (!$menu['parentisbuiltinmenu']))) {
						$hookSuffix = add_submenu_page($this->getProjectObj()->getSettingInternalName().'_'.$menu['parentid'], $menu['title'], $menu['title'], $menu['capability'], $id, $callback);
					} else {
						switch ($menu['parentid']) {
							case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::DASHBOARD:
								$hookSuffix = add_dashboard_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::POSTS:
								$hookSuffix = add_posts_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::MEDIA:
								$hookSuffix = add_media_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::LINKS:
								$hookSuffix = add_links_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::PAGES:
								$hookSuffix = add_pages_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::COMMENTS:
								$hookSuffix = add_comments_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::THEMES:
								$hookSuffix = add_theme_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::PLUGINS:
								$hookSuffix = add_plugins_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::USERS:
								$hookSuffix = add_users_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::TOOLS:
								$hookSuffix = add_management_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V1a\Enums\SettingsMenuParentType::OPTIONS:
								$hookSuffix = add_options_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
						}
					}
					if ($hookSuffix !== false) {
						$this->settingsMenuHookSuffix[$key] = $hookSuffix;
					}
				}
			}
		}
	}

	/**
	 * Filter for actions links on plugins list
	 *
	 * @access public
	 * @param string $links Actions links list
	 * @param string $filename Plugin filename
	 * @return string Actions links list
	 */
	public function filterPluginActionLinks($links, $filename) {
		// show settings on plugin list
		if ((plugin_basename($this->getProjectObj()->getSettingMainFilename()) === $filename) && (!empty($this->settingsMenu))) {
			foreach (array_reverse($this->settingsMenu) as $id => $settingsMenu) {
				if ((isset($settingsMenu['onpluginslist'])) && ($settingsMenu['onpluginslist'])) {
					array_unshift($links, \KocujInternalLib\V1a\Classes\Helper::getInstance()->getLinkAnchor(get_admin_url().'?page='.$this->getProjectObj()->getSettingInternalName().'_'.$id, (((isset($settingsMenu['pluginslisttitle']) && (isset($settingsMenu['pluginslisttitle'][0]) /* strlen($settingsMenu['pluginslisttitle']) > 0 */ ))) ?
							$settingsMenu['pluginslisttitle'] :
							$settingsMenu['title']
						)));
				}
			}
		}
		// exit
		return $links;
	}
}
