<?php

/**
 * base.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html
 * @copyright Copyright (c) 2013-2016 Dominik Kocuj
 * @package kocuj_sitemap
 */

// set namespace
namespace KocujSitemapPlugin\Classes;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Base class
 *
 * @access public
 */
class Base {
	/**
	 * Singleton instance
	 *
	 * @access private
	 * @var object
	 */
	private static $instance = NULL;

	/**
	 * Plugin major version
	 *
	 * @access private
	 * @var int
	 */
	private $versionMajor = 2;

	/**
	 * Plugin minor version
	 *
	 * @access private
	 * @var int
	 */
	private $versionMinor = 0;

	/**
	 * Plugin revision version
	 *
	 * @access private
	 * @var int
	 */
	private $versionRevision = 6;

	/**
	 * Plugin URL
	 *
	 * @access private
	 * @var string
	 */
	private $pluginUrl = '';

	/**
	 * Plugin directory
	 *
	 * @access private
	 * @var string
	 */
	private $pluginDir = '';

	/**
	 * Plugin base name
	 *
	 * @access private
	 * @var string
	 */
	private $pluginBaseName = '';

	/**
	 * Plugin filename
	 *
	 * @access private
	 * @var string
	 */
	private $pluginFilename = '';

	/**
	 * Kocuj Internal Lib object instance
	 *
	 * @access private
	 * @var object
	 */
	private $kocujInternalLibObj = NULL;

	/**
	 * Kocuj API Add Thanks Client object instance
	 *
	 * @access private
	 * @var object
	 */
	private $kocujApiAddThanksClientObj = NULL;

	/**
	 * Constructor
	 *
	 * @access private
	 * @param string $pluginFile Plugin file path
	 * @return void
	 */
	private function __construct($pluginFile) {
		// get plugin URL
		$this->pluginUrl = plugins_url('', $pluginFile);
		// get plugin directory
		$this->pluginDir = dirname($pluginFile);
		// get plugin base name
		$this->pluginBaseName = plugin_basename($pluginFile);
		// get plugin filename
		$this->pluginFilename = $pluginFile;
		// add initialize
		add_action('plugins_loaded', array($this, 'init'), 10);
	}

	/**
	 * Disable cloning of object
	 *
	 * @access private
	 * @return void
	 */
	private function __clone() {
	}

	/**
	 * Get singleton instance
	 *
	 * @access public
	 * @param string $pluginFile Plugin file path; first execution of "getInstance()" method should set this to plugin file path, but all others executions can ommit this parameter - default: empty
	 * @return object Singleton instance
	 */
	public static function getInstance($pluginFile = '') {
		// optionally create new instance
		if (!self::$instance) {
			self::$instance = new \KocujSitemapPlugin\Classes\Base($pluginFile);
		}
		// exit
		return self::$instance;
	}

	/**
	 * Initialize
	 *
	 * @access public
	 * @return void
	 */
	public function init() {
		// load translations
		$domain = 'kocuj-sitemap';
		$locale = apply_filters('plugin_locale', get_locale(), $domain);
		load_textdomain($domain, WP_LANG_DIR.'/plugins/'.$domain.'-'.$locale.'.mo');
		load_plugin_textdomain($domain, false, dirname($this->pluginBaseName).'/languages');
		// initialize Kocuj Internal Lib
		include $this->pluginDir.'/lib/kocuj-internal-lib/kocuj-internal-lib.php';
		$this->kocujInternalLibObj = new \KocujInternalLib\V1a\Classes\Project(array(
			'type'          => \KocujInternalLib\V1a\Enums\ProjectType::PLUGIN,
			'mainfilename'  => $this->pluginFilename,
			'title'         => __('Sitemap', 'kocuj-sitemap'),
			'licensename'   => 'GPL',
			'name'          => 'Kocuj Sitemap plugin',
			'version'       => $this->getVersion(),
			'url'           => 'http://kocujsitemap.wpplugin.kocuj.pl/',
			'titleoriginal' => 'Kocuj Sitemap',
			'author'        => array(
				'name'  => 'Dominik Kocuj',
				'email' => 'dominik@kocuj.pl',
				'url'   => 'http://kocuj.pl/',
			),
			'support'       => array(
				'email' => 'dominik@kocuj.pl',
				'url'   => 'https://wordpress.org/support/plugin/kocuj-sitemap/',
			),
			'newschannels'  => array(
				'facebook' => 'http://www.facebook.com/kocujsitemap/',
				'rss'      => 'http://kocujsitemap.wpplugin.kocuj.pl/en/feed/',
			),
			'tellothers'    => array(
				'facebook'    => true,
				'twitter'     => __('Very good #wordpress plugin for creating a #sitemap with #multilingualism and #cache.', 'kocuj-sitemap'),
			),
		), array(
			'admin'                  => '\\KocujSitemapPlugin\\Classes\\InternalLibAdminStrings',
			'config'                 => '\\KocujSitemapPlugin\\Classes\\InternalLibConfigStrings',
			'backend_license'        => '\\KocujSitemapPlugin\\Classes\\InternalLibBackendLicenseStrings',
			'backend_page-about'     => '\\KocujSitemapPlugin\\Classes\\InternalLibBackendPageAboutStrings',
			'backend_review-message' => '\\KocujSitemapPlugin\\Classes\\InternalLibBackendReviewMessageStrings',
			'backend_settings-menu'  => '\\KocujSitemapPlugin\\Classes\\InternalLibBackendSettingsMenuStrings',
			'backend_version-check'  => '\\KocujSitemapPlugin\\Classes\\InternalLibBackendVersionCheckStrings',
		), array(
			'admin'  => '\\KocujSitemapPlugin\\Classes\\Admin',
			'config' => '\\KocujSitemapPlugin\\Classes\\Config',
		));
		// initialize Kocuj API Add Thanks Client
		include $this->pluginDir.'/lib/kocuj-api-add-thanks-client/kocuj-api-add-thanks-client.php';
		if ((is_admin()) || (is_network_admin())) {
			$this->kocujApiAddThanksClientObj = new \KocujApiAddThanksClient\V1a\Classes\Project($this->kocujInternalLibObj, 'api.kocuj.pl/', 'kocujsitemapplugin', 'K0cmDi_98XifA', array(
				'kocuj.pl',
				'kocujsitemap.wpplugin.kocuj.pl',
			), array(
				'backend_add-thanks'                => '\\KocujSitemapPlugin\\Classes\\ApiAddThanksClientBackendAddThanksStrings',
				'backend_add-thanks-ajax'           => '\\KocujSitemapPlugin\\Classes\\ApiAddThanksClientBackendAddThanksAjaxStrings',
				'backend_page-about-add-thanks'     => '\\KocujSitemapPlugin\\Classes\\ApiAddThanksClientBackendPageAboutAddThanksStrings',
				'backend_review-message-add-thanks' => '\\KocujSitemapPlugin\\Classes\\ApiAddThanksClientBackendReviewMessageAddThanksStrings',
			));
		}
		// make some backend things
		if ((is_admin()) || (is_network_admin())) {
			// set plugin version
			$this->kocujInternalLibObj->getObj('backend_version-check')->setCurrentVersion($this->getVersion());
			// update version data
			$value = get_option('kocujsitemap_version'); // for compatibility with 1.x.x
			if ($value !== false) { // for compatibility with 1.x.x
				\KocujInternalLib\V1a\Classes\Helper::getInstance()->addOrUpdateOptionForNetworkOrSite($this->kocujInternalLibObj->getObj('backend_version-check')->getVersionOptionName(), $value); // for compatibility with 1.x.x
				delete_option('kocujsitemap_version'); // for compatibility with 1.x.x
			} // for compatibility with 1.x.x
			// set plugin update callback
			$this->kocujInternalLibObj->getObj('backend_version-check')->setUpdateCallback(array($this, 'updatePlugin'));
			// set review message
			$this->kocujInternalLibObj->getObj('backend_review-message')->setDays(30);
			$this->kocujInternalLibObj->getObj('backend_review-message')->setVoteUrl('http://wordpress.org/support/view/plugin-reviews/kocuj-sitemap');
		}
		// initialize some classes
		\KocujSitemapPlugin\Classes\Sitemap::getInstance();
		\KocujSitemapPlugin\Classes\Widget::getInstance();
	}

	/**
	 * Get major version
	 *
	 * @access public
	 * @return int Major version
	 */
	public function getVersionMajor() {
		// get major version
		return $this->versionMajor;
	}

	/**
	 * Get minor version
	 *
	 * @access public
	 * @return int Minor version
	 */
	public function getVersionMinor() {
		// get minor version
		return $this->versionMinor;
	}

	/**
	 * Get revision version
	 *
	 * @access public
	 * @return int Revision version
	 */
	public function getVersionRevision() {
		// get revision version
		return $this->versionRevision;
	}

	/**
	 * Get version
	 *
	 * @access public
	 * @return string Version
	 */
	public function getVersion() {
		// get version
		return $this->versionMajor.'.'.$this->versionMinor.'.'.$this->versionRevision;
	}

	/**
	 * Get plugin URL
	 *
	 * @access public
	 * @return string Plugin URL
	 */
	public function getPluginUrl() {
		// get plugin URL
		return $this->pluginUrl;
	}

	/**
	 * Get plugin directory
	 *
	 * @access public
	 * @return string Plugin directory
	 */
	public function getPluginDir() {
		// get plugin directory
		return $this->pluginDir;
	}

	/**
	 * Get plugin base name
	 *
	 * @access public
	 * @return string Plugin base name
	 */
	public function getPluginBaseName() {
		// get plugin base name
		return $this->pluginBaseName;
	}

	/**
	 * Get plugin filename
	 *
	 * @access public
	 * @return string Plugin filename
	 */
	public function getPluginFilename() {
		// get plugin filename
		return $this->pluginFilename;
	}

	/**
	 * Get Kocuj Internal Lib object
	 *
	 * @access public
	 * @return object Kocuj Internal Lib object
	 */
	public function getKocujInternalLibObj() {
		// get Kocuj Internal Lib object
		return $this->kocujInternalLibObj;
	}

	/**
	 * Get Kocuj API Add Thanks Client object
	 *
	 * @access public
	 * @return object Kocuj API Add Thanks Client object
	 */
	public function getKocujApiAddThanksClientObj() {
		// get Kocuj API Add Thanks Client object
		return $this->kocujApiAddThanksClientObj;
	}

	/**
	 * Plugin activation
	 *
	 * @access public
	 * @return void
	 */
	public function activate() {
		// initialize
		$this->init();
		// add activation initialization
		add_action('init', array($this, 'initActivate'));
	}

	/**
	 * Plugin activation initialization
	 *
	 * @access public
	 * @return void
	 */
	public function initActivate() {
		// create cache
		\KocujSitemapPlugin\Classes\Cache::getInstance()->createCache();
	}

	/**
	 * Plugin deactivation
	 *
	 * @access public
	 * @return void
	 */
	public function deactivate() {
	}

	/**
	 * Plugin update - call after plugin update and plugin activation
	 *
	 * @access public
	 * @param string $fromVersion Old plugin version
	 * @param string $toVersion New plugin version
	 * @return void
	 */
	public function updatePlugin($fromVersion, $toVersion) {
		// update from 1.x.x
		if (version_compare($fromVersion, '2.0.0', '<')) { // for compatibility with 1.x.x
			// remove cron schedule
			wp_clear_scheduled_hook('kocujsitemapcron'); // for compatibility with 1.x.x
			// remove database values for old plugin versions
			delete_option('kocujsitemap_2_0_0_get_ready'); // for compatibility with 1.x.x (from transition version 1.3.2 and newer 1.3.3)
			delete_option('kocujsitemap_plugin_data_version'); // for compatibility with 1.x.x
			delete_option('kocujsitemap_thanks_date'); // for compatibility with 1.x.x
			// update sending website address ("thanks") status
			$value = get_option('kocujsitemap_thanks'); // for compatibility with 1.x.x
			if ($value !== false) { // for compatibility with 1.x.x
				\KocujInternalLib\V1a\Classes\Helper::getInstance()->addOrUpdateOptionForNetworkOrSite($this->kocujApiAddThanksClientObj->getObj('backend_add-thanks')->getThanksAddedOptionName(), '1'); // for compatibility with 1.x.x
				delete_option('kocujsitemap_thanks');  // for compatibility with 1.x.x
			} // for compatibility with 1.x.x
			// update old options
			$exist = $this->kocujInternalLibObj->getObj('config')->checkOptionExists('Order'); // for compatibility with 1.x.x
			if ($exist) { // for compatibility with 1.x.x
				$this->kocujInternalLibObj->getObj('config')->addOption('Order', '', 'string', '', false, array(), false, 0, false); // for compatibility with 1.x.x
				$value = $this->kocujInternalLibObj->getObj('config')->getOption('Order'); // for compatibility with 1.x.x
				if (isset($value[0]) /* strlen($value) > 0 */ ) { // for compatibility with 1.x.x
					$def = str_split($value); // for compatibility with 1.x.x
					if (!in_array('M', $def)) { // for compatibility with 1.x.x
						array_unshift($def, 'M'); // for compatibility with 1.x.x
					} // for compatibility with 1.x.x
					$def[] = 'C'; // for compatibility with 1.x.x
					$def[] = 'A'; // for compatibility with 1.x.x
					$def[] = 'T'; // for compatibility with 1.x.x
					$this->kocujInternalLibObj->getObj('config')->removeOption('Order'); // for compatibility with 1.x.x
					$this->kocujInternalLibObj->getObj('config')->deleteOption('Order'); // for compatibility with 1.x.x
					$output = ''; // for compatibility with 1.x.x
					$this->kocujInternalLibObj->getObj('config')->updateOption('OrderList', $def, $output); // for compatibility with 1.x.x
				} // for compatibility with 1.x.x
			} // for compatibility with 1.x.x
			// disable displaying sections if it is an update from 1.x.x
			$output = ''; // for compatibility with 1.x.x
			$this->kocujInternalLibObj->getObj('config')->updateOption('DisplaySections', '0', $output); // for compatibility with 1.x.x
		} // for compatibility with 1.x.x
		// purge cache
		\KocujSitemapPlugin\Classes\Cache::getInstance()->purgeCache();
	}

	/**
	 * Get prefix for deprecated information; for compatibility with 1.x.x
	 *
	 * @access private
	 * @return string Prefix for deprecated information
	 */
	private function getOldInfoPrefix() {
		// exit
		/* translators: %s: Name of this plugin ("Kocuj Sitemap") */
		return sprintf(__('%s plugin', 'kocuj-sitemap'), 'Kocuj Sitemap'); // for compatibility with 1.x.x
	}
	
	/**
	 * Information about deprecated filter; for compatibility with 1.x.x
	 *
	 * @access public
	 * @param string $filterOld Old filter name
	 * @param string $filterNew New filter name which replace old filter - default: empty
	 * @param string $version Version from which filter has been deprecated - default: 2.0.0
	 * @return void
	 */
	public function filterOldInfo($filterOld, $filterNew = '', $version = '2.0.0') { // for compatibility with 1.x.x
		// check filter
		if (has_filter($filterOld)) { // for compatibility with 1.x.x
			// show information
			$newInfo = ''; // for compatibility with 1.x.x
			if (isset($filterNew[0]) /* strlen($filterNew) > 0 */ ) { // for compatibility with 1.x.x
				/* translators: %s: Filter name */
				$newInfo = sprintf(__('the "%s" filter', 'kocuj-sitemap'), $filterNew); // for compatibility with 1.x.x
			} // for compatibility with 1.x.x
			/* translators: %s: Filter name */
			_deprecated_function($this->getOldInfoPrefix().': '.sprintf(__('The "%s" filter', 'kocuj-sitemap'), $filterOld), $version, $newInfo); // for compatibility with 1.x.x
		} // for compatibility with 1.x.x
	} // for compatibility with 1.x.x
}
