<?php

/**
 * multiple-languages.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2013-2016 Dominik Kocuj
 * @package kocuj_sitemap
 */

// set namespace
namespace KocujSitemapPlugin\Classes;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Multiple languages plugin class
 *
 * @access public
 */
class MultipleLanguages {
	/**
	 * Singleton instance
	 *
	 * @access private
	 * @var object
	 */
	private static $instance = NULL;

	/**
	 * Languages cache
	 *
	 * @access private
	 * @var array
	 */
	private $languagesCache = array();

	/**
	 * Selected class
	 *
	 * @access private
	 * @var string
	 */
	private $selectedClass = '';

	/**
	 * Class has been already selected (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $isSelectedClass = false;

	/**
	 * Constructor
	 *
	 * @access private
	 * @return void
	 */
	private function __construct() {
	}

	/**
	 * Disable cloning of object
	 *
	 * @access private
	 * @return void
	 */
	private function __clone() {
	}

	/**
	 * Get singleton instance
	 *
	 * @access public
	 * @return object Singleton instance
	 */
	public static function getInstance() {
		// optionally create new instance
		if (!self::$instance) {
			self::$instance = new self();
		}
		// exit
		return self::$instance;
	}

	/**
	 * Get selectedy class name
	 *
	 * @access private
	 * @return string Selected class name
	 */
	private function getSelectedClass() {
		// check if class has not been already selected
		if ($this->isSelectedClass) {
			return $this->selectedClass;
		}
		// get data
		$data = MultipleLanguagesData::getInstance()->getData();
		// select translation plugin
		$value = str_replace('/', '\\', Base::getInstance()->getKocujInternalLibObj()->getObj('config')->getOption('Multilang'));
		if ((isset($value[0]) /* strlen($value) > 0 */ ) && ($value !== '0')) {
			if (isset($data[$value])) {
				$this->selectedClass = $value;
			}
		} else {
			// search for the lowest priority
			if (!empty($data)) {
				$prior = -1;
				$pos = '';
				foreach ($data as $key => $val) {
					if (($prior === -1) || ($val['prior'] < $prior)) {
						$prior = $val['prior'];
						$pos = $key;
					}
				}
				$this->selectedClass = $pos;
			}
		}
		// set that class has been already selected
		$this->isSelectedClass = true;
		// exit
		return $this->selectedClass;
	}

	/**
	 * Process callback function for the selected translation plugin
	 *
	 * @access private
	 * @param string $callback Callback function
	 * @param array $parameters Callback parameters
	 * @return any Returned callback data
	 */
	private function processCallback($callback, array $parameters = array()) {
		// get data
		$data = MultipleLanguagesData::getInstance()->getData();
		// execute callback
		$selectedClass = $this->getSelectedClass();
		if ((isset($selectedClass[0]) /* strlen($selectedClass) > 0 */ ) && (method_exists($data[$selectedClass]['instance'], $callback))) {
			if (method_exists($data[$selectedClass]['instance'], $callback)) {
				return call_user_func_array(array($data[$selectedClass]['instance'], $callback), $parameters);
			}
		}
		// optionally return first parameter
		if (isset($parameters[0])) {
			return $parameters[0];
		}
		// exit
		return NULL;
	}

	/**
	 * Get selected translation plugin name
	 *
	 * @access public
	 * @return string Selected translation plugin name or empty string if no plugin has been selected
	 */
	public function getSelectedPluginName() {
		// get data
		$data = MultipleLanguagesData::getInstance()->getData();
		// get selected class
		$selectedClass = $this->getSelectedClass();
		if ((isset($selectedClass[0]) /* strlen($selectedClass) > 0 */ ) && (isset($data[$selectedClass]['admin_name']))) {
			return $data[$selectedClass]['admin_name'];
		}
		// exit
		return '';
	}

	/**
	 * Get all languages list
	 *
	 * @access public
	 * @return array Languages list
	 */
	public function getLanguages() {
		// check if it is not in cache
		if (!empty($this->languagesCache)) {
			return $this->languagesCache;
		}
		// initialize
		$languages = array();
		// get languages
		$selectedClass = $this->getSelectedClass();
		if (isset($selectedClass[0]) /* strlen($selectedClass) > 0 */ ) {
			$data = MultipleLanguagesData::getInstance()->getData();
			$languages = array_keys($data[$selectedClass]['instance']->getLanguages());
		}
		// optionally set current language
		if (empty($languages)) {
			$languages[] = get_locale();
		}
		// set languages cache
		$this->languagesCache = $languages;
		// exit
		return $languages;
	}

	/**
	 * Get path to language flag
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return string Path to language flag
	 */
	public function getLanguageFlag($locale) {
		// get languages
		$languages = $this->getLanguages();
		// get flag
		$selectedClass = $this->getSelectedClass();
		if ((isset($selectedClass[0]) /* strlen($selectedClass) > 0 */ ) && (in_array($locale, $languages))) {
			$data = MultipleLanguagesData::getInstance()->getData();
			return $data[$selectedClass]['instance']->getLanguageFlag($locale);
		}
		// exit
		return '';
	}

	/**
	 * Something to do before get blog name
	 *
	 * @access public
	 * @return void
	 */
	public function beforeGetBlogName() {
		// something to do before get blog name
		$this->processCallback('beforeGetBlogName');
	}

	/**
	 * Something to do after get blog name
	 *
	 * @access public
	 * @return void
	 */
	public function afterGetBlogName() {
		// something to do after get blog name
		$this->processCallback('afterGetBlogName');
	}

	/**
	 * Get translated blog name
	 *
	 * @access public
	 * @return string Translated blog name
	 */
	public function getTranslatedBlogName() {
		// get original blog name
		$name = get_bloginfo('name');
		$origName = $name;
		// get translated blog name
		$name = $this->processCallback('getTranslatedBlogName', array(
			$origName,
		));
		if (!isset($name[0]) /* strlen($name) === 0 */ ) {
			$name = $origName;
		}
		// exit
		return $name;
	}

	/**
	 * Something to do before get home URL
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetHomeURL($locale) {
		// something to do before get home URL
		$this->processCallback('beforeGetHomeURL', array(
			$locale,
		));
	}

	/**
	 * Something to do after get home URL
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetHomeURL($locale) {
		// something to do after get home URL
		$this->processCallback('afterGetHomeURL', array(
			$locale,
		));
	}

	/**
	 * Get translated home URL
	 *
	 * @access public
	 * @param string $origURL Original home URL
	 * @param string $locale Language locale
	 * @return string Translated home URL
	 */
	public function getTranslatedHomeURL($origURL, $locale) {
		// get original home URL
		$url = $origURL;
		// get translated home URL
		$url = $this->processCallback('getTranslatedHomeURL', array(
			$origURL,
			$locale,
		));
		if (!isset($url[0]) /* strlen($url) === 0 */ ) {
			$url = $origURL;
		}
		// exit
		return $url;
	}

	/**
	 * Something to do before get pages
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetPages($locale) {
		// something to do before get pages
		$this->processCallback('beforeGetPages', array(
			$locale,
		));
	}

	/**
	 * Something to do after get pages
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetPages($locale) {
		// something to do after get pages
		$this->processCallback('afterGetPages', array(
			$locale,
		));
	}

	/**
	 * Something to do before get page item
	 *
	 * @access public
	 * @param int $pageId Page id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetPageItem($pageId, $locale) {
		// something to do before get page item
		$this->processCallback('beforeGetPageItem', array(
			$pageId,
			$locale,
		));
	}

	/**
	 * Something to do after get page item
	 *
	 * @access public
	 * @param int $pageId Page id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetPageItem($pageId, $locale) {
		// something to do after get page item
		$this->processCallback('afterGetPageItem', array(
			$pageId,
			$locale,
		));
	}

	/**
	 * Get translated page URL
	 *
	 * @access public
	 * @param string $origURL Original page URL
	 * @param int $pageId Page id
	 * @param string $locale Language locale
	 * @return string Translated page URL
	 */
	public function getTranslatedPageURL($origURL, $pageId, $locale) {
		// get original page URL
		$url = $origURL;
		// get translated page URL
		$url = $this->processCallback('getTranslatedPageURL', array(
			$origURL,
			$pageId,
			$locale,
		));
		if (!isset($url[0]) /* strlen($url) === 0 */ ) {
			$url = $origURL;
		}
		// exit
		return $url;
	}

	/**
	 * Get translated page title
	 *
	 * @access public
	 * @param string $origTitle Original page title
	 * @param int $pageId Page id
	 * @param string $locale Language locale
	 * @return string Translated page title
	 */
	public function getTranslatedPageTitle($origTitle, $pageId, $locale) {
		// get original page title
		$title = $origTitle;
		// get translated page URL
		$title = $this->processCallback('getTranslatedPageTitle', array(
			$origTitle,
			$pageId,
			$locale,
		));
		if (!isset($title[0]) /* strlen($title) === 0 */ ) {
			$title = $origTitle;
		}
		// exit
		return $title;
	}

	/**
	 * Something to do before get posts
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetPosts($locale) {
		// something to do before get posts
		$this->processCallback('beforeGetPosts', array(
			$locale,
		));
	}

	/**
	 * Something to do after get posts
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetPosts($locale) {
		// something to do after get posts
		$this->processCallback('afterGetPosts', array(
			$locale,
		));
	}

	/**
	 * Something to do before get post item
	 *
	 * @access public
	 * @param int $postId Post id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetPostItem($postId, $locale) {
		// something to do before get post item
		$this->processCallback('beforeGetPostItem', array(
			$postId,
			$locale,
		));
	}

	/**
	 * Something to do after get post item
	 *
	 * @access public
	 * @param int $postId Post id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetPostItem($postId, $locale) {
		// something to do after get post item
		$this->processCallback('afterGetPostItem', array(
			$postId,
			$locale,
		));
	}

	/**
	 * Get translated post URL
	 *
	 * @access public
	 * @param string $origURL Original post URL
	 * @param int $postId Post id
	 * @param string $locale Language locale
	 * @return string Translated post URL
	 */
	public function getTranslatedPostURL($origURL, $postId, $locale) {
		// get original post URL
		$url = $origURL;
		// get translated post URL
		$url = $this->processCallback('getTranslatedPostURL', array(
			$origURL,
			$postId,
			$locale,
		));
		if (!isset($url[0]) /* strlen($url) === 0 */ ) {
			$url = $origURL;
		}
		// exit
		return $url;
	}

	/**
	 * Get translated post title
	 *
	 * @access public
	 * @param string $origTitle Original post title
	 * @param int $postId Post id
	 * @param string $locale Language locale
	 * @return string Translated post title
	 */
	public function getTranslatedPostTitle($origTitle, $postId, $locale) {
		// get original post title
		$title = $origTitle;
		// get translated post URL
		$title = $this->processCallback('getTranslatedPostTitle', array(
			$origTitle,
			$postId,
			$locale,
		));
		if (!isset($title[0]) /* strlen($title) === 0 */ ) {
			$title = $origTitle;
		}
		// exit
		return $title;
	}

	/**
	 * Something to do before get categories
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetCategories($locale) {
		// something to do before get categories
		$this->processCallback('beforeGetCategories', array(
			$locale,
		));
	}

	/**
	 * Something to do after get categories
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetCategories($locale) {
		// something to do after get categories
		$this->processCallback('afterGetCategories', array(
			$locale,
		));
	}

	/**
	 * Something to do before get category item
	 *
	 * @access public
	 * @param int $categoryId Category id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetCategoryItem($categoryId, $locale) {
		// something to do before get category item
		$this->processCallback('beforeGetCategoryItem', array(
			$categoryId,
			$locale,
		));
	}

	/**
	 * Something to do after get category item
	 *
	 * @access public
	 * @param int $categoryId Category id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetCategoryItem($categoryId, $locale) {
		// something to do after get category item
		$this->processCallback('afterGetCategoryItem', array(
			$categoryId,
			$locale,
		));
	}

	/**
	 * Get translated category URL
	 *
	 * @access public
	 * @param string $origURL Original category URL
	 * @param int $categoryId Category id
	 * @param string $locale Language locale
	 * @return string Translated category URL
	 */
	public function getTranslatedCategoryURL($origURL, $categoryId, $locale) {
		// get original category URL
		$url = $origURL;
		// get translated category URL
		$url = $this->processCallback('getTranslatedCategoryURL', array(
			$origURL,
			$categoryId,
			$locale,
		));
		if (!isset($url[0]) /* strlen($url) === 0 */ ) {
			$url = $origURL;
		}
		// exit
		return $url;
	}

	/**
	 * Get translated category title
	 *
	 * @access public
	 * @param string $origTitle Original category title
	 * @param int $categoryId Category id
	 * @param string $locale Language locale
	 * @return string Translated category title
	 */
	public function getTranslatedCategoryTitle($origTitle, $categoryId, $locale) {
		// get original category title
		$title = $origTitle;
		// get translated category URL
		$title = $this->processCallback('getTranslatedCategoryTitle', array(
			$origTitle,
			$categoryId,
			$locale,
		));
		if (!isset($title[0]) /* strlen($title) === 0 */ ) {
			$title = $origTitle;
		}
		// exit
		return $title;
	}

	/**
	 * Something to do before get taxonomies
	 *
	 * @access public
	 * @param string $taxonomy Taxonomy type
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetTaxonomies($taxonomy, $locale) {
		// something to do before get taxonomies
		$this->processCallback('beforeGetTaxonomies', array(
			$taxonomy,
			$locale,
		));
	}

	/**
	 * Something to do after get taxonomies
	 *
	 * @access public
	 * @param string $taxonomy Taxonomy type
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetTaxonomies($taxonomy, $locale) {
		// something to do after get taxonomies
		$this->processCallback('afterGetTaxonomies', array(
			$taxonomy,
			$locale,
		));
	}

	/**
	 * Something to do before get taxonomy item
	 *
	 * @access public
	 * @param int $termId Term id
	 * @param string $taxonomy Taxonomy type
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetTaxonomyItem($termId, $taxonomy, $locale) {
		// something to do before get taxonomy item
		$this->processCallback('beforeGetTaxonomyItem', array(
			$termId,
			$taxonomy,
			$locale,
		));
	}

	/**
	 * Something to do after get taxonomy item
	 *
	 * @access public
	 * @param int $termId Term id
	 * @param string $taxonomy Taxonomy type
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetTaxonomyItem($termId, $taxonomy, $locale) {
		// something to do after get taxonomy item
		$this->processCallback('afterGetTaxonomyItem', array(
			$termId,
			$taxonomy,
			$locale,
		));
	}

	/**
	 * Get translated taxonomy URL
	 *
	 * @access public
	 * @param string $origURL Original taxonomy URL
	 * @param int $termId Term id
	 * @param string $taxonomy Taxonomy type
	 * @param string $locale Language locale
	 * @return string Translated taxonomy URL
	 */
	public function getTranslatedTaxonomyURL($origURL, $termId, $taxonomy, $locale) {
		// get original taxonomy URL
		$url = $origURL;
		// get translated taxonomy URL
		$url = $this->processCallback('getTranslatedTaxonomyURL', array(
			$origURL,
			$termId,
			$taxonomy,
			$locale,
		));
		if (!isset($url[0]) /* strlen($url) === 0 */ ) {
			$url = $origURL;
		}
		// exit
		return $url;
	}

	/**
	 * Get translated taxonomy title
	 *
	 * @access public
	 * @param string $origTitle Original taxonomy title
	 * @param int $termId Term id
	 * @param string $taxonomy Taxonomy type
	 * @param string $locale Language locale
	 * @return string Translated taxonomy title
	 */
	public function getTranslatedTaxonomyTitle($origTitle, $termId, $taxonomy, $locale) {
		// get original taxonomy title
		$title = $origTitle;
		// get translated taxonomy URL
		$title = $this->processCallback('getTranslatedTaxonomyTitle', array(
			$origTitle,
			$termId,
			$taxonomy,
			$locale,
		));
		if (!isset($title[0]) /* strlen($title) === 0 */ ) {
			$title = $origTitle;
		}
		// exit
		return $title;
	}

	/**
	 * Something to do before get menu
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetMenu($locale) {
		// something to do before get menu
		$this->processCallback('beforeGetMenu', array(
			$locale,
		));
	}

	/**
	 * Something to do after get menu
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetMenu($locale) {
		// something to do after get menu
		$this->processCallback('afterGetMenu', array(
			$locale,
		));
	}

	/**
	 * Something to do before get menu item
	 *
	 * @access public
	 * @param int $itemId Menu item id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetMenuItem($itemId, $locale) {
		// something to do before get menu item
		$this->processCallback('beforeGetMenuItem', array(
			$itemId,
			$locale,
		));
	}

	/**
	 * Something to do after get menu item
	 *
	 * @access public
	 * @param int $itemId Menu item id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetMenuItem($itemId, $locale) {
		// something to do after get menu item
		$this->processCallback('afterGetMenuItem', array(
			$itemId,
			$locale,
		));
	}

	/**
	 * Get translated menu URL
	 *
	 * @access public
	 * @param string $origURL Original menu URL
	 * @param int $menuId Menu id
	 * @param string $locale Language locale
	 * @return string Translated menu URL
	 */
	public function getTranslatedMenuURL($origURL, $menuId, $locale) {
		// get original menu URL
		$url = $origURL;
		// get translated menu URL
		$url = $this->processCallback('getTranslatedMenuURL', array(
			$origURL,
			$menuId,
			$locale,
		));
		if (!isset($url[0]) /* strlen($url) === 0 */ ) {
			$url = $origURL;
		}
		// exit
		return $url;
	}

	/**
	 * Get translated menu title
	 *
	 * @access public
	 * @param string $origTitle Original menu title
	 * @param int $menuId Menu id
	 * @param string $locale Language locale
	 * @return string Translated menu title
	 */
	public function getTranslatedMenuTitle($origTitle, $menuId, $locale) {
		// get original menu title
		$title = $origTitle;
		// get translated menu URL
		$title = $this->processCallback('getTranslatedMenuTitle', array(
			$origTitle,
			$menuId,
			$locale,
		));
		if (!isset($title[0]) /* strlen($title) === 0 */ ) {
			$title = $origTitle;
		}
		// exit
		return $title;
	}

	/**
	 * Something to do before get authors
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetAuthors($locale) {
		// something to do before get authors
		$this->processCallback('beforeGetAuthors', array(
			$locale,
		));
	}

	/**
	 * Something to do after get authors
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetAuthors($locale) {
		// something to do after get authors
		$this->processCallback('afterGetAuthors', array(
			$locale,
		));
	}

	/**
	 * Something to do before get author item
	 *
	 * @access public
	 * @param int $authorId Author id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetAuthorItem($authorId, $locale) {
		// something to do before get author item
		$this->processCallback('beforeGetAuthorItem', array(
			$authorId,
			$locale,
		));
	}

	/**
	 * Something to do after get author item
	 *
	 * @access public
	 * @param int $authorId Author id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetAuthorItem($authorId, $locale) {
		// something to do after get author item
		$this->processCallback('afterGetAuthorItem', array(
			$authorId,
			$locale,
		));
	}

	/**
	 * Get translated author URL
	 *
	 * @access public
	 * @param string $origURL Original author URL
	 * @param int $authorId Author id
	 * @param string $locale Language locale
	 * @return string Translated author URL
	 */
	public function getTranslatedAuthorURL($origURL, $authorId, $locale) {
		// get original author URL
		$url = $origURL;
		// get translated author URL
		$url = $this->processCallback('getTranslatedAuthorURL', array(
			$origURL,
			$authorId,
			$locale,
		));
		if (!isset($url[0]) /* strlen($url) === 0 */ ) {
			$url = $origURL;
		}
		// exit
		return $url;
	}

	/**
	 * Get translated author name
	 *
	 * @access public
	 * @param string $origName Original author name
	 * @param int $authorId Author id
	 * @param string $locale Language locale
	 * @return string Translated author name
	 */
	public function getTranslatedAuthorName($origName, $authorId, $locale) {
		// get original author name
		$name = $origName;
		// get translated author URL
		$name = $this->processCallback('getTranslatedAuthorName', array(
			$origName,
			$authorId,
			$locale,
		));
		if (!isset($name[0]) /* strlen($name) === 0 */ ) {
			$name = $origName;
		}
		// exit
		return $name;
	}

	/**
	 * Something to do before get tags
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetTags($locale) {
		// something to do before get tags
		$this->processCallback('beforeGetTags', array(
			$locale,
		));
	}

	/**
	 * Something to do after get tags
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetTags($locale) {
		// something to do after get tags
		$this->processCallback('afterGetTags', array(
			$locale,
		));
	}

	/**
	 * Something to do before get tag item
	 *
	 * @access public
	 * @param int $tagId Tag id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetTagItem($tagId, $locale) {
		// something to do before get tag item
		$this->processCallback('beforeGetTagItem', array(
			$tagId,
			$locale,
		));
	}

	/**
	 * Something to do after get tag item
	 *
	 * @access public
	 * @param int $tagId Tag id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetTagItem($tagId, $locale) {
		// something to do after get tag item
		$this->processCallback('afterGetTagItem', array(
			$tagId,
			$locale,
		));
	}

	/**
	 * Get translated tag URL
	 *
	 * @access public
	 * @param string $origURL Original tag URL
	 * @param int $tagId Tag id
	 * @param string $locale Language locale
	 * @return string Translated tag URL
	 */
	public function getTranslatedTagURL($origURL, $tagId, $locale) {
		// get original tag URL
		$url = $origURL;
		// get translated tag URL
		$url = $this->processCallback('getTranslatedTagURL', array(
			$origURL,
			$tagId,
			$locale,
		));
		if (!isset($url[0]) /* strlen($url) === 0 */ ) {
			$url = $origURL;
		}
		// exit
		return $url;
	}

	/**
	 * Get translated tag name
	 *
	 * @access public
	 * @param string $origName Original tag name
	 * @param int $tagId Tag id
	 * @param string $locale Language locale
	 * @return string Translated tag name
	 */
	public function getTranslatedTagName($origName, $tagId, $locale) {
		// get original tag name
		$name = $origName;
		// get translated tag URL
		$name = $this->processCallback('getTranslatedTagName', array(
			$origName,
			$tagId,
			$locale,
		));
		if (!isset($name[0]) /* strlen($name) === 0 */ ) {
			$name = $origName;
		}
		// exit
		return $name;
	}

	/**
	 * Something to do before get custom posts
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetCustomPosts($locale) {
		// something to do before get custom posts
		$this->processCallback('beforeGetCustomPosts', array(
			$locale,
		));
	}

	/**
	 * Something to do after get custom posts
	 *
	 * @access public
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetCustomPosts($locale) {
		// something to do after get custom posts
		$this->processCallback('afterGetCustomPosts', array(
			$locale,
		));
	}

	/**
	 * Something to do before get custom post item
	 *
	 * @access public
	 * @param int $customPostId Custom post id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function beforeGetCustomPostItem($customPostId, $locale) {
		// something to do before get custom post item
		$this->processCallback('beforeGetCustomPostItem', array(
			$customPostId,
			$locale,
		));
	}

	/**
	 * Something to do after get custom post item
	 *
	 * @access public
	 * @param int $customPostId Custom post id
	 * @param string $locale Language locale
	 * @return void
	 */
	public function afterGetCustomPostItem($customPostId, $locale) {
		// something to do after get custom post item
		$this->processCallback('afterGetCustomPostItem', array(
			$customPostId,
			$locale,
		));
	}

	/**
	 * Get translated custom post URL
	 *
	 * @access public
	 * @param string $origURL Original custom post URL
	 * @param int $customPostId Custom post id
	 * @param string $locale Language locale
	 * @return string Translated custom post URL
	 */
	public function getTranslatedCustomPostURL($origURL, $customPostId, $locale) {
		// get original custom post URL
		$url = $origURL;
		// get translated custom post URL
		$url = $this->processCallback('getTranslatedCustomPostURL', array(
			$origURL,
			$customPostId,
			$locale,
		));
		if (!isset($url[0]) /* strlen($url) === 0 */ ) {
			$url = $origURL;
		}
		// exit
		return $url;
	}

	/**
	 * Get translated custom post title
	 *
	 * @access public
	 * @param string $origTitle Original custom post title
	 * @param int $customPostId Custom post id
	 * @param string $locale Language locale
	 * @return string Translated custom post title
	 */
	public function getTranslatedCustomPostTitle($origTitle, $customPostId, $locale) {
		// get original custom post title
		$title = $origTitle;
		// get translated custom post URL
		$title = $this->processCallback('getTranslatedCustomPostTitle', array(
			$origTitle,
			$customPostId,
			$locale,
		));
		if (!isset($title[0]) /* strlen($title) === 0 */ ) {
			$title = $origTitle;
		}
		// exit
		return $title;
	}
}
