<?php

/**
 * js-helper.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujInternalLib\V2a\Classes;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * JavaScript helper class
 *
 * @access public
 */
final class JsHelper {
	/**
	 * Singleton instance
	 *
	 * @access private
	 * @var object
	 */
	private static $instance = NULL;

	/**
	 * Action for adding scripts has been already executed (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $actionExecuted = false;

	/**
	 * Add "exception.js" and "exception-code.js" scripts in adding scripts action (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $addExceptionJS = false;

	/**
	 * Add "exception.js" and "exception-code.js" scripts in footer (true) or in header (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $addExceptionJSInFooter = false;

	/**
	 * Scripts "exception.js" and "exception-code.js" has been added to queue (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $addedException = false;

	/**
	 * Add "helper.js" script in adding scripts action (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $addHelperJS = false;

	/**
	 * Add "helper.js" script in footer (true) or in header (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $addHelperJSInFooter = false;

	/**
	 * Script "helper.js" has been added to queue (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $addedHelper = false;

	/**
	 * Add "modal.js" script in adding scripts action (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $addModalJS = false;

	/**
	 * Add "modal.js" script in footer (true) or in header (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $addModalJSInFooter = false;

	/**
	 * Script "modal.js" has been added to queue (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $addedModal = false;

	/**
	 * Constructor
	 *
	 * @access private
	 * @return void
	 */
	private function __construct() {
		// set scripts actions
		if ((is_admin()) || (is_network_admin())) {
			add_action('admin_enqueue_scripts', array($this, 'actionAddScripts'), 1);
		} else {
			add_action('wp_enqueue_scripts', array($this, 'actionAddScripts'), 1);
		}
	}

	/**
	 * Disable cloning of object
	 *
	 * @access private
	 * @return void
	 */
	private function __clone() {
	}

	/**
	 * Get singleton instance
	 *
	 * @access public
	 * @return object Singleton instance
	 */
	public static function getInstance() {
		// optionally create new instance
		if (!self::$instance) {
			self::$instance = new self();
		}
		// exit
		return self::$instance;
	}

	/**
	 * Get minified JavaScript script filename if WordPress allows for it
	 *
	 * @access public
	 * @param string $filename Filename of script without extension
	 * @return string Minified JavaScript script filename if WordPress allows for it
	 */
	public function getMinJSFilename($filename) {
		// exit
		return $filename.'.'.((((defined('SCRIPT_DEBUG')) && (SCRIPT_DEBUG)) || ((!is_admin()) && (!is_network_admin()))) ?
				'js' :
				'min.js'
			);
	}

	/**
	 * Check if throw errors in JavaScript
	 *
	 * @access public
	 * @return bool Throw errors in JavaScript (true) or not (false)
	 */
	public function getThrowErrors() {
		// exit
		return (((defined('WP_DEBUG')) && (WP_DEBUG)) || ((defined('SCRIPT_DEBUG')) && (SCRIPT_DEBUG)));
	}

	/**
	 * Add script
	 *
	 * @access public
	 * @param string $handle Script handle
	 * @param string $urlToDirectory URL to script directory
	 * @param string $filenameWithoutExt Script filename without extension
	 * @param array $deps Script dependencies - default: empty
	 * @param string $ver Script version - default: empty
	 * @param bool $inFooter Show script in footer (true) or in header (false) - default: false
	 * @param bool $useMin Use minimized script (true) or standard (false) - default: true
	 * @param string $localizeVar Variable for localization; if empty, there will be no localization - default: empty
	 * @param array $localizeData Localization data; used only if $localizeVar is not empty - default: empty
	 * @param bool $addThrowToLocalizeData Add settings "throwErrors" to localization data with value indicated if script should throw errors (true) or does not add anything (false) - default: false
	 * @return void
	 */
	public function addScript($handle, $urlToDirectory, $filenameWithoutExt, array $deps = array(), $ver = '', $inFooter = false, $useMin = true, $localizeVar = '', array $localizeData = array(), $addThrowToLocalizeData = false) {
		// add script
		wp_enqueue_script($handle, $urlToDirectory.'/'.(($useMin) ?
				$this->getMinJSFilename($filenameWithoutExt) :
				$filenameWithoutExt.'.js'
			), $deps, $ver, $inFooter);
		// add script localization
		if (isset($localizeVar[0]) /* strlen($localizeVar) > 0 */ ) {
			if ($addThrowToLocalizeData) {
				$localizeData['throwErrors'] = ($this->getThrowErrors()) ?
					'1' :
					'0';
			}
			wp_localize_script($handle, $localizeVar, $localizeData);
		}
	}

	/**
	 * Add "exception.js" and "exception-code.js" scripts
	 *
	 * @access public
	 * @param bool $inFooter Show script in footer (true) or in header (false) - default: false
	 * @return void
	 */
	public function addExceptionJS($inFooter = false) {
		// add script
		if (!$this->addedException) {
			if (!$this->actionExecuted) {
				$this->addExceptionJS = true;
				$this->addExceptionJSInFooter = $inFooter;
			} else {
				$this->addExceptionJSNow($inFooter);
			}
		}
	}

	/**
	 * Add "exception.js" and "exception-code.js" scripts now
	 *
	 * @access private
	 * @param bool $inFooter Show script in footer (true) or in header (false) - default: false
	 * @return void
	 */
	private function addExceptionJSNow($inFooter = false) {
		// add script now
		$this->addHelperJSNow(($this->addHelperJS) ?
			$this->addHelperJSInFooter :
			$this->addExceptionJSInFooter
		);
		if (!$this->addedException) {
			if ($this->getThrowErrors()) {
				$this->addScript(Helper::getInstance()->getPrefix().'-exception-code', LibUrls::getInstance()->getJavaScriptUrl(), 'exception-code', array(), Version::getInstance()->getVersion(), $inFooter);
				$this->addScript(Helper::getInstance()->getPrefix().'-exception', LibUrls::getInstance()->getJavaScriptUrl(), 'exception', array(
					Helper::getInstance()->getPrefix().'-helper',
					Helper::getInstance()->getPrefix().'-exception-code',
				), Version::getInstance()->getVersion(), $inFooter);
			}
			$this->addedException = true;
		}
	}

	/**
	 * Add "helper.js" script
	 *
	 * @access public
	 * @param bool $inFooter Show script in footer (true) or in header (false) - default: false
	 * @return void
	 */
	public function addHelperJS($inFooter = false) {
		// add script
		if (!$this->addedHelper) {
			if (!$this->actionExecuted) {
				$this->addHelperJS = true;
				$this->addHelperJSInFooter = $inFooter;
			} else {
				$this->addHelperJSNow($inFooter);
			}
		}
	}

	/**
	 * Add "helper.js" script now
	 *
	 * @access private
	 * @param bool $inFooter Show script in footer (true) or in header (false) - default: false
	 * @return void
	 */
	private function addHelperJSNow($inFooter = false) {
		// add script now
		if (!$this->addedHelper) {
			$this->addScript(Helper::getInstance()->getPrefix().'-helper', LibUrls::getInstance()->getJavaScriptUrl(), 'helper', array(), Version::getInstance()->getVersion(), $inFooter);
			$this->addedHelper = true;
		}
	}

	/**
	 * Add "modal.js" script
	 *
	 * @access public
	 * @param bool $inFooter Show script in footer (true) or in header (false) - default: false
	 * @return void
	 */
	public function addModalJS($inFooter = false) {
		// add script
		if (!$this->addedModal) {
			if (!$this->actionExecuted) {
				$this->addModalJS = true;
				$this->addModalJSInFooter = $inFooter;
			} else {
				$this->addModalJSNow($inFooter);
			}
		}
	}

	/**
	 * Add "modal.js" script now
	 *
	 * @access private
	 * @param bool $inFooter Show script in footer (true) or in header (false) - default: false
	 * @return void
	 */
	private function addModalJSNow($inFooter = false) {
		// add script now
		$this->addHelperJSNow(($this->addHelperJS) ?
			$this->addHelperJSInFooter :
			$this->addModalJSInFooter
		);
		$this->addExceptionJSNow(($this->addExceptionJS) ?
			$this->addExceptionJSInFooter :
			$this->addModalJSInFooter
		);
		if (!$this->addedModal) {
			$deps = array(
				'jquery',
				Helper::getInstance()->getPrefix().'-helper',
			);
			if ($this->getThrowErrors()) {
				$deps[] = Helper::getInstance()->getPrefix().'-exception';
			}
			$this->addScript(Helper::getInstance()->getPrefix().'-modal', LibUrls::getInstance()->getJavaScriptUrl(), 'modal', $deps, Version::getInstance()->getVersion(), $inFooter, true, 'kocujILV2aModalVals', array(
				'prefix' => Helper::getInstance()->getPrefix(),
			), true);
			$this->addedModal = true;
		}
	}

	/**
	 * Action for adding scripts
	 *
	 * @access public
	 * @return void
	 */
	public function actionAddScripts() {
		// add scripts
		if (($this->addHelperJS) || ($this->addModalJS)) {
			$this->addHelperJSNow(($this->addHelperJS) ?
				$this->addHelperJSInFooter :
				$this->addModalJSInFooter
			);
		}
		if ($this->addExceptionJS) {
			$this->addExceptionJSNow($this->addExceptionJSInFooter);
		}
		if ($this->addModalJS) {
			$this->addModalJSNow($this->addModalJSInFooter);
		}
		// set action as executed
		$this->actionExecuted = true;
	}
}
