/**
 * @file Modal window
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj <dominik@kocuj.pl>
 */

(function() {})(); // empty function for correct minify with comments
//'use strict'; // for jshint uncomment this and comment line above

/* jshint strict: true */
/* jshint -W034 */
/* jshint -W107 */

/* global document */
/* global jQuery */
/* global window */

/* global kocujILV2aHelper */

/* global kocujILV2aModalVals */

/* global kocujILV2aCException */
/* global kocujILV2aExceptionCode */

/**
 * Modal window types
 *
 * @namespace kocujILV2aModalType
 * @public
 */
var kocujILV2aModalType = {
	/**
	 * Window without title and status bar
	 *
	 * @public
	 * @const {number}
	 */
	DEFAULT : 0,

	/**
	 * Window with title and without status bar
	 *
	 * @public
	 * @const {number}
	 */
	INFO : 1,

	/**
	 * Window with title and status bar
	 *
	 * @public
	 * @const {number}
	 */
	INFO_WITH_STATUS : 2,

	/**
	 * Information about maximum constant value; it should not be used in executing the modal script methods
	 *
	 * @public
	 * @const {number}
	 */
	LAST : 2
};

/**
 * Modal window prototype constructor
 *
 * @constructs
 * @namespace kocujILV2aCModal
 * @public
 * @return {void}
 */
function kocujILV2aCModal() {
	'use strict';
	/* jshint validthis: true */
	// get this object
	var self = this;
	// initialize objects
	self._objHelper = kocujILV2aHelper;
	// get current script filename
	self._thisFilename = document.scripts[document.scripts.length-1].src;
	// get settings
	if (typeof kocujILV2aModalVals !== 'undefined') {
		var vals = kocujILV2aModalVals;
		if (vals.throwErrors !== undefined) {
			if (vals.throwErrors === '1') {
				self._valsThrowErrors = true;
			} else {
				self._valsThrowErrors = false;
			}
		}
		if (vals.prefix !== undefined) {
			self._valsPrefix = vals.prefix;
		}
	}
}

/**
 * Modal window prototype
 *
 * @namespace kocujILV2aCModal
 * @public
 */
kocujILV2aCModal.prototype = {
	/**
	 * Object kocujILV2aHelper
	 *
	 * @private
	 * @type {Object}
	 */
	_objHelper : null,

	/**
	 * Current script filename
	 *
	 * @private
	 * @type {string}
	 */
	_thisFilename : '',

	/**
	 * Projects list
	 *
	 * @private
	 * @type {Array}
	 */
	_prj : [],

	/**
	 * Script settings - throw errors (true) or not (false)
	 *
	 * @private
	 * @type {string}
	 */
	_valsThrowErrors : false,

	/**
	 * Script settings - prefix
	 *
	 * @private
	 * @type {string}
	 */
	_valsPrefix : '',

	/**
	 * Add project
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {number} internalMargins Internal margins; can be empty; if not set, the default value 15 will be used
	 * @param {number} borderWidth Border width; can be empty; if not set, the default value 5 will be used
	 * @return {void}
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.PROJECT_ALREADY_EXISTS if project identifier entered in projectId already exists
	 */
	addProject : function(projectId, internalMargins, borderWidth) {
		'use strict';
		// parse arguments
		var args = this._checkAddProject(projectId, internalMargins, borderWidth);
		// add project
		if (this._prj['prj_' + args.projectId] === undefined) {
			this.addProjectIfNotExists(args.projectId, args.internalMargins, args.borderWidth);
		} else {
			this._throwError('PROJECT_ALREADY_EXISTS', args.projectId);
			return;
		}
	},

	/**
	 * Add project if not exists
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {number} internalMargins Internal margins; can be empty; if not set, the default value 15 will be used
	 * @param {number} borderWidth Border width; can be empty; if not set, the default value 5 will be used
	 * @return {void}
	 */
	addProjectIfNotExists : function(projectId, internalMargins, borderWidth) {
		'use strict';
		// parse arguments
		var args = this._checkAddProject(projectId, internalMargins, borderWidth);
		// add project
		if (this._prj['prj_' + args.projectId] === undefined) {
			this._prj['prj_' + args.projectId] = {
				internalMargins : args.internalMargins,
				borderWidth     : args.borderWidth,
				preparedData    : [],
				visible         : [],
				opacity         : [],
				closeEvents     : []
			};
		}
	},

	/**
	 * Get HTML selector for modal mask
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @return {string} HTML selector for modal mask
	 */
	getHTMLSelectorModalMask : function(projectId, id) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		id = this._objHelper.initString(id);
		// exit
		return '#' + this._getHTMLNameModalMask(projectId) + this._changeIdToSuffix(id);
	},

	/**
	 * Get HTML selector for modal container
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @return {string} HTML selector for modal container
	 */
	getHTMLSelectorModalContainer : function(projectId, id) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		id = this._objHelper.initString(id);
		// exit
		return '#' + this._getHTMLNameModalContainer(projectId) + this._changeIdToSuffix(id);
	},

	/**
	 * Get HTML selector for modal dialog
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @return {string} HTML selector for modal dialog
	 */
	getHTMLSelectorModalDialog : function(projectId, id) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		id = this._objHelper.initString(id);
		// exit
		return '#' + this._getHTMLNameModalDialog(projectId) + this._changeIdToSuffix(id);
	},

	/**
	 * Get HTML selector for inside modal dialog
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @return {string} HTML selector for inside modal dialog
	 */
	getHTMLSelectorModalDialogInside : function(projectId, id) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		id = this._objHelper.initString(id);
		// exit
		return '#' + this._getHTMLNameModalDialogInside(projectId) + this._changeIdToSuffix(id);
	},

	/**
	 * Get HTML selector for content inside modal dialog
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @return {string} HTML selector for content inside modal dialog
	 */
	getHTMLSelectorModalDialogContentInside : function(projectId, id) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		id = this._objHelper.initString(id);
		// exit
		return '#' + this._getHTMLNameModalDialogContentInside(projectId) + this._changeIdToSuffix(id);
	},

	/**
	 * Get HTML selector for modal dialog elements
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @return {string} HTML selector for modal dialog elements
	 */
	getHTMLSelectorModalDialogElements : function(projectId, id) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		id = this._objHelper.initString(id);
		// exit
		return '#' + this._getHTMLNameModalDialogElements(projectId) + this._changeIdToSuffix(id);
	},

	/**
	 * Get HTML selector for modal dialog title
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @return {string} HTML selector for modal dialog title
	 */
	getHTMLSelectorModalDialogTitle : function(projectId, id) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		id = this._objHelper.initString(id);
		// exit
		return '#' + this._getHTMLNameModalDialogTitle(projectId) + this._changeIdToSuffix(id);
	},

	/**
	 * Get HTML selector for modal dialog status
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @return {string} HTML selector for modal dialog status
	 */
	getHTMLSelectorModalDialogStatus : function(projectId, id) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		id = this._objHelper.initString(id);
		// exit
		return '#' + this._getHTMLNameModalDialogStatus(projectId) + this._changeIdToSuffix(id);
	},

	/**
	 * Add close event to element
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @param {string} elementPath DOM path to element
	 * @param {function} [callback] Callback function execute after clicking on element
	 * @return {void}
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.EMPTY_ELEMENT_PATH if DOM path to element entered in elementPath is empty
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.ELEMENT_DOES_NOT_EXIST if DOM path to element entered in elementPath does not exist
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.MODAL_WINDOW_CLOSE_EVENT_EXISTS_ALREADY if event for closing window exists already in DOM path to element entered in elementPath
	 */
	addCloseEvent : function(projectId, id, elementPath, callback) {
		'use strict';
		// get this object
		var self = this;
		(function($) {
			// parse arguments
			projectId = self._parseProjectId(projectId);
			id = self._objHelper.initString(id);
			elementPath = self._objHelper.initString(elementPath);
			if (elementPath === '') {
				self._throwError('EMPTY_ELEMENT_PATH', id);
				return;
			}
			callback = self._objHelper.initFunction(callback);
			// check if window has been prepared
			self._checkIdWithError(projectId, id, true);
			// check selector
			var selectorElementPath = $(elementPath);
			if (selectorElementPath.length === 0) {
				self._throwError('ELEMENT_DOES_NOT_EXIST', elementPath);
				return;
			}
			// check if close event does not exist already
			if (self.checkCloseEvent(projectId, id, elementPath)) {
				self._throwError('MODAL_WINDOW_CLOSE_EVENT_EXISTS_ALREADY', elementPath);
				return;
			}
			// add close event
			selectorElementPath.attr('href', 'javascript:void(0);');
			selectorElementPath.bind('click.' + self._getEventNamePrefix(projectId) + 'modal_close_' + self._changeIdToSuffix(id), {
				projectId : projectId,
				id        : id,
				callback  : callback,
			}, function(event) {
				// disable default event
				event.preventDefault();
				// check if window is visible
				if (self.checkVisible(event.data.projectId, event.data.id)) {
					// hide modal window
					$(self.getHTMLSelectorModalDialog(event.data.projectId, event.data.id)).fadeOut('slow', function() {
						// hide window mask
						$(self.getHTMLSelectorModalMask(event.data.projectId, event.data.id)).fadeOut('fast', function() {
							// hide all elements
							$(self.getHTMLSelectorModalContainer(event.data.projectId, event.data.id)).hide();
							$(self.getHTMLSelectorModalMask(event.data.projectId, event.data.id)).hide();
							// optionally execute callback
							if (event.data.callback !== false) {
								/* jshint evil: true */
								eval(event.data.callback);
							}
							// clear content
							$(self.getHTMLSelectorModalDialogContentInside(event.data.projectId, event.data.id)).html('');
						});
					});
					// set window as hidden
					self._prj['prj_' + event.data.projectId].visible['win_' + event.data.id] = false;
					// remove window prepared data
					delete self._prj['prj_' + event.data.projectId].preparedData['win_' + event.data.id];
				}
			});
			// save close event
			self._prj['prj_' + projectId].closeEvents['win_' + id]['sel_' + elementPath] = elementPath;
		}(jQuery));
	},

	/**
	 * Add close event to mask behind modal window
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @param {function} [callback] Callback function execute after clicking on mask
	 * @return {void}
	 */
	addCloseEventToMask : function(projectId, id, callback) {
		'use strict';
		// remove close event from mask
		this.addCloseEvent(projectId, id, this.getHTMLSelectorModalMask(projectId, id), callback);
	},

	/**
	 * Remove close event from element
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @param {string} elementPath DOM path to element
	 * @return {void}
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.EMPTY_ELEMENT_PATH if DOM path to element entered in elementPath is empty
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.ELEMENT_DOES_NOT_EXIST if DOM path to element entered in elementPath does not exist
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.MODAL_WINDOW_CLOSE_EVENT_DOES_NOT_EXIST if event for closing window does not exist in DOM path to element entered in elementPath
	 */
	removeCloseEvent : function(projectId, id, elementPath) {
		'use strict';
		// get this object
		var self = this;
		(function($) {
			// parse arguments
			projectId = self._parseProjectId(projectId);
			id = self._objHelper.initString(id);
			elementPath = self._objHelper.initString(elementPath);
			if (elementPath === '') {
				self._throwError('EMPTY_ELEMENT_PATH', id);
				return;
			}
			// check if window has been prepared
			self._checkIdWithError(projectId, id, true);
			// check selector
			var selectorElementPath = $(elementPath);
			if (selectorElementPath.length === 0) {
				self._throwError('ELEMENT_DOES_NOT_EXIST', elementPath);
				return;
			}
			// check if close event does exist
			if (!self.checkCloseEvent(projectId, id, elementPath)) {
				self._throwError('MODAL_WINDOW_CLOSE_EVENT_DOES_NOT_EXIST', elementPath);
				return;
			}
			// remove close event
			selectorElementPath.unbind('click.' + self._getEventNamePrefix(projectId) + 'modal_close_' + self._changeIdToSuffix(id));
			delete self._prj['prj_' + projectId].closeEvents['win_' + id]['sel_' + elementPath];
		}(jQuery));
	},

	/**
	 * Remove close event from mask behind modal window
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @return {void}
	 */
	removeCloseEventFromMask : function(projectId, id) {
		'use strict';
		// remove close event from mask
		this.removeCloseEvent(projectId, id, this.getHTMLSelectorModalMask(projectId, id));
	},

	/**
	 * Prepare modal window
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @param {string} [content] Window content
	 * @param {Object} [attr] Window attributes; there are the following attributes available: "height" (int type; height of the window), "opacity" (float type; opacity of the window), "status" (string type; information in status bar); "title" (string type: title text); "type" (int type; window type from constants from kocujILV2aModalType); "width" (int type; with of the window)
	 * @return {void}
	 */
	prepareModal : function(projectId, id, content, attr) {
		'use strict';
		// get this object
		var self = this;
		(function($) {
			// parse arguments
			projectId = self._parseProjectId(projectId);
			id = self._objHelper.initString(id);
			content = self._objHelper.initString(content);
			// check if window has not been prepared already
			self._checkIdWithError(projectId, id, false);
			// save prepared window settings
			self._savePreparedModalAttr(projectId, id, attr);
			// get prepared window data
			var preparedData = self._prj['prj_' + projectId].preparedData['win_' + id];
			// append window
			var add = '<div id="' + self._getHTMLNameModalContainer(projectId) + self._changeIdToSuffix(id) + '">' +
				'<div id="' + self._getHTMLNameModalDialog(projectId) + self._changeIdToSuffix(id) + '">';
			if ((preparedData.type === kocujILV2aModalType.INFO) || (preparedData.type === kocujILV2aModalType.INFO_WITH_STATUS)) {
				add += '<div id="' + self._getHTMLNameModalDialogElements(projectId) + self._changeIdToSuffix(id) + '">' +
					'<div id="' + self._getHTMLNameModalDialogTitle(projectId) + self._changeIdToSuffix(id) + '">' +
					preparedData.title +
					'</div>' +
					'<div id="' + self._getHTMLNameModalDialogInside(projectId) + self._changeIdToSuffix(id) + '">' +
					'<div id="' + self._getHTMLNameModalDialogContentInside(projectId) + self._changeIdToSuffix(id) + '">' +
					'</div>' +
					'</div>';
				if (preparedData.type === kocujILV2aModalType.INFO_WITH_STATUS) {
					add += '<div id="' + self._getHTMLNameModalDialogStatus(projectId) + self._changeIdToSuffix(id) + '">' +
						'</div>';
				}
				add += '</div>';
			} else {
				add += '<div id="' + self._getHTMLNameModalDialogInside(projectId) + self._changeIdToSuffix(id) + '">' +
					'</div>';
			}
			add += '</div>' +
				'<div id="' + self._getHTMLNameModalMask(projectId) + self._changeIdToSuffix(id) + '">' +
				'</div>' +
				'</div>';
			$('body').prepend(add);
			$(self.getHTMLSelectorModalContainer(projectId, id)).hide();
			// get selectors
			var selectorModalDialogInside = $(self.getHTMLSelectorModalContainer(projectId, id) + ' ' + self.getHTMLSelectorModalDialogInside(projectId, id));
			var selectorModalDialog = $(self.getHTMLSelectorModalContainer(projectId, id) + ' ' + self.getHTMLSelectorModalDialog(projectId, id));
			// set styles
			selectorModalDialog.css({
				'position'         : 'absolute',
				'display'          : 'none',
				'z-index'          : '999999',
				'background-color' : '#000000',
				'left'             : ($(window).width()/2)-((preparedData.width+(self._prj['prj_' + projectId].internalMargins*2)+(self._prj['prj_' + projectId].borderWidth*2))/2),
				'top'              : ($(window).height()/2)-((preparedData.height+(self._prj['prj_' + projectId].internalMargins*2)+(self._prj['prj_' + projectId].borderWidth*2))/2)
			});
			// set additional styles
			var stylesInsideOrElements = {
				'background-color' : '#ffffff',
				'box-sizing'       : 'content-box',
				'margin-left'      : parseInt(self._prj['prj_' + projectId].borderWidth, 10) + 'px',
				'margin-right'     : parseInt(self._prj['prj_' + projectId].borderWidth, 10) + 'px',
				'margin-top'       : parseInt(self._prj['prj_' + projectId].borderWidth, 10) + 'px',
				'margin-bottom'    : parseInt(self._prj['prj_' + projectId].borderWidth, 10) + 'px'
			};
			if ((preparedData.type === kocujILV2aModalType.INFO) || (preparedData.type === kocujILV2aModalType.INFO_WITH_STATUS)) {
				// set styles
				var backgroundColor = '#464646';
				$(self.getHTMLSelectorModalContainer(projectId, id) + ' ' + self.getHTMLSelectorModalDialogElements(projectId, id)).css(stylesInsideOrElements);
				$(self.getHTMLSelectorModalContainer(projectId, id) + ' ' + self.getHTMLSelectorModalDialogTitle(projectId, id)).css({
					'box-sizing'       : 'content-box',
					'font-size'        : '13px',
					'text-align'       : 'center',
					'font-weight'      : 'normal',
					'line-height'      : '28px',
					'width'            : parseInt(preparedData.width, 10) + 'px',
					'height'           : '28px',
					'color'            : '#cccccc',
					'background-color' : backgroundColor
				});
				var insideHeight = preparedData.height-28,
					borderBottomWidth = 0,
					statusHeight = 0;
				if (preparedData.type === kocujILV2aModalType.INFO_WITH_STATUS) {
					statusHeight = 40;
					insideHeight -= statusHeight;
					borderBottomWidth = 1;
				}
				selectorModalDialogInside.css({
					'box-sizing'          : 'content-box',
					'font-size'           : '12px',
					'text-align'          : 'left',
					'font-weight'         : 'normal',
					'width'               : parseInt(preparedData.width, 10) + 'px',
					'height'              : parseInt(insideHeight, 10) + 'px',
					'padding'             : '0',
					'color'               : backgroundColor,
					'background-color'    : '#ffffff',
					'border-bottom-color' : '#000000',
					'border-bottom-style' : 'solid',
					'border-bottom-width' : borderBottomWidth + 'px',
					'overflow'            : 'auto'
				});
				$(self.getHTMLSelectorModalContainer(projectId, id) + ' ' + self.getHTMLSelectorModalDialogContentInside(projectId, id)).css({
					'box-sizing'       : 'content-box',
					'font-size'        : '12px',
					'text-align'       : 'left',
					'font-weight'      : 'normal',
					'width'            : 'auto',
					'padding'          : parseInt(self._prj['prj_' + projectId].internalMargins, 10) + 'px ' + parseInt(self._prj['prj_' + projectId].internalMargins, 10) + 'px ' + parseInt(self._prj['prj_' + projectId].internalMargins, 10) + 'px ' + parseInt(self._prj['prj_' + projectId].internalMargins, 10) + 'px',
					'color'            : backgroundColor,
					'background-color' : '#ffffff',
					'overflow'         : 'hidden'
				});
				if (preparedData.type === kocujILV2aModalType.INFO_WITH_STATUS) {
					$(self.getHTMLSelectorModalContainer(projectId, id) + ' ' + self.getHTMLSelectorModalDialogStatus(projectId, id)).css({
						'box-sizing'       : 'content-box',
						'font-size'        : '12px',
						'text-align'       : 'center',
						'vertical-align'   : 'middle',
						'font-weight'      : 'normal',
						'width'            : parseInt(preparedData.width, 10) + 'px',
						'height'           : statusHeight + 'px',
						'color'            : '#21759b',
						'background-color' : '#ececec'
					});
				}
			} else {
				// set styles
				selectorModalDialogInside.css(stylesInsideOrElements);
			}
			// prepare mask
			$(self.getHTMLSelectorModalMask(projectId, id)).css({
				'box-sizing'       : 'content-box',
				'position'         : 'absolute',
				'z-index'          : 999998,
				'background-color' : '#000000',
				'display'          : 'none',
				'left'             : 0,
				'top'              : 0,
				'width'            : $(window).width(),
				'height'           : $(document).height()
			});
			// set new window size
			if (preparedData.type === kocujILV2aModalType.DEFAULT) {
				selectorModalDialogInside.width(preparedData.width);
				selectorModalDialogInside.height(preparedData.height);
			}
			// change window content
			if (preparedData.type !== kocujILV2aModalType.DEFAULT) {
				$(self.getHTMLSelectorModalContainer(projectId, id) + ' ' + self.getHTMLSelectorModalDialogContentInside(projectId, id)).html(content);
			}
			// add close event
			self._prj['prj_' + projectId].closeEvents['win_' + id] = [];
			self.addCloseEvent(projectId, id, self.getHTMLSelectorModalMask(projectId, id));
		}(jQuery));
	},

	/**
	 * Remove data for preparation of modal window
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @return {void}
	 */
	unprepareModal : function(projectId, id) {
		'use strict';
		// get this object
		var self = this;
		(function($) {
			// parse arguments
			projectId = self._parseProjectId(projectId);
			id = self._objHelper.initString(id);
			// check if window has been prepared
			self._checkIdWithError(projectId, id, true);
			// check if window is not visible
			self._checkNotVisibleWithError(projectId, id);
			// remove all close events
			$.each(self._prj['prj_' + projectId].closeEvents['win_' + id], function(key, value) {
				self.removeCloseEvent(projectId, id, value);
			});
			self._prj['prj_' + projectId].closeEvents['win_' + id] = [];
			// remove window elements
			$(self.getHTMLSelectorModalContainer(projectId, id)).remove();
			// remove window prepared data
			delete self._prj['prj_' + projectId].preparedData['win_' + id];
		}(jQuery));
	},

	/**
	 * Show prepared modal window
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @param {number} [x] X position; can be empty for set Y position automatically
	 * @param {number} [y] Y position; can be empty for set Y position automatically
	 * @return {void}
	 */
	showPreparedModal : function(projectId, id, x, y) {
		'use strict';
		// get this object
		var self = this;
		(function($) {
			// parse arguments
			projectId = self._parseProjectId(projectId);
			id = self._objHelper.initString(id);
			x = self._objHelper.initNumeric(x);
			y = self._objHelper.initNumeric(y);
			// check if window has been prepared
			self._checkIdWithError(projectId, id, true);
			// check if window is not visible
			self._checkNotVisibleWithError(projectId, id);
			// get prepared window data
			var preparedData = self._prj['prj_' + projectId].preparedData['win_' + id];
			// show modal window box
			$(self.getHTMLSelectorModalContainer(projectId, id)).show();
			// show mask
			$(self.getHTMLSelectorModalMask(projectId, id)).fadeTo('fast', 0.8);
			// get selectors
			var selectorModalDialog = $(self.getHTMLSelectorModalDialog(projectId, id));
			// set window position
			if (x !== '') {
				selectorModalDialog.css('left', x);
			}
			if (y !== '') {
				selectorModalDialog.css('top', y);
			}
			if (preparedData.width !== '') {
				selectorModalDialog.width(preparedData.width+(self._prj['prj_' + projectId].borderWidth*2));
			}
			if (preparedData.height !== '') {
				selectorModalDialog.height(preparedData.height+(self._prj['prj_' + projectId].borderWidth*2));
			}
			// show window
			self._prj['prj_' + projectId].opacity['win_' + id] = preparedData.opacity;
			$(self.getHTMLSelectorModalDialog(projectId, id)).fadeTo('slow', preparedData.opacity);
			// set visible status
			self._prj['prj_' + projectId].visible['win_' + id] = true;
		}(jQuery));
	},

	/**
	 * Show modal window
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @param {string} [content] Window content
	 * @param {number} [x] X position; can be empty for set Y position automatically
	 * @param {number} [y] Y position; can be empty for set Y position automatically
	 * @param {Object} [attr] Window attributes; there are the following attributes available: "height" (int type; height of the window), "opacity" (float type; opacity of the window), "status" (string type; information in status bar); "title" (string type: title text); "type" (int type; window type from constants from kocujILV2aModalType); "width" (int type; with of the window)
	 * @return {void}
	 */
	showModal : function(projectId, id, content, x, y, attr) {
		'use strict';
		// show modal window
		this.prepareModal(projectId, id, content, attr);
		this.showPreparedModal(projectId, id, x, y);
	},

	/**
	 * Change prepared modal attributes
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @param {Object} [attr] Window attributes; there are the following attributes available: "height" (int type; height of the window), "opacity" (float type; opacity of the window), "status" (string type; information in status bar); "title" (string type: title text); "type" (int type; window type from constants from kocujILV2aModalType); "width" (int type; with of the window)
	 * @return {void}
	 */
	changePreparedModalAttr : function(projectId, id, attr) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		id = this._objHelper.initString(id);
		attr = this._objHelper.initObject(attr);
		// check if window has been prepared
		this._checkIdWithError(projectId, id, true);
		// check if window is not visible
		this._checkNotVisibleWithError(projectId, id);
		// save prepared window settings
		this._savePreparedModalAttr(projectId, id, attr);
	},

	/**
	 * Change modal window content
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @param {string} [content] Window content
	 * @return {void}
	 */
	changeModalContent : function(projectId, id, content) {
		'use strict';
		// get this object
		var self = this;
		(function($) {
			// parse arguments
			projectId = self._parseProjectId(projectId);
			id = self._objHelper.initString(id);
			content = self._objHelper.initString(content);
			// check if window has been prepared
			self._checkIdWithError(projectId, id, true);
			// change content
			$(self.getHTMLSelectorModalContainer(projectId, id) + ' ' + self.getHTMLSelectorModalDialogContentInside(projectId, id)).html(content);
		}(jQuery));
	},

	/**
	 * Check if window is visible
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @return {boolean} Window is visible (true) or hidden (false)
	 */
	checkVisible : function(projectId, id) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		id = this._objHelper.initString(id);
		// check if modal window is visible
		if (this._prj['prj_' + projectId].visible['win_' + id] === undefined) {
			return false;
		} else {
			return this._prj['prj_' + projectId].visible['win_' + id];
		}
	},

	/**
	 * Check if window exists
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @return {boolean} Window exists (true) or not (false)
	 */
	checkId : function(projectId, id) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		id = this._objHelper.initString(id);
		// check identifier
		if (this._prj['prj_' + projectId].preparedData['win_' + id] !== undefined) {
			return true;
		}
		// exit
		return false;
	},

	/**
	 * Check if close event exists in the selected DOM element path
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @param {string} elementPath DOM path to element
	 * @return {boolean} Close event exists (true) or not (false)
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.EMPTY_ELEMENT_PATH if DOM path to element entered in elementPath is empty
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.ELEMENT_DOES_NOT_EXIST if DOM path to element entered in elementPath does not exist
	 */
	checkCloseEvent : function(projectId, id, elementPath) {
		'use strict';
		// get this object
		var self = this;
		(function($) {
			// parse arguments
			projectId = self._parseProjectId(projectId);
			id = self._objHelper.initString(id);
			elementPath = self._objHelper.initString(elementPath);
			if (elementPath === '') {
				self._throwError('EMPTY_ELEMENT_PATH', id);
				return;
			}
			// check if window has been prepared
			self._checkIdWithError(projectId, id, true);
			// check selector
			var selectorElementPath = $(elementPath);
			if (selectorElementPath.length === 0) {
				self._throwError('ELEMENT_DOES_NOT_EXIST', elementPath);
				return;
			}
			// check if close event exists
			if ((self._prj['prj_' + projectId].closeEvents['win_' + id] !== undefined) && (self._prj['prj_' + projectId].closeEvents['win_' + id]['sel_' + elementPath] !== undefined)) {
				return true;
			}
			// exit
			return false;
		}(jQuery));
	},

	/**
	 * Get window type
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @return {number} Window type
	 */
	getType : function(projectId, id) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		id = this._objHelper.initString(id);
		// check if window has been prepared
		this._checkIdWithError(projectId, id, true);
		// exit
		return this._prj['prj_' + projectId].preparedData['win_' + id].type;
	},

	/**
	 * Get window width
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @return {number} Window width
	 */
	getWidth : function(projectId, id) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		id = this._objHelper.initString(id);
		// check if window has been prepared
		this._checkIdWithError(projectId, id, true);
		// exit
		return parseInt(this._prj['prj_' + projectId].preparedData['win_' + id].width, 10);
	},

	/**
	 * Get window height
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @return {number} Window height
	 */
	getHeight : function(projectId, id) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		id = this._objHelper.initString(id);
		// check if window has been prepared
		this._checkIdWithError(projectId, id, true);
		// exit
		return parseInt(this._prj['prj_' + projectId].preparedData['win_' + id].height, 10);
	},

	/**
	 * Get window opacity
	 *
	 * @public
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @return {number} Window opacity
	 */
	getOpacity : function(projectId, id) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		id = this._objHelper.initString(id);
		// check if window has been prepared
		this._checkIdWithError(projectId, id, true);
		// exit
		return parseFloat(this._prj['prj_' + projectId].preparedData['win_' + id].opacity);
	},

	/**
	 * Get event name prefix
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @return {string} Event name prefix
	 */
	_getEventNamePrefix : function(projectId) {
		'use strict';
		// exit
		return this._valsPrefix + '_' + projectId + '__';
	},

	/**
	 * Get HTML prefix
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @return {string} HTML prefix
	 */
	_getHTMLPrefix : function(projectId) {
		'use strict';
		// exit
		return this._valsPrefix + '_' + projectId + '__';
	},

	/**
	 * Get HTML name for modal mask
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @return {string} HTML name for modal mask
	 */
	_getHTMLNameModalMask : function(projectId) {
		'use strict';
		// exit
		return this._getHTMLPrefix(projectId) + 'modal_mask';
	},

	/**
	 * Get HTML name for modal container
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @return {string} HTML name for modal container
	 */
	_getHTMLNameModalContainer : function(projectId) {
		'use strict';
		// exit
		return this._getHTMLPrefix(projectId) + 'modal_container';
	},

	/**
	 * Get HTML name for modal dialog
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @return {string} HTML name for modal dialog
	 */
	_getHTMLNameModalDialog : function(projectId) {
		'use strict';
		// exit
		return this._getHTMLPrefix(projectId) + 'modal_dialog';
	},

	/**
	 * Get HTML name for inside modal dialog
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @return {string} HTML name for inside modal dialog
	 */
	_getHTMLNameModalDialogInside : function(projectId) {
		'use strict';
		// exit
		return this._getHTMLPrefix(projectId) + 'modal_dialog_inside';
	},

	/**
	 * Get HTML name for content inside modal dialog
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @return {string} HTML name for content inside modal dialog
	 */
	_getHTMLNameModalDialogContentInside : function(projectId) {
		'use strict';
		// exit
		return this._getHTMLPrefix(projectId) + 'modal_dialog_content_inside';
	},

	/**
	 * Get HTML name for modal dialog elements
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @return {string} HTML name for modal dialog elements
	 */
	_getHTMLNameModalDialogElements : function(projectId) {
		'use strict';
		// exit
		return this._getHTMLPrefix(projectId) + 'modal_dialog_elements';
	},

	/**
	 * Get HTML name for modal dialog title
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @return {string} HTML name for modal dialog title
	 */
	_getHTMLNameModalDialogTitle : function(projectId) {
		'use strict';
		// exit
		return this._getHTMLPrefix(projectId) + 'modal_dialog_title';
	},

	/**
	 * Get HTML name for modal dialog status
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @return {string} HTML name for modal dialog status
	 */
	_getHTMLNameModalDialogStatus : function(projectId) {
		'use strict';
		// exit
		return this._getHTMLPrefix(projectId) + 'modal_dialog_status';
	},

	/**
	 * Parse project identifier
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @return {string} Parsed project identifier
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.EMPTY_PROJECT_ID if project identifier entered in projectId is empty
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.PROJECT_DOES_NOT_EXIST if project identifier entered in projectId does not exist
	 */
	_parseProjectId : function(projectId) {
		'use strict';
		// parse project identifier
		projectId = this._objHelper.initString(projectId);
		if (projectId === '') {
			this._throwError('EMPTY_PROJECT_ID');
			return;
		}
		// check if project exists
		if (this._prj['prj_' + projectId] === undefined) {
			this._throwError('PROJECT_DOES_NOT_EXIST', projectId);
			return;
		}
		// exit
		return projectId;
	},

	/**
	 * Check arguments for adding project
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @param {number} internalMargins Internal margins; can be empty; if not set, the default value 15 will be used
	 * @param {number} borderWidth Border width; can be empty; if not set, the default value 5 will be used
	 * @return {Object} Parsed arguments for adding project
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.EMPTY_PROJECT_ID if project identifier entered in projectId is empty
	 */
	_checkAddProject : function(projectId, internalMargins, borderWidth) {
		'use strict';
		// parse arguments
		projectId = this._objHelper.initString(projectId);
		if (projectId === '') {
			this._throwError('EMPTY_PROJECT_ID');
			return;
		}
		internalMargins = this._objHelper.initNumeric(internalMargins);
		if (internalMargins === '') {
			internalMargins = 15;
		}
		if (internalMargins < 0) {
			internalMargins = 0;
		}
		borderWidth = this._objHelper.initNumeric(borderWidth);
		if (borderWidth === '') {
			borderWidth = 5;
		}
		if (borderWidth < 0) {
			borderWidth = 0;
		}
		// exit
		return {
			projectId       : projectId,
			internalMargins : internalMargins,
			borderWidth     : borderWidth
		};
	},

	/**
	 * Throw an error if debugging is enabled
	 *
	 * @private
	 * @param {string} codeString Error code in string format
	 * @param {string} [param] Parameter for error information
	 * @return {void}
	 */
	_throwError : function(codeString, param) {
		'use strict';
		// parse arguments
		codeString = this._objHelper.initString(codeString);
		if (codeString === '') {
			return;
		}
		param = this._objHelper.initString(param);
		// throw an error
		if (this._valsThrowErrors) {
			/* jshint evil: true */
			eval('throw new kocujILV2aCException(kocujILV2aExceptionCode.' + codeString + ', this._thisFilename, param);');
		}
	},

	/**
	 * Change identifier to suffix
	 *
	 * @private
	 * @param {string} id Window identifier
	 * @return {string} Suffix
	 */
	_changeIdToSuffix : function(id) {
		'use strict';
		// change identifier to suffix
		var idString = '';
		if (id !== '') {
			idString = '_' + id;
		}
		// exit
		return idString;
	},

	/**
	 * Check if window is not visible; if it is, throw an error
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @return {void}
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.MODAL_WINDOW_VISIBLE if window with identifier entered in id is visible
	 */
	_checkNotVisibleWithError : function(projectId, id) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		id = this._objHelper.initString(id);
		// check if modal window is visible
		if (this.checkVisible(projectId, id)) {
			this._throwError('MODAL_WINDOW_VISIBLE', id);
			return;
		}
	},

	/**
	 * Check if window exists or not with throwing an error
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @param {string} [id] Window identifier
	 * @param {bool} shouldExists Window identifier should exists (true) or should not exists (false)
	 * @return {void}
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.MODAL_WINDOW_HAS_NOT_BEEN_PREPARED if window with identifier entered in id has not been prepared already
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.MODAL_WINDOW_HAS_BEEN_PREPARED_ALREADY if window with identifier entered in id has been prepared already
	 */
	_checkIdWithError : function(projectId, id, shouldExists) {
		'use strict';
		// parse arguments
		projectId = this._parseProjectId(projectId);
		id = this._objHelper.initString(id);
		shouldExists = this._objHelper.initBoolean(shouldExists);
		// check identifier
		if ((shouldExists) && (!this.checkId(projectId, id))) {
			this._throwError('MODAL_WINDOW_HAS_NOT_BEEN_PREPARED', id);
			return;
		}
		if ((!shouldExists) && (this.checkId(projectId, id))) {
			this._throwError('MODAL_WINDOW_HAS_BEEN_PREPARED_ALREADY', id);
			return;
		}
	},

	/**
	 * Save prepared modal window attributes
	 *
	 * @private
	 * @param {string} projectId Project identifier
	 * @param {string} id Window identifier
	 * @param {Object} [attr] Window attributes; there are the following attributes available: "height" (int type; height of the window), "opacity" (float type; opacity of the window), "status" (string type; information in status bar); "title" (string type: title text); "type" (int type; window type from constants from kocujILV2aModalType); "width" (int type; with of the window)
	 * @return {void}
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.MODAL_WINDOW_WRONG_TYPE if window attributes entered in attr have wrong type
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.MODAL_WINDOW_WRONG_WIDTH if window attributes entered in attr have wrong width
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.MODAL_WINDOW_WRONG_HEIGHT if window attributes entered in attr have wrong height
	 * @throws {kocujILV2aCException} kocujILV2aExceptionCode.MODAL_WINDOW_WRONG_OPACITY if window attributes entered in attr have wrong opacity
	 */
	_savePreparedModalAttr : function(projectId, id, attr) {
		'use strict';
		// get this object
		var self = this;
		(function($) {
			// parse arguments
			attr = self._objHelper.initObject(attr);
			var type = kocujILV2aModalType.DEFAULT;
			var width = 600;
			var height = parseInt($(window).height(), 10)-200;
			if (height < 140) {
				height = 140;
			}
			var opacity = 1.0;
			var title = '';
			var status = '';
			if (attr !== '') {
				if ('type' in attr) {
					type = self._objHelper.initNumeric(attr.type);
					if ((type < 0) || (type > kocujILV2aModalType.LAST)) {
						self._throwError('MODAL_WINDOW_WRONG_TYPE', type);
						return;
					}
				}
				if ('width' in attr) {
					width = self._objHelper.initNumeric(attr.width);
					if (width <= 50) {
						self._throwError('MODAL_WINDOW_WRONG_WIDTH', width);
						return;
					}
				}
				if ('height' in attr) {
					height = self._objHelper.initNumeric(attr.height);
					if (height <= 50) {
						self._throwError('MODAL_WINDOW_WRONG_HEIGHT', height);
						return;
					}
				}
				if ('opacity' in attr) {
					opacity = self._objHelper.initNumericFloat(attr.opacity);
					if ((opacity <= 0.0) || (opacity > 1.0)) {
						self._throwError('MODAL_WINDOW_WRONG_OPACITY', opacity);
						return;
					}
				}
				if ('title' in attr) {
					title = self._objHelper.initString(attr.title);
				}
				if ('status' in attr) {
					status = self._objHelper.initString(attr.status);
				}
			}
			// save prepared window settings
			self._prj['prj_' + projectId].preparedData['win_' + id] = {
				type    : type,
				width   : width,
				height  : height,
				opacity : opacity,
				title   : title,
				status  : status
			};
		}(jQuery));
	}
};

// initialize
var kocujILV2aModal = new kocujILV2aCModal();
