<?php

/**
 * base.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2013-2016 Dominik Kocuj
 * @package kocuj_sitemap
 */

// set namespace
namespace KocujSitemapPlugin\Classes;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Base class
 *
 * @access public
 */
class Base {
	/**
	 * Singleton instance
	 *
	 * @access private
	 * @var object
	 */
	private static $instance = NULL;

	/**
	 * Plugin major version
	 *
	 * @access private
	 * @var int
	 */
	private $versionMajor = 2;

	/**
	 * Plugin minor version
	 *
	 * @access private
	 * @var int
	 */
	private $versionMinor = 2;

	/**
	 * Plugin revision version
	 *
	 * @access private
	 * @var int
	 */
	private $versionRevision = 2;

	/**
	 * Plugin URL
	 *
	 * @access private
	 * @var string
	 */
	private $pluginUrl = '';

	/**
	 * Plugin directory
	 *
	 * @access private
	 * @var string
	 */
	private $pluginDir = '';

	/**
	 * Plugin base name
	 *
	 * @access private
	 * @var string
	 */
	private $pluginBaseName = '';

	/**
	 * Plugin filename
	 *
	 * @access private
	 * @var string
	 */
	private $pluginFilename = '';

	/**
	 * Kocuj Internal Lib object instance
	 *
	 * @access private
	 * @var object
	 */
	private $kocujInternalLibObj = NULL;

	/**
	 * Kocuj.pl Lib object instance
	 *
	 * @access private
	 * @var object
	 */
	private $kocujPlLibObj = NULL;

	/**
	 * Constructor
	 *
	 * @access private
	 * @param string $pluginFile Plugin file path
	 * @return void
	 */
	private function __construct($pluginFile) {
		// get plugin URL
		$this->pluginUrl = plugins_url('', $pluginFile);
		// get plugin directory
		$this->pluginDir = dirname($pluginFile);
		// get plugin base name
		$this->pluginBaseName = plugin_basename($pluginFile);
		// get plugin filename
		$this->pluginFilename = $pluginFile;
		// add initialize
		add_action('plugins_loaded', array($this, 'init'), 10);
	}

	/**
	 * Disable cloning of object
	 *
	 * @access private
	 * @return void
	 */
	private function __clone() {
	}

	/**
	 * Get singleton instance
	 *
	 * @access public
	 * @param string $pluginFile Plugin file path; first execution of "getInstance()" method should set this to plugin file path, but all others executions can ommit this parameter - default: empty
	 * @return object Singleton instance
	 */
	public static function getInstance($pluginFile = '') {
		// optionally create new instance
		if (!self::$instance) {
			self::$instance = new self($pluginFile);
		}
		// exit
		return self::$instance;
	}

	/**
	 * Initialize
	 *
	 * @access public
	 * @return void
	 */
	public function init() {
		// load translations
		$domain = 'kocuj-sitemap';
		$locale = apply_filters('plugin_locale', get_locale(), $domain);
		load_textdomain($domain, WP_LANG_DIR.'/plugins/'.$domain.'-'.$locale.'.mo');
		load_plugin_textdomain($domain, false, dirname($this->pluginBaseName).'/languages');
		// initialize Kocuj Internal Lib
		include $this->pluginDir.'/libs/kocuj-internal-lib/kocuj-internal-lib.php';
		$this->kocujInternalLibObj = new \KocujInternalLib\V4a\Classes\Project(array(
			'all'      => array(
				'admin',
				'config',
				'js-ajax',
			),
			'frontend' => array(
				'admin',
			),
			'backend'  => array(
				'admin',
				'editor-text-buttons',
				'editor-visual-buttons',
				'installation-date',
				'license',
				'message',
				'page-about',
				'plugin-uninstall',
				'review-message',
				'settings-help',
				'settings-menu',
				'settings-meta-boxes',
				'version-check',
			),
		), array(
			'type'          => \KocujInternalLib\V4a\Enums\ProjectType::PLUGIN,
			'mainfilename'  => $this->pluginFilename,
			'licensename'   => 'GPL v2',
			'title'         => __('Sitemap', 'kocuj-sitemap'),
			'name'          => 'Kocuj Sitemap plugin',
			'version'       => $this->getVersion(),
			'url'           => 'http://kocujsitemap.wpplugin.kocuj.pl/',
			'titleoriginal' => 'Kocuj Sitemap',
		), array(
			'author'        => array(
				'name'  => 'Dominik Kocuj',
				'email' => 'dominik@kocuj.pl',
				'url'   => 'http://kocuj.pl/',
			),
			'support'       => array(
				'email' => 'dominik@kocuj.pl',
				'url'   => 'https://wordpress.org/support/plugin/kocuj-sitemap/',
			),
			'newschannels'  => array(
				'facebook' => 'http://www.facebook.com/kocujsitemap/',
				'rss'      => 'http://kocujsitemap.wpplugin.kocuj.pl/en/feed/',
			),
			'tellothers'    => array(
				'facebook' => 'http://www.facebook.com/sharer/sharer.php?u='.urlencode('http://kocujsitemap.wpplugin.kocuj.pl/'),
				'twitter'  => 'http://twitter.com/share?url='.urlencode('http://kocujsitemap.wpplugin.kocuj.pl/').'&text='.urlencode(__('Very good #wordpress plugin for creating a #sitemap with #multilingualism and #cache.', 'kocuj-sitemap')),
			),
		), array(
			'all'     => array(
				'admin'  => '\\KocujSitemapPlugin\\Classes\\KocujInternalLibStrings\\All\\Admin',
				'config' => '\\KocujSitemapPlugin\\Classes\\KocujInternalLibStrings\\All\\Config',
			),
			'backend' => array(
				'license'        => '\\KocujSitemapPlugin\\Classes\\KocujInternalLibStrings\\Backend\\License',
				'page-about'     => '\\KocujSitemapPlugin\\Classes\\KocujInternalLibStrings\\Backend\\PageAbout',
				'review-message' => '\\KocujSitemapPlugin\\Classes\\KocujInternalLibStrings\\Backend\\ReviewMessage',
				'settings-menu'  => '\\KocujSitemapPlugin\\Classes\\KocujInternalLibStrings\\Backend\\SettingsMenu',
				'version-check'  => '\\KocujSitemapPlugin\\Classes\\KocujInternalLibStrings\\Backend\\VersionCheck',
			),
		), array(
			'all' => array(
				'admin'  => '\\KocujSitemapPlugin\\Classes\\Admin',
				'config' => '\\KocujSitemapPlugin\\Classes\\Config',
			),
		));
		// initialize Kocuj.pl Lib
		include $this->pluginDir.'/libs/kocuj-pl-lib/kocuj-pl-lib.php';
		if ((is_admin()) || (is_network_admin())) {
			$this->kocujPlLibObj = new \KocujPlLib\V4a\Classes\Project($this->kocujInternalLibObj, array(
				'backend' => array(
					'add-thanks',
					'page-about-add-thanks',
					'review-message-add-thanks',
					'settings-adv-meta-boxes',
				),
			), array(
				'api'                => array(
					'url'      => 'http://api.kocuj.pl/',
					'login'    => 'kocujsitemapplugin',
					'password' => 'K0cmDi_98XifA',
				),
				'additionalwebsites' => array(
					'kocujsitemap.wpplugin.kocuj.pl',
				),
			), array(
				'backend' => array(
					'add-thanks'                => '\\KocujSitemapPlugin\\Classes\\KocujPlLibStrings\\Backend\\AddThanks',
					'add-thanks-ajax'           => '\\KocujSitemapPlugin\\Classes\\KocujPlLibStrings\\Backend\\AddThanksAjax',
					'settings-adv-meta-boxes'   => '\\KocujSitemapPlugin\\Classes\\KocujPlLibStrings\\Backend\\SettingsAdvMetaBoxes',
					'page-about-add-thanks'     => '\\KocujSitemapPlugin\\Classes\\KocujPlLibStrings\\Backend\\PageAboutAddThanks',
					'review-message-add-thanks' => '\\KocujSitemapPlugin\\Classes\\KocujPlLibStrings\\Backend\\ReviewMessageAddThanks',
				),
			), array(), array(
				'KocujInternalLib' => $this->kocujInternalLibObj,
			));
		}
		// initialize some classes
		Sitemap::getInstance();
		Widget::getInstance();
		// set backend
		if ((is_admin()) || (is_network_admin())) {
			// set plugin version
			$this->kocujInternalLibObj->getComponent('backend', 'version-check')->setCurrentVersion($this->getVersion());
			// update version data
			$versions = array( // for compatibility with older versions
				'do_not_check_for_1_x_x' => 'kocujsitemap_version', // for compatibility with older versions
				'do_not_check_for_2_1_0' => $this->kocujInternalLibObj->getMainSettingInternalName().'__version', // for compatibility with older versions
			); // for compatibility with older versions
			foreach ($versions as $versionFlag => $versionOptionName) { // for compatibility with older versions
				$value = $this->kocujInternalLibObj->getComponent('core', 'meta')->getOption($versionFlag); // for compatibility with older versions
				if ($value === false) { // for compatibility with older versions
					$value = get_option($versionOptionName); // for compatibility with older versions
					if ($value !== false) { // for compatibility with older versions
						$this->kocujInternalLibObj->getComponent('core', 'meta')->addOrUpdateOption($this->kocujInternalLibObj->getComponent('backend', 'version-check')->OPTION_NAME_VERSION, $value); // for compatibility with older versions
						delete_option($versionOptionName); // for compatibility with older versions
					} // for compatibility with older versions
					$this->kocujInternalLibObj->getComponent('core', 'meta')->addOrUpdateOption($versionFlag, '1'); // for compatibility with older versions
				} // for compatibility with older versions
			} // for compatibility with older versions
			// set plugin update callback
			$this->kocujInternalLibObj->getComponent('backend', 'version-check')->setUpdateCallback(array($this, 'updatePlugin'));
			// set review message
			$this->kocujInternalLibObj->getComponent('backend', 'review-message')->setDays(30);
			$this->kocujInternalLibObj->getComponent('backend', 'review-message')->setVoteUrl('http://wordpress.org/support/view/plugin-reviews/kocuj-sitemap');
			// show information about author's website
			$this->kocujPlLibObj->getComponent('backend', 'settings-adv-meta-boxes')->setAuthorInfo(array(
				'pl_PL',
			));
		}
	}

	/**
	 * Get major version
	 *
	 * @access public
	 * @return int Major version
	 */
	public function getVersionMajor() {
		// get major version
		return $this->versionMajor;
	}

	/**
	 * Get minor version
	 *
	 * @access public
	 * @return int Minor version
	 */
	public function getVersionMinor() {
		// get minor version
		return $this->versionMinor;
	}

	/**
	 * Get revision version
	 *
	 * @access public
	 * @return int Revision version
	 */
	public function getVersionRevision() {
		// get revision version
		return $this->versionRevision;
	}

	/**
	 * Get version
	 *
	 * @access public
	 * @return string Version
	 */
	public function getVersion() {
		// get version
		return $this->versionMajor.'.'.$this->versionMinor.'.'.$this->versionRevision;
	}

	/**
	 * Get plugin URL
	 *
	 * @access public
	 * @return string Plugin URL
	 */
	public function getPluginUrl() {
		// get plugin URL
		return $this->pluginUrl;
	}

	/**
	 * Get plugin directory
	 *
	 * @access public
	 * @return string Plugin directory
	 */
	public function getPluginDir() {
		// get plugin directory
		return $this->pluginDir;
	}

	/**
	 * Get plugin base name
	 *
	 * @access public
	 * @return string Plugin base name
	 */
	public function getPluginBaseName() {
		// get plugin base name
		return $this->pluginBaseName;
	}

	/**
	 * Get plugin filename
	 *
	 * @access public
	 * @return string Plugin filename
	 */
	public function getPluginFilename() {
		// get plugin filename
		return $this->pluginFilename;
	}

	/**
	 * Get Kocuj Internal Lib object
	 *
	 * @access public
	 * @return object Kocuj Internal Lib object
	 */
	public function getKocujInternalLibObj() {
		// get Kocuj Internal Lib object
		return $this->kocujInternalLibObj;
	}

	/**
	 * Get Kocuj.pl Lib object
	 *
	 * @access public
	 * @return object Kocuj.pl Lib object
	 */
	public function getKocujPlLibObj() {
		// get Kocuj.pl Lib object
		return $this->kocujPlLibObj;
	}

	/**
	 * Plugin activation
	 *
	 * @access public
	 * @return void
	 */
	public function activate() {
		// initialize
		$this->init();
		// add activation initialization
		add_action('init', array($this, 'initActivate'));
	}

	/**
	 * Plugin activation initialization
	 *
	 * @access public
	 * @return void
	 */
	public function initActivate() {
		// create cache
		try {
			Cache::getInstance()->createCache();
		} catch (Exception $e) {
		}
	}

	/**
	 * Plugin deactivation
	 *
	 * @access public
	 * @return void
	 */
	public function deactivate() {
	}

	/**
	 * Plugin update - call after plugin update and plugin activation
	 *
	 * @access public
	 * @param string $fromVersion Old plugin version
	 * @param string $toVersion New plugin version
	 * @return void
	 */
	public function updatePlugin($fromVersion, $toVersion) {
		// update from 1.x.x
		if (version_compare($fromVersion, '2.0.0', '<')) { // for compatibility with 1.x.x
			// remove cron schedule
			wp_clear_scheduled_hook('kocujsitemapcron'); // for compatibility with 1.x.x
			// remove database values for old plugin versions
			delete_option('kocujsitemap_2_0_0_get_ready'); // for compatibility with 1.x.x (from transition version 1.3.2 and newer 1.3.3)
			delete_option('kocujsitemap_plugin_data_version'); // for compatibility with 1.x.x
			delete_option('kocujsitemap_thanks_date'); // for compatibility with 1.x.x
			// update sending website address ("thanks") status
			$value = get_option('kocujsitemap_thanks'); // for compatibility with 1.x.x
			if ($value !== false) { // for compatibility with 1.x.x
				$this->kocujInternalLibObj->getComponent('core', 'meta')->addOrUpdateOption($this->kocujPlLibObj->getComponent('backend', 'add-thanks')->OPTION_NAME_THANKS_ADDED, '1'); // for compatibility with 1.x.x
				delete_option('kocujsitemap_thanks');  // for compatibility with 1.x.x
			} // for compatibility with 1.x.x
			// update old options
			$exist = $this->kocujInternalLibObj->getComponent('all', 'config')->checkOptionExists('Order'); // for compatibility with 1.x.x
			if ($exist) { // for compatibility with 1.x.x
				$this->kocujInternalLibObj->getComponent('all', 'config')->addOption('Order', '', 'string', '', false, array(), false, 0, false); // for compatibility with 1.x.x
				$value = $this->kocujInternalLibObj->getComponent('all', 'config')->getOption('Order'); // for compatibility with 1.x.x
				if (isset($value[0]) /* strlen($value) > 0 */ ) { // for compatibility with 1.x.x
					$def = str_split($value); // for compatibility with 1.x.x
					if (!in_array('M', $def)) { // for compatibility with 1.x.x
						array_unshift($def, 'M'); // for compatibility with 1.x.x
					} // for compatibility with 1.x.x
					$def[] = 'C'; // for compatibility with 1.x.x
					$def[] = 'A'; // for compatibility with 1.x.x
					$def[] = 'T'; // for compatibility with 1.x.x
					$this->kocujInternalLibObj->getComponent('all', 'config')->removeOption('Order'); // for compatibility with 1.x.x
					$this->kocujInternalLibObj->getComponent('all', 'config')->deleteOption('Order'); // for compatibility with 1.x.x
					$output = ''; // for compatibility with 1.x.x
					$this->kocujInternalLibObj->getComponent('all', 'config')->updateOption('OrderList', $def, $output); // for compatibility with 1.x.x
				} // for compatibility with 1.x.x
			} // for compatibility with 1.x.x
			// disable displaying sections if it is an update from 1.x.x
			$output = ''; // for compatibility with 1.x.x
			$this->kocujInternalLibObj->getComponent('all', 'config')->updateOption('DisplaySections', '0', $output); // for compatibility with 1.x.x
		} // for compatibility with 1.x.x
		// update from 2.0.x
		if (version_compare($fromVersion, '2.1.0', '<')) { // for compatibility with 2.0.x
			$value = $this->kocujInternalLibObj->getComponent('all', 'config')->getOption('Multilang'); // for compatibility with 2.0.x
			if ((isset($value[0]) /* strlen($value) > 0 */ ) && ($value !== '0')) { // for compatibility with 2.0.x
				$value = str_replace('____B_A_C_K_S_P_A_C_E____', '/', $value); // for compatibility with 2.0.x
				$output = ''; // for compatibility with 2.0.x
				$this->kocujInternalLibObj->getComponent('all', 'config')->updateOption('Multilang', $value, $output); // for compatibility with 2.0.x
			}
		} // for compatibility with 2.0.x
		if (version_compare($fromVersion, '2.2.1', '<')) { // for compatibility with 2.1.x
			$names = array( // for compatibility with 2.1.x
				$this->kocujInternalLibObj->getMainSettingInternalName().'__version' => $this->kocujInternalLibObj->getComponent('backend', 'version-check')->OPTION_NAME_VERSION, // for compatibility with 2.1.x
				$this->kocujInternalLibObj->getMainSettingInternalName().'__req_update_info' => $this->kocujInternalLibObj->getComponent('backend', 'version-check')->OPTION_NAME_REQ_UPDATE_INFO, // for compatibility with 2.1.x
				$this->kocujInternalLibObj->getMainSettingInternalName().'__installation_date' => $this->kocujInternalLibObj->getComponent('backend', 'installation-date')->OPTION_NAME_INSTALL_DATE, // for compatibility with 2.1.x
				$this->kocujInternalLibObj->getMainSettingInternalName().'__review_message_closed' => $this->kocujInternalLibObj->getComponent('backend', 'review-message')->OPTION_NAME_REVIEW_MSG_CLOSED, // for compatibility with 2.1.x
				$this->kocujInternalLibObj->getMainSettingInternalName().'__license_accept' => $this->kocujInternalLibObj->getComponent('backend', 'license')->OPTION_NAME_LICENSE_ACCEPT, // for compatibility with 2.1.x
				$this->kocujInternalLibObj->getMainSettingInternalName().'__thanks_added' => $this->kocujPlLibObj->getComponent('backend', 'add-thanks')->OPTION_NAME_THANKS_ADDED, // for compatibility with 2.1.x
			); // for compatibility with 2.1.x
			foreach ($names as $oldName => $newName) { // for compatibility with 2.1.x
				$value = \KocujInternalLib\V4a\Classes\Helper::getInstance()->getOptionForNetworkOrSite($oldName); // for compatibility with 2.1.x
				if ($value !== false) { // for compatibility with 2.1.x
					\KocujInternalLib\V4a\Classes\Helper::getInstance()->deleteOptionForNetworkOrSite($oldName); // for compatibility with 2.1.x
					$this->kocujInternalLibObj->getComponent('core', 'meta')->addOrUpdateOption($newName, $value); // for compatibility with 2.1.x
				} // for compatibility with 2.1.x
			} // for compatibility with 2.1.x
			$this->kocujInternalLibObj->getComponent('core', 'meta')->forceRealUpdateNow(); // for compatibility with 2.1.x
		} // for compatibility with 2.1.x
		// purge cache
		try {
			Cache::getInstance()->clearCache(true);
		} catch (Exception $e) {
		}
	}

	/**
	 * Get prefix for deprecated information; for compatibility with 1.x.x
	 *
	 * @access private
	 * @return string Prefix for deprecated information
	 */
	private function getOldInfoPrefix() {
		// exit
		/* translators: %s: Name of this plugin ("Kocuj Sitemap") */
		return sprintf(__('%s plugin', 'kocuj-sitemap'), 'Kocuj Sitemap'); // for compatibility with 1.x.x
	}
	
	/**
	 * Information about deprecated filter; for compatibility with 1.x.x
	 *
	 * @access public
	 * @param string $filterOld Old filter name
	 * @param string $filterNew New filter name which replace old filter - default: empty
	 * @param string $version Version from which filter has been deprecated - default: 2.0.0
	 * @return void
	 */
	public function filterOldInfo($filterOld, $filterNew = '', $version = '2.0.0') { // for compatibility with 1.x.x
		// check filter
		if (has_filter($filterOld)) { // for compatibility with 1.x.x
			// show information
			$newInfo = ''; // for compatibility with 1.x.x
			if (isset($filterNew[0]) /* strlen($filterNew) > 0 */ ) { // for compatibility with 1.x.x
				/* translators: %s: Filter name */
				$newInfo = sprintf(__('the "%s" filter', 'kocuj-sitemap'), $filterNew); // for compatibility with 1.x.x
			} // for compatibility with 1.x.x
			/* translators: %s: Filter name */
			_deprecated_function($this->getOldInfoPrefix().': '.sprintf(__('The "%s" filter', 'kocuj-sitemap'), $filterOld), $version, $newInfo); // for compatibility with 1.x.x
		} // for compatibility with 1.x.x
	} // for compatibility with 1.x.x
}
