<?php

/**
 * component.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujInternalLib\V4a\Classes\Project\Components\Backend\Admin;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Admin class
 *
 * @access public
 */
class Component extends \KocujInternalLib\V4a\Classes\ComponentObject {
	/**
	 * Random value used for unique simulated methods names and for some forms
	 *
	 * @access private
	 * @var int
	 */
	private $random = 0;

	/**
	 * Controllers list
	 *
	 * @access private
	 * @var array
	 */
	private $controller = array();

	/**
	 * External data callbacks
	 *
	 * @access private
	 * @var array
	 */
	private $externalData = array();

	/**
	 * Input helpers list
	 *
	 * @access private
	 * @var array
	 */
	private $inputHelper = array();

	/**
	 * Block helpers id list
	 *
	 * @access private
	 * @var array
	 */
	private $blockHelperId = array();

	/**
	 * Meta boxes containers list
	 *
	 * @access private
	 * @var array
	 */
	private $metaBox = array();

	/**
	 * Widgets settings containers list
	 *
	 * @access private
	 * @var array
	 */
	private $widgets = array();

	/**
	 * Form started (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $formStarted = false;

	/**
	 * Tabs containers list
	 *
	 * @access private
	 * @var array
	 */
	private $tabsContainers = array();

	/**
	 * Tabs list
	 *
	 * @access private
	 * @var array
	 */
	private $tabs = array();

	/**
	 * Tabs container started (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $tabsContainerStarted = false;

	/**
	 * Id of tabs container started
	 *
	 * @access private
	 * @var string
	 */
	private $tabsContainerStartedId = '';

	/**
	 * Tab started (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $tabStarted = false;

	/**
	 * Table started (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $tableStarted = false;

	/**
	 * It is widget settings (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $isWidget = false;

	/**
	 * Widgets has been added (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $widgetsAdded = false;

	/**
	 * Table with settings is empty (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $tableEmpty = false;

	/**
	 * Old block type
	 *
	 * @access private
	 * @var string
	 */
	private $oldBlockType = '';

	/**
	 * Plugin activation hook callback
	 *
	 * @access protected
	 * @var string|array
	 */
	protected $pluginActivationHook = array();

	/**
	 * Plugin deactivation hook callback
	 *
	 * @access protected
	 * @var string|array
	 */
	protected $pluginDeactivationHook = array();

	/**
	 * Nonce action
	 *
	 * @access protected
	 * @var string
	 */
	protected $nonceAction = '';

	/**
	 * Image for up arrow
	 *
	 * @access protected
	 * @var string
	 */
	protected $arrowUpImage = 'arrow-up.png';

	/**
	 * Width of image for up arrow
	 *
	 * @access protected
	 * @var int
	 */
	protected $arrowUpImageWidth = 0;

	/**
	 * Height of image for up arrow
	 *
	 * @access protected
	 * @var int
	 */
	protected $arrowUpImageHeight = 0;

	/**
	 * Image for down arrow
	 *
	 * @access protected
	 * @var string
	 */
	protected $arrowDownImage = 'arrow-down.png';

	/**
	 * Width of image for down arrow
	 *
	 * @access protected
	 * @var int
	 */
	protected $arrowDownImageWidth = 0;

	/**
	 * Height of image for down arrow
	 *
	 * @access protected
	 * @var int
	 */
	protected $arrowDownImageHeight = 0;

	/**
	 * Image for close button
	 *
	 * @access protected
	 * @var string
	 */
	protected $closeImage = 'close.png';

	/**
	 * Width of image for close button
	 *
	 * @access protected
	 * @var int
	 */
	protected $closeImageWidth = 0;

	/**
	 * Height of image for close button
	 *
	 * @access protected
	 * @var int
	 */
	protected $closeImageHeight = 0;

	/**
	 * "InitAfter()" method has been finished (true) or not (false)
	 *
	 * @access protected
	 * @var string
	 */
	private $initAfterFinished = false;

	/**
	 * Constructor
	 *
	 * @access public
	 * @param object $projectObj \KocujInternalLib\V4a\Classes\Project object for current project
	 * @return void
	 */
	public function __construct($projectObj) {
		// execute parent constructor
		parent::__construct($projectObj);
		// set random value
		$this->random = time().rand(1111, 9999);
		// set initialize actions
		add_action('init', array($this, 'init'), 1);
		if (is_multisite()) {
			add_action('network_admin_menu', array($this, 'initAfter'), 1);
		}
		add_action('admin_init', array($this, 'initAfter'), 1);
		// register plugin activation hook
		if ($this->getProjectObj()->getMainSettingType() === \KocujInternalLib\V4a\Enums\ProjectType::PLUGIN) {
			register_activation_hook($this->getProjectObj()->getMainSettingMainFilename(), array($this, 'activatePlugin'));
		}
		add_action('admin_enqueue_scripts', array($this, 'actionEnqueueLibScripts'), 10);
	}

	/**
	 * Get random value used for unique simulated methods names
	 *
	 * @access public
	 * @return int Random value used for unique simulated methods names
	 */
	public function getRandom() {
		// get random value used for unique simulated methods names
		return $this->random;
	}

	/**
	 * Get form started (true) or not (false)
	 *
	 * @access public
	 * @return bool Form started (true) or not (false)
	 */
	public function getFormStarted() {
		// get form started (true) or not (false)
		return $this->formStarted;
	}

	/**
	 * Get tabs container started (true) or not (false)
	 *
	 * @access public
	 * @return bool Tabs container started (true) or not (false)
	 */
	public function getTabsContainerStarted() {
		// get tabs container started (true) or not (false)
		return $this->tabsContainerStarted;
	}

	/**
	 * Get tab started (true) or not (false)
	 *
	 * @access public
	 * @return bool Tab started (true) or not (false)
	 */
	public function getTabStarted() {
		// get tab started (true) or not (false)
		return $this->tabStarted;
	}

	/**
	 * Get table started (true) or not (false)
	 *
	 * @access public
	 * @return bool Table started (true) or not (false)
	 */
	public function getTableStarted() {
		// get table started (true) or not (false)
		return $this->tableStarted;
	}

	/**
	 * Initialize
	 *
	 * @access public
	 * @return void
	 */
	public function init() {
		// initialize configuration
		$this->getProjectObj()->getComponent('all', 'config')->initAfterAddToAdmin();
		// add controllers
		if ((is_admin()) || (is_network_admin())) {
			$this->controller['save'] = array(
				array($this, 'controllerSave'),
				$this->getProjectObj()->getStringsObj('all', 'admin')->getString('INIT_SETTINGS_SAVED'),
			);
			$this->controller['saveexternal'] = array(
				array($this, 'controllerSaveExternal'),
				$this->getProjectObj()->getStringsObj('all', 'admin')->getString('INIT_DATA_SAVED'),
			);
			$this->controller['reset'] = array(
				array($this, 'controllerReset'),
				$this->getProjectObj()->getStringsObj('all', 'admin')->getString('INIT_SETTINGS_RESET'),
			);
			$this->controller['uninstall'] = array(
				array($this, 'controllerUninstall'),
				$this->getProjectObj()->getStringsObj('all', 'admin')->getString('INIT_SETTINGS_UNINSTALLED'),
			);
		}
		// add input helpers
		if ((is_admin()) || (is_network_admin())) {
			$this->inputHelper['hidden'] = array(
				array($this, 'inputHelperHidden'),
				NULL,
				true,
				false,
			);
			$this->inputHelper['text'] = array(
				array($this, 'inputHelperText'),
				NULL,
				true,
				true,
			);
			$this->inputHelper['textdisabled'] = array(
				array($this, 'inputHelperTextDisabled'),
				NULL,
				true,
				true,
			);
			$this->inputHelper['textarea'] = array(
				array($this, 'inputHelperTextarea'),
				NULL,
				true,
				true,
			);
			$this->inputHelper['checkbox'] = array(
				array($this, 'inputHelperCheckbox'),
				NULL,
				true,
				false,
			);
			$this->inputHelper['select'] = array(
				array($this, 'inputHelperSelect'),
				NULL,
				true,
				true,
			);
			$this->inputHelper['select_end'] = array(
				array($this, 'inputHelperSelectEnd'),
				NULL,
				true,
				true
			);
			$this->inputHelper['option'] = array(
				array($this, 'inputHelperOption'),
				NULL,
				true,
				true,
			);
			$this->inputHelper['selectmenu'] = array(
				array($this, 'inputHelperSelectmenu'),
				array($this, 'checkBlockHelperLoopSelectmenu'),
				true,
				true,
			);
			$this->inputHelper['selectcustomposts'] = array(
				array($this, 'inputHelperSelectcustomposts'),
				NULL,
				true,
				true,
			);
			$this->inputHelper['submit'] = array(
				array($this, 'inputHelperSubmit'),
				NULL,
				false,
				true,
			);
			$this->inputHelper['submitexternal'] = array(
				array($this, 'inputHelperSubmitExternal'),
				NULL,
				false,
				true,
			);
			$this->inputHelper['reset'] = array(
				array($this, 'inputHelperReset'),
				NULL,
				false,
				true,
			);
			$this->inputHelper['uninstall'] = array(
				array($this, 'inputHelperUninstall'),
				NULL,
				false,
				true,
			);
		}
		// add actions and filters
		if ((is_admin()) || (is_network_admin())) {
			// add admin header
			add_action('admin_head', array($this, 'addHeader'));
			// add admin menu
			add_action('current_screen', array($this, 'pageController'));
			// add meta boxes
			add_action('add_meta_boxes', array($this, 'filterAddMetaBox'), \KocujInternalLib\V4a\Classes\Helper::getInstance()->calculateMaxPriority('add_meta_boxes'));
			add_action('save_post', array($this, 'filterSaveMetaBox'), \KocujInternalLib\V4a\Classes\Helper::getInstance()->calculateMaxPriority('save_post'));
			// process output
			add_action('admin_head', array($this, 'processOutput'));
		}
		// add widgets settings
		if ((isset($_SERVER['REQUEST_URI'])) && (substr($_SERVER['REQUEST_URI'], -15, 15) === '/admin-ajax.php')) {
			add_action('check_ajax_referer', array($this, 'actionAddWidgetsSettings'), \KocujInternalLib\V4a\Classes\Helper::getInstance()->calculateMaxPriority('check_ajax_referer'));
		} else {
			$this->actionAddWidgetsSettings();
		}
	}

	/**
	 * Initialize - phase 2
	 *
	 * @access public
	 * @return void
	 */
	public function initAfter() {
		// check if execute this method
		if ($this->initAfterFinished) {
			return;
		}
		// set this method as finished
		$this->initAfterFinished = true;
		// check and correct settings
		if (!isset($this->nonceAction[0]) /* strlen($this->nonceAction) === 0 */ ) {
			$this->nonceAction = 'kocuj_internal_lib';
		}
		// register plugin deactivation hook
		if ($this->getProjectObj()->getMainSettingType() === \KocujInternalLib\V4a\Enums\ProjectType::PLUGIN) {
			register_deactivation_hook($this->getProjectObj()->getMainSettingMainFilename(), array($this, 'deactivatePlugin'));
		}
	}

	/**
	 * Action for adding JavaScript scripts and CSS styles
	 *
	 * @access public
	 * @return void
	 */
	public function actionEnqueueLibScripts() {
		// add styles and scripts
		if ($this->getProjectObj()->getComponent('backend', 'settings-menu')->checkCurrentPageIsSettingsForProject()) {
			wp_enqueue_style(\KocujInternalLib\V4a\Classes\Helper::getInstance()->getPrefix().'-jquery-ui', \KocujInternalLib\V4a\Classes\LibUrls::getInstance()->getCSSUrl().'/jquery-ui/jquery-ui.css', array(), \KocujInternalLib\V4a\Classes\Version::getInstance()->getVersion());
			wp_enqueue_script('jquery-ui-core');
			wp_enqueue_script('jquery-ui-tabs');
		}
	}

	/**
	 * Call handler
	 *
	 * @access public
	 * @param string $name Method name
	 * @param array $argument Method arguments
	 * @return any|void Value returned by called method
	 */
	public function __call($name, array $arguments) {
		// call handler
		$this->callHandler($name, $arguments);
	}

	/**
	 * Call handler helper method
	 *
	 * @access protected
	 * @param string $name Method name
	 * @param array $argument Method arguments
	 * @return any|void Value returned by called method
	 */
	protected function callHandler($name, array $arguments) {
		// get method name
		$div = explode('_', $name);
		if (isset($div[2]) /* count($div) > 2 */ ) {
			$div0 = $div[0];
			$div1 = $div;
			unset($div1[0]);
			$divEnd = implode('_', $div1);
			$div = array(
				$div0,
				$divEnd,
			);
		}
		$divCount = count($div);
		// methods for meta boxes
		if (($divCount === 2) && ($div[0] === 'metaBox'.$this->random) && (count($arguments) > 0)) {
			$this->showMetaBox($div[1], $arguments[0]);
		}
		// methods for widgets settings
		if (($divCount === 2) && ($div[0] === 'widget'.$this->random)) {
			$this->showWidgetSettings($div[1]);
		}
	}

	/**
	 * Plugin activation hook
	 *
	 * @access public
	 * @return void
	 */
	public function activatePlugin() {
		// call plugin activation hook callback
		if (!empty($this->pluginActivationHook)) {
			call_user_func($this->pluginActivationHook);
		}
	}

	/**
	 * Plugin deactivation hook
	 *
	 * @access public
	 * @return void
	 */
	public function deactivatePlugin() {
		// call plugin deactivation hook callback
		if (!empty($this->pluginDeactivationHook)) {
			call_user_func($this->pluginDeactivationHook);
		}
	}

	/**
	 * Add controller
	 *
	 * @access protected
	 * @param string $name Controller name
	 * @param string $function Callback method name - must be in this or child class
	 * @param string $text Text to display after executing controller
	 * @return void
	 */
	protected function addController($name, $function, $text) {
		// add controller
		$this->controller[$name] = array(
			array($this, $function),
			$text,
		);
	}

	/**
	 * Add external data callbacks
	 *
	 * @access protected
	 * @param string $name External data
	 * @param string $functionGetValue Callback method name for get data value - must be in this or child class
	 * @param string $functionCheckArray Callback method name for check if value is array - must be in this or child class
	 * @param string $functionDefaultValue Callback method name for default value - must be in this or child class
	 * @param string $label Option label
	 * @param array $arrayOption Array options - default: empty
	 * @param string $functionAddData Callback method name for add data - must be in this or child class - default: NULL
	 * @param string $functionUpdateData Callback method name for update data - must be in this or child class - default: NULL
	 * @param string $functionDeleteData Callback method name for delete data - must be in this or child class - default: NULL
	 * @return void
	 */
	protected function addExternalData($name, $functionGetValue, $functionCheckArray, $functionDefaultValue, $label, array $arrayOptions = array(), $functionAddData = NULL, $functionUpdateData = NULL, $functionDeleteData = NULL) {
		// add external data callbacks
		$this->externalData[$name] = array(
			array($this, $functionGetValue),
			array($this, $functionCheckArray),
			array($this, $functionDefaultValue),
			$label,
			$arrayOptions,
			((is_string($functionAddData)) && (isset($functionAddData[0]) /* strlen($functionAddData) > 0 */ )) ?
				array($this, $functionAddData) :
				NULL,
			((is_string($functionUpdateData)) && (isset($functionUpdateData[0]) /* strlen($functionUpdateData) > 0 */ )) ?
				array($this, $functionUpdateData) :
				NULL,
			((is_string($functionUpdateData)) && (isset($functionDeleteData[0]) /* strlen($functionDeleteData) > 0 */ )) ?
				array($this, $functionDeleteData) :
				NULL,
		);
	}

	/**
	 * Add input helper
	 *
	 * @access protected
	 * @param string $type Input helper type
	 * @param string $function Callback method name - must be in this or child class
	 * @param string $checkBlockHelperFunction Callback method name for checking block helper loop - must be in this or child class - default: NULL
	 * @param bool $allowInWidget Option is allowed to display in widget (true) or not (false) - default: false
	 * @param bool $labelBefore Label for option will be before option (true) or not (false) - default: true
	 * @return void
	 */
	protected function addInputHelper($type, $function, $checkBlockHelperLoopFunction = NULL, $allowInWidget = false, $labelBefore = true) {
		// add input helper
		$this->inputHelper[$type] = array(
			array($this, $function),
			((is_string($checkBlockHelperLoopFunction)) && (isset($checkBlockHelperLoopFunction[0]) /* strlen($checkBlockHelperLoopFunction) > 0 */ )) ?
				array($this, $checkBlockHelperLoopFunction) :
				NULL,
			$allowInWidget,
			$labelBefore,
		);
	}

	/**
	 * Add meta box
	 *
	 * @access protected
	 * @param string $id Meta box id
	 * @param string $title Meta box title
	 * @param array $places Places with this meta box, for example, "post" or "page"
	 * @param string $context Part of page with meta box, for example, "normal"
	 * @param string $priority Priority of meta box, for example, "default"
	 * @param string $functionShow Callback method name for show meta box - must be in this or child class
	 * @param string $functionUpdate Callback method name for update meta box - must be in this or child class
	 * @param array $metaKeys Meta keys to update - default: empty
	 * @param array $metaKeysCheckbox Meta keys with checkbox value to update - default: empty
	 * @return void
	 */
	protected function addMetaBox($id, $title, array $places, $context, $priority, $functionShow, $functionUpdate, array $metaKeys = array(),
		array $metaKeysCheckbox = array()) {
		// add meta box
		$this->metaBox[] = array(
			$id,
			$title,
			$places,
			$context,
			$priority,
			$id.'_'.md5($id),
			array($this, $functionShow),
			array($this, $functionUpdate),
			$metaKeys,
			$metaKeysCheckbox,
		);
	}

	/**
	 * Add widget settings
	 *
	 * @access protected
	 * @param string $id Widget id
	 * @param string $title Widget title
	 * @param string $functionShow Callback method name for show meta box - must be in this or child class
	 * @param string $functionUpdate Callback method name for update meta box - must be in this or child class
	 * @return void
	 */
	protected function addWidget($id, $title, $functionShow, $functionUpdate) {
		// add widget
		$this->widgets[] = array(
			$id,
			$title,
			array($this, $functionShow),
			array($this, $functionUpdate),
			$id.'_'.md5($id),
		);
	}

	/**
	 * Register tabs container
	 *
	 * @access protected
	 * @param string $name Container name
	 * @param int &$containerId Returned container id
	 * @return bool Status - true or false
	 */
	protected function registerTabsContainer($name, &$containerId) {
		// check if container with the selected name does not exists
		if (in_array($name, $this->tabsContainers)) {
			return false;
		}
		// add container
		$containerId = count($this->tabsContainers);
		$this->tabsContainers[$containerId] = $name;
		// show initialization script
		?>
			<script type="text/javascript">
			/* <![CDATA[ */
				(function($) {
					$(function() {
						$('<?php echo esc_js('#'.$name.'_tabs'); ?>').tabs();
					});
				}(jQuery));
			/* ]]> */
			</script>
		<?php
		// exit
		return true;
	}

	/**
	 * Register tab
	 *
	 * @access protected
	 * @param int $containerId Container id
	 * @param string $title Tab title
	 * @param int &$tabId Returned tab id
	 * @return bool Status - true or false
	 */
	protected function registerTab($containerId, $title, &$tabId) {
		// check if container exists
		if ((!isset($this->tabsContainers[$containerId])) || (!isset($title[0]) /* strlen($title) === 0 */ )) {
			return false;
		}
		// add tab
		$tabId = count($this->tabs);
		$this->tabs[$tabId] = array(
			'CONTAINERID' => $containerId,
			'TITLE'       => $title,
		);
		// exit
		return true;
	}

	/**
	 * Show form start
	 *
	 * @access protected
	 * @return bool Status - true or false
	 */
	protected function showFormStart() {
		// show form start
		$this->getProjectObj()->getComponent('core', 'project-helper')->doAction('before_form_div');
		echo '<div'.$this->getProjectObj()->getComponent('core', 'project-helper')->applyFiltersForHTMLStyleAndClass('form_div').' id="kocujinternallib_div_form">';
		$this->getProjectObj()->getComponent('core', 'project-helper')->doAction('before_form');
		echo '<form method="post" action="#" name="'.$this->getProjectObj()->getMainSettingInternalName().'" id="'.$this->getProjectObj()->getMainSettingInternalName().'">';
		$this->getProjectObj()->getComponent('core', 'project-helper')->doAction('inside_form_begin');
		// set nonce
		wp_nonce_field($this->nonceAction, $this->getProjectObj()->getMainSettingInternalName().'_'.md5($this->getProjectObj()->getMainSettingInternalName()));
		// set form to start
		$this->formStarted = true;
		// exit
		return true;
	}

	/**
	 * Show form end
	 *
	 * @access protected
	 * @param string $forceAction Force this action if it is not empty - default: empty
	 * @param string $forceExternalVar Force this external variable if it is not empty - default: empty
	 * @return bool Status - true or false
	 */
	protected function showFormEnd($forceAction = '', $forceExternalVar = '') {
		// check if form has started
		if (!$this->formStarted) {
			return false;
		}
		// show action and external data field
		echo '<input type="hidden" name="action" id="action" value="'.esc_attr((isset($forceAction[0]) /* strlen($forceAction) > 0 */ ) ?
				$forceAction :
				''
			).'" />';
		echo '<input type="hidden" name="externalvar" id="externalvar" value="'.esc_attr((isset($forceExternalVar[0]) /* strlen($forceExternalVar) > 0 */ ) ?
				$forceExternalVar :
				''
			).'" />';
		// show form end
		$this->getProjectObj()->getComponent('core', 'project-helper')->doAction('inside_form_end');
		echo '</form>';
		$this->getProjectObj()->getComponent('core', 'project-helper')->doAction('after_form');
		echo '</div>';
		$this->getProjectObj()->getComponent('core', 'project-helper')->doAction('after_form_div');
		// set form to end
		$this->formStarted = false;
		// exit
		return true;
	}

	/**
	 * Show tabs container start
	 *
	 * @access protected
	 * @param int $containerId Container id
	 * @return bool Status - true or false
	 */
	protected function showTabsContainerStart($containerId) {
		// check if container exists
		if (!isset($this->tabsContainers[$containerId])) {
			return false;
		}
		// show container start
		echo '<div id="'.esc_attr($this->tabsContainers[$containerId].'_tabs').'">';
		// show tabs
		$ul = false;
		if (!empty($this->tabs)) {
			foreach ($this->tabs as $key => $tab) {
				if ($tab['CONTAINERID'] === $containerId) {
					if (!$ul) {
						echo '<ul>';
						$ul = true;
					}
					echo '<li>';
					echo \KocujInternalLib\V4a\Classes\HtmlHelper::getInstance()->getLinkAnchor('#'.$this->tabsContainers[$containerId].'-'.$key, $tab['TITLE']);
					echo '</li>';
				}
			}
		}
		if ($ul) {
			echo '</ul>';
		}
		// set tabs container to start
		$this->tabsContainerStarted = true;
		$this->tabsContainerStartedId = $this->tabsContainers[$containerId].'_tabs';
		// exit
		return true;
	}

	/**
	 * Show tabs container end
	 *
	 * @access protected
	 * @return bool Status - true or false
	 */
	protected function showTabsContainerEnd() {
		// check if container has started
		if (!$this->tabsContainerStarted) {
			return false;
		}
		// show container end
		echo '</div>';
		// set tabs container to end
		$this->tabsContainerStarted = false;
		$this->tabsContainerStartedId = '';
		// exit
		return true;
	}

	/**
	 * Show tab start
	 *
	 * @access protected
	 * @param int $tabId Tab id
	 * @return bool Status - true or false
	 */
	protected function showTabStart($tabId) {
		// check if tab exists
		if (!isset($this->tabs[$tabId])) {
			return false;
		}
		// check if tab has not started
		if ($this->tabStarted) {
			return false;
		}
		// show tab start
		echo '<div id="'.$this->tabsContainers[$this->tabs[$tabId]['CONTAINERID']].'-'.$tabId.'">';
		// set tab to start
		$this->tabStarted = true;
		// exit
		return true;
	}

	/**
	 * Show tab end
	 *
	 * @access protected
	 * @return bool Status - true or false
	 */
	protected function showTabEnd() {
		// check if tab has started
		if (!$this->tabStarted) {
			return false;
		}
		// show tab end
		echo '</div>';
		// set tab to end
		$this->tabStarted = false;
		// exit
		return true;
	}

	/**
	 * Get table start
	 *
	 * @access public
	 * @param string $id Table id
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @return string Table start
	 */
	public function getTableStart($id, $isWidget = false) {
		// check if table has not started
		if ($this->tableStarted) {
			return '';
		}
		// set if it is widget
		$this->isWidget = $isWidget;
		// set that table is empty
		$this->tableEmpty = true;
		// get table start
		$output = '';
		if ($isWidget) {
			$output .= '<p>';
		} else {
			if (!isset($id[0]) /* strlen($id) === 0 */ ) {
				$id = time().rand(1111, 9999);
			}
			$output .= '<table id="'.esc_attr('admin_form_'.$id).'" class="form-table kocujinternallib-form-table"><tbody>';
			$output .= '<tr style="height:1px;">';
			$output .= '<td class="kocujinternallib-label" style="width:300px;height:1px;line-height:1px;margin:0;padding:0;">&nbsp;</td>';
			$output .= '<td class="kocujinternallib-content" style="height:1px;line-height:1px;margin:0;padding:0;">&nbsp;</td>';
			$output .= '</tr>';
		}
		// set tab to end
		$this->tableStarted = true;
		// exit
		return $output;
	}

	/**
	 * Show table start
	 *
	 * @access public
	 * @param string $id Table id
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @return void
	 */
	public function showTableStart($id, $isWidget = false) {
		// show table start
		echo $this->getTableStart($id, $isWidget);
	}

	/**
	 * Get table end
	 *
	 * @access public
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @return string Table end
	 */
	public function getTableEnd($isWidget = false) {
		// check if table has started
		if (!$this->tableStarted) {
			return '';
		}
		// clear if it is widget
		$this->isWidget = false;
		// clear that table is empty
		$this->tableEmpty = false;
		// show table end
		$output = ($isWidget) ?
			'</p>' :
			'</tbody></table>';
		// set tab to end
		$this->tableStarted = false;
		// exit
		return $output;
	}

	/**
	 * Show table end
	 *
	 * @access public
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @return void
	 */
	public function showTableEnd($isWidget = false) {
		// show table end
		echo $this->getTableEnd($isWidget);
	}

	/**
	 * Get input helper
	 *
	 * @access public
	 * @param string $var Field name
	 * @param string $type Field type
	 * @param string $description Field description - default: empty
	 * @param string $value Field value - default: empty
	 * @param string $selected Selected field - default: empty
	 * @param bool $checked Checked field - default: false
	 * @param string $tooltip Tooltip text - default: empty
	 * @param bool $isArray Option is array - default: false
	 * @param string $eventOnChange Event onChange - default: empty
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
	 * @param string $forcedName Forced name if $isForcedName is true
	 * @return string Input helper
	 */
	public function getInputHelper($var, $type, $description = '', $value = '', $selected = '', $checked = false, $tooltip = '',
		$isArray = false, $eventOnChange = '', $isWidget = false, $isForcedName = false, $forcedName = '') {
		// check parameters
		if (!isset($this->inputHelper[$type])) {
			return '';
		}
		// optionally divide tooltip
		if (isset($tooltip[0]) /* strlen($tooltip) > 0 */ ) {
			$tooltip = str_replace('"', '`', $tooltip);
		}
		// show input field
		$tip = (isset($tooltip[0]) /* strlen($tooltip) > 0 */ ) ?
			'title="'.esc_attr($tooltip).'"' :
			'';
		// call method
		$output = PHP_EOL.call_user_func_array($this->inputHelper[$type][0], array(
			$var,
			$description,
			stripslashes($value),
			$selected,
			$checked,
			$tip,
			$isArray,
			$eventOnChange,
			$isWidget,
			$isForcedName,
			$forcedName,
		));
		// exit
		return $output;
	}

	/**
	 * Show input helper
	 *
	 * @access public
	 * @param string $var Field name
	 * @param string $type Field type
	 * @param string $description Field description - default: empty
	 * @param string $value Field value - default: empty
	 * @param string $selected Selected field - default: empty
	 * @param bool $checked Checked field - default: false
	 * @param string $tooltip Tooltip text - default: empty
	 * @param bool $isArray Option is array - default: false
	 * @param string $eventOnChange Event onChange - default: empty
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
	 * @param string $forcedName Forced name if $isForcedName is true
	 * @return void
	 */
	public function inputHelper($var, $type, $description = '', $value = '', $selected = '', $checked = false, $tooltip = '',
		$isArray = false, $eventOnChange = '', $isWidget = false, $isForcedName = false, $forcedName = '') {
		// show input helper
		echo $this->getInputHelper($var, $type, $description, $value, $selected, $checked, $tooltip, $isArray, $eventOnChange, $isWidget, $isForcedName, $forcedName);
	}

	/**
	 * Get input helper - hidden
	 *
	 * @access private
	 * @param string $var Field name
	 * @param string $description Field description - default: empty
	 * @param string $value Field value - default: empty
	 * @param string $selected Selected field - default: empty
	 * @param bool $checked Checked field - default: false
	 * @param string $tip Event for mouseover displaying tooltip - default: empty
	 * @param bool $isArray Option is array - default: false
	 * @param string $eventOnChange Event onChange - default: empty
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
	 * @param string $forcedName Forced name if $isForcedName is true
	 * @return string Input helper
	 */
	private function inputHelperHidden($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
		$isArray = false, $eventOnChange = '', $isWidget = false, $isForcedName = false, $forcedName = '') {
		// prepare onChange event
		$change = (isset($eventOnChange[0]) /* strlen($eventOnChange) > 0 */ ) ?
			'onchange="'.esc_attr($eventOnChange).'"' :
			'';
		// prepare id
		$idString = (strpos($var, '[]') !== false) ?
			'' :
			' id="'.esc_attr($var).'"';
		// prepare name
		$name = ($isForcedName) ?
			$forcedName :
			$var;
		// get input helper
		return '<input name="'.esc_attr($name).'"'.$idString.' type="hidden" value="'.esc_attr($value).'" '.$change.' />';
	}

	/**
	 * Get input helper - text
	 *
	 * @access private
	 * @param string $var Field name
	 * @param string $description Field description - default: empty
	 * @param string $value Field value - default: empty
	 * @param string $selected Selected field - default: empty
	 * @param bool $checked Checked field - default: false
	 * @param string $tip Event for mouseover displaying tooltip - default: empty
	 * @param bool $isArray Option is array - default: false
	 * @param string $eventOnChange Event onChange - default: empty
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
	 * @param string $forcedName Forced name if $isForcedName is true
	 * @return string Input helper
	 */
	private function inputHelperText($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
		$isArray = false, $eventOnChange = '', $isWidget = false, $isForcedName = false, $forcedName = '') {
		// prepare onChange event
		$change = (isset($eventOnChange[0]) /* strlen($eventOnChange) > 0 */ ) ?
			'onchange="'.esc_attr($eventOnChange).'"' :
			'';
		// prepare id
		$idString = (strpos($var, '[]') !== false) ?
			'' :
			' id="'.esc_attr($var).'"';
		// prepare additional styles
		$additional = ($isWidget) ?
			' class="widefat"' :
			' style="width:300px;"';
		// prepare name
		$name = ($isForcedName) ?
			$forcedName :
			$var;
		// get input helper
		return '<input name="'.esc_attr($name).'"'.$idString.' type="text" value="'.esc_attr($value).'"'.$additional.' '.$tip.' '.$change.' />';
	}

	/**
	 * Get input helper - text disabled
	 *
	 * @access private
	 * @param string $var Field name
	 * @param string $description Field description - default: empty
	 * @param string $value Field value - default: empty
	 * @param string $selected Selected field - default: empty
	 * @param bool $checked Checked field - default: false
	 * @param string $tip Event for mouseover displaying tooltip - default: empty
	 * @param bool $isArray Option is array - default: false
	 * @param string $eventOnChange Event onChange - default: empty
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
	 * @param string $forcedName Forced name if $isForcedName is true
	 * @return string Input helper
	 */
	private function inputHelperTextDisabled($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
		$isArray = false, $eventOnChange = '', $isWidget = false, $isForcedName = false, $forcedName = '') {
		// prepare id
		$idString = (strpos($var, '[]') !== false) ?
			'' :
			' id="'.esc_attr($var).'"';
		// prepare additional styles
		$additional = ($isWidget) ?
			' style="display:inline;position:absolute;" class="widefat"' :
			' style="width:300px;display:inline;position:absolute;"';
		// prepare name
		$name = ($isForcedName) ?
			$forcedName :
			$var;
		// get input helper
		return '<input name="'.esc_attr($name).'"'.$idString.' type="hidden" value="'.esc_attr($value).'" style="width:1px;height:1px;margin:0;padding:0;position:absolute;" /><div'.$additional.'>'.$value.'</div>';
	}

	/**
	 * Get input helper - textarea
	 *
	 * @access private
	 * @param string $var Field name
	 * @param string $description Field description - default: empty
	 * @param string $value Field value - default: empty
	 * @param string $selected Selected field - default: empty
	 * @param bool $checked Checked field - default: false
	 * @param string $tip Event for mouseover displaying tooltip - default: empty
	 * @param bool $isArray Option is array - default: false
	 * @param string $eventOnChange Event onChange - default: empty
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
	 * @param string $forcedName Forced name if $isForcedName is true
	 * @return string Input helper
	 */
	private function inputHelperTextarea($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
		$isArray = false, $eventOnChange = '', $isWidget = false, $isForcedName = false, $forcedName = '') {
		// prepare onChange event
		$change = (isset($eventOnChange[0]) /* strlen($eventOnChange) > 0 */ ) ?
			'onchange="'.esc_attr($eventOnChange).'"' :
			'';
		// prepare id
		$idString = (strpos($var, '[]') !== false) ?
			'' :
			' id="'.esc_attr($var).'"';
		// prepare additional styles
		$additional = ($isWidget) ?
			' style="resize:none;" class="widefat"' :
			' style="width:300px;resize:none;"';
		// prepare name
		$name = ($isForcedName) ?
			$forcedName :
			$var;
		// get input helper
		return '<textarea name="'.esc_attr($name).'"'.$idString.' rows="15" cols=""'.$additional.' '.$tip.' '.$change.'>'.htmlspecialchars($value).'</textarea>';
	}

	/**
	 * Get input helper - checkbox
	 *
	 * @access private
	 * @param string $var Field name
	 * @param string $description Field description - default: empty
	 * @param string $value Field value - default: empty
	 * @param string $selected Selected field - default: empty
	 * @param bool $checked Checked field - default: false
	 * @param string $tip Event for mouseover displaying tooltip - default: empty
	 * @param bool $isArray Option is array - default: false
	 * @param string $eventOnChange Event onChange - default: empty
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
	 * @param string $forcedName Forced name if $isForcedName is true
	 * @return string Input helper
	 */
	private function inputHelperCheckbox($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
		$isArray = false, $eventOnChange = '', $isWidget = false, $isForcedName = false, $forcedName = '') {
		// prepare onChange event
		$change = (isset($eventOnChange[0]) /* strlen($eventOnChange) > 0 */ ) ?
			'onchange="'.esc_attr($eventOnChange).'"' :
			'';
		// prepare checked
		$extra = ($checked) ?
			'checked="checked"' :
			'';
		// prepare id
		$idString = (strpos($var, '[]') !== false) ?
			'' :
			' id="'.esc_attr($var).'"';
		// prepare additional styles
		$additional = ($isWidget) ?
			' class="checkbox"' :
			' style="border-width:0;"';
		// prepare name
		$name = ($isForcedName) ?
			$forcedName :
			$var;
		// get input helper
		return '<input type="hidden" name="'.esc_attr($name).'" value="0" /><input name="'.esc_attr($name).'"'.$idString.' type="checkbox" value="'.esc_attr($value).'"'.$additional.' '.$extra.' '.$tip.' '.$change.' />';
	}

	/**
	 * Get input helper - select
	 *
	 * @access private
	 * @param string $var Field name
	 * @param string $description Field description - default: empty
	 * @param string $value Field value - default: empty
	 * @param string $selected Selected field - default: empty
	 * @param bool $checked Checked field - default: false
	 * @param string $tip Event for mouseover displaying tooltip - default: empty
	 * @param bool $isArray Option is array - default: false
	 * @param string $eventOnChange Event onChange - default: empty
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
	 * @param string $forcedName Forced name if $isForcedName is true
	 * @return string Input helper
	 */
	private function inputHelperSelect($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
		$isArray = false, $eventOnChange = '', $isWidget = false, $isForcedName = false, $forcedName = '') {
		// prepare onChange event
		$change = (isset($eventOnChange[0]) /* strlen($eventOnChange) > 0 */ ) ?
			'onchange="'.esc_attr($eventOnChange).'"' :
			'';
		// prepare id
		$idString = (strpos($var, '[]') !== false) ?
			'' :
			' id="'.esc_attr($var).'"';
		// prepare additional styles
		$additional = ($isWidget) ?
			' class="widefat"' :
			' style="width:300px;"';
		// prepare name
		$name = ($isForcedName) ?
			$forcedName :
			$var;
		// get input helper
		return '<select name="'.esc_attr($name).'"'.$idString.$additional.' '.$tip.' '.$change.' >';
	}

	/**
	 * Get input helper - select_end
	 *
	 * @access private
	 * @param string $var Field name
	 * @param string $description Field description - default: empty
	 * @param string $value Field value - default: empty
	 * @param string $selected Selected field - default: empty
	 * @param bool $checked Checked field - default: false
	 * @param string $tip Event for mouseover displaying tooltip - default: empty
	 * @param bool $isArray Option is array - default: false
	 * @param string $eventOnChange Event onChange - default: empty
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
	 * @param string $forcedName Forced name if $isForcedName is true
	 * @return string Input helper
	 */
	private function inputHelperSelectEnd($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
		$isArray = false, $eventOnChange = '', $isWidget = false, $isForcedName = false, $forcedName = '') {
		// get input helper
		return '</select>';
	}

	/**
	 * Get input helper - option
	 *
	 * @access private
	 * @param string $var Field name
	 * @param string $description Field description - default: empty
	 * @param string $value Field value - default: empty
	 * @param string $selected Selected field - default: empty
	 * @param bool $checked Checked field - default: false
	 * @param string $tip Event for mouseover displaying tooltip - default: empty
	 * @param bool $isArray Option is array - default: false
	 * @param string $eventOnChange Event onChange - default: empty
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
	 * @param string $forcedName Forced name if $isForcedName is true
	 * @return string Input helper
	 */
	private function inputHelperOption($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
		$isArray = false, $eventOnChange = '', $isWidget = false, $isForcedName = false, $forcedName = '') {
		// get input helper
		return '<option value="'.esc_attr($value).'" '.(((string)$selected === (string)$value) ?
				'selected="selected"' :
				''
			).' >'.$description.'</option>';
	}

	/**
	 * Get input helper - selectmenu
	 *
	 * @access private
	 * @param string $var Field name
	 * @param string $description Field description - default: empty
	 * @param string $value Field value - default: empty
	 * @param string $selected Selected field - default: empty
	 * @param bool $checked Checked field - default: false
	 * @param string $tip Event for mouseover displaying tooltip - default: empty
	 * @param bool $isArray Option is array - default: false
	 * @param string $eventOnChange Event onChange - default: empty
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
	 * @param string $forcedName Forced name if $isForcedName is true
	 * @return string Input helper
	 */
	private function inputHelperSelectmenu($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
		$isArray = false, $eventOnChange = '', $isWidget = false, $isForcedName = false, $forcedName = '') {
		// get menus
		$menus = get_terms('nav_menu');
		// get input helper
		$output = $this->inputHelperSelect($var, $description, $value, $selected, $checked, $tip, false, $eventOnChange, $isWidget, $isForcedName, $forcedName);
		if ($isArray) {
			$output .= $this->inputHelperOption($var, '----', '', $selected, $checked, $tip, false, $eventOnChange, $isWidget, $isForcedName, $forcedName);
		}
		if (!empty($menus)) {
			foreach ($menus as $menu) {
				$output .= $this->inputHelperOption($var, $menu->name, $menu->term_id, $selected, $checked, $tip, false, $eventOnChange, $isWidget, $isForcedName, $forcedName);
			}
		}
		$output .= $this->inputHelperSelectEnd($var, $description, $value, $selected, $checked, $tip, false, $eventOnChange, $isWidget, $isForcedName, $forcedName);
		// exit
		return $output;
	}

	/**
	 * Get input helper - selectcustomposts
	 *
	 * @access private
	 * @param string $var Field name
	 * @param string $description Field description - default: empty
	 * @param string $value Field value - default: empty
	 * @param string $selected Selected field - default: empty
	 * @param bool $checked Checked field - default: false
	 * @param string $tip Event for mouseover displaying tooltip - default: empty
	 * @param bool $isArray Option is array - default: false
	 * @param string $eventOnChange Event onChange - default: empty
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
	 * @param string $forcedName Forced name if $isForcedName is true
	 * @return string Input helper
	 */
	private function inputHelperSelectcustomposts($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
		$isArray = false, $eventOnChange = '', $isWidget = false, $isForcedName = false, $forcedName = '') {
		// get custom posts types
		$types = get_post_types(array(
			'public'   => true,
			'_builtin' => false,
		), 'object');
		// get input helper
		$output = $this->inputHelperSelect($var, $description, $value, $selected, $checked, $tip, false, $eventOnChange, $isWidget, $isForcedName, $forcedName);
		if ($isArray) {
			$output .= $this->inputHelperOption($var, '----', '', $selected, $checked, $tip, false, $eventOnChange, $isWidget, $isForcedName, $forcedName);
		}
		if (!empty($types)) {
			foreach ($types as $type => $val) {
				$output .= $this->inputHelperOption($var, $val->labels->name, $type, $selected, $checked, $tip, false, $eventOnChange, $isWidget, $isForcedName, $forcedName);
			}
		}
		$output .= $this->inputHelperSelectEnd($var, $description, $value, $selected, $checked, $tip, false, $eventOnChange, $isWidget, $isForcedName, $forcedName);
		// exit
		return $output;
	}

	/**
	 * Get input helper - submit
	 *
	 * @access private
	 * @param string $var Field name
	 * @param string $description Field description - default: empty
	 * @param string $value Field value - default: empty
	 * @param string $selected Selected field - default: empty
	 * @param bool $checked Checked field - default: false
	 * @param string $tip Event for mouseover displaying tooltip - default: empty
	 * @param bool $isArray Option is array - default: false
	 * @param string $eventOnChange Event onChange - default: empty
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
	 * @param string $forcedName Forced name if $isForcedName is true
	 * @return string Input helper
	 */
	private function inputHelperSubmit($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
		$isArray = false, $eventOnChange = '', $isWidget = false, $isForcedName = false, $forcedName = '') {
		// prepare name
		$name = ($isForcedName) ?
			$forcedName :
			$var;
		// exit
		return '<input name="'.esc_attr($name).'" id="'.esc_attr($var).'" type="button" class="button-primary" value="'.esc_attr($value).'" '.$tip.' />'.
			'<script type="text/javascript">'.PHP_EOL.
			'/* <![CDATA[ */'.PHP_EOL.
			'(function($) {'.PHP_EOL.
			'$(document).ready(function() {'.PHP_EOL.
			'$(\''.esc_js('#'.$var).'\').click(function(event) {'.PHP_EOL.
			'event.preventDefault();'.PHP_EOL.
			'$(\'#action\').val(\'save\');'.PHP_EOL.
			'$(\''.esc_js('#'.$this->getProjectObj()->getMainSettingInternalName()).'\').submit();'.PHP_EOL.
			'});'.PHP_EOL.
			'});'.PHP_EOL.
			'}(jQuery));'.PHP_EOL.
			'/* ]]> */'.PHP_EOL.
			'</script>'.PHP_EOL;
	}

	/**
	 * Get input helper - submit external
	 *
	 * @access private
	 * @param string $var Field name
	 * @param string $description Field description - default: empty
	 * @param string $value Field value - default: empty
	 * @param string $selected Selected field - default: empty
	 * @param bool $checked Checked field - default: false
	 * @param string $tip Event for mouseover displaying tooltip - default: empty
	 * @param bool $isArray Option is array - default: false
	 * @param string $eventOnChange Event onChange - default: empty
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
	 * @param string $forcedName Forced name if $isForcedName is true
	 * @return string Input helper
	 */
	private function inputHelperSubmitExternal($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
		$isArray = false, $eventOnChange = '', $isWidget = false, $isForcedName = false, $forcedName = '') {
		// prepare name
		$name = ($isForcedName) ?
			$forcedName :
			$var;
		// exit
		return '<input name="'.esc_attr($name).'_submit" id="'.esc_attr($var.'_submit').'" type="button" class="button-primary" value="'.esc_attr($value).'" '.$tip.' />'.
			'<script type="text/javascript">'.PHP_EOL.
			'/* <![CDATA[ */'.PHP_EOL.
			'(function($) {'.PHP_EOL.
			'$(document).ready(function() {'.PHP_EOL.
			'$(\''.esc_js('#'.$var.'_submit').'\').click(function(event) {'.PHP_EOL.
			'event.preventDefault();'.PHP_EOL.
			'$(\'#action\').val(\'saveexternal\');'.PHP_EOL.
			'$(\'#externalvar\').val(\''.esc_js($var).'\');'.PHP_EOL.
			'$(\''.esc_js('#'.$this->getProjectObj()->getMainSettingInternalName()).'\').submit();'.PHP_EOL.
			'});'.PHP_EOL.
			'});'.PHP_EOL.
			'}(jQuery));'.PHP_EOL.
			'/* ]]> */'.PHP_EOL.
			'</script>'.PHP_EOL;
	}

	/**
	 * Get input helper - reset
	 *
	 * @access private
	 * @param string $var Field name
	 * @param string $description Field description - default: empty
	 * @param string $value Field value - default: empty
	 * @param string $selected Selected field - default: empty
	 * @param bool $checked Checked field - default: false
	 * @param string $tip Event for mouseover displaying tooltip - default: empty
	 * @param bool $isArray Option is array - default: false
	 * @param string $eventOnChange Event onChange - default: empty
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
	 * @param string $forcedName Forced name if $isForcedName is true
	 * @return string Input helper
	 */
	private function inputHelperReset($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
		$isArray = false, $eventOnChange = '', $isWidget = false, $isForcedName = false, $forcedName = '') {
		// prepare name
		$name = ($isForcedName) ?
			$forcedName :
			$var;
		// exit
		return '<input name="'.esc_attr($name).'" id="'.esc_attr($var).'" type="button" class="button" value="'.esc_attr($value).'" '.$tip.' />'.
			'<script type="text/javascript">'.PHP_EOL.
			'/* <![CDATA[ */'.PHP_EOL.
			'(function($) {'.PHP_EOL.
			'$(document).ready(function() {'.PHP_EOL.
			'$(\''.esc_js('#'.$var).'\').click(function(event) {'.PHP_EOL.
			'event.preventDefault();'.PHP_EOL.
			'if (confirm(\''.esc_js($this->getProjectObj()->getStringsObj('all', 'admin')->getString('INPUT_HELPER_RESET_CONFIRM')).'\')) {'.PHP_EOL.
			'$(\'#action\').val(\'reset\');'.PHP_EOL.
			'$(\''.esc_js('#'.$this->getProjectObj()->getMainSettingInternalName()).'\').submit();'.PHP_EOL.
			'}'.PHP_EOL.
			'});'.PHP_EOL.
			'});'.PHP_EOL.
			'}(jQuery));'.PHP_EOL.
			'/* ]]> */'.PHP_EOL.
			'</script>'.PHP_EOL;
	}

	/**
	 * Get input helper - uninstall
	 *
	 * @access private
	 * @param string $var Field name
	 * @param string $description Field description - default: empty
	 * @param string $value Field value - default: empty
	 * @param string $selected Selected field - default: empty
	 * @param bool $checked Checked field - default: false
	 * @param string $tip Event for mouseover displaying tooltip - default: empty
	 * @param bool $isArray Option is array - default: false
	 * @param string $eventOnChange Event onChange - default: empty
	 * @param bool $isWidget Settings for widget (true) or not (false) - default: false
	 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
	 * @param string $forcedName Forced name if $isForcedName is true
	 * @return string Input helper
	 */
	private function inputHelperUninstall($var, $description = '', $value = '', $selected = '', $checked = false, $tip = '',
		$isArray = false, $eventOnChange = '', $isWidget = false, $isForcedName = false, $forcedName = '') {
		// prepare name
		$name = ($isForcedName) ?
			$forcedName :
			$var;
		// exit
		return '<input name="'.esc_attr($name).'" id="'.esc_attr($var).'" type="button" class="button" value="'.esc_attr($value).'" '.$tip.' />'.
			'<script type="text/javascript">'.PHP_EOL.
			'/* <![CDATA[ */'.PHP_EOL.
			'(function($) {'.PHP_EOL.
			'$(document).ready(function() {'.PHP_EOL.
			'$(\''.esc_js('#'.$var).'\').click(function(event) {'.PHP_EOL.
			'event.preventDefault();'.PHP_EOL.
			'if (confirm(\''.esc_js($this->getProjectObj()->getStringsObj('all', 'admin')->getString('INPUT_HELPER_UNINSTALL_CONFIRM')).'\')) {'.PHP_EOL.
			'$(\'#action\').val(\'uninstall\');'.PHP_EOL.
			'$(\''.esc_js('#'.$this->getProjectObj()->getMainSettingInternalName()).'\').submit();'.PHP_EOL.
			'}'.PHP_EOL.
			'});'.PHP_EOL.
			'});'.PHP_EOL.
			'}(jQuery));'.PHP_EOL.
			'/* ]]> */'.PHP_EOL.
			'</script>'.PHP_EOL;
	}

	/**
	 * Check block helper loop
	 *
	 * @access public
	 * @param any $value Option value
	 * @param int $position Position in loop
	 * @param int $loopCount Loop count
	 * @return bool Allow to display in loop (true) or not (false)
	 */
	public function checkBlockHelperLoopSelectmenu($value, $position, $loopCount) {
		// exit
		return !((!wp_get_nav_menu_object($value)) && ($position < $loopCount-1));
	}

	/**
	 * Get block helper
	 *
	 * @access public
	 * @param string $id Configuration id
	 * @param string $type Field type
	 * @param string $tooltip Tooltip text or, if $type is "statictext", "submitext" or "submitexternal", text to show - default: empty
	 * @param array $options Options for select - default: empty array
	 * @param string $additionalLabel Additional label text - default: empty
	 * @param string $additional Additional text - default: empty
	 * @param bool $showLabel Show label (true) or not (false) - default: true
	 * @param string $eventOnChange Event onChange - default: empty
	 * @param bool $externalData External data istead of data from configuration (true) or not (false)
	 * @param bool $isForcedValue Is forced value (true) or not (false) - default: false
	 * @param any $forcedValue Forced value if $isForcedValue is true
	 * @param bool $addSetToId Add "set_" to field id (true) or not (false) - default: true;
	 * @param bool $isForcedLabel Is forced label (true) or not (false) - default: false
	 * @param string $forcedLabel Forced label if $isForcedLabel is true
	 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
	 * @param string $forcedName Forced name if $isForcedName is true
	 * @return string
	 */
	public function getBlockHelper($id, $type, $tooltip = '', array $options = array(), $additionalLabel = '', $additional = '', $showLabel = true, $eventOnChange = '', $externalData = false, $isForcedValue = false, $forcedValue = '', $addSetToId = true, $isForcedLabel = false, $forcedLabel = '', $isForcedName = false, $forcedName = '') {
		// check if show input helper
		if (($this->isWidget) && (!$this->tableStarted)) {
			return '';
		}
		// initialize
		$output = '';
		$setString = ($addSetToId) ?
			'set_' :
			'';
		// check if type exists
		if (($type !== 'submittext') && ($type !== 'submitexternal') && ($type !== 'statictext') && ($type !== 'emptyline') && (!isset($this->inputHelper[$type]))) {
			return '';
		}
		// check if external data id exists
		if (($externalData) && (!isset($this->externalData[$id]))) {
			return '';
		}
		// get configuration value
		if (!$isForcedValue) {
			$value = ($externalData) ?
				call_user_func($this->externalData[$id][0]) :
				$this->getProjectObj()->getComponent('all', 'config')->getOption($id);
		} else {
			$value = $forcedValue;
		}
		// check option for multisite mode
		if ((!$externalData) && (!$this->isWidget) && (is_multisite()) && ($this->getProjectObj()->getComponent('all', 'config')->checkOptionForMultisite($id)) && (!is_main_site())) {
			return '';
		}
		// prepare values
		$array = ($externalData) ?
			call_user_func($this->externalData[$id][1]) :
			$this->getProjectObj()->getComponent('all', 'config')->checkOptionArray($id);
		if (!$isForcedValue) {
			if ((!empty($array)) && ($value === false)) {
				$value = ($externalData) ?
					call_user_func($this->externalData[$id][2]) :
					$this->getProjectObj()->getComponent('all', 'config')->getOptionDefaultValue($id);
			}
		}
		$arrayText = '';
		if (empty($array)) {
			$tmp = $value;
			$value = array(
				$tmp,
			);
		} else {
			$arrayText = '[]';
			$value[] = ($externalData) ?
					array(
						'ID'    => '',
						'VALUE' => '',
					) :
					'';
		}
		// get images sizes
		if (!$externalData) {
			$checkArrayReorganized = $this->getProjectObj()->getComponent('all', 'config')->checkOptionArrayReorganized($id);
		} else {
			$checkArrayReorganized = (isset($this->externalData[$id][4]['allowreorganize'])) ?
				$this->externalData[$id][4]['allowreorganize'] :
				false;
		}
		if (($checkArrayReorganized) && (isset($this->arrowUpImage[0]) /* strlen($this->arrowUpImage) > 0 */ ) && (isset($this->arrowDownImage[0]) /* strlen($this->arrowDownImage) > 0 */ )) {
			if ((empty($this->arrowUpImageWidth)) && (empty($this->arrowUpImageHeight))) {
				$size = getimagesize(\KocujInternalLib\V4a\Classes\LibDirs::getInstance()->getImagesDir().'/'.$this->arrowUpImage);
				if (!empty($size)) {
					$this->arrowUpImageWidth = $size[0];
					$this->arrowUpImageHeight = $size[1];
				}
			}
			if ((empty($this->arrowDownImageWidth)) && (empty($this->arrowDownImageHeight))) {
				$size = getimagesize(\KocujInternalLib\V4a\Classes\LibDirs::getInstance()->getImagesDir().'/'.$this->arrowDownImage);
				if (!empty($size)) {
					$this->arrowDownImageWidth = $size[0];
					$this->arrowDownImageHeight = $size[1];
				}
			}
		}
		if (isset($this->closeImage[0]) /* strlen($this->closeImage) > 0 */ ) {
			if ((empty($this->closeImageWidth)) && (empty($this->closeImageHeight))) {
				$size = getimagesize(\KocujInternalLib\V4a\Classes\LibDirs::getInstance()->getImagesDir().'/'.$this->closeImage);
				if (!empty($size)) {
					$this->closeImageWidth = $size[0];
					$this->closeImageHeight = $size[1];
				}
			}
		}
		// show block
		$labelDisplayed = false;
		$orderString = '';
		$label = '';
		$loopCount = count($value);
		$check = true;
		$allowClose = true;
		if ((isset($this->inputHelper[$type])) && ($this->isWidget) && (!$this->inputHelper[$type][2])) {
			$check = false;
		}
		if (($this->isWidget) && (!empty($array))) {
			$check = false;
		}
		if ($check) {
			if (($this->isWidget) && (!$this->tableEmpty) && ($this->oldBlockType !== 'emptyline') && ($type !== 'emptyline')) {
				$output .= '<br />';
			}
			if (!$externalData) {
				$checkOptionArrayAddButton = $this->getProjectObj()->getComponent('all', 'config')->checkOptionArrayAddButton($id);
			} else {
				$checkOptionArrayAddButton = (isset($this->externalData[$id][4]['addbutton'])) ?
					$this->externalData[$id][4]['addbutton'] :
					false;
			}
			for ($z=0; $z<$loopCount; $z++) {
				// check if display this position
				if ((isset($this->inputHelper[$type])) && (!empty($this->inputHelper[$type][1]))) {
					$check = ($externalData) ?
						call_user_func_array($this->inputHelper[$type][1], array(
							$value[$z]['VALUE'],
							$z,
							$loopCount,
						)) :
						call_user_func_array($this->inputHelper[$type][1], array(
							$value[$z],
							$z,
							$loopCount,
						));
				} else {
					$check = true;
				}
				if ($check) {
					// prepare values
					$checked = false;
					$selected = '';
					if ($type === 'checkbox') {
						if (!$externalData) {
							$checked = $value[$z];
							$value[$z] = '1';
						} else {
							$checked = $value[$z]['VALUE'];
							$value[$z]['VALUE'] = '1';
						}
					}
					if (($type === 'select') || ($type === 'selectmenu') || ($type === 'selectcustomposts')) {
						$checked = '';
						$selected = ($externalData) ?
							$value[$z]['VALUE'] :
							$value[$z];
					}
					$labelAdd = (isset($additionalLabel[0]) /* strlen($additionalLabel) > 0 */ ) ?
						'<br />'.$additionalLabel :
						'';
					// prepare block id
					$blockId = 0;
					if (isset($this->blockHelperId[$id])) {
						++$this->blockHelperId[$id];
						$blockId = $this->blockHelperId[$id];
					}
					$blockIdNr = $blockId;
					$blockId = ($blockId === 0) ?
						$id :
						$id.'_'.$blockId;
					if (!isset($this->blockHelperId[$id])) {
						$this->blockHelperId[$id] = 0;
					}
					// get block
					if (($type === 'statictext') || ($type === 'submittext') || ($type === 'submitexternal') || ($type === 'emptyline')) {
						if (!$this->isWidget) {
							$output .= '<tr id="'.esc_attr('kocujinternallib_field_'.$blockId).'">';
							$classToAdd = '';
							if ($type === 'statictext') {
								$classToAdd = ' static';
							}
							if ($type === 'submittext') {
								$classToAdd = ' submit';
							}
							if ($type === 'submitexternal') {
								$classToAdd = ' submit';
							}
							$output .= '<td class="'.esc_attr('kocujinternallib-'.$type.$classToAdd).'" colspan="2">';
						}
						if ($type !== 'emptyline') {
							$output .= $tooltip;
						}
						if (!$this->isWidget) {
							$output .= '</td>'.
								'</tr>';
						}
						if (($this->isWidget) && ($type === 'emptyline')) {
							echo '</p><p>';
						}
					} else {
						$labelValue = '';
						if (($showLabel) && (!$labelDisplayed)) {
							if (!isset($arrayText[0]) /* strlen($arrayText) === 0 */ ) {
								$labelValue .= '<label for="'.$setString.$id.'">';
							}
							if (!$isForcedLabel) {
								$label = ($externalData) ?
									$this->externalData[$id][3] :
									$this->getProjectObj()->getComponent('all', 'config')->getOptionLabel($id).$labelAdd;
							} else {
								$label = $forcedLabel;
							}
							$labelValue .= $label;
							if (!isset($arrayText[0]) /* strlen($arrayText) === 0 */ ) {
								if ($this->isWidget) {
									$labelValue .= ':';
								}
								$labelValue .= '</label>';
							}
							$labelDisplayed = true;
						}
						$labelBefore = !(($this->isWidget) && (!$this->inputHelper[$type][3]));
						if (!$this->isWidget) {
							$output .= '<tr id="'.esc_attr('kocujinternallib_field_'.$blockId).'">'.
								'<td class="kocujinternallib-label" id="'.esc_attr('kocujinternallib_field_label_'.$blockId).'">';
						}
						if ($labelBefore) {
							if (isset($labelValue[0]) /* strlen($labelValue) > 0 */ ) {
								$output .= $labelValue;
								if ($this->isWidget) {
									$output .= '<br />';
								}
							} else {
								$output .= '&nbsp;';
							}
						}
						if (!$this->isWidget) {
							$output .= '</td>'.
								'<td class="kocujinternallib-content">';
						}
						$fragment = ($externalData) ?
							$this->getInputHelper($setString.$id.$arrayText, $type, '', $value[$z]['VALUE'], $selected, $checked, $tooltip, $array, $eventOnChange, $this->isWidget, $isForcedName, $forcedName) :
							$this->getInputHelper($setString.$id.$arrayText, $type, '', $value[$z], $selected, $checked, $tooltip, $array, $eventOnChange, $this->isWidget, $isForcedName, $forcedName);
						if ($type === 'select') {
							if (!empty($array)) {
								$fragment .= $this->getInputHelper($setString.$id.$arrayText, 'option', '', '', '', false, '', false, $eventOnChange, $this->isWidget, $isForcedName, $forcedName);
							}
							if (!empty($options)) {
								foreach ($options as $key => $option) {
									$fragment .= ($externalData) ?
										$this->getInputHelper($setString.$id.$arrayText, 'option', $option, $key, $value[$z]['VALUE'], false, '', false, $eventOnChange, $this->isWidget, $isForcedName, $forcedName) :
										$this->getInputHelper($setString.$id.$arrayText, 'option', $option, $key, $value[$z], false, '', false, $eventOnChange, $this->isWidget, $isForcedName, $forcedName);
								}
							}
							$fragment .= $this->getInputHelper($setString.$id.$arrayText, 'select_end', '', '', '', false, '', false, $eventOnChange, $this->isWidget, $isForcedName, $forcedName);
						}
						if (($externalData) && (!empty($array))) {
							$fragment .= '<input type="hidden" name="'.esc_attr('ID_'.$id.$arrayText).'" id="'.esc_attr('ID_'.$id.$arrayText).'" value="'.esc_attr($value[$z]['ID']).'" />';
						}
						if (!empty($array)) {
							$leftButton = 0;
							if (!$externalData) {
								$leftMargin = $this->getProjectObj()->getComponent('all', 'config')->getOptionArrayMarginLeft($id);
								$verticalCenter = $this->getProjectObj()->getComponent('all', 'config')->getOptionArrayVerticalCenter($id);
								$allowClose = $this->getProjectObj()->getComponent('all', 'config')->checkOptionArrayClose($id);
							} else {
								$leftMargin = (isset($this->externalData[$id][4]['marginleft'])) ?
									$this->externalData[$id][4]['marginleft'] :
									5;
								$verticalCenter = (isset($this->externalData[$id][4]['verticalcenter'])) ?
									$this->externalData[$id][4]['verticalcenter'] :
									12;
								$allowClose = (isset($this->externalData[$id][4]['allowclose'])) ?
									$this->externalData[$id][4]['allowclose'] :
									false;
							}
							if (($checkArrayReorganized) && (isset($this->arrowUpImage[0]) /* strlen($this->arrowUpImage) > 0 */ ) && (isset($this->arrowDownImage[0]) /* strlen($this->arrowDownImage) > 0 */ )) {
								$fragment .= \KocujInternalLib\V4a\Classes\HtmlHelper::getInstance()->getHTMLImage(\KocujInternalLib\V4a\Classes\LibUrls::getInstance()->getImagesUrl().'/'.$this->arrowUpImage, array(
									'id'    => 'kocujinternallibfragmentarrowup_##%%ARROW_ID%%##',
									'style' => 'position:absolute;margin-top:'.($verticalCenter-($this->arrowUpImageWidth/2)).'px;margin-left:'.$leftMargin.'px;cursor:pointer;',
								)).
								\KocujInternalLib\V4a\Classes\HtmlHelper::getInstance()->getHTMLImage(\KocujInternalLib\V4a\Classes\LibUrls::getInstance()->getImagesUrl().'/'.$this->arrowDownImage, array(
									'id'    => 'kocujinternallibfragmentarrowdown_##%%ARROW_ID%%##',
									'style' => 'position:absolute;margin-top:'.($verticalCenter-($this->arrowDownImageWidth/2)).'px;margin-left:'.($leftMargin+$this->arrowUpImageWidth+2).'px;cursor:pointer;',
								));
								$leftButton = $leftMargin+$this->arrowUpImageWidth+2+$this->arrowDownImageWidth;
							}
							if (($allowClose) && (isset($this->closeImage[0]) /* strlen($this->closeImage) > 0 */ )) {
								$fragment .= \KocujInternalLib\V4a\Classes\HtmlHelper::getInstance()->getHTMLImage(\KocujInternalLib\V4a\Classes\LibUrls::getInstance()->getImagesUrl().'/'.$this->closeImage, array(
									'id'    => 'kocujinternallibfragmentclose_##%%CLOSE_ID%%##',
									'style' => 'position:absolute;margin-top:'.($verticalCenter-($this->closeImageWidth/2)).'px;margin-left:'.($leftMargin+$leftButton).'px;cursor:pointer;',
								));
							}
							$output .= str_replace('##%%CLOSE_ID%%##', $blockId, str_replace('##%%ARROW_ID%%##', $blockId, $fragment));
						} else {
							$output .= $fragment;
						}
						$additionalDiv = '';
						if ($z === $loopCount-1) {
							$lastFragmentNr = $blockIdNr;
							if (!$this->isWidget) {
								if (isset($additional[0]) /* strlen($additional) > 0 */ ) {
									$additionalDiv = '<div id="'.esc_attr('kocujinternallib-additional-'.$id).'">'.$additional.'</div>';
									$output .= $additionalDiv;
								}
							}
						} else {
							$lastFragmentNr = 0;
						}
						if ($z > 0) {
							$orderString .= ',';
						}
						$orderString .= $z;
						if (!$labelBefore) {
							$output .= (isset($labelValue[0]) /* strlen($labelValue) > 0 */ ) ?
								'&nbsp;'.$labelValue :
								'&nbsp;';
						}
						if ((!empty($array)) && ($z === $loopCount-1)) {
							$output .= '<script type="text/javascript">'.PHP_EOL.
								'/* <![CDATA[ */'.PHP_EOL.
								'(function($) {'.PHP_EOL.
								'$(document).ready(function() {'.PHP_EOL.
								'var kocujinternalliblabel_'.$id.' = \''.esc_js(str_replace(array(
									"\r\n",
									"\n",
									"\r",
								), '', $label)).'\';'.PHP_EOL.
								'var kocujinternallibfragment_'.$id.' = \''.str_replace(array(
									"\r\n",
									"\n",
									"\r",
								), '', str_replace('\'', '\\\'', $fragment)).'\';'.PHP_EOL.
								'var kocujinternallibfragmenttrprefix_'.$id.' = \''.esc_js('kocujinternallib_field_'.$id).'\';'.PHP_EOL.
								'var kocujinternalliblastfragmenttrnr_'.$id.' = '.$lastFragmentNr.';'.PHP_EOL;
							if (isset($additionalDiv[0]) /* strlen($additionalDiv) > 0 */ ) {
								$output .= 'var kocujinternallibadditionaldiv_'.$id.' = \''.str_replace(array(
									"\r\n",
									"\n",
									"\r",
								), '', str_replace('\'', '\\\'', $additionalDiv)).'\';'.PHP_EOL;
							}
							$output .= 'var kocujinternallibfragmentsorder_'.$id.' = ['.$orderString.'];'.PHP_EOL.
								'function kocujinternallib_get_id_string_'.$id.'(id) {'.PHP_EOL.
								'var id = parseInt(id, 10);'.PHP_EOL.
								'var idString = \'_\' + id;'.PHP_EOL.
								'if (id === 0) {'.PHP_EOL.
								'idString = \'\';'.PHP_EOL.
								'}'.PHP_EOL.
								'return idString;'.PHP_EOL.
								'}'.PHP_EOL.
								'function kocujinternallib_find_id_in_order_'.$id.'(id) {'.PHP_EOL.
								'var id = parseInt(id, 10);'.PHP_EOL.
								'if (kocujinternallibfragmentsorder_'.$id.'.length > 0) {'.PHP_EOL.
								'for (var z=0; z<kocujinternallibfragmentsorder_'.$id.'.length; z++) {'.PHP_EOL.
								'if (parseInt(kocujinternallibfragmentsorder_'.$id.'[z], 10) === id) {'.PHP_EOL.
								'return z;'.PHP_EOL.
								'}'.PHP_EOL.
								'}'.PHP_EOL.
								'}'.PHP_EOL.
								'return -1;'.PHP_EOL.
								'}'.PHP_EOL.
								'function kocujinternallib_event_change_add_without_val_check_'.$id.'(event) {'.PHP_EOL.
								'kocujinternalliblastfragmenttrnr_'.$id.' = parseInt(kocujinternalliblastfragmenttrnr_'.$id.', 10)+1;'.PHP_EOL.
								'kocujinternallibfragmentsorder_'.$id.'[kocujinternallibfragmentsorder_'.$id.'.length] = parseInt(kocujinternalliblastfragmenttrnr_'.$id.', 10);'.PHP_EOL.
								'kocujinternallib_set_remove_events_'.$id.'();'.PHP_EOL;
							if (isset($additionalDiv[0]) /* strlen($additionalDiv) > 0 */ ) {
								$output .= '$(\'div#kocujinternallib-additional-'.$id.'\').remove();'.PHP_EOL;
							}
							if (($checkArrayReorganized) && (isset($this->arrowUpImage[0]) /* strlen($this->arrowUpImage) > 0 */ ) && (isset($this->arrowDownImage[0]) /* strlen($this->arrowDownImage) > 0 */ )) {
								$output .= '$(\'#kocujinternallib_empty_field_'.$id.'\').before(\'<tr id="\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternalliblastfragmenttrnr_'.$id.') + \'"><td class="kocujinternallib-label" id="kocujinternallib_field_label_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternalliblastfragmenttrnr_'.$id.') + \'">&nbsp;</td><td class="kocujinternallib-content">\' + kocujinternallibfragment_'.$id.'.replace(\'##%%ARROW_ID%%##\', \''.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternalliblastfragmenttrnr_'.$id.')).replace(\'##%%ARROW_ID%%##\', \''.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternalliblastfragmenttrnr_'.$id.')).replace(\'##%%CLOSE_ID%%##\', \''.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternalliblastfragmenttrnr_'.$id.')) + \''.$additionalDiv.'</td></tr>\');'.PHP_EOL.
									'kocujinternallib_set_arrows_'.$id.'();'.PHP_EOL;
							} else {
								$output .= '$(\'#kocujinternallib_empty_field_'.$id.'\').before(\'<tr id="\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternalliblastfragmenttrnr_'.$id.') + \'"><td class="kocujinternallib-label" id="kocujinternallib_field_label_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternalliblastfragmenttrnr_'.$id.') + \'">&nbsp;</td><td class="kocujinternallib-content">\' + kocujinternallibfragment_'.$id.'.replace(\'##%%CLOSE_ID%%##\', \''.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternalliblastfragmenttrnr_'.$id.')) + \''.$additionalDiv.'</td></tr>\');'.PHP_EOL;
							}
							if ((!empty($allowClose)) && (isset($this->closeImage[0]) /* strlen($this->closeImage) > 0 */ )) {
								$output .= 'kocujinternallib_set_close_'.$id.'();'.PHP_EOL;
							}
							$output .= 'if (parseInt(kocujinternalliblastfragmenttrnr_'.$id.', 10)-1 >= 0) {'.PHP_EOL.
								'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternalliblastfragmenttrnr_'.$id.'-1) + \' [name="set_'.$id.'[]"]\').unbind(\'change.kocujinternallibfragment\');'.PHP_EOL.
								'}'.PHP_EOL;
							if (!$externalData) {
								$checkOptionArrayAddOnLastNotEmpty = $this->getProjectObj()->getComponent('all', 'config')->checkOptionArrayAddOnLastNotEmpty($id);
							} else {
								$checkOptionArrayAddOnLastNotEmpty = (isset($this->externalData[$id][4]['addonlastnotempty'])) ?
									$this->externalData[$id][4]['addonlastnotempty'] :
									false;
							}
							if ($checkOptionArrayAddOnLastNotEmpty) {
								$output .= '$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternalliblastfragmenttrnr_'.$id.') + \' [name="'.$setString.$id.'[]"]\').bind(\'change.kocujinternallibfragment\', kocujinternallib_event_change_add_'.$id.');'.PHP_EOL;
							}
							$output .= '}'.PHP_EOL;
							if ($checkOptionArrayAddOnLastNotEmpty) {
								$output .= 'function kocujinternallib_event_change_add_'.$id.'(event) {'.PHP_EOL.
									'if ($(this).val() !== \'\') {'.PHP_EOL.
									'kocujinternallib_event_change_add_without_val_check_'.$id.'(event);'.PHP_EOL.
									'}'.PHP_EOL.
									'}'.PHP_EOL;
							}
							$output .= 'function kocujinternallib_event_change_remove_without_val_check_'.$id.'(event) {'.PHP_EOL.
								'var id = parseInt(event.data.id, 10);'.PHP_EOL.
								'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(id)).remove();'.PHP_EOL.
								'var orderpos = parseInt(kocujinternallib_find_id_in_order_'.$id.'(id), 10);'.PHP_EOL.
								'if (orderpos === 0) {'.PHP_EOL.
								'var newid = parseInt(kocujinternallibfragmentsorder_'.$id.'[1], 10);'.PHP_EOL.
								'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(newid) + \' #kocujinternallib_field_label_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(newid)).html(kocujinternalliblabel_'.$id.');'.PHP_EOL.
								'}'.PHP_EOL.
								'kocujinternallibfragmentsorder_'.$id.'.splice(orderpos, 1);'.PHP_EOL;
							if (($checkArrayReorganized) && (isset($this->arrowUpImage[0]) /* strlen($this->arrowUpImage) > 0 */ ) && (isset($this->arrowDownImage[0]) /* strlen($this->arrowDownImage) > 0 */ )) {
								$output .= 'kocujinternallib_set_arrows_'.$id.'();'.PHP_EOL;
							}
							if (($allowClose) && (isset($this->closeImage[0]) /* strlen($this->closeImage) > 0 */ )) {
								$output .= 'kocujinternallib_set_close_'.$id.'();'.PHP_EOL;
							}
							$output .= '}'.PHP_EOL;
							if (!$externalData) {
								$checkOptionArrayRemoveOnEmpty = $this->getProjectObj()->getComponent('all', 'config')->checkOptionArrayRemoveOnEmpty($id);
							} else {
								$checkOptionArrayRemoveOnEmpty = (isset($this->externalData[$id][4]['removeonempty'])) ?
									$this->externalData[$id][4]['removeonempty'] :
									false;
							}
							if ($checkOptionArrayRemoveOnEmpty) {
								$output .= 'function kocujinternallib_event_change_remove_'.$id.'(event) {'.PHP_EOL.
									'if ($(this).val() === \'\') {'.PHP_EOL.
									'kocujinternallib_event_change_remove_without_val_check_'.$id.'(event);'.PHP_EOL.
									'}'.PHP_EOL.
									'}'.PHP_EOL;
							}
							$output .= 'function kocujinternallib_set_remove_events_'.$id.'() {'.PHP_EOL.
								'if (parseInt(kocujinternalliblastfragmenttrnr_'.$id.', 10) >= 1) {'.PHP_EOL.
								'for (var z=0; z<parseInt(kocujinternalliblastfragmenttrnr_'.$id.', 10); z++) {'.PHP_EOL.
								'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(z) + \' [name="'.$setString.$id.'[]"]\').unbind(\'change.kocujinternallibfragmentremove\');'.PHP_EOL;
							if ($checkOptionArrayRemoveOnEmpty) {
								$output .= '$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(z) + \' [name="'.$setString.$id.'[]"]\').bind(\'change.kocujinternallibfragmentremove\', {id: z}, kocujinternallib_event_change_remove_'.$id.');'.PHP_EOL;
							}
							$output .= '}'.PHP_EOL.
								'}'.PHP_EOL.
								'}'.PHP_EOL;
							if (($allowClose) && (isset($this->closeImage[0]) /* strlen($this->closeImage) > 0 */ )) {
								$output .= 'function kocujinternallib_set_close_'.$id.'() {'.PHP_EOL.
									'if (kocujinternallibfragmentsorder_'.$id.'.length > 1) {'.PHP_EOL.
									'for (var z=0; z<kocujinternallibfragmentsorder_'.$id.'.length-1; z++) {'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z]) + \' #kocujinternallibfragmentclose_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z])).unbind(\'click.kocujinternallibclose\');'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z]) + \' #kocujinternallibfragmentclose_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z])).bind(\'click.kocujinternallibclose\', {id: kocujinternallibfragmentsorder_'.$id.'[z]}, kocujinternallib_event_change_remove_without_val_check_'.$id.');'.PHP_EOL.
									'}'.PHP_EOL.
									'for (var z=0; z<kocujinternallibfragmentsorder_'.$id.'.length-1; z++) {'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z]) + \' #kocujinternallibfragmentclose_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z])).css(\'display\', \'inline\');'.PHP_EOL.
									'}'.PHP_EOL.
									'}'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternalliblastfragmenttrnr_'.$id.') + \' #kocujinternallibfragmentclose_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternalliblastfragmenttrnr_'.$id.')).hide();'.PHP_EOL.
									'}'.PHP_EOL;
							}
							if (($checkArrayReorganized) && (isset($this->arrowUpImage[0]) /* strlen($this->arrowUpImage) > 0 */ ) && (isset($this->arrowDownImage[0]) /* strlen($this->arrowDownImage) > 0 */ )) {
								$output .= 'function kocujinternallib_event_arrow_up_click_'.$id.'(event) {'.PHP_EOL.
									'var id = parseInt(event.data.id, 10);'.PHP_EOL.
									'var fromid = parseInt(kocujinternallibfragmentsorder_'.$id.'[id], 10);'.PHP_EOL.
									'var toid = parseInt(kocujinternallibfragmentsorder_'.$id.'[id-1], 10);'.PHP_EOL.
									'if (parseInt(id-1, 10) === 0) {'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(fromid) + \' #kocujinternallib_field_label_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(fromid)).html(kocujinternalliblabel_'.$id.');'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(toid) + \' #kocujinternallib_field_label_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(toid)).html(\'&nbsp;\');'.PHP_EOL.
									'}'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(fromid)).insertBefore(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(toid));'.PHP_EOL.
									'var tmp = kocujinternallibfragmentsorder_'.$id.'[id];'.PHP_EOL.
									'kocujinternallibfragmentsorder_'.$id.'[id] = kocujinternallibfragmentsorder_'.$id.'[id-1];'.PHP_EOL.
									'kocujinternallibfragmentsorder_'.$id.'[id-1] = tmp;'.PHP_EOL.
									'kocujinternallib_set_arrows_'.$id.'();'.PHP_EOL;
								if (($allowClose) && (isset($this->closeImage[0]) /* strlen($this->closeImage) > 0 */ )) {
									$output .= 'kocujinternallib_set_close_'.$id.'();'.PHP_EOL;
								}
								$output .= '}'.PHP_EOL.
									'function kocujinternallib_event_arrow_down_click_'.$id.'(event) {'.PHP_EOL.
									'var id = parseInt(event.data.id, 10);'.PHP_EOL.
									'var fromid = parseInt(kocujinternallibfragmentsorder_'.$id.'[id], 10);'.PHP_EOL.
									'var toid = parseInt(kocujinternallibfragmentsorder_'.$id.'[id+1], 10);'.PHP_EOL.
									'if (parseInt(id, 10) === 0) {'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(fromid) + \' #kocujinternallib_field_label_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(fromid)).html(\'&nbsp;\');'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(toid) + \' #kocujinternallib_field_label_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(toid)).html(kocujinternalliblabel_'.$id.');'.PHP_EOL.
									'}'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(fromid)).insertAfter(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(toid));'.PHP_EOL.
									'var tmp = kocujinternallibfragmentsorder_'.$id.'[id];'.PHP_EOL.
									'kocujinternallibfragmentsorder_'.$id.'[id] = kocujinternallibfragmentsorder_'.$id.'[id+1];'.PHP_EOL.
									'kocujinternallibfragmentsorder_'.$id.'[id+1] = tmp;'.PHP_EOL.
									'kocujinternallib_set_arrows_'.$id.'();'.PHP_EOL;
								if (($allowClose) && (isset($this->closeImage[0]) /* strlen($this->closeImage) > 0 */ )) {
									$output .= 'kocujinternallib_set_close_'.$id.'();'.PHP_EOL;
								}
								$output .= '}'.PHP_EOL.
									'function kocujinternallib_set_arrows_'.$id.'() {'.PHP_EOL.
									'for (var z=0; z<kocujinternallibfragmentsorder_'.$id.'.length-1; z++) {'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z]) + \' #kocujinternallibfragmentarrowup_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z])).unbind(\'click.kocujinternallibarrowup\');'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z]) + \' #kocujinternallibfragmentarrowdown_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z])).unbind(\'click.kocujinternallibarrowdown\');'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z]) + \' #kocujinternallibfragmentarrowup_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z])).bind(\'click.kocujinternallibarrowup\', {id: z}, kocujinternallib_event_arrow_up_click_'.$id.');'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z]) + \' #kocujinternallibfragmentarrowdown_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z])).bind(\'click.kocujinternallibarrowdown\', {id: z}, kocujinternallib_event_arrow_down_click_'.$id.');'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z]) + \' #kocujinternallibfragmentarrowup_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z])).css(\'display\', \'inline\');'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z]) + \' #kocujinternallibfragmentarrowdown_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z])).css(\'display\', \'inline\');'.PHP_EOL.
									'if (z === 0) {'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z]) + \' #kocujinternallibfragmentarrowup_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z])).hide();'.PHP_EOL.
									'} else {'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z]) + \' #kocujinternallibfragmentarrowup_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z])).css(\'display\', \'inline\');'.PHP_EOL.
									'}'.PHP_EOL.
									'if (z === kocujinternallibfragmentsorder_'.$id.'.length-2) {'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z]) + \' #kocujinternallibfragmentarrowdown_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z])).hide();'.PHP_EOL.
									'} else {'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z]) + \' #kocujinternallibfragmentarrowdown_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternallibfragmentsorder_'.$id.'[z])).css(\'display\', \'inline\');'.PHP_EOL.
									'}'.PHP_EOL.
									'}'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternalliblastfragmenttrnr_'.$id.') + \' #kocujinternallibfragmentarrowup_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternalliblastfragmenttrnr_'.$id.')).hide();'.PHP_EOL.
									'$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternalliblastfragmenttrnr_'.$id.') + \' #kocujinternallibfragmentarrowdown_'.$id.'\' + kocujinternallib_get_id_string_'.$id.'(kocujinternalliblastfragmenttrnr_'.$id.')).hide();'.PHP_EOL.
									'}'.PHP_EOL.
									'kocujinternallib_set_arrows_'.$id.'();'.PHP_EOL;
							}
							if (($allowClose) && (isset($this->closeImage[0]) /* strlen($this->closeImage) > 0 */ )) {
								$output .= 'kocujinternallib_set_close_'.$id.'();'.PHP_EOL;
							}
							$output .= 'kocujinternallib_set_remove_events_'.$id.'();'.PHP_EOL;
							if ($checkOptionArrayAddOnLastNotEmpty) {
								$output .= '$(\'#\' + kocujinternallibfragmenttrprefix_'.$id.' + kocujinternallib_get_id_string_'.$id.'(kocujinternalliblastfragmenttrnr_'.$id.') + \' [name="'.$setString.$id.'[]"]\').bind(\'change.kocujinternallibfragment\', kocujinternallib_event_change_add_'.$id.');'.PHP_EOL;
							}
							if ($checkOptionArrayAddButton) {
								$output .= '$(\'#kocujinternallib_addbutton_field_'.$id.' #kocujinternallibfragmentadd_'.$id.'\').bind(\'click.kocujinternallibadd\', kocujinternallib_event_change_add_without_val_check_'.$id.');'.PHP_EOL;
							}
							$output .= '});'.PHP_EOL.
								'}(jQuery));'.PHP_EOL.
								'/* ]]> */'.PHP_EOL.
								'</script>'.PHP_EOL;
						}
						if (!$this->isWidget) {
							$output .= '</td>'.
								'</tr>';
						}
						$this->tableEmpty = false;
					}
				}
			}
			// optionally add empty element
			if (!empty($array)) {
				$output .= '<tr id="'.esc_attr('kocujinternallib_empty_field_'.$id).'" style="display:none;">'.
					'<td colspan="2">'.
					'</td>'.
					'</tr>';
			}
			// optionally add adding button
			if ((!empty($array)) && ($checkOptionArrayAddButton)) {
				$output .= '<tr id="'.esc_attr('kocujinternallib_addbutton_field_'.$id).'">'.
					'<td class="kocujinternallib-label">'.
					'&nbsp;'.
					'</td>'.
					'<td class="kocujinternallib-content">'.
					'<input id="'.esc_attr('kocujinternallibfragmentadd_'.$id).'" type="button" class="button" value="'.esc_attr($this->getProjectObj()->getStringsObj('all', 'admin')->getString('GET_BLOCK_HELPER_ADD_NEW_ELEMENT')).'" />'.
					'</td>'.
					'</tr>';
			}
		}
		// save old type
		$this->oldBlockType = $type;
		// exit
		return $output;
	}

	/**
	 * Show block helper
	 *
	 * @access public
	 * @param string $id Configuration id
	 * @param string $type Field type
	 * @param string $tooltip Tooltip text - default: empty
	 * @param array $options Options for select - default: empty arraz
	 * @param string $additionalLabel Additional label text - default: empty
	 * @param string $additional Additional text - default: empty
	 * @param bool $showLabel Show label (true) or not (false) - default: true
	 * @param string $eventOnChange Event onChange - default: empty
	 * @param bool $externalData External data istead of data from configuration (true) or not (false)
	 * @param bool $isForcedValue Is forced value (true) or not (false) - default: false
	 * @param any $forcedValue Forced value if $isForcedValue is true
	 * @param bool $addSetToId Add "set_" to field id (true) or not (false) - default: true;
	 * @param bool $isForcedLabel Is forced label (true) or not (false) - default: false
	 * @param string $forcedLabel Forced label if $isForcedLabel is true
	 * @param bool $isForcedName Name is forced (true) or not (false) - default: false
	 * @param string $forcedName Forced name if $isForcedName is true
	 * @return void
	 */
	public function blockHelper($id, $type, $tooltip = '', array $options = array(), $additionalLabel = '', $additional = '', $showLabel = true, $eventOnChange = '', $externalData = false, $isForcedValue = false, $forcedValue = '', $addSetToId = true, $isForcedLabel = false, $forcedLabel = '', $isForcedName = false, $forcedName = '') {
		// show block helper
		echo $this->getBlockHelper($id, $type, $tooltip, $options, $additionalLabel, $additional, $showLabel, $eventOnChange, $externalData, $isForcedValue, $forcedValue, $addSetToId, $isForcedLabel, $forcedLabel, $isForcedName, $forcedName);
	}

	/**
	 * Add header
	 *
	 * @access public
	 * @return void
	 */
	public function addHeader() {
		// optionally add script to open all external links in a new window
		if ($this->getProjectObj()->getComponent('backend', 'settings-menu')->checkCurrentPageIsSettingsForProject()) {
			?>
				<script type="text/javascript">
				/* <![CDATA[ */
					(function($) {
						$(document).ready(function() {
							$('a[rel=external]').attr('target', '_blank');
						});
					}(jQuery));
				/* ]]> */
				</script>
			<?php
		}
		// show style
		if ($this->getProjectObj()->getComponent('backend', 'settings-menu')->checkCurrentPageIsSettingsForProject()) {
			?>
			<style type="text/css" media="all">
			<!--
				.kocujinternallib-form-table {
					width: 100%;
				}

				.kocujinternallib-label,
				.kocujinternallib-label label {
					text-align: left;
					vertical-align: top;
					width: 300px;
					margin: 0;
					padding: 0;
				}

				.kocujinternallib-content {
					text-align: left;
					vertical-align: top;
					margin: 0;
					padding: 0;
				}
			-->
			</style>
			<?php
		}
	}

	/**
	 * Page controller
	 *
	 * @access public
	 * @return void
	 */
	public function pageController() {
		// change options
		if (($this->getProjectObj()->getComponent('backend', 'settings-menu')->checkCurrentPageIsSettingsForProject()) && (isset($_POST['action']))) {
			if (isset($this->controller[$_POST['action']][0])) {
				$errors = ((isset($_POST[$this->getProjectObj()->getMainSettingInternalName().'_'.md5($this->getProjectObj()->getMainSettingInternalName())])) && (wp_verify_nonce($_POST[$this->getProjectObj()->getMainSettingInternalName().'_'.md5($this->getProjectObj()->getMainSettingInternalName())], $this->nonceAction))) ?
					call_user_func($this->controller[$_POST['action']][0]) :
					$this->getProjectObj()->getStringsObj('all', 'admin')->getString('PAGE_CONTROLLER_UNKNOWN_FORM_ERROR');
				$page = (isset($_GET['page'])) ?
					$_GET['page'] :
					'';
				$url = \KocujInternalLib\V4a\Classes\Helper::getInstance()->getScriptUrl();
				$div = explode('?', $url);
				if (isset($div[0]) /* count($div) > 0 */ ) {
					$url = $div[0];
				}
				header('Location: '.$url.'?page='.urlencode($page).'&'.urlencode($_POST['action']).'=true&errors='.urlencode($errors));
				exit();
			} else {
				echo $this->getProjectObj()->getStringsObj('all', 'admin')->getString('PAGE_CONTROLLER_WRONG_PARAMETERS');
			}
		}
	}

	/**
	 * Save controller
	 *
	 * @access public
	 * @return string Output status
	 */
	public function controllerSave() {
		// save options
		$errors = $this->getProjectObj()->getComponent('all', 'config')->updateOptions();
		// exit
		return $errors;
	}

	/**
	 * Save external data controller
	 *
	 * @access public
	 * @return string Output status
	 */
	public function controllerSaveExternal() {
		// get external type
		$var = '';
		if ((isset($_POST['externalvar'])) && (isset($_POST['externalvar'][0]) /* strlen($_POST['externalvar']) > 0 */ )) {
			$var = $_POST['externalvar'];
		}
		$ok = true;
		if ((isset($var[0]) /* strlen($var) > 0 */ ) && (!isset($this->externalData[$var]))) {
			$ok = false;
		}
		if ((!isset($var[0]) /* strlen($var) === 0 */ ) || (!$ok)) {
			return $this->getProjectObj()->getStringsObj('all', 'admin')->getString('CONTROLLER_SAVE_EXTERNAL_FORM_ERRORS');
		}
		$settings = $this->externalData[$var];
		// get all id to update
		$ids = array();
		if ((isset($_POST['set_'.$var])) && (isset($_POST['ID_'.$var]))) {
			$loopCount = count($_POST['set_'.$var]);
			for ($z=0; $z<$loopCount; $z++) {
				if ((isset($_POST['ID_'.$var][$z])) && (is_numeric($_POST['ID_'.$var][$z]))) {
					$ids[] = $_POST['ID_'.$var][$z];
				}
			}
		}
		// delete unneeded data
		$ok = true;
		if (!empty($settings[7])) {
			$status = call_user_func_array($settings[7], array(
				$ids,
			));
			if (!$status) {
				$ok = false;
			}
		}
		// add or update data
		if ((isset($_POST['set_'.$var])) && (isset($_POST['ID_'.$var]))) {
			$loopCount = count($_POST['set_'.$var]);
			for ($z=0; $z<$loopCount; $z++) {
				if (isset($_POST['set_'.$var][$z])) {
					if ((isset($_POST['ID_'.$var][$z])) && (is_numeric($_POST['ID_'.$var][$z]))) {
						// update data
						if (!empty($settings[6])) {
							$status = call_user_func_array($settings[6], array(
								$_POST['ID_'.$var][$z],
								$_POST['set_'.$var][$z],
							));
							if (!$status) {
								$ok = false;
							}
						}
					} else {
						// add data
						if (!empty($settings[5])) {
							$status = call_user_func_array($settings[5], array(
								$_POST['set_'.$var][$z],
							));
							if (!$status) {
								$ok = false;
							}
						}
					}
				}
			}
		}
		if (!$ok) {
			return $this->getProjectObj()->getStringsObj('all', 'admin')->getString('CONTROLLER_SAVE_EXTERNAL_DATA_ERRORS');
		}
		// exit
		return $errors;
	}

	/**
	 * Reset controller
	 *
	 * @access public
	 * @return string Output status
	 */
	public function controllerReset() {
		// reset options
		$this->getProjectObj()->getComponent('all', 'config')->deleteOptions();
		// exit
		return '';
	}

	/**
	 * Uninstall controller
	 *
	 * @access public
	 * @return string Output status
	 */
	public function controllerUninstall() {
		// uninstall
		$this->getProjectObj()->getComponent('all', 'config')->uninstallOptions();
		// exit
		return '';
	}

	/**
	 * Processing output
	 *
	 * @access public
	 * @return void
	 */
	public function processOutput() {
		// process output
		if ($this->getProjectObj()->getComponent('backend', 'settings-menu')->checkCurrentPageIsSettingsForProject()) {
			// get errors text
			$errors = ((isset($_REQUEST['errors'])) && ($_REQUEST['errors'])) ?
				$_REQUEST['errors'] :
				'';
			// show save message
			if (!empty($this->controller)) {
				foreach ($this->controller as $key => $controller) {
					if ((isset($_REQUEST[$key])) && ($_REQUEST[$key]) && (isset($this->controller[$key][1])) && (isset($this->controller[$key][1][0]) /* strlen($this->controller[$key][1]) > 0 */ )) {
						$this->getProjectObj()->getComponent('backend', 'message')->addMessage('settings_updated', '<strong>'.$this->controller[$key][1].'</strong>');
					}
				}
			}
			if ((isset($_REQUEST['uninstall'])) && ($_REQUEST['uninstall'])) {
				return;
			}
			// optionally show errors
			if (isset($errors[0]) /* strlen($errors) > 0 */ ) {
				$this->getProjectObj()->getComponent('backend', 'message')->addMessage('settings_update_error', $this->getProjectObj()->getStringsObj('all', 'admin')->getString('PROCESS_OUTPUT_ERRORS').'</p><p>'.stripslashes($errors), \KocujInternalLib\V4a\Enums\Project\Components\Backend\Message\Type::ERROR);
			}
		}
	}

	/**
	 * Add meta boxes
	 *
	 * @access public
	 * @return void
	 */
	public function filterAddMetaBox() {
		// add meta boxes
		if (!empty($this->metaBox)) {
			foreach ($this->metaBox as $key => $metaBox) {
				foreach ($metaBox[2] as $place) {
					add_meta_box($metaBox[0], $metaBox[1], array($this, 'metaBox'.$this->random.'_'.$key), $place, $metaBox[3], $metaBox[4]);
				}
			}
		}
	}

	/**
	 * Show meta box
	 *
	 * @access private
	 * @param int $pos Position in metaBox array
	 * @param object $post Post object
	 * @return void
	 */
	private function showMetaBox($pos, $post) {
		// get post or page id
		if (isset($post->ID)) {
			$id = $post->ID;
		} else {
			return;
		}
		// get meta box data
		$data = $this->metaBox[$pos];
		// add hidden security field
		wp_nonce_field($this->nonceAction, $data[5]);
		// show meta box
		call_user_func_array($data[6], array(
			$id,
			$post,
		));
	}

	/**
	 * Update meta box
	 *
	 * @access private
	 * @param int $id Post id
	 * @param string $metaKey Meta key
	 * @param bool $checkbox It is checkbox (true) or not (false)
	 * @return void
	 */
	private function updateMetaBox($id, $metaKey, $checkbox) {
		// get new value
		$newValue = (isset($_POST[$metaKey])) ?
			$_POST[$metaKey] :
			'';
		if ($checkbox) {
			$newValue = (!isset($newValue[0]) /* strlen($newValue) === 0 */ ) ?
				'0' :
				'1';
		}
		// add or update
		$value = get_post_meta($id, '_'.$metaKey, false);
		if (isset($value[0]) /* count($value) > 0 */ ) {
			update_post_meta($id, '_'.$metaKey, $newValue);
		} else {
			add_post_meta($id, '_'.$metaKey, $newValue);
		}
	}

	/**
	 * Saving of meta box
	 *
	 * @access public
	 * @return void
	 */
	public function filterSaveMetaBox() {
		// check if not autosave
		if ((defined('DOING_AUTOSAVE')) && (DOING_AUTOSAVE)) {
			return;
		}
		// declare globals
		global $post;
		// get post id
		if (isset($post->ID)) {
			$id = $post->ID;
		} else {
			return;
		}
		// loop
		if (!empty($this->metaBox)) {
			foreach ($this->metaBox as $metaBox) {
				// check security
				if ((isset($_POST[$metaBox[5]])) && (wp_verify_nonce($_POST[$metaBox[5]], $this->nonceAction)) && (isset($_POST['post_type'])) && (in_array($_POST['post_type'], $metaBox[2]))) {
					// check permissions
					$ok = true;
					foreach ($metaBox[2] as $place) {
						$type = ($place === 'post') ?
							'edit_post' :
							'edit_page';
						if (!current_user_can($type, $id)) {
							$ok = false;
							break;
						}
					}
					if ($ok) {
						// callback function for update post meta
						call_user_func_array($metaBox[7], array(
							$id,
						));
						// update post meta
						foreach ($metaBox[8] as $metaKey) {
							$this->updateMetaBox($id, $metaKey, false);
						}
						// update post meta with checkbox
						foreach ($metaBox[9] as $metaKey) {
							$this->updateMetaBox($id, $metaKey, true);
						}
					}
				}
			}
		}
	}

	/**
	 * Add widgets settings
	 *
	 * @access public
	 * @return void
	 */
	public function actionAddWidgetsSettings() {
		// check if widgets has been added
		if ($this->widgetsAdded) {
			return;
		}
		// add widgets settings
		if (!empty($this->widgets)) {
			foreach ($this->widgets as $key => $widget) {
				wp_register_widget_control($widget[0], $widget[1], array($this, 'widget'.$this->random.'_'.$key));
			}
		}
		// set that widgets has been added
		$this->widgetsAdded = true;
	}

	/**
	 * Show widget settings
	 *
	 * @access private
	 * @param int $pos Position in widgets array
	 * @return void
	 */
	private function showWidgetSettings($pos) {
		// get widget data
		$data = $this->widgets[$pos];
		// optionally update widget options
		if ((isset($_POST['kocujinternallib_widget_'.$pos])) && ($_POST['kocujinternallib_widget_'.$pos] === '1') && (wp_verify_nonce($_POST[$data[4]], $this->nonceAction))) {
			call_user_func($data[3]);
		}
		// add hidden security field
		wp_nonce_field($this->nonceAction, $data[4]);
		// set field to send
		echo '<input type="hidden" name="'.esc_attr('kocujinternallib_widget_'.$pos).'" value="1" />';
		// show table start
		$this->showTableStart('widget_'.$pos, true);
		// show meta box
		call_user_func($data[2]);
		// show table end
		$this->showTableEnd(true);
	}
}
