<?php

/**
 * component.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujInternalLib\V5a\Classes\Project\Components\Backend\SettingsMenu;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Settings menu class
 *
 * @access public
 */
class Component extends \KocujInternalLib\V5a\Classes\ComponentObject {
	/**
	 * Settings menus list
	 *
	 * @access private
	 * @var array
	 */
	private $settingsMenu = array();

	/**
	 * Settings menus hooks suffixes
	 *
	 * @access private
	 * @var array
	 */
	private $settingsMenuHookSuffix = array();

	/**
	 * Settings pages screen names
	 *
	 * @access private
	 * @var array
	 */
	private $pageScreenNames = array();

	/**
	 * Settings alternative pages screen names
	 *
	 * @access private
	 * @var array
	 */
	private $alternativePageScreenNames = array();

	/**
	 * Call handler for non-existing methods
	 *
	 * @access public
	 * @param string $name Method name
	 * @param array $argument Method arguments
	 * @return array|bool|float|int|string|void Value returned by called method
	 */
	public function __call($name, array $arguments) {
		// get type and identifier based on method name
		$div = explode('_', $name);
		if (count($div) > 1) {
			$type = $div[0];
			unset($div[0]);
			$id = implode('_', $div);
			// check type and id
			if (($type === 'settingsMenu') && (isset($this->settingsMenu[$id]))) {
				// check user permission
				if (!current_user_can($this->settingsMenu[$id]['capability'])) {
					wp_die($this->getProjectObj()->getStringsObj('backend', 'settings-menu')->getString('CALL_NO_PERMISSION'));
				}
				// show header
				$this->getProjectObj()->getComponent('core', 'project-helper')->doAction('before_wrap_div');
				?>
					<div<?php echo $this->getProjectObj()->getComponent('core', 'project-helper')->applyFiltersForHTMLStyleAndClass('wrap_div', '', array(
						'defaultclass' => 'wrap',
					)); ?>>
					<?php
						echo '<h1>'.$this->getProjectObj()->getMainSettingTitle().'</h1>';
						$title = (isset($this->settingsMenu[$id]['fulltitle'])) ?
							$this->settingsMenu[$id]['fulltitle'] :
							$this->settingsMenu[$id]['title'];
						echo '<h2>'.$title.'</h2>';
						$this->getProjectObj()->getComponent('core', 'project-helper')->doAction('before_wrapinside_div');
					?>
					<div id="<?php echo esc_attr($this->getProjectObj()->getComponent('core', 'project-helper')->getPrefix().'__div_wrapinside'); ?>" <?php echo $this->getProjectObj()->getComponent('core', 'project-helper')->applyFiltersForHTMLStyleAndClass('wrapinside_div', '', array()); ?>>
				<?php
				// execute method
				call_user_func($this->settingsMenu[$id]['function']);
				// show footer
				?>
					</div>
				<?php
				$this->getProjectObj()->getComponent('core', 'project-helper')->doAction('after_wrapinside_div');
				?>
					</div>
				<?php
				$this->getProjectObj()->getComponent('core', 'project-helper')->doAction('after_wrap_div');
			}
		}
	}

	/**
	 * Get settings menus data
	 *
	 * @access public
	 * @return array Settings menus data; each settings menu data have the following fields: "alternativepagescreenname" (string type; alternative screen name for this page), "capability" (string type; capability required for access to this menu), "function" (array or string type; callback function or method name), "menutype" (int type; it is one of the following: \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuType::SITE or \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuType::NETWORK), "pagescreenname" (string type; screen name for this page), "title" (string type; menu title); there are also the following fields which should exists or not: "firstoptiontitle" (title for first option if current option menu is without parent; only for menu - not submenu), "icon" (string type; icon name for settings option; only for menu - not submenu), "onpluginslist" (bool type; there will be link to this settings on this plugin information in plugins list with title from this option or from "pluginslisttitle" attribute if exists), "pluginslisttitle" (string type)
	 */
	public function getSettingsMenus() {
		// exit
		return $this->settingsMenu;
	}

	/**
	 * Get the selected settings menu data
	 *
	 * @access public
	 * @param string $id Settings menu identifier
	 * @return array|bool Select settings menu data or false if not exists; settings menu data have the following fields: "alternativepagescreenname" (string type; alternative screen name for this page), "capability" (string type; capability required for access to this menu), "function" (array or string type; callback function or method name), "menutype" (int type; it is one of the following: \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuType::SITE or \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuType::NETWORK), "pagescreenname" (string type; screen name for this page), "title" (string type; menu title); there are also the following fields which should exists or not: "firstoptiontitle" (title for first option if current option menu is without parent; only for menu - not submenu), "icon" (string type; icon name for settings option; only for menu - not submenu), "onpluginslist" (bool type; there will be link to this settings on this plugin information in plugins list with title from this option or from "pluginslisttitle" attribute if exists), "pluginslisttitle" (string type)
	 */
	public function getSettingsMenu($id) {
		// exit
		return (isset($this->settingsMenu[$id])) ?
			$this->settingsMenu[$id] :
			false;
	}

	/**
	 * Get hook suffix for the selected settings menu
	 *
	 * @access public
	 * @param string $id Settings menu identifier
	 * @return bool|string Hook suffix for the selected settings menu or false if not exists
	 */
	public function getSettingsMenuHookSuffix($id) {
		// exit
		return (isset($this->settingsMenuHookSuffix[$id])) ?
			$this->settingsMenuHookSuffix[$id] :
			false;
	}

	/**
	 * Check if current page is for settings for current project
	 *
	 * @access public
	 * @return bool Current page is for settings for current project (true) or not (false)
	 */
	public function checkCurrentPageIsSettingsForProject() {
		// get screen identifier
		$screenId = \KocujInternalLib\V5a\Classes\CurrentScreenIdHelper::getInstance()->get();
		// exit
		return ((!empty($screenId)) && (((!empty($this->pageScreenNames)) && (in_array($screenId['original'], $this->pageScreenNames))) || ((!empty($this->alternativePageScreenNames)) && (in_array($screenId['alternative'], $this->alternativePageScreenNames)))));
	}

	/**
	 * Get current settings menu page
	 *
	 * @access public
	 * @return bool|string Current settings menu page or false if it is not page for current project
	 */
	public function getCurrentSettingsMenu() {
		// get current settings menu
		if (!empty($this->settingsMenu)) {
			// get screen identifier
			$screenId = \KocujInternalLib\V5a\Classes\CurrentScreenIdHelper::getInstance()->get();
			if (!empty($screenId)) {
				// get current settings menu
				foreach ($this->settingsMenu as $key => $val) {
					if (($val['pagescreenname'] === $screenId['original']) || ($val['alternativepagescreenname'] === $screenId['alternative'])) {
						return $key;
					}
				}
			}
		}
		// exit
		return false;
	}

	/**
	 * Add settings menu, submenu or submenu to built-in menu
	 *
	 * @access private
	 * @param string $title Menu title
	 * @param string $capability Capability required for access to this menu
	 * @param string $id Menu id
	 * @param array|string $function Callback function or method name; can be global function or method from any class
	 * @param bool $parentIsBuiltInMenu Parent is built-in menu (true) or is settings menu for this project if exists (false)
	 * @param string $parentId Parent menu id (if $parentIsBuiltInMenu=false) or parent menu type (if $parentIsBuiltInMenu=true); for parent menu type it must be one of the following constants: \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::DASHBOARD, \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::POSTS, \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::\KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::MEDIA, \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::LINKS, \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::PAGES, \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::COMMENTS, \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::THEMES, \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::PLUGINS, \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::USERS, \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::TOOLS or \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::OPTIONS - default: \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::DASHBOARD
	 * @param int $menuType Menu type - site or network; must be one of the following constants: \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuType::SITE or \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuType::NETWORK
	 * @param array $attr Additional attributes; there are available the following attributes: "firstoptiontitle" (title for first option if current option menu is without parent; only for menu - not submenu), "fulltitle" (string type; full title of page), "icon" (string type; icon name for settings option; only for menu - not submenu), "onpluginslist" (bool type; there will be link to this settings on this plugin information in plugins list with title from this option or from "pluginslisttitle" attribute if exists), "pluginslisttitle" (string type)
	 * @return void
	 */
	private function addSettingsMenuOrSubmenu($title, $capability, $id, $function, $parentIsBuiltInMenu, $parentId, $menuType, array $attr) {
		// check if settings can be add
		if (($this->getProjectObj()->getMainSettingType() !== \KocujInternalLib\V5a\Enums\ProjectType::PLUGIN) && ((($parentIsBuiltInMenu) && ($parentId !== \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::THEMES)) || (!$parentIsBuiltInMenu))) {
			return;
		}
		// set page screen name
		$pageScreenSuffix = '';
		$alternativePageScreenSuffix = '';
		if (!$parentIsBuiltInMenu) {
			$parentTitle = ((isset($parentId[0]) /* strlen($parentId) > 0 */ ) && (isset($this->settingsMenu[$parentId]))) ?
				$this->settingsMenu[$parentId]['title'] :
				'';
			$pageScreenName = (((!isset($parentId[0]) /* strlen($parentId) === 0 */ ) || (!isset($parentTitle[0]) /* strlen($parentTitle) === 0 */ )) ?
				'toplevel' :
				sanitize_title($parentTitle)
			);
			$alternativePageScreenName = 'admin';
			if ($menuType === \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuType::NETWORK) {
				$alternativePageScreenSuffix = '-network';
			}
		} else {
			switch ($parentId) {
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::NETWORK_DASHBOARD:
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::NETWORK_SITES:
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::NETWORK_USERS:
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::NETWORK_THEMES:
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::NETWORK_PLUGINS:
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::NETWORK_OPTIONS:
					$menuType = \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuType::NETWORK;
					break;
				default:
					$menuType = \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuType::SITE;
			}
			switch ($parentId) {
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::DASHBOARD:
					$pageScreenName = 'dashboard';
					break;
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::POSTS:
					$pageScreenName = 'posts';
					break;
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::MEDIA:
					$pageScreenName = 'media';
					break;
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::LINKS:
					$pageScreenName = 'admin';
					break;
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::PAGES:
					$pageScreenName = 'pages';
					break;
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::COMMENTS:
					$pageScreenName = 'comments';
					break;
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::THEMES:
					$pageScreenName = 'appearance';
					break;
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::PLUGINS:
					$pageScreenName = 'plugins';
					break;
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::USERS:
					$pageScreenName = 'users';
					break;
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::TOOLS:
					$pageScreenName = 'tools';
					break;
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::OPTIONS:
					$pageScreenName = 'settings';
					break;
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::NETWORK_DASHBOARD:
					$pageScreenName = 'index';
					break;
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::NETWORK_SITES:
					$pageScreenName = 'sites';
					break;
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::NETWORK_USERS:
					$pageScreenName = 'users';
					break;
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::NETWORK_THEMES:
					$pageScreenName = 'themes';
					break;
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::NETWORK_PLUGINS:
					$pageScreenName = 'plugins';
					break;
				case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::NETWORK_OPTIONS:
					$pageScreenName = 'settings';
					break;
				default:
					$pageScreenName = 'admin';
			}
			$suffix = ($menuType === \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuType::NETWORK) ?
				'-network' :
				'';
			$pageScreenSuffix = $suffix;
			$alternativePageScreenName = $pageScreenName;
			$alternativePageScreenSuffix = $suffix;
		}
		$toAdd = '_page_'.$this->getProjectObj()->getMainSettingInternalName().'_'.$id;
		$pageScreenName .= $toAdd.$pageScreenSuffix;
		$alternativePageScreenName .= $toAdd.$alternativePageScreenSuffix;
		// add settings menu
		$this->settingsMenu[$id] = array(
			'title'                     => $title,
			'capability'                => $capability,
			'function'                  => $function,
			'menutype'                  => $menuType,
			'pagescreenname'            => $pageScreenName,
			'alternativepagescreenname' => $alternativePageScreenName,
		);
		if (isset($attr['fulltitle'])) {
			$this->settingsMenu[$id]['fulltitle'] = $attr['fulltitle'];
		}
		if (!$parentIsBuiltInMenu) {
			if (isset($parentId[0]) /* strlen($parentId) > 0 */ ) {
				$this->settingsMenu[$id]['parentid'] = $parentId;
			}
		} else {
			$this->settingsMenu[$id]['parentid'] = $parentId;
			$this->settingsMenu[$id]['parentisbuiltinmenu'] = true;
		}
		if ((!$parentIsBuiltInMenu) && (isset($attr['icon']))) {
			$this->settingsMenu[$id]['icon'] = $attr['icon'];
		}
		if (isset($attr['onpluginslist'])) {
			$this->settingsMenu[$id]['onpluginslist'] = $attr['onpluginslist'];
			if (isset($attr['pluginslisttitle'])) {
				$this->settingsMenu[$id]['pluginslisttitle'] = $attr['pluginslisttitle'];
			}
		}
		if (!$parentIsBuiltInMenu) {
			if ((!isset($parentId[0]) /* strlen($parentId) === 0 */ ) && (isset($attr['firstoptiontitle']))) {
				$this->settingsMenu[$id]['firstoptiontitle'] = $attr['firstoptiontitle'];
			}
		}
		// add page internal name
		if (!in_array($pageScreenName, $this->pageScreenNames)) {
			$this->pageScreenNames[] = $pageScreenName;
		}
		if (!in_array($alternativePageScreenName, $this->alternativePageScreenNames)) {
			$this->alternativePageScreenNames[] = $alternativePageScreenName;
		}
	}

	/**
	 * Add settings menu
	 *
	 * @access public
	 * @param string $title Menu title
	 * @param string $capability Capability required for access to this menu
	 * @param string $id Menu id
	 * @param array|string $function Callback function or method name; can be global function or method from any class
	 * @param string $parentId Parent menu id - default: empty
	 * @param int $menuType Menu type - site or network; must be one of the following constants: \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuType::SITE or \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuType::NETWORK - default: \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuType::SITE
	 * @param array $attr Additional attributes; there are available the following attributes: "firstoptiontitle" (title for first option if current option menu is without parent; only for menu - not submenu), "fulltitle" (string type; full title of page), "icon" (string type; icon name for settings option; only for menu - not submenu), "onpluginslist" (bool type; there will be link to this settings on this plugin information in plugins list with title from this option or from "pluginslisttitle" attribute if exists), "pluginslisttitle" (string type) - default: empty
	 * @return void
	 */
	public function addSettingsMenu($title, $capability, $id, $function, $parentId = '', $menuType = \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuType::SITE, array $attr = array()) {
		// add settings menu
		$this->addSettingsMenuOrSubmenu($title, $capability, $id, $function, false, $parentId, $menuType, $attr);
	}

	/**
	 * Add settings menu to built in menu
	 *
	 * @access public
	 * @param string $title Menu title
	 * @param string $capability Capability required for access to this menu
	 * @param string $id Menu id
	 * @param array|string $function Callback function or method name; can be global function or method from any class
	 * @param int $parentType Parent menu type; must be one of the following constants: \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::DASHBOARD, \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::POSTS, \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::\KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::MEDIA, \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::LINKS, \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::PAGES, \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::COMMENTS, \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::THEMES, \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::PLUGINS, \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::USERS, \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::TOOLS or \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::OPTIONS - default: \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::DASHBOARD
	 * @param array $attr Additional attributes; there are available the following attributes: "firstoptiontitle" (title for first option if current option menu is without parent; only for menu - not submenu), "fulltitle" (string type; full title of page), "icon" (string type; icon name for settings option; only for menu - not submenu), "onpluginslist" (bool type; there will be link to this settings on this plugin information in plugins list with title from this option or from "pluginslisttitle" attribute if exists), "pluginslisttitle" (string type) - default: empty
	 * @return void
	 */
	public function addSettingsMenuBuiltIn($title, $capability, $id, $function, $parentType = \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::DASHBOARD, array $attr = array()) {
		// add settings menu
		$this->addSettingsMenuOrSubmenu($title, $capability, $id, $function, true, $parentType, 0, $attr);
	}

	/**
	 * Action for adding menu
	 *
	 * @access public
	 * @return void
	 */
	public function actionAdminMenu() {
		// add settings menu
		if (!empty($this->settingsMenu)) {
			// set required value of menu type
			$reqMenuType = (is_network_admin()) ?
				\KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuType::NETWORK :
				\KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuType::SITE;
			// add settings main menus
			foreach ($this->settingsMenu as $key => $menu) {
				if (($menu['menutype'] === $reqMenuType) && (!isset($menu['parentid']))) {
					$hookSuffix = add_menu_page($menu['title'], $menu['title'], $menu['capability'], $this->getProjectObj()->getMainSettingInternalName().'_'.$key, array($this, 'settingsMenu_'.$key), (isset($menu['icon'])) ?
						$menu['icon'] :
						''
					);
					if ($hookSuffix !== false) {
						$this->settingsMenuHookSuffix[$key] = $hookSuffix;
					}
					if (isset($menu['firstoptiontitle'])) {
						$hookSuffix = add_submenu_page($this->getProjectObj()->getMainSettingInternalName().'_'.$key, $menu['firstoptiontitle'], $menu['firstoptiontitle'], $menu['capability'], $this->getProjectObj()->getMainSettingInternalName().'_'.$key, array($this, 'settingsMenu_'.$key));
						if ($hookSuffix !== false) {
							$this->settingsMenuHookSuffix[$key] = $hookSuffix;
						}
					}
				}
			}
			// add settings submenus
			foreach ($this->settingsMenu as $key => $menu) {
				if (($menu['menutype'] === $reqMenuType) && (isset($menu['parentid']))) {
					$id = $this->getProjectObj()->getMainSettingInternalName().'_'.$key;
					$callback = array($this, 'settingsMenu_'.$key);
					if ((!isset($menu['parentisbuiltinmenu'])) || ((isset($menu['parentisbuiltinmenu'])) && (!$menu['parentisbuiltinmenu']))) {
						$hookSuffix = add_submenu_page($this->getProjectObj()->getMainSettingInternalName().'_'.$menu['parentid'], $menu['title'], $menu['title'], $menu['capability'], $id, $callback);
					} else {
						switch ($menu['parentid']) {
							case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::DASHBOARD:
							case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::NETWORK_DASHBOARD:
								$hookSuffix = add_dashboard_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::POSTS:
								$hookSuffix = add_posts_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::MEDIA:
								$hookSuffix = add_media_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::LINKS:
								$hookSuffix = add_links_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::PAGES:
								$hookSuffix = add_pages_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::COMMENTS:
								$hookSuffix = add_comments_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::THEMES:
							case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::NETWORK_THEMES:
								$hookSuffix = add_theme_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::PLUGINS:
							case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::NETWORK_PLUGINS:
								$hookSuffix = add_plugins_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::USERS:
							case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::NETWORK_USERS:
								$hookSuffix = add_users_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::TOOLS:
								$hookSuffix = add_management_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::OPTIONS:
								$hookSuffix = add_options_page($menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::NETWORK_SITES:
								$hookSuffix = add_submenu_page('sites.php', $menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
							case \KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuParentType::NETWORK_OPTIONS:
								$hookSuffix = add_submenu_page('settings.php', $menu['title'], $menu['title'], $menu['capability'], $id, $callback);
								break;
						}
					}
					if ($hookSuffix !== false) {
						$this->settingsMenuHookSuffix[$key] = $hookSuffix;
					}
				}
			}
		}
	}

	/**
	 * Filter for actions links on plugins list
	 *
	 * @access public
	 * @param string $links Actions links list
	 * @param string $filename Plugin filename
	 * @return string Actions links list
	 */
	public function filterPluginActionLinks($links, $filename) {
		// show settings on plugin list
		if ((plugin_basename($this->getProjectObj()->getMainSettingMainFilename()) === $filename) && (!empty($this->settingsMenu))) {
			// set required value of menu type
			$reqMenuType = (is_network_admin()) ?
				\KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuType::NETWORK :
				\KocujInternalLib\V5a\Enums\Project\Components\Backend\SettingsMenu\MenuType::SITE;
			// show settings on plugin list
			foreach (array_reverse($this->settingsMenu) as $id => $settingsMenu) {
				if (($settingsMenu['menutype'] === $reqMenuType) && (isset($settingsMenu['onpluginslist'])) && ($settingsMenu['onpluginslist'])) {
					array_unshift($links, \KocujInternalLib\V5a\Classes\HtmlHelper::getInstance()->getLinkAnchor(network_admin_url('admin.php?page='.$this->getProjectObj()->getMainSettingInternalName().'_'.$id), (((isset($settingsMenu['pluginslisttitle']) && (isset($settingsMenu['pluginslisttitle'][0]) /* strlen($settingsMenu['pluginslisttitle']) > 0 */ ))) ?
							$settingsMenu['pluginslisttitle'] :
							$settingsMenu['title']
						)));
				}
			}
		}
		// exit
		return $links;
	}
}
