<?php

/**
 * component.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujInternalLib\V5a\Classes\Project\Components\Backend\VersionCheck;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Version check class
 *
 * @access public
 */
class Component extends \KocujInternalLib\V5a\Classes\ComponentObject {
	/**
	 * Option name - version
	 *
	 */
	const OPTION_NAME_VERSION = 'version';

	/**
	 * Option name - requested update information
	 *
	 */
	const OPTION_NAME_REQ_UPDATE_INFO = 'req_update_info';

	/**
	 * Current version of project
	 *
	 * @access private
	 * @var string
	 */
	private $currentVersion = '';

	/**
	 * Update callback
	 *
	 * @access private
	 * @var array|string
	 */
	private $updateCallback = '';

	/**
	 * Update informations
	 *
	 * @access private
	 * @var array
	 */
	private $updateInformation = array();

	/**
	 * Dividing string for multiple information for one version update
	 *
	 * @access private
	 * @var string
	 */
	private $divideString = '<br /><br /><hr /><br />';

	/**
	 * Information is set to be displayed  (true) or information will not be displayed (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $informationDisplay = false;

	/**
	 * Data for information to display
	 *
	 * @access private
	 * @var array
	 */
	private $informationDisplayData = array();

	/**
	 * Get version number from database
	 *
	 * @access public
	 * @return string Version number from database
	 */
	public function getVersionOptionValue() {
		// exit
		return $this->getProjectObj()->getComponent('core', 'meta')->getOption(self::OPTION_NAME_VERSION);
	}

	/**
	 * Get request for update information from database
	 *
	 * @access public
	 * @return array Request for update information from database
	 */
	public function getRequestUpdateInfoValue() {
		// exit
		return $this->getProjectObj()->getComponent('core', 'meta')->getOption(self::OPTION_NAME_REQ_UPDATE_INFO);
	}

	/**
	 * Set current version of project
	 *
	 * @access public
	 * @param string $version Current version of project
	 * @return void
	 */
	public function setCurrentVersion($currentVersion) {
		// set current version of project
		$this->currentVersion = $currentVersion;
	}

	/**
	 * Get current version of project
	 *
	 * @access public
	 * @return string Current version of project
	 */
	public function getCurrentVersion() {
		// exit
		return $this->currentVersion;
	}

	/**
	 * Set update callback
	 *
	 * @access public
	 * @param array|string $updateCallback Callback function or method for update
	 * @return void
	 */
	public function setUpdateCallback($updateCallback) {
		// set update callback
		$this->updateCallback = $updateCallback;
	}

	/**
	 * Get update callback
	 *
	 * @access public
	 * @return array|string Update callback
	 */
	public function getUpdateCallback() {
		// exit
		return $this->updateCallback;
	}

	/**
	 * Set dividing string for multiple information for one version update
	 *
	 * @access public
	 * @param string $divideString Dividing string
	 * @return void
	 */
	public function setDivideString($divideString) {
		// set dividing string
		$this->divideString = $divideString;
	}

	/**
	 * Add information for update from the selected version
	 *
	 * @access public
	 * @param string $fromVersion Version from which update will show information; it can be only fragment of version number, but then it must be ended with dot character ("."), for example, "1.", "1.0.", etc.
	 * @param string $information Information to display
	 * @return void
	 */
	public function addUpdateInformation($fromVersion, $information) {
		// set update information
		if (isset($this->updateInformation[$fromVersion])) {
			$this->updateInformation[$fromVersion] .= $this->divideString;
		} else {
			$this->updateInformation[$fromVersion] = '';
		}
		$this->updateInformation[$fromVersion] .= $information;
	}

	/**
	 * Get information for update from the selected version
	 *
	 * @access public
	 * @param string $fromVersion Version from which update will show information
	 * @return bool|string Information for the selected version or false if not exists
	 */
	public function getInformation($fromVersion) {
		// get information for version
		if (!empty($this->updateInformation)) {
			foreach ($this->updateInformation as $key => $val) {
				$keyLength = strlen($key);
				if ($key[$keyLength-1] !== '.') {
					if ($key === $fromVersion) {
						return $val;
					}
				} else {
					if ($key === substr($fromVersion, 0, $keyLength)) {
						return $val;
					}
				}
			}
		}
		// exit
		return false;
	}

	/**
	 * Action for checking version of project
	 *
	 * @access public
	 * @return void
	 */
	public function actionAdminInit() {
		// check if current version of project is set
		if (!isset($this->currentVersion[0]) /* strlen($this->currentVersion) === 0 */ ) {
			return;
		}
		// optionally show information about this update
		$value = $this->getRequestUpdateInfoValue();
		if (($value !== false) && ((current_user_can('manage_network')) || (current_user_can('manage_options')))) {
			if ($this->getInformation($value['from']) !== false) {
				\KocujInternalLib\V5a\Classes\JsHelper::getInstance()->addHelperJS(true);
				\KocujInternalLib\V5a\Classes\JsHelper::getInstance()->addExceptionJS(true);
				\KocujInternalLib\V5a\Classes\JsHelper::getInstance()->addModalJS(true);
				$this->getProjectObj()->getComponent('all', 'js-ajax')->addAjaxJS();
				$deps = array(
					'jquery',
					\KocujInternalLib\V5a\Classes\Helper::getInstance()->getPrefix().'-helper',
					\KocujInternalLib\V5a\Classes\Helper::getInstance()->getPrefix().'-all-js-ajax',
					\KocujInternalLib\V5a\Classes\Helper::getInstance()->getPrefix().'-modal',
				);
				if (\KocujInternalLib\V5a\Classes\JsHelper::getInstance()->getThrowErrors()) {
					$deps[] = \KocujInternalLib\V5a\Classes\Helper::getInstance()->getPrefix().'-exception';
				}
				\KocujInternalLib\V5a\Classes\JsHelper::getInstance()->addScript(\KocujInternalLib\V5a\Classes\Helper::getInstance()->getPrefix().'-backend-version-check', \KocujInternalLib\V5a\Classes\LibUrls::getInstance()->getJavaScriptUrl().'/project/components/backend/version-check', 'version-check', $deps, \KocujInternalLib\V5a\Classes\Version::getInstance()->getVersion(), true, true, 'kocujILV5aBackendVersionCheckVals', array(
					'prefix'           => \KocujInternalLib\V5a\Classes\Helper::getInstance()->getPrefix(),
					'security'         => wp_create_nonce(\KocujInternalLib\V5a\Classes\Helper::getInstance()->getPrefix().'__version_info'),
					'textLoading'      => $this->getProjectObj()->getStringsObj('backend', 'version-check')->getString('ACTION_ADMIN_INIT_SCRIPT_LOADING'),
					'textLoadingError' => $this->getProjectObj()->getStringsObj('backend', 'version-check')->getString('ACTION_ADMIN_INIT_SCRIPT_LOADING_ERROR'),
					'textTitlePlugin'  => $this->getProjectObj()->getStringsObj('backend', 'version-check')->getString('ACTION_ADMIN_INIT_SCRIPT_TITLE_PLUGIN'),
					'textTitleTheme'   => $this->getProjectObj()->getStringsObj('backend', 'version-check')->getString('ACTION_ADMIN_INIT_SCRIPT_TITLE_THEME'),
				), true);
				// set information to display
				$this->informationDisplay = true;
				// set data for information to display
				$this->informationDisplayData = $value;
			} else {
				// remove request for information about update
				$this->getProjectObj()->getComponent('core', 'meta')->deleteOption(self::OPTION_NAME_REQ_UPDATE_INFO);
			}
		}
		// check version of project
		$version = $this->getVersionOptionValue();
		if ($version === false) {
			// update version of project in database
			$this->getProjectObj()->getComponent('core', 'meta')->addOrUpdateOption(self::OPTION_NAME_VERSION, $this->currentVersion);
		} else {
			if (version_compare($version, $this->currentVersion, '<')) {
				// execute update callback
				if ($this->updateCallback !== '') {
					call_user_func_array($this->updateCallback, array(
						$version,
						$this->currentVersion,
					));
				}
				// save request for information about this update
				$value = array(
					'from' => $version,
					'to'   => $this->currentVersion,
				);
				$this->getProjectObj()->getComponent('core', 'meta')->addOrUpdateOption(self::OPTION_NAME_REQ_UPDATE_INFO, $value);
				// update version of project in database
				$this->getProjectObj()->getComponent('core', 'meta')->addOrUpdateOption(self::OPTION_NAME_VERSION, $this->currentVersion);
			}
		}
	}

	/**
	 * Action for displaying update information
	 *
	 * @access public
	 * @return void
	 */
	public function actionPrintFooterScripts() {
		// initialize information script
		if ($this->informationDisplay) {
			?>
				<script type="text/javascript">
				/* <![CDATA[ */
					(function($) {
						$(document).ready(function() {
							kocujILV5aBackendVersionCheck.addProject('<?php echo esc_js($this->getProjectObj()->getMainSettingInternalName()); ?>', '<?php echo esc_js($this->getProjectObj()->getMainSettingTitleOriginal()); ?>', '<?php echo ($this->getProjectObj()->getMainSettingType() === \KocujInternalLib\V5a\Enums\ProjectType::PLUGIN) ?
									'kocujILV5aBackendVersionCheckProjectType.PLUGIN' :
									'kocujILV5aBackendVersionCheckProjectType.THEME'
								?>', '<?php echo esc_js($this->informationDisplayData['from']); ?>', '<?php echo esc_js($this->informationDisplayData['to']); ?>');
							kocujILV5aBackendVersionCheck.showVersionCheck('<?php echo esc_js($this->getProjectObj()->getMainSettingInternalName()); ?>');
						});
					}(jQuery));
				/* ]]> */
				</script>
			<?php
		}
	}

	/**
	 * Action for uninstall plugin
	 *
	 * @access public
	 * @return void
	 */
	public function actionPluginUninstall() {
		// remove option
		$this->getProjectObj()->getComponent('core', 'meta')->deleteOption(self::OPTION_NAME_REQ_UPDATE_INFO);
		$this->getProjectObj()->getComponent('core', 'meta')->deleteOption(self::OPTION_NAME_VERSION);
	}
}
