<?php

/**
 * component.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujIL\V6a\Classes\Project\Components\Backend\VersionCheck;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Version check class
 *
 * @access public
 */
class Component extends \KocujIL\V6a\Classes\ComponentObject {
	/**
	 * Option name - version
	 *
	 */
	const OPTION_NAME_VERSION = 'version';

	/**
	 * Option name - requested update information
	 *
	 */
	const OPTION_NAME_REQ_UPDATE_INFO = 'req_update_info';

	/**
	 * Current version of project
	 *
	 * @access private
	 * @var string
	 */
	private $currentVersion = '';

	/**
	 * Update callback
	 *
	 * @access private
	 * @var array|string
	 */
	private $updateCallback = '';

	/**
	 * Update informations
	 *
	 * @access private
	 * @var array
	 */
	private $updateInformation = array();

	/**
	 * Dividing string for multiple information for one version update
	 *
	 * @access private
	 * @var string
	 */
	private $divideString = '<br /><br /><hr /><br />';

	/**
	 * Information is set to be displayed  (true) or information will not be displayed (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $informationDisplay = false;

	/**
	 * Data for information to display
	 *
	 * @access private
	 * @var array
	 */
	private $informationDisplayData = array();

	/**
	 * Get version number from database
	 *
	 * @access public
	 * @return string Version number from database
	 */
	public function getVersionOptionValue() {
		// exit
		return $this->getComponent('core', 'meta')->get(self::OPTION_NAME_VERSION);
	}

	/**
	 * Get request for update information from database
	 *
	 * @access public
	 * @return array Request for update information from database
	 */
	public function getRequestUpdateInfoValue() {
		// exit
		return $this->getComponent('core', 'meta')->get(self::OPTION_NAME_REQ_UPDATE_INFO);
	}

	/**
	 * Set current version of project
	 *
	 * @access public
	 * @param string $version Current version of project
	 * @return void
	 */
	public function setCurrentVersion($currentVersion) {
		// set current version of project
		$this->currentVersion = $currentVersion;
	}

	/**
	 * Get current version of project
	 *
	 * @access public
	 * @return string Current version of project
	 */
	public function getCurrentVersion() {
		// exit
		return $this->currentVersion;
	}

	/**
	 * Set update callback
	 *
	 * @access public
	 * @param array|string $updateCallback Callback function or method for update
	 * @return void
	 */
	public function setUpdateCallback($updateCallback) {
		// set update callback
		$this->updateCallback = $updateCallback;
	}

	/**
	 * Get update callback
	 *
	 * @access public
	 * @return array|string Update callback
	 */
	public function getUpdateCallback() {
		// exit
		return $this->updateCallback;
	}

	/**
	 * Set dividing string for multiple information for one version update
	 *
	 * @access public
	 * @param string $divideString Dividing string
	 * @return void
	 */
	public function setDivideString($divideString) {
		// set dividing string
		$this->divideString = $divideString;
	}

	/**
	 * Add information for update from the selected version
	 *
	 * @access public
	 * @param string $fromVersion Version from which update will show information; it can be only fragment of version number, but then it must be ended with dot character ("."), for example, "1.", "1.0.", etc.
	 * @param string $information Information to display
	 * @param string $useTopMessage Use top message with link to information to display (true) or just show information (false) - default: false
	 * @return void
	 */
	public function addUpdateInformation($fromVersion, $information, $useTopMessage = false) {
		// set update information
		if (isset($this->updateInformation[$fromVersion])) {
			$this->updateInformation[$fromVersion]['information'] .= $this->divideString.$information;
		} else {
			$this->updateInformation[$fromVersion] = array(
				'information'   => $information,
				'usetopmessage' => $useTopMessage,
			);
			uasort($this->updateInformation, function($a, $b) {
				return -strcasecmp($a['information'], $b['information']);
			});
		}
	}

	/**
	 * Get information or top message for update from the selected version
	 *
	 * @access private
	 * @param bool $isInformation Information will be returned (true) or top message (false)
	 * @param string $fromVersion Version from which update will show information
	 * @return bool|string Information text or check top message for the selected version or false if not exists
	 */
	private function getInformationOrTopMessage($isInformation, $fromVersion) {
		// initialize
		$output = '';
		// get information for version
		if (!empty($this->updateInformation)) {
			$index = ($isInformation) ?
				'information' :
				'usetopmessage';
			foreach ($this->updateInformation as $key => $val) {
				$addText = '';
				$keyLength = strlen($key);
				if ($key[$keyLength-1] !== '.') {
					if ($key === $fromVersion) {
						if ($isInformation) {
							$addText = $val[$index];
						} else {
							return $val[$index];
						}
					}
				} else {
					if ($key === substr($fromVersion, 0, $keyLength)) {
						if ($isInformation) {
							$addText = $val[$index];
						} else {
							return $val[$index];
						}
					}
				}
				if (isset($addText[0]) /* strlen($addText) > 0 */ ) {
					if (isset($output[0]) /* strlen($output) > 0 */ ) {
						$output .= $this->divideString;
					}
					$output .= $addText;
				}
			}
		}
		// exit
		return (isset($output[0]) /* strlen($output) > 0 */ ) ?
			$output :
			false;
	}

	/**
	 * Check if information for update from the selected version exists
	 *
	 * @access public
	 * @param string $fromVersion Version from which update will show information
	 * @return bool Information for update from the selected version exists (true) or not (false)
	 */
	public function checkInformation($fromVersion) {
		// get information
		$information = $this->getInformationOrTopMessage(true, $fromVersion);
		// exit
		return (isset($information[0]) /* strlen($information) > 0 */ );
	}

	/**
	 * Get information for update from the selected version
	 *
	 * @access public
	 * @param string $fromVersion Version from which update will show information
	 * @return bool|string Information for the selected version or false if not exists
	 */
	public function getInformation($fromVersion) {
		// exit
		return $this->getInformationOrTopMessage(true, $fromVersion);
	}

	/**
	 * Check top message for update from the selected version
	 *
	 * @access public
	 * @param string $fromVersion Version from which update will show information
	 * @return bool Top message for the selected version exists (true) or not (false)
	 */
	public function checkTopMessage($fromVersion) {
		// exit
		return $this->getInformationOrTopMessage(false, $fromVersion);
	}

	/**
	 * Show information or message about this update if needed
	 *
	 * @access private
	 * @return void
	 */
	private function showUpdateInfoNow() {
		// optionally show information or message about this update
		$value = $this->getRequestUpdateInfoValue();
		if (($value !== false) && ((current_user_can('manage_network')) || (current_user_can('manage_options')))) {
			if ((isset($value['f'])) && ($this->checkInformation($value['f']))) {
				// load scripts
				$this->getComponent('all', 'js-ajax')->addAjaxJs();
				\KocujIL\V6a\Classes\JsHelper::getInstance()->addLibScript('backend-version-check', 'project/components/backend/version-check', 'version-check', array(
					'helper',
					'modal',
				), array(
					\KocujIL\V6a\Classes\Helper::getInstance()->getPrefix().'-all-js-ajax',
				), 'kocujILV6aBackendVersionCheckVals', array(
					'prefix'           => \KocujIL\V6a\Classes\Helper::getInstance()->getPrefix(),
					'security'         => wp_create_nonce(\KocujIL\V6a\Classes\Helper::getInstance()->getPrefix().'__version_info'),
					'textLoading'      => $this->getStrings('backend', 'version-check')->getString('ACTION_ADMIN_INIT_SCRIPT_LOADING'),
					'textLoadingError' => $this->getStrings('backend', 'version-check')->getString('ACTION_ADMIN_INIT_SCRIPT_LOADING_ERROR'),
					'textTitlePlugin'  => $this->getStrings('backend', 'version-check')->getString('ACTION_ADMIN_INIT_SCRIPT_TITLE_PLUGIN'),
					'textTitleTheme'   => $this->getStrings('backend', 'version-check')->getString('ACTION_ADMIN_INIT_SCRIPT_TITLE_THEME'),
				));
				// set data for information to display
				$this->informationDisplayData = $value;
				// set information to display
				$this->informationDisplay = true;
				// optionally show top message
				if ((isset($value['m'])) && ($value['m'])) {
					if ($this->getProjectObj()->getMainSettingType() === \KocujIL\V6a\Enums\ProjectType::PLUGIN) {
						$message = $this->getStrings('backend', 'version-check')->getString('ACTION_ADMIN_INIT_TOP_MESSAGE_PLUGIN');
					}
					if ($this->getProjectObj()->getMainSettingType() === \KocujIL\V6a\Enums\ProjectType::THEME) {
						$message = $this->getStrings('backend', 'version-check')->getString('ACTION_ADMIN_INIT_TOP_MESSAGE_THEME');
					}
					$this->getComponent('backend', 'message')->addMessageForAllPages('version_check', sprintf($message, $this->getProjectObj()->getMainSettingTitleOriginal(), \KocujIL\V6a\Classes\HtmlHelper::getInstance()->getLinkBegin('#', array(
						'id' => $this->getComponent('core', 'project-helper')->getPrefix().'__version_check_link',
					)), \KocujIL\V6a\Classes\HtmlHelper::getInstance()->getLinkEnd()), \KocujIL\V6a\Enums\Project\Components\Backend\Message\Type::INFORMATION, \KocujIL\V6a\Enums\Project\Components\Backend\Message\Closable::CLOSABLE_TEMPORARY);
				}
			} else {
				// remove request for information about update
				$this->getComponent('core', 'meta')->delete(self::OPTION_NAME_REQ_UPDATE_INFO);
			}
		}
	}

	/**
	 * Action for checking version of project
	 *
	 * @access public
	 * @return void
	 */
	public function actionAdminInit() {
		// check if current version of project is set
		if (!isset($this->currentVersion[0]) /* strlen($this->currentVersion) === 0 */ ) {
			return;
		}
		// optionally show information about this update
		$this->showUpdateInfoNow();
		// check version of project
		$version = $this->getVersionOptionValue();
		if ($version === false) {
			// update version of project in database
			$this->getComponent('core', 'meta')->addOrUpdate(self::OPTION_NAME_VERSION, $this->currentVersion);
		} else {
			if (version_compare($version, $this->currentVersion, '<')) {
				// execute update callback
				if ($this->updateCallback !== '') {
					call_user_func_array($this->updateCallback, array(
						$version,
						$this->currentVersion,
					));
				}
				// check if message should be displayed
				$displayMessage = $this->checkTopMessage($version);
				// save request for information about this update
				$value = array(
					'f' => $version,
					't' => $this->currentVersion,
					'm' => $displayMessage,
				);
				$this->getComponent('core', 'meta')->addOrUpdate(self::OPTION_NAME_REQ_UPDATE_INFO, $value);
				// update version of project in database
				$this->getComponent('core', 'meta')->addOrUpdate(self::OPTION_NAME_VERSION, $this->currentVersion);
				// optionally force to display message without delay
				if ($displayMessage) {
					$this->showUpdateInfoNow();
				}
			}
		}
	}

	/**
	 * Action for displaying update information
	 *
	 * @access public
	 * @return void
	 */
	public function actionPrintFooterScripts() {
		// initialize information script
		if ($this->informationDisplay) {
			?>
				<script type="text/javascript">
				/* <![CDATA[ */
					(function($) {
						$(document).ready(function() {
							kocujILV6aBackendVersionCheck.addProject('<?php echo esc_js($this->getProjectObj()->getMainSettingInternalName()); ?>', '<?php echo esc_js($this->getProjectObj()->getMainSettingTitleOriginal()); ?>', '<?php echo ($this->getProjectObj()->getMainSettingType() === \KocujIL\V6a\Enums\ProjectType::PLUGIN) ?
									'kocujILV6aBackendVersionCheckProjectType.PLUGIN' :
									'kocujILV6aBackendVersionCheckProjectType.THEME'
								?>', '<?php echo esc_js($this->informationDisplayData['f']); ?>', '<?php echo esc_js($this->informationDisplayData['t']); ?>');
							<?php if ((isset($this->informationDisplayData['m'])) && ($this->informationDisplayData['m'])) : ?>
								$('#<?php echo esc_js($this->getComponent('core', 'project-helper')->getPrefix().'__version_check_link'); ?>').click(function() {
									kocujILV6aBackendVersionCheck.showVersionCheck('<?php echo esc_js($this->getProjectObj()->getMainSettingInternalName()); ?>');
								});
								$('#<?php echo esc_js($this->getComponent('core', 'project-helper')->getPrefix().'__message_version_check'); ?> .notice-dismiss').click(function() {
									kocujILV6aBackendVersionCheck.sendCloseEvent('<?php echo esc_js($this->getProjectObj()->getMainSettingInternalName()); ?>');
								});
							<?php else : ?>
								kocujILV6aBackendVersionCheck.showVersionCheck('<?php echo esc_js($this->getProjectObj()->getMainSettingInternalName()); ?>');
							<?php endif; ?>
						});
					}(jQuery));
				/* ]]> */
				</script>
			<?php
		}
	}
}
