<?php

/**
 * exception.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujIL\V6a\Classes;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Exception class
 *
 * @access public
 */
class Exception extends \Exception {
	/**
	 * Errors data
	 *
	 * @access private
	 * @var array
	 */
	private $errors = array();

	/**
	 * Error data
	 *
	 * @access private
	 * @var array
	 */
	private $data = array();

	/**
	 * Constructor
	 *
	 * @access public
	 * @param int $code Error code
	 * @param string $file Filename where there was an error; should be set to __FILE__ during throwing an exception
	 * @param int $line Line number where there was an error; should be set to __LINE__ during throwing an exception
	 * @param string $param Optional argument for error message - default: empty
	 * @return void
	 */
	public function __construct($code, $file, $line, $param = '') {
		// initialize errors
		$this->setErrors();
		// get error constant name
		$name = '';
		if (isset($this->errors[$code])) {
			$class = new \ReflectionClass(get_class($this));
			$constants = array_flip($class->getConstants());
			if (isset($constants[$code])) {
				$name = $constants[$code];
			}
		}
		// save parameters
		$this->data = array(
			'name'  => $name,
			'param' => $param,
		);
		// prepare message
		$message = '['.$this->getExceptionName().'] [code: '.$code.', name: '.$name.', file: '.$file.', line: '.$line.'] ';
		$message .= (isset($this->errors[$code])) ?
			$this->errors[$code] :
			'Unknown error';
		if (isset($param[0]) /* strlen($param) > 0 */ ) {
			$message .= ' ('.$param.')';
		}
		// execute parent constructor
		parent::__construct($message, $code);
		// save parent parameters
		$this->file = $file;
		$this->line = $line;
	}

	/**
	 * Get errors data
	 *
	 * @access protected
	 * @return string Exception name
	 */
	protected function getExceptionName() {
		// exit
		return \KocujIL\V6a\Classes\Helper::getInstance()->getPrefix();
	}

	/**
	 * Set errors data
	 *
	 * @access protected
	 * @return void
	 */
	protected function setErrors() {
		// initialize errors
		$this->errors = array(
			ExceptionCode::OK                                   => 'OK',
			ExceptionCode::NO_REQUIRED_SETTING_DATA             => 'No required setting data',
			ExceptionCode::NO_REQUIRED_LIBRARY                  => 'No required library',
			ExceptionCode::NO_REQUIRED_COMPONENT                => 'No required component',
			ExceptionCode::CANNOT_USE_COMPONENTS_AFTER_SHUTDOWN => 'Cannot use components after shutdown',
		);
	}

	/**
	 * Get errors
	 *
	 * @access protected
	 * @return array Errors
	 */
	protected function getErrors() {
		// exit
		return $this->errors;
	}

	/**
	 * Get error name
	 *
	 * @access public
	 * @return string Error name
	 */
	public function getName() {
		// exit
		return $this->data['name'];
	}

	/**
	 * Get optional argument for error message
	 *
	 * @access public
	 * @return string Optional argument for error message
	 */
	public function getParam() {
		// exit
		return $this->data['param'];
	}
}
