<?php

/**
 * component.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujIL\V6a\Classes\Project\Components\Core\Meta;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Meta class
 *
 * @access public
 */
class Component extends \KocujIL\V6a\Classes\ComponentObject {
	/**
	 * Meta data
	 *
	 * @access private
	 * @var array
	 */
	private $data = array();

	/**
	 * Clear meta data (true) or not (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $clear = false;

	/**
	 * Constructor
	 *
	 * @access public
	 * @param object $projectObj \KocujIL\V6a\Classes\Project object for current project
	 * @return void
	 */
	public function __construct($projectObj) {
		// execute parent constructor
		parent::__construct($projectObj);
		// get meta data
		$value = \KocujIL\V6a\Classes\DbDataHelper::getInstance()->getOption($this->getName());
		if ($value !== false) {
			$this->data = maybe_unserialize($value);
		}
	}

	/**
	 * Destructor
	 *
	 * @access public
	 * @return void
	 */
	public function __destruct() {
		// update or clear meta data
		if ($this->clear) {
			$this->forceRealClearDataNow();
		} else {
			$this->forceRealUpdateNow();
		}
	}

	/**
	 * Get meta option name
	 *
	 * @access public
	 * @return string Meta option name
	 */
	public function getName() {
		// exit
		return $this->getProjectObj()->getMainSettingInternalName().'__meta';
	}

	/**
	 * Add or update meta
	 *
	 * @access public
	 * @param string $name Meta name
	 * @param array|bool|float|int|string $value Meta value
	 * @return void
	 */
	public function addOrUpdate($name, $value) {
		// set meta to not clear
		$this->clear = false;
		// add or update meta
		$this->data[$name] = $value;
	}

	/**
	 * Delete meta
	 *
	 * @access public
	 * @param string $name Meta name
	 * @return void
	 */
	public function delete($name) {
		// delete meta
		if (isset($this->data[$name])) {
			unset($this->data[$name]);
		}
	}

	/**
	 * Get meta value
	 *
	 * @access public
	 * @param string $name Meta name
	 * @param array|bool|float|int|string $defaultValue Default meta value - default: false
	 * @return array|bool|float|int|string Meta value
	 */
	public function get($name, $defaultValue = false) {
		// exit
		return (isset($this->data[$name])) ?
			$this->data[$name] :
			$defaultValue;
	}

	/**
	 * Clear meta
	 *
	 * @access public
	 * @return void
	 */
	public function clear() {
		// set meta to clear
		$this->clear = true;
		// clear meta in memory
		$this->data = array();
	}

	/**
	 * Force real update of the entire meta data now
	 *
	 * @access public
	 * @return void
	 */
	public function forceRealUpdateNow() {
		// update meta data
		\KocujIL\V6a\Classes\DbDataHelper::getInstance()->addOrUpdateOption($this->getName(), $this->data);
	}

	/**
	 * Force real clear entire meta data now
	 *
	 * @access public
	 * @return void
	 */
	public function forceRealClearDataNow() {
		// clear meta data
		$this->clear();
		\KocujIL\V6a\Classes\DbDataHelper::getInstance()->deleteOption($this->getName());
	}
}
