<?php

/**
 * component.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujIL\V7a\Classes\Project\Components\Backend\Message;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Message class
 *
 * @access public
 */
class Component extends \KocujIL\V7a\Classes\ComponentObject {
	/**
	 * Option name - message closed
	 *
	 */
	const OPTION_NAME_MSG_CLOSED = 'msg_closed';

	/**
	 * Messages
	 *
	 * @access private
	 * @var array
	 */
	private $messages = array();

	/**
	 * Add message
	 *
	 * @access private
	 * @param string $id Message identifier; must be unique
	 * @param string $content Message content
	 * @param int $type Message type; must be one of the following constants from \KocujIL\V7a\Enums\Project\Components\Backend\Message\Type: INFORMATION (for information message), WARNING (for warning message), ERROR (for error message) or SUCCESS (for success message)
	 * @param int $closable Message closable status; must be one of the following constants from \KocujIL\V7a\Enums\Project\Components\Backend\Message\Closable: NOT_CLOSABLE (for message without closing button) or CLOSABLE (for message with closing button)
	 * @param bool $allPages Message will be displayed on all pages in administration panel (true) or not (false)
	 * @param array $pages Settings pages list for current project where this message should be displayed; it should be identifiers from settings page for current project only; if it is empty, this message will be displayed on all settings pages for current project
	 * @param array $attr Additional attributes; there are available the following attributes: "class" (string type; CSS class for message div), "closecallback" (string type; if message is closable, it is callback executed during closing the message), "permissions" (array type; permissions for message to display; if empty, no permissions are required to display this message), "style" (string type; CSS style for message div)
	 * @return void
	 */
	private function addMessageForProjectOrAllPages($id, $content, $type, $closable, $allPages, array $pages, array $attr) {
		// check if this id does not already exists
		if (isset($this->messages[$id])) {
			return;
		}
		// add message
		$this->messages[$id] = array(
			'content'  => $content,
			'type'     => $type,
			'closable' => $closable,
		);
		if ($allPages) {
			$this->messages[$id]['allpages'] = true;
		} else {
			$this->messages[$id]['pages'] = $pages;
		}
		if ((isset($attr['class'])) && (isset($attr['class'][0]) /* strlen($attr['class']) > 0 */ )) {
			$this->messages[$id]['class'] = $attr['class'];
		}
		if ((isset($attr['style'])) && (isset($attr['style'][0]) /* strlen($attr['style']) > 0 */ )) {
			$this->messages[$id]['style'] = $attr['style'];
		}
		if (isset($attr['permissions'])) {
			$this->messages[$id]['permissions'] = $attr['permissions'];
		}
		if ((($closable === \KocujIL\V7a\Enums\Project\Components\Backend\Message\Closable::CLOSABLE) || ($closable === \KocujIL\V7a\Enums\Project\Components\Backend\Message\Closable::CLOSABLE_TEMPORARY)) && (isset($attr['closecallback']))) {
			$this->messages[$id]['closecallback'] = $attr['closecallback'];
		}
	}

	/**
	 * Add message
	 *
	 * @access public
	 * @param string $id Message identifier; must be unique
	 * @param string $content Message content
	 * @param int $type Message type; must be one of the following constants from \KocujIL\V7a\Enums\Project\Components\Backend\Message\Type: INFORMATION (for information message), WARNING (for warning message), ERROR (for error message) or SUCCESS (for success message) - default: \KocujIL\V7a\Enums\Project\Components\Backend\Message\Type::INFORMATION
	 * @param int $closable Message closable status; must be one of the following constants from \KocujIL\V7a\Enums\Project\Components\Backend\Message\Closable: NOT_CLOSABLE (for message without closing button) or CLOSABLE (for message with closing button) - default: \KocujIL\V7a\Enums\Project\Components\Backend\Message\Closable::NOT_CLOSABLE
	 * @param array $pages Settings pages list for current project where this message should be displayed; it should be identifiers from settings page for current project only; if it is empty, this message will be displayed on all settings pages for current project - default: empty
	 * @param array $attr Additional attributes; there are available the following attributes: "class" (string type; CSS class for message div), "closecallback" (string type; if message is closable, it is callback executed during closing the message), "permissions" (array type; permissions for message to display; if empty, no permissions are required to display this message), "style" (string type; CSS style for message div) - default: empty
	 * @return void
	 */
	public function addMessage($id, $content, $type = \KocujIL\V7a\Enums\Project\Components\Backend\Message\Type::INFORMATION, $closable = \KocujIL\V7a\Enums\Project\Components\Backend\Message\Closable::NOT_CLOSABLE, array $pages = array(), array $attr = array()) {
		// add message
		$this->addMessageForProjectOrAllPages($id, $content, $type, $closable, false, $pages, $attr);
	}

	/**
	 * Add message for all pages in administration panel
	 *
	 * @access public
	 * @param string $id Message identifier; must be unique
	 * @param string $content Message content
	 * @param int $type Message type; must be one of the following constants from \KocujIL\V7a\Enums\Project\Components\Backend\Message\Type: INFORMATION (for information message), WARNING (for warning message), ERROR (for error message) or SUCCESS (for success message) - default: \KocujIL\V7a\Enums\Project\Components\Backend\Message\Type::INFORMATION
	 * @param int $closable Message closable status; must be one of the following constants from \KocujIL\V7a\Enums\Project\Components\Backend\Message\Closable: NOT_CLOSABLE (for message without closing button) or CLOSABLE (for message with closing button) - default: \KocujIL\V7a\Enums\Project\Components\Backend\Message\Closable::NOT_CLOSABLE
	 * @param array $attr Additional attributes; there are available the following attributes: "class" (string type; CSS class for message div), "closecallback" (string type; if message is closable, it is callback executed during closing the message), "permissions" (array type; permissions for message to display; if empty, no permissions are required to display this message), "style" (string type; CSS style for message div) - default: empty
	 * @return void
	 */
	public function addMessageForAllPages($id, $content, $type = \KocujIL\V7a\Enums\Project\Components\Backend\Message\Type::INFORMATION, $closable = \KocujIL\V7a\Enums\Project\Components\Backend\Message\Closable::NOT_CLOSABLE, array $attr = array()) {
		// add message
		$this->addMessageForProjectOrAllPages($id, $content, $type, $closable, true, array(), $attr);
	}

	/**
	 * Get messages data
	 *
	 * @access public
	 * @param string $id Message identifier
	 * @return array Message data; each message data have the following fields: "content" (content of message), "type" (type of message); there are also the following fields which should exists or not: "allpages" (if it is set to true, message will be displayed in all pages in administration panel), "closecallback" (string type; if message is closable, it is callback executed during closing the message), "style" (string type; CSS style for message div), "class" (string type; CSS class for message div), "pages" (settings pages list for current project on which message is displayed or empty if it is displayed on all settings pages for current project), "permissions" (array type; permissions for message to display; if empty, no permissions are required to display this message)
	 */
	public function getMessages() {
		// exit
		return $this->messages;
	}

	/**
	 * Get message data by id
	 *
	 * @access public
	 * @param string $id Message identifier
	 * @return array|bool Message data or false if not exists; message data have the following fields: "content" (content of message), "type" (type of message); there are also the following fields which should exists or not: "allpages" (is it is set to true, message will be displayed in all pages in administration panel), "closecallback" (string type; if message is closable, it is callback executed during closing the message), "style" (string type; CSS style for message div), "class" (string type; CSS class for message div), "pages" (settings pages list for current project on which message is displayed or empty if it is displayed on all settings pages for current project), "permissions" (array type; permissions for message to display; if empty, no permissions are required to display this message)
	 */
	public function getMessage($id) {
		// exit
		return (isset($this->messages[$id])) ?
			$this->messages[$id] :
			false;
	}

	/**
	 * Check permissions to display message
	 *
	 * @access public
	 * @param string $id Message identifier
	 * @return bool Permission to display this message are correct (true) or not (false)
	 */
	public function checkMessagePermissions($id) {
		// check permissions to display message
		if (isset($this->messages[$id]['permissions'])) {
			return \KocujIL\V7a\Classes\Helper::getInstance()->checkCurrentPermissions($this->messages[$id]['permissions']);
		}
		// exit
		return true;
	}

	/**
	 * Check if message will be displayed
	 *
	 * @access public
	 * @param string $id Message identifier
	 * @return bool Message will be displayed (true) or not (false)
	 */
	public function checkMessageToDisplay($id) {
		// check if message will be displayed
		$output = true;
		if (!((isset($this->messages[$id]['allpages'])) && ($this->messages[$id]['allpages']))) {
			$output = (empty($this->messages[$id]['pages'])) ?
				$this->getComponent('backend', 'settings-menu')->checkCurrentPageIsSettingsForProject() :
				in_array($this->getComponent('backend', 'settings-menu')->getCurrentSettingsMenu(), $this->messages[$id]['pages']);
		}
		if ($output) {
			$output = (($this->checkMessagePermissions($id)) && (!$this->checkMessageClosed($id)));
		}
		// exit
		return $output;
	}

	/**
	 * Close message
	 *
	 * @access public
	 * @param string $id Message identifier
	 * @param bool $onlyIfExists Close message only if exists (true) or not (false) - default: true
	 * @return void
	 */
	public function closeMessage($id, $onlyIfExists = true) {
		// close message
		if ((($onlyIfExists) && (isset($this->messages[$id]))) || (!$onlyIfExists)) {
			$this->getComponent('core', 'meta')->addOrUpdate(self::OPTION_NAME_MSG_CLOSED.'__'.$id, '1');
		}
	}

	/**
	 * Restore closed message
	 *
	 * @access public
	 * @param string $id Message identifier
	 * @return void
	 */
	public function restoreClosedMessage($id) {
		// restore closed message
		if (isset($this->messages[$id])) {
			$this->getComponent('core', 'meta')->delete(self::OPTION_NAME_MSG_CLOSED.'__'.$id);
		}
	}

	/**
	 * Check if message is closed
	 *
	 * @access public
	 * @param string $id Message identifier
	 * @return bool Message is closed (true) or not (false)
	 */
	public function checkMessageClosed($id) {
		// exit
		return ((isset($this->messages[$id])) && ($this->getComponent('core', 'meta')->get(self::OPTION_NAME_MSG_CLOSED.'__'.$id) === '1'));
	}

	/**
	 * Action for admin header
	 *
	 * @access public
	 * @return void
	 */
	public function actionAdminHead() {
		// check if add script
		$display = false;
		if (!empty($this->messages)) {
			foreach ($this->messages as $id => $data) {
				if (($this->checkMessageToDisplay($id)) && (($data['closable'] === \KocujIL\V7a\Enums\Project\Components\Backend\Message\Closable::CLOSABLE) || ($data['closable'] === \KocujIL\V7a\Enums\Project\Components\Backend\Message\Closable::CLOSABLE_TEMPORARY))) {
					$display = true;
				}
			}
		}
		// add script
		if ($display) {
			$this->getComponent('all', 'js-ajax')->addAjaxJs();
			\KocujIL\V7a\Classes\JsHelper::getInstance()->addLibScript('backend-message', 'project/components/backend/message', 'message', array(
				'helper',
			), array(
				\KocujIL\V7a\Classes\Helper::getInstance()->getPrefix().'-all-js-ajax',
			), 'kocujILV7aBackendMessageVals', array(
				'prefix'   => \KocujIL\V7a\Classes\Helper::getInstance()->getPrefix(),
				'security' => wp_create_nonce(\KocujIL\V7a\Classes\Helper::getInstance()->getPrefix().'__message'),
			));
		}
	}

	/**
	 * Action for admin notices
	 *
	 * @access public
	 * @return void
	 */
	public function actionAdminNotices() {
		// show messages
		if (!empty($this->messages)) {
			foreach ($this->messages as $id => $data) {
				// show message
				if ($this->checkMessageToDisplay($id)) {
					// get message class
					$addClass = array(
						\KocujIL\V7a\Enums\Project\Components\Backend\Message\Type::INFORMATION => 'notice-info',
						\KocujIL\V7a\Enums\Project\Components\Backend\Message\Type::WARNING     => 'notice-warning',
						\KocujIL\V7a\Enums\Project\Components\Backend\Message\Type::ERROR       => 'notice-error',
						\KocujIL\V7a\Enums\Project\Components\Backend\Message\Type::SUCCESS     => 'notice-success',
					);
					$class = 'notice'.((isset($addClass[$data['type']])) ?
							' '.$addClass[$data['type']] :
							''
						);
					if (($data['closable'] === \KocujIL\V7a\Enums\Project\Components\Backend\Message\Closable::CLOSABLE) || ($data['closable'] === \KocujIL\V7a\Enums\Project\Components\Backend\Message\Closable::CLOSABLE_TEMPORARY)) {
						$class .= ' is-dismissible';
					}
					// prepare style
					$style = (isset($data['style'])) ?
						' style="'.esc_attr($data['style']).'"' :
						'';
					// prepare additional class
					if (isset($data['class'])) {
						$class .= ' '.$data['class'];
					}
					// set message identifier
					$messageId = $this->getComponent('core', 'project-helper')->getPrefix().'__message_'.$id;
					// show message
					echo '<div'.$style.' id="'.esc_attr($messageId).'" class="'.esc_attr($class).'"><p>'.$data['content'].'</p></div>';
					// add or change message transient data for close message
					if ($data['closable'] === \KocujIL\V7a\Enums\Project\Components\Backend\Message\Closable::CLOSABLE) {
						\KocujIL\V7a\Classes\DbDataHelper::getInstance()->mergeTransientArray('kocuj_il_'.\KocujIL\V7a\Classes\Version::getInstance()->getVersionInternal().'_messages_close', $id, array(
							'closecallback' => (isset($data['closecallback'])) ?
								$data['closecallback'] :
								'',
							'closable'      => $data['closable'],
						), 3600);
					}
				}
			}
		}
	}

	/**
	 * Action for admin footer scripts
	 *
	 * @access public
	 * @return void
	 */
	public function actionPrintFooterScripts() {
		// show scripts
		if (!empty($this->messages)) {
			foreach ($this->messages as $id => $data) {
				// show script for closing window
				if (($this->checkMessageToDisplay($id)) && (($data['closable'] === \KocujIL\V7a\Enums\Project\Components\Backend\Message\Closable::CLOSABLE) || ($data['closable'] === \KocujIL\V7a\Enums\Project\Components\Backend\Message\Closable::CLOSABLE_TEMPORARY))) {
					$display = true;
					?>
						<script type="text/javascript">
						/* <![CDATA[ */
							(function($) {
								$(document).ready(function() {
									kocujILV7aBackendMessage.addProjectIfNotExists('<?php echo esc_js($this->getProjectObj()->getMainSettingInternalName()); ?>');
									kocujILV7aBackendMessage.addMessageCloseButton('<?php echo esc_js($this->getProjectObj()->getMainSettingInternalName()); ?>', '<?php echo esc_js($id); ?>');
								});
							}(jQuery));
						/* ]]> */
						</script>
					<?php
				}
			}
		}
	}
}
