<?php

/**
 * component.class.php
 *
 * @author Dominik Kocuj <dominik@kocuj.pl>
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU General Public License v2 or later
 * @copyright Copyright (c) 2016 Dominik Kocuj
 * @package kocuj_internal_lib
 */

// set namespace
namespace KocujIL\V7a\Classes\Project\Components\Backend\ReviewMessage;

// security
if ((!defined('ABSPATH')) || ((isset($_SERVER['SCRIPT_FILENAME'])) && (basename($_SERVER['SCRIPT_FILENAME']) === basename(__FILE__)))) {
	header('HTTP/1.1 404 Not Found');
	die();
}

/**
 * Review message class
 *
 * @access public
 */
class Component extends \KocujIL\V7a\Classes\ComponentObject {
	/**
	 * Days count after which message about review will be displayed; 0 means never
	 *
	 * @access private
	 * @var int
	 */
	private $days = 0;

	/**
	 * Vote URL
	 *
	 * @access private
	 * @var string
	 */
	private $voteUrl = '';

	/**
	 * Review message is set to be displayed (true) or review message will not be displayed (false)
	 *
	 * @access private
	 * @var bool
	 */
	private $reviewMessageDisplay = false;

	/**
	 * Set days count after which message about review will be displayed; 0 means never
	 *
	 * @access public
	 * @param int $days Days count after which message about review will be displayed; 0 means never
	 * @return void
	 */
	public function setDays($days) {
		// set days count
		$this->days = $days;
	}

	/**
	 * Get days count after which message about review will be displayed; 0 means never
	 *
	 * @access public
	 * @return int Days count after which message about review will be displayed; 0 means never
	 */
	public function getDays() {
		// exit
		return $this->days;
	}

	/**
	 * Set vote URL
	 *
	 * @access public
	 * @param string $url Vote url
	 * @return void
	 */
	public function setVoteUrl($url) {
		// set value
		$this->voteUrl = $url;
	}

	/**
	 * Set vote URL
	 *
	 * @access public
	 * @return string Vote url
	 */
	public function getVoteUrl() {
		// exit
		return $this->voteUrl;
	}

	/**
	 * Get window has been closed (true) or not (false)
	 *
	 * @access public
	 * @return bool Window has been closed (true) or not (false)
	 */
	public function getClosedWindowOptionValue() {
		// exit
		return ($this->getComponent('backend', 'message')->checkMessageClosed('review'));
	}

	/**
	 * Get message element in "div" container
	 *
	 * @access private
	 * @param string $element Message element
	 * @param bool $smaller Message element is smaller than normal (true) or not (false) - default: false
	 * @return string Message element in "div" container
	 */
	private function getMessageElement($element, $smaller = false) {
		// set horizontal margins
		$margin = ($smaller) ?
			2 :
			5;
		// exit
		return '<div'.$this->getComponent('core', 'project-helper')->applyFiltersForHTMLStyleAndClass('review_message_element'.(($smaller) ?
				'_small' :
				''
			), '', array(
				'defaultstyle' => 'float:left;margin-left:'.$margin.'px;margin-right:'.$margin.'px;vertical-align:top;',
				'defaultclass' => $this->getComponent('core', 'project-helper')->getPrefix().'__review_message_element_'.(($smaller) ?
						'small_' :
						''
					).'div',
			)).'>'.$element.'</div>';
	}

	/**
	 * Add additional elements
	 *
	 * @access private
	 * @param array $elements Additional elements to add
	 * @return string Additional elements in string
	 */
	private function addAdditionalElements(array $elements) {
		// check if there are any additional elements
		if (empty($elements)) {
			return '';
		}
		// add additional elements
		$output = '';
		foreach ($elements as $element) {
			$output .= $this->getMessageElement($element);
		}
		// exit
		return $output;
	}

	/**
	 * Action for display review message
	 *
	 * @access public
	 * @return void
	 */
	public function actionAdminHead() {
		// check if display review message
		if ((!isset($this->voteUrl[0]) /* strlen($this->voteUrl) === 0 */ ) || ($this->getClosedWindowOptionValue()) || ($this->days === 0)) {
			return;
		}
		// get installation date
		$date = $this->getComponent('backend', 'installation-date')->getInstallationDateOptionValue();
		if ($date === false) {
			return;
		}
		// check days count
		$days = (int)((strtotime(date('Y-m-d').' 23:59:59')-strtotime($date.' 00:00:00'))/86400);
		// optionally show review message
		if ($days > $this->days) {
			// add CSS stylesheet
			?>
				<style type="text/css" media="all">
				<!--
					#<?php echo $this->getComponent('core', 'project-helper')->getPrefix(); ?>__review_message_news_channels_header_div {
						font-size: 11px;
					}

					.<?php echo $this->getComponent('core', 'project-helper')->getPrefix(); ?>__review_message_element_div {
						height: 40px;
					}

					@media screen and (max-width: 782px) {
						#<?php echo $this->getComponent('core', 'project-helper')->getPrefix(); ?>__review_message_news_channels_header_div {
							font-size: 12px;
						}
					}
				-->
				</style>
			<?php
			// show review message beginning
			$message = $this->getComponent('core', 'project-helper')->applyFilters('before_review_message');
			if ($this->getProjectObj()->getMainSettingType() === \KocujIL\V7a\Enums\ProjectType::PLUGIN) {
				$message .= sprintf($this->getStrings('backend', 'review-message')->getString('ACTION_ADMIN_HEAD_PLUGIN_DAYS'), $this->getProjectObj()->getMainSettingTitleOriginal(), $this->days);
				$message .= ' '.$this->getStrings('backend', 'review-message')->getString('ACTION_ADMIN_HEAD_THINGS_SUPPORT_PLUGIN').' ';
			} else {
				$message .= sprintf($this->getStrings('backend', 'review-message')->getString('ACTION_ADMIN_HEAD_THEME_DAYS'), $this->getProjectObj()->getMainSettingTitleOriginal(), $this->days);
				$message .= ' '.$this->getStrings('backend', 'review-message')->getString('ACTION_ADMIN_HEAD_THINGS_SUPPORT_THEME').' ';
			}
			$message .= '<br />';
			$elements = $this->getComponent('core', 'project-helper')->applyFilters('review_message_elements_before', '', '', array());
			// show buttons for telling others
			$message .= $this->addAdditionalElements($elements);
			$message .= $this->getMessageElement('<input type="button" class="button button-small" id="'.esc_attr($this->getComponent('core', 'project-helper')->getPrefix().'__review_message_vote').'" value="'.esc_attr(($this->getProjectObj()->getMainSettingType() === \KocujIL\V7a\Enums\ProjectType::PLUGIN) ?
					$this->getStrings('backend', 'review-message')->getString('ACTION_ADMIN_HEAD_VOTE_PLUGIN') :
					$this->getStrings('backend', 'review-message')->getString('ACTION_ADMIN_HEAD_VOTE_THEME')
				).'" />');
			if ($this->getProjectObj()->getSettingArray('tellothers', 'facebook')) {
				$message .= $this->getMessageElement('<input type="button" class="button button-small" id="'.esc_attr($this->getComponent('core', 'project-helper')->getPrefix().'__review_message_facebook').'" value="'.esc_attr($this->getStrings('backend', 'review-message')->getString('ACTION_ADMIN_HEAD_FACEBOOK')).'" />');
			}
			if ($this->getProjectObj()->getSettingArray('tellothers', 'twitter')) {
				$message .= $this->getMessageElement('<input type="button" class="button button-small" id="'.esc_attr($this->getComponent('core', 'project-helper')->getPrefix().'__review_message_twitter').'" value="'.esc_attr($this->getStrings('backend', 'review-message')->getString('ACTION_ADMIN_HEAD_TWITTER')).'" />');
			}
			$elements = $this->getComponent('core', 'project-helper')->applyFilters('review_message_elements_after', '', '', array());
			$message .= $this->addAdditionalElements($elements);
			$message .= '<div style="clear:both;"></div>';
			// show news channels links
			$linkRSS = $this->getProjectObj()->getSettingArray('newschannels', 'rss');
			$linkFacebook = $this->getProjectObj()->getSettingArray('newschannels', 'facebook');
			$linkTwitter = $this->getProjectObj()->getSettingArray('newschannels', 'twitter');
			if ((isset($linkRSS[0]) /* strlen($linkRSS) > 0 */ ) || (isset($linkFacebook[0]) /* strlen($linkFacebook) > 0 */ ) || (isset($linkTwitter[0]) /* strlen($linkTwitter) > 0 */ )) {
				if ($this->getProjectObj()->getMainSettingType() === \KocujIL\V7a\Enums\ProjectType::PLUGIN) {
					$text = $this->getStrings('backend', 'review-message')->getString('ACTION_ADMIN_HEAD_NEWS_CHANNELS_TEXT_PLUGIN');
				} else {
					$text = $this->getStrings('backend', 'review-message')->getString('ACTION_ADMIN_HEAD_NEWS_CHANNELS_TEXT_THEME');
				}
				$message .= $this->getMessageElement('<div id="'.$this->getComponent('core', 'project-helper')->getPrefix().'__review_message_news_channels_header_div"><em>'.$text.':</em></div>', true);
				if (isset($linkRSS[0]) /* strlen($linkRSS) > 0 */ ) {
					$message .= $this->getMessageElement(\KocujIL\V7a\Classes\HtmlHelper::getInstance()->getLink($linkRSS, \KocujIL\V7a\Classes\HtmlHelper::getInstance()->getImage(\KocujIL\V7a\Classes\LibUrls::getInstance()->get('images').'/project/components/backend/review-message/icon-small-rss.png'), array(
						'external' => true,
					)), true);
				}
				if (isset($linkFacebook[0]) /* strlen($linkFacebook) > 0 */ ) {
					$message .= $this->getMessageElement(\KocujIL\V7a\Classes\HtmlHelper::getInstance()->getLink($linkFacebook, \KocujIL\V7a\Classes\HtmlHelper::getInstance()->getImage(\KocujIL\V7a\Classes\LibUrls::getInstance()->get('images').'/project/components/backend/review-message/icon-small-facebook.png'), array(
						'external' => true,
					)), true);
				}
				if (isset($linkTwitter[0]) /* strlen($linkTwitter) > 0 */ ) {
					$message .= $this->getMessageElement(\KocujIL\V7a\Classes\HtmlHelper::getInstance()->getLink($linkTwitter, \KocujIL\V7a\Classes\HtmlHelper::getInstance()->getImage(\KocujIL\V7a\Classes\LibUrls::getInstance()->get('images').'/project/components/backend/review-message/icon-small-twitter.png'), array(
						'external' => true,
					)), true);
				}
				$message .= '<div style="clear:both;"></div>';
			}
			// show review message end
			$message .= $this->getComponent('core', 'project-helper')->applyFilters('after_review_message');
			$this->getComponent('backend', 'message')->addMessageForAllPages('review', $message, \KocujIL\V7a\Enums\Project\Components\Backend\Message\Type::INFORMATION, \KocujIL\V7a\Enums\Project\Components\Backend\Message\Closable::CLOSABLE);
			// add scripts
			\KocujIL\V7a\Classes\JsHelper::getInstance()->addLibScript('backend-review-message', 'project/components/backend/review-message', 'review-message', array(
				'helper',
			), array(), 'kocujILV7aBackendReviewMessageVals', array(
				'prefix'   => \KocujIL\V7a\Classes\Helper::getInstance()->getPrefix(),
				'security' => wp_create_nonce(\KocujIL\V7a\Classes\Helper::getInstance()->getPrefix().'__review_message'),
			));
			// set review message to be displayed
			$this->reviewMessageDisplay = true;
		}
	}

	/**
	 * Action for forcing displaying of review message
	 *
	 * @access public
	 * @return void
	 */
	public function actionPrintFooterScripts() {
		// initialize review message script
		if ($this->reviewMessageDisplay) {
			$url = $this->getProjectObj()->getMainSettingUrl();
			$tellFacebook = $this->getProjectObj()->getSettingArray('tellothers', 'facebook');
			$tellTwitter = $this->getProjectObj()->getSettingArray('tellothers', 'twitter');
			?>
				<script type="text/javascript">
				/* <![CDATA[ */
					(function($) {
						$(document).ready(function() {
							kocujILV7aBackendReviewMessage.addProject('<?php echo esc_js($this->getProjectObj()->getMainSettingInternalName()); ?>', '<?php echo esc_js($this->voteUrl); ?>', '<?php echo ((isset($url[0]) /* strlen($url) > 0 */ ) && (isset($tellFacebook[0]) /* strlen($tellFacebook) > 0 */ )) ?
								esc_js($tellFacebook) :
								''
							; ?>', '<?php echo ((isset($url[0]) /* strlen($url) > 0 */ ) && (isset($tellTwitter[0]) /* strlen($tellTwitter) > 0 */ )) ?
								esc_js($tellTwitter) :
								''
							; ?>');
							kocujILV7aBackendReviewMessage.setEvents('<?php echo esc_js($this->getProjectObj()->getMainSettingInternalName()); ?>');
						});
					}(jQuery));
				/* ]]> */
				</script>
			<?php
		}
	}
}
